package Application::Model::Requisites;

use qbit;

use base qw(
  QBit::Application::Model
  );

use File::Basename qw(dirname);
use File::Spec;
use Cwd qw(abs_path);

use lib::abs qw(
  ../../../api/lib/
  );
use Branches;
use Submit;

sub accessor {'requisites'}

__PACKAGE__->model_accessors(
    api_blackbox => 'QBit::Application::Model::API::Yandex::BlackBox',
    api_balance  => 'QBit::Application::Model::API::Yandex::Balance',
    documents    => 'Application::Model::Documents',
    partner_db   => 'Application::Model::PartnerDB',
    users        => 'Application::Model::Users',
);

my $READONLY_FIELDS = {
    account_owner                       => 1,
    adfox_account                       => 1,
    ag_ur_organization_name             => 1,
    bank_name                           => 1,
    belarus_bank_name                   => 1,
    cyrillic_accountant                 => 1,
    cyrillic_accountant_first_name      => 1,
    cyrillic_accountant_last_name       => 1,
    cyrillic_accountant_patronymic_name => 1,
    cyrillic_first_name                 => 1,
    cyrillic_last_name                  => 1,
    cyrillic_patronymic_name            => 1,
    currency                            => 1,
    inn                                 => 1,
    inn_ip                              => 1,
    inn_ph                              => 1,
    inn_ur                              => 1,
    latin_lastname                      => 1,
    latin_name                          => 1,
    latin_bank_name                     => 1,
    legaladdress                        => 1,
    name                                => 1,
    organization_name                   => 1,
    organization_name_belarus           => 1,
    payoneer_payee                      => 1,
    pfr                                 => 1,
    registration_address                => 1,
    registrationaddress                 => 1,
    representative                      => 1,
    unp                                 => 1,
};

my $OPTIONAL_FIELDS = {
    legaladdress         => 1,
    payoneer_payee       => 1,
    registrationaddress  => 1,
    registration_address => 1,
    yandex_money_details => 1,
};

my $PERSONAL_INFO_FIELDS = {
    accountant_email                    => 1,
    accountant_phone                    => 1,
    adfox_account                       => 1,
    cyrillic_accountant                 => 1,
    cyrillic_accountant_first_name      => 1,
    cyrillic_accountant_last_name       => 1,
    cyrillic_accountant_patronymic_name => 1,
    cyrillic_first_name                 => 1,
    cyrillic_last_name                  => 1,
    cyrillic_patronymic_name            => 1,
    email                               => 1,
    latin_lastname                      => 1,
    latin_name                          => 1,
    name_change_warning                 => 1,
    newsletter                          => 1,
    phone                               => 1,
    representative                      => 1,
};

my $SPECIAL_ORDER_FIELDS = {
    bik_with_account_or_yandex_money => 1,
    bank_name                        => 1,
    belarus_bank_name                => 1,
    latin_bank_name                  => 1,
    bik                              => 2,
    ur_swift                         => 2,
    swift                            => 2,
    account                          => 3,
    iban_or_account                  => 3,
};

my $SUBMIT_FIELD_DEPENDENCIES = {
    account   => [qw(bank_name bik)],
    bank_name => [qw(account bik)],
    bik       => [qw(account bank_name)],

    belarus_bank_name => [qw(iban_or_account ur_swift)],
    iban_or_account   => [qw(belarus_bank_name latin_bank_name swift ur_swift)],
    latin_bank_name   => [qw(iban_or_account swift ur_swift)],
    swift             => [qw(iban_or_account latin_bank_name)],
    ur_swift          => [qw(belarus_bank_name iban_or_account latin_bank_name)],

    representative_position_other => [
        qw(
          representative_position_ur
          )
    ],
    representative_position_ur => [
        qw(
          representative_position_other
          )
    ],

    cyrillic_accountant_last_name => [
        qw(
          cyrillic_accountant_first_name
          cyrillic_accountant_patronymic_name
          )
    ],
    cyrillic_accountant_first_name => [
        qw(
          cyrillic_accountant_last_name
          cyrillic_accountant_patronymic_name
          )
    ],
    cyrillic_accountant_patronymic_name => [
        qw(
          cyrillic_accountant_last_name
          cyrillic_accountant_first_name
          )
    ],

    cyrillic_representative_last_name => [
        qw(
          cyrillic_representative_first_name
          cyrillic_representative_patronymic_name
          )
    ],
    cyrillic_representative_first_name => [
        qw(
          cyrillic_representative_last_name
          cyrillic_representative_patronymic_name
          )
    ],
    cyrillic_representative_patronymic_name => [
        qw(
          cyrillic_representative_last_name
          cyrillic_representative_first_name
          )
    ],
};

sub fields {
    my ($self, $user_id) = @_;

    my $user = $self->app->get_option('cur_user', {});
    my $language = $self->app->get_option('locale', 'ru');

    $user_id //= $user->{id};

    throw Exception 'User not found' unless $user_id;

    my $branches = Branches->new();
    my @fields;

    my $tmp_rights = $self->app->add_all_tmp_rights();
    my $user_data  = $self->users->get($user_id, fields => [qw(client_id cooperation_form)]);
    my $contract   = $self->documents->get_active_contract({client_id => $user_data->{client_id}, ignore_filter => 1});
    my $balance_data = $contract->{Person} // {};
    $balance_data->{currency} = $contract->{Contract}{currency};
    my $branch_id = $self->users->get_branch_id_by_person($balance_data, $user_data->{cooperation_form});
    return {} unless ($branch_id && scalar keys %$balance_data);

    if (defined $balance_data->{bank_type}) {
        if ($balance_data->{bank_type} == 3) {
            delete @$balance_data{
                qw(
                  account
                  bik
                  iban
                  payoneer_payee
                  swift
                  )
              };
            $balance_data->{yamoney_wallet} //= '';
        } elsif ($balance_data->{bank_type} == 7) {
            delete @$balance_data{
                qw(
                  account
                  bik
                  iban
                  swift
                  yamoney_wallet
                  )
              };
            $balance_data->{payoneer_payee} //= '';
        } else {
            delete @$balance_data{
                qw(
                  payoneer_payee
                  yamoney_wallet
                  )
              };
        }
    }
    delete $balance_data->{legaladdress} if $balance_data->{legal_fias_guid};

    my $field_order = 10;
    my @branches;
    if ($branch_id =~ /(.*_part)\d/) {
        @branches = map {$1 . $_} (1, 2);
    } else {
        @branches = ($branch_id);
    }

    foreach my $branchname (@branches) {
        my $data = Submit::_get_converted_fields(
            converter => $branches->get_converter_requisites($branchname),
            fields    => $balance_data,
        );

        foreach my $key (keys %{$data}) {
            if ($key =~ /\//) {
                my $value = delete $data->{$key};
                next unless $value;
                $key =~ s/^\///;
                my @keys = split '/', $key;
                next unless scalar @keys >= 2;
                next unless (not exists $data->{$keys[0]} or ref $data->{$keys[0]} eq 'HASH');
                my $result;
                while (my $key = pop @keys) {
                    $result = $result ? {$key => $result,} : {$key => $value,};
                }
                if (exists $data->{(keys %$result)[0]}) {
                    $data->{(keys %$result)[0]} = {%{$data->{(keys %$result)[0]}}, %{$result->{(keys %$result)[0]}}};
                } else {
                    $data = {%$data, %$result};
                }
            }
        }

        if ($data->{bik}) {
            try {
                my $bank = $self->api_balance->get_bank(Bik => $data->{bik})->[0];
                $data->{bank_name} = $bank->{bank_name} if $bank->{bank_name};
            };
        }

        if (my $swift = ($data->{swift} || $data->{ur_swift})) {
            try {
                my $bank = $self->api_balance->get_bank(Swift => $swift)->[0];
                if ($bank->{name}) {
                    $data->{latin_bank_name}   = $bank->{name};
                    $data->{belarus_bank_name} = $bank->{name};
                }
            };
        }

        my @branch_fields = $branch_id ? $branches->get_fields($branchname, 'requisites_fields') : ();
        my %branch_fields;
        @branch_fields{@branch_fields} = (TRUE) x scalar @branch_fields;
        foreach my $field_id (@branch_fields) {
            my $p = "Field::$field_id";
            my $f = $p->new(language => $language);

            my $field_data = $f->get_api_description();

            push @fields,
              {
                %$field_data,
                field_order =>
                  (exists $SPECIAL_ORDER_FIELDS->{$field_id} ? $SPECIAL_ORDER_FIELDS->{$field_id} : $field_order++),
                editable => ($READONLY_FIELDS->{$field_id} || !$contract ? JSON::XS::false : JSON::XS::true),
                (
                    $SUBMIT_FIELD_DEPENDENCIES->{$field_id} && !$READONLY_FIELDS->{$field_id}
                    ? (
                        submit_with => [
                            grep {$branch_fields{$_} && !$READONLY_FIELDS->{$_}}
                              @{$SUBMIT_FIELD_DEPENDENCIES->{$field_id}}
                        ]
                      )
                    : ()
                ),
                tab_id => (
                    $PERSONAL_INFO_FIELDS->{$field_id} ? 'personal_info'
                    : 'financial_info'
                ),
                (
                    exists $data->{$field_id} ? (value => $data->{$field_id})
                    : ()
                )
              }
              if !$OPTIONAL_FIELDS->{$field_id} || $data->{$field_id};
        }
    }

    return {fields => \@fields};

}

1;
