package Application::Model::Resources;

use qbit;
use base qw(QBit::Application::Model RestApi::SimpleModel);

use PiConstants qw($M_RU_TEXT_USER_ID :ROLES :ROLES_GROUPS $YAN_MANAGER_ROLE_ID);

use Exception::Validation::BadArguments;

sub accessor {'resources'}

__PACKAGE__->register_rights(
    [
        {
            name        => 'rights_without_model',
            description => d_gettext('Right without model'),
            rights      => {
                statistics_constructor_view => d_gettext('Right to view statistics constructor'),
                financial_documents_view    => d_gettext('Right to view financial documents'),
                resources_view              => d_gettext('Right to view resources'),
            }
        }
    ]
);

__PACKAGE__->model_accessors(
    assistants              => 'Application::Model::Assistants',
    users                   => 'Application::Model::Users',
    rbac                    => 'Application::Model::RBAC',
    vip_partners            => 'Application::Model::VIP_Partners',
    partner_db              => 'Application::Model::PartnerDB',
    statistics              => 'Application::Model::Statistics',
    product_manager         => 'Application::Model::ProductManager',
    memcached               => 'QBit::Application::Model::Memcached',
    search_on_site_campaign => 'Application::Model::Product::AN::SearchOnSite::Campaign',
);

my $MOBILE_MEDIATION_RESOURCES = {
    bk_statistics            => TRUE,
    financial_documents      => TRUE,
    inviter                  => TRUE,
    mobile_mediation         => TRUE,
    partner_acts             => TRUE,
    partner_documents        => TRUE,
    production_stage_warning => TRUE,
    statistics               => TRUE,
    statistics_reports       => TRUE,
    statistics_constructor   => TRUE,
    test_stage_warning       => TRUE,
};

sub get_product_name {gettext('resources')}

#API

sub api_available_fields {
    return qw(title available_fields available_actions methods);
}

sub api_available_actions {
    return qw();
}

sub api_check_public_id {$_[1] =~ /^[a-z_]+$/}

sub api_get_all {
    my ($self, $controller, $resource, $one_object) = @_;

    $controller->check_params(($one_object ? 'public_id' : ()), "fields[$resource]");

    my @fields = $controller->get_fields($resource);

    my $url_simple_model_get = $controller->get_abs_for_sprintf('simple_model__resource_get', qw(resource public_id));

    my %fields = map {$_ => TRUE} @fields;

    my $resources = $self->get_api_resources();

    my @result = ();
    foreach my $accessor (sort keys(%$resources)) {
        my $res = {
            id   => "$accessor",
            type => $resource,
            ($one_object ? () : (links => {self => sprintf($url_simple_model_get, $resource, $accessor),})),
            meta => {show_in_menu => $resources->{$accessor}->{'show_in_menu'}},
        };

        $res->{'attributes'}{'title'} = $self->app->$accessor->get_product_name() if $fields{'title'};

        $res->{'attributes'}{'available_fields'} = [sort $self->app->$accessor->api_available_fields()]
          if $fields{'available_fields'};

        $res->{'attributes'}{'available_actions'} = [sort $self->app->$accessor->api_available_actions()]
          if $fields{'available_actions'};

        $res->{'attributes'}{'methods'} = [sort @{$resources->{$accessor}->{'methods'}}]
          if $fields{'methods'};

        push(@result, $res);
    }

    return \@result;
}

sub get_api_resources {
    my ($self, $resource) = @_;

    my %resources = ();
    foreach my $r (
        {
            resource     => 'users',
            methods      => ['GET', ($self->users->api_can_edit() ? 'PATCH' : ())],
            show_in_menu => \0,
        },
        @{
            $self->get_available_resources(
                defined($resource) ? ($resource =~ /action_log$/ ? undef : [$resource]) : undef
            )
        }
      )
    {
        my $model = $r->{'resource'};

        if ($model =~ /(.+?)_action_log$/) {
            $model = $self->app->$model->_action_model_accessor();
        }

        next
          unless $self->app->can($model)
              && (   $self->app->{$model}->isa('RestApi::DBModel')
                  || $self->app->{$model}->isa('RestApi::SimpleModel'));

        $resources{$r->{'resource'}}{$_} = $r->{$_} foreach (qw(methods show_in_menu));
    }

    return \%resources;
}

sub _cache_key {
    my ($self, $user_id) = @_;

    my $user = $user_id ? {id => $user_id} : ($self->get_option('cur_user', {id => -1}));
    my $user_roles = $user->{'roles'} // $self->rbac->get_roles_by_user_id($user->{id}) // {};
    my @user_role_ids = sort {$a <=> $b} keys(%$user_roles);

    return "user_id: $user->{id}, roles: [" . join(',', @user_role_ids) . "]";
}

sub invalidate_cache {
    my ($self, $user_id) = @_;

    $self->memcached->delete(available_resources => $self->_cache_key($user_id));
}

sub get_assistant_resources {
    my ($self, $user_id, $cur_user_roles) = @_;

    my %assistants_pages =
      map {$_->{'page_id'} => $_->{'can_edit'}}
      @{$self->assistants->get_all(fields => [qw(page_id can_edit)], filter => {user_id => $user_id})};

    my $assistant_resources = {};

    return sub {
        my ($model, $key) = @_;

        unless ($assistant_resources->{$model}) {
            my $res = $assistant_resources->{$model} = {};

            my $pages = $self->app->$model->get_all(
                fields => [qw(id page_id owner_id)],
                filter => {page_id => [sort keys %assistants_pages]}
            );

            $res->{'view'} = TRUE if @$pages;

            my $can_edit = 0;
            my %owners;
            my %editable_pages;
            foreach my $page (@$pages) {
                $owners{$page->{'owner_id'}}++;

                if ($assistants_pages{$page->{'page_id'}}) {
                    $can_edit++;
                    $editable_pages{$page->{'owner_id'}}++;
                }
            }
            $res->{'edit'} = TRUE if $can_edit;

            if ($model eq 'context_on_site_campaign') {
                if (%owners) {
                    my $adblock_users = $self->partner_db->user_role->get_all(
                        fields => [qw(user_id)],
                        filter => {user_id => [sort keys(%owners)], role_id => $ADBLOCK_PARTNER_ROLE_ID},
                    );
                    $res->{'adblock_view'} = TRUE if @$adblock_users;

                    foreach my $user (@$adblock_users) {
                        if ($editable_pages{$user->{'user_id'}}) {
                            $res->{'adblock_edit'} = TRUE;
                            last;
                        }
                    }
                }
            }
        }
        return $assistant_resources->{$model}{$key};
    };
}

sub _has_search_on_site_campaigns {
    my ($self) = @_;

    my $campaigns = $self->search_on_site_campaign->get_all(
        fields => qw(id),
        filter => {multistate => 'not deleted'},
        limit  => 1
    );

    if (@$campaigns) {
        return TRUE;
    } else {
        return FALSE;
    }
}

sub _has_permission_view_search_on_site_campaigns {
    my ($self, $has_role_from_group_dev, $cur_user_roles) = @_;

    if ($has_role_from_group_dev || defined($cur_user_roles->{$YAN_MANAGER_ROLE_ID})) {
        return TRUE;
    }

    return $self->_has_search_on_site_campaigns;
}

=encoding UTF-8

=head2 get_available_resources

Возвращает массив хешей. В каждом хеше содержится описание части интерфейса к которому
у пользователя есть доступ. На основании этих данных фронтенд создает меню. Этот метод
возвращает плоский список. Данные о иерархии содержатся на фронтенде. Так же фронтенд
содержит ссылки на страницы.

Вид доступа кодируется с помощью значений methods. GET ознчает что есть право читать
ресурс, а POST означает что есть право создавать новый объект в этом ресурсе.
Порядок элеметов в methods не определен. В methods может быть ли GET, либо POST,
либо и GET, и POST. Ключ methods всегда есть и у него всегда есть хоть какое-то
значение.

Из ответа этого метода нельзя однознано определить какой perl моделе соответствует
ресурс (и вообще есть ли перловая модель для этого ресурса).

Пример данныых, которые отдает этот метод:

    [
        # Эта часть интерфейса соответствует модели 'context_on_site_rtb'
        {
            resource => 'context_on_site_rtb',
            methods => ['GET', 'POST'],
        },

        # Эта часть интерфейса соответствует удаленным элементам модели 'context_on_site_rtb'
        {
            model => 'context_on_site_rtb_deleted',
            methods => ['GET'],
        },

        # А это часть интрфейса, для которой нет отдельной модели
        {
            resource => 'drop_user',
            methods => ['GET'],
        },
        ...
    ]

Поля resource уникальны.

Порядок элементов в массиве не определен, разные вызовы этого метода get_available_units
могут создавть разный порядок элеметов.

Через веб можно увидеть результат работы этого метода на странице
/v2/call/frontend/get_available_resources (работате только на бетах и не работает в проде).

=cut

sub get_available_resources {
    my ($self, $need_resources) = @_;

    my $cur_user = $self->get_option('cur_user', {});
    my $user_id = $cur_user->{'id'} // -1;

    my $cur_user_roles = $cur_user->{'roles'} // $self->rbac->get_cur_user_roles() // {};
    my @cur_user_role_ids = sort {$a <=> $b} keys(%$cur_user_roles);

    my %need_resources = map {$_ => TRUE} @{$need_resources // []};

    my $cache_key        = $self->_cache_key();
    my $can_common_offer = defined $cur_user->{common_offer};

    if ($can_common_offer and $cur_user->{common_offer} < 0) {
        %need_resources = map {$_ => TRUE} qw(
          common_offer
          resources
          users
          simple_notification
          user_notifications
          );
    } elsif (my $resources = $self->memcached->get(available_resources => $cache_key)) {
        return $resources;
    }

    my $cur_user_is_internal = !!grep {$cur_user_roles->{$_}{'is_internal'}} @cur_user_role_ids;
    my %cur_user_groups_hash = map {$cur_user_roles->{$_}{'group'} => TRUE} @cur_user_role_ids;

    my $cur_user_is_linked_with_adfox = $cur_user->{'has_adfox'};

    my $has_role_from_group_dev        = $cur_user_groups_hash{$GROUP_DEV};
    my $has_role_from_group_yan        = $cur_user_groups_hash{$GROUP_YAN};
    my $has_role_from_group_dev_or_yan = $has_role_from_group_dev
      || $has_role_from_group_yan;
    my $has_role_from_group_dev_or_dsp = $has_role_from_group_dev
      || $cur_user_groups_hash{$GROUP_DSP};

    my $has_assistant_role_without_an_partner =
      $self->check_rights('is_assistant') && !exists($cur_user_roles->{$SITE_PARTNER_ROLE_ID});
    my $has_assistant_role_without_video_partner =
      $self->check_rights('is_assistant') && !exists($cur_user_roles->{$VIDEO_PARTNER_ROLE_ID});
    my $has_assistant_role_without_mobile_partner =
      $self->check_rights('is_assistant') && !exists($cur_user_roles->{$MOBILE_PARTNER_ROLE_ID});
    my $has_assistant_role_without_adblock_partner =
      $self->check_rights('is_assistant') && !exists($cur_user_roles->{$ADBLOCK_PARTNER_ROLE_ID});

    my $assistant_resources;
    my $assistant_rights;
    my $is_assistant_only;

    if ($self->check_rights('is_assistant')) {
        #
        # Get only those pages where this user is an assistant.
        # Get info:
        # 1. Are there any pages belonging to the model where he is an assistant.
        #    If so, he gets 'view' resource.
        # 2. Can he edit some of these pages as an assistant.
        #    If so, he gets 'edit' resource.
        # 3. Who are the owners of these pages.
        #    Are these owners adfox or adblock partners.
        #    If they are, partner assistants can add and edit adblock or adfox.
        #    Manager assistants don't pay attention to this.
        #
        $assistant_resources = $self->get_assistant_resources($user_id, $cur_user_roles);
        $assistant_rights = $self->assistants->get_rights_for_assistant($user_id, $cur_user_roles);
        $is_assistant_only = (
            0 == grep {exists($cur_user_roles->{$_})} (
                $SITE_PARTNER_ROLE_ID,    $VIDEO_PARTNER_ROLE_ID, $MOBILE_PARTNER_ROLE_ID,
                $ADBLOCK_PARTNER_ROLE_ID, $TUTBY_ROLE_ID
            )
        );

    } else {
        $assistant_resources = sub {FALSE};
    }

    my $has_edit_context_campaign = sub {$assistant_resources->('context_on_site_campaign', 'edit')};
    my $has_edit_search_campaign  = sub {$assistant_resources->('search_on_site_campaign',  'edit')};
    my $has_edit_mobile_campaign  = sub {$assistant_resources->('mobile_app_settings',      'edit')};
    my $has_edit_video_campaign   = sub {$assistant_resources->('video_an_site',            'edit')};

    my $can_view_adblock = sub {$assistant_resources->('context_on_site_campaign', 'adblock_view')};
    my $can_edit_adblock = sub {$assistant_resources->('context_on_site_campaign', 'adblock_edit')};

    my $is_dev_or_ts = in_array($self->get_option('stage', 'unknown'), [qw(dev test)]);
    my $is_dev     = $self->get_option('stage', 'unknown') eq 'dev';
    my $is_preprod = $self->get_option('stage', 'unknown') eq 'preprod';

    my $has_permission_view_search_on_site_campaigns;

    my @permisions = (
        # Business Rules
        {
            resource   => 'business_rules',
            get        => 'business_rules_view',
            post       => 'do_business_rules_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },

        # Источники трафика Сайты
        {
            resource   => 'site',
            get        => 'site_view',
            post       => 'do_site_add',
            properties => {has_action_log => 1,},
        },

        # Simple notification
        {
            resource   => 'simple_notification',
            get        => 'simple_notification_view',
            post       => 'do_simple_notification_action_add',
            properties => {has_action_log => 1,},
        },

        {
            resource   => 'notification',
            get        => 'notification_view',
            post       => 'do_notification_action_add',
            properties => {has_action_log => 1,},
        },

        {
            resource => 'user_notifications',
            get      => 'user_notifications_view',
            post     => 'do_user_notifications_add'
        },

        # Thematic filters
        {
            resource => 'thematic_filters',
            get      => 'thematic_filters_view',
            post     => 'do_thematic_filters_action_add',
        },

        # Продукты Рекламные Платформы Документация
        {
            resource   => 'dsp_documentation',
            get        => 'dsp_documentation_view',
            post       => 'do_dsp_documentation_add',
            properties => {has_action_log => 1,},
        },

        # SSP модерация
        {
            resource   => 'ssp_link_mobile_app',
            get        => 'ssp_moderation_view_all',
            properties => {has_action_log => 1,},
        },

        # SSP модерация вебок
        {
            resource   => 'ssp_link_context_rtb',
            get        => 'ssp_moderation_view_all',
            properties => {has_action_log => 1,},
        },

        # SSP модерация видео
        {
            resource   => 'ssp_link_video_app',
            get        => 'ssp_moderation_view_all',
            properties => {has_action_log => 1,},
        },

        # Приглашалка
        {
            resource   => 'inviter',
            get        => 'inviter_view',
            post       => 'inviter_add',
            properties => {has_action_log => 1,},
        },

        # Пользователи
        {
            resource   => 'users',
            get        => 'users_view',
            post       => 'do_user_action_add',
            properties => {has_action_log => 1,},
        },

        # Новости
        {
            resource => 'news',
            get      => 'news_list_view',
            post     => 'news_add',
        },

        # Разработчику Информация Очередь
        {
            resource   => 'queue',
            get        => 'queue_view',
            properties => {has_action_log => 1,},
        },

        # Разработчику Управление Шаблоны текстов
        {
            resource   => 'text_template',
            get        => 'text_template_view',
            post       => 'do_text_template_add',
            properties => {has_action_log => 1,},
        },

        # Разработчику Управление ACL'ы внутреннего API
        {
            resource => 'intapi_acl',
            get      => 'intapi_acl_view',
        },

        # Разработчику БК Языки
        {
            resource => 'bk_language',
            get      => 'bk_language_view',
            post     => 'bk_language_add',
        },

        # Продукты Видеосеть Справочник Контент
        {
            resource   => 'internal_content',
            get        => 'internal_content_view_all',
            post       => 'internal_content_add',
            properties => {has_action_log => 1,},
        },

        # Продукты Видеосеть Справочник Модерация контента
        {
            resource   => 'partner_content',
            get        => 'partner_content_view',
            properties => {has_action_log => 1,},
        },

        # Продукты Видеосеть Справочник Жанры
        {
            resource   => 'internal_genre',
            get        => 'internal_genre_view_all',
            properties => {has_action_log => 1,},
        },

        # Продукты Видеосеть Справочник Модерация жанров
        {
            resource   => 'partner_genre',
            get        => 'partner_genre_view',
            post       => 'internal_genre_add',
            properties => {has_action_log => 1,},
        },

        # Продукты Рекламные Платформы Подключения Инструменты Песочница
        {
            resource => 'dsp_testing',
            get      => sub {!!@{shift->dsp->get_all_dsp_for_testing(limit => 1)};},
        },

        # Продукты Рекламные Платформы Подключения Инструменты Статистика ответов
        {
            resource => 'statistics_dsp_response',
            get      => sub {
                my ($app) = @_;
                return $app->check_rights('dsp_statistics_responses')
                  && !!@{$app->dsp->get_all(filter => {multistate => 'created_in_bk and not deleted'}, limit => 1)};
            },
        },

        # Статистика
        {
            resource => 'statistics',
            get      => sub {
                $_[0]->check_rights('statistics_view');
            },
        },

        {
            resource => 'video_stat_files',
            get      => sub {!!@{shift->video_stat_files->get_all(fields => [qw(id)], limit => 1)};},
        },

        # финансовые документы
        {
            resource => 'partner_acts',
            get      => 'partner_acts_view',
        },

        {
            resource => 'partner_documents',
            get      => 'partner_documents_view',
        },

        # Статистика Отчеты
        {
            resource   => 'statistics_reports',
            get        => 'statistics_reports_view',
            delete     => 'statistics_reports_view',
            post       => 'statistics_reports_view',
            properties => {has_action_log => 1,},
        },

        # CookieMatch
        {
            resource   => 'cookie_match',
            get        => 'cookie_match_view',
            post       => 'do_cookie_match_add',
            properties => {has_action_log => 1,},
        },

        # Список брендов
        {
            resource => 'tns_dict_brand',
            get      => 'tns_dict_brand_view',
        },

        # Список articles
        {
            resource => 'tns_dict_article',
            get      => 'tns_dict_article_view',
        },

        # Список категорий
        {
            resource => 'picategories_dict',
            get      => 'picategories_dict_view',
        },
        # Модерация
        {
            resource   => 'moderation',
            get        => 'moderation_view',
            post       => 'moderation_add',
            properties => {has_action_log => 1,},
        },

        {
            resource   => 'block_tags',
            get        => sub {TRUE;},
            post       => sub {TRUE;},
            delete     => sub {TRUE;},
            properties => {has_action_log => 1,},
        },

        # Статистика из БК (МОЛ, расширенная статистика)
        {
            resource => 'bk_statistics',
            get      => 'bk_statistics_view',
        },

        # Список ресурсов
        {
            resource => 'resources',
            get      => sub {TRUE;},
        },

        # Гео база
        {
            resource => 'geo_base',
            get      => sub {TRUE;},
        },

        # Список ошибок в АПИ
        {
            resource => 'errors',
            get      => sub {TRUE;},
        },

        # Источники трафика Приложения
        {
            resource   => 'mobile_app',
            get        => 'mobile_app_view',
            post       => 'do_mobile_app_add',
            properties => {has_action_log => 1,},
        },

        # Мобильные приложения
        {
            resource   => 'mobile_app_settings',
            get        => 'mobile_app_settings_view',
            post       => 'do_mobile_app_settings_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },

        # Продукты РСЯ Мобильные приложения
        {
            resource => 'mobile_app_rtb',
            get      => 'mobile_app_rtb_view',
            post     => sub {
                my ($app) = @_;

                return $app->check_rights('do_mobile_app_rtb_add')
                  ? ($has_assistant_role_without_mobile_partner ? $has_edit_mobile_campaign->() : TRUE)
                  : FALSE;
            },
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        # трафареты
        {
            resource   => 'design_templates',
            get        => sub {TRUE;},
            post       => 'do_design_templates_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },

        # Продукты РСЯ Контекст
        {
            resource   => 'context_on_site_campaign',
            get        => 'context_on_site_campaign_view',
            post       => 'do_context_on_site_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource => 'context_on_site_rtb',
            get      => 'context_on_site_rtb_view',
            post     => sub {
                my ($app) = @_;

                #PI-8722
                return FALSE if $user_id == $M_RU_TEXT_USER_ID;

                return $app->check_rights('do_context_on_site_rtb_add')
                  ? ($has_assistant_role_without_an_partner ? $has_edit_context_campaign->() : TRUE)
                  : FALSE;
            },
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource => 'context_on_site_content',
            get      => _get_sub_check_right_assistant_aware(
                'context_on_site_content_view',
                $is_assistant_only, $assistant_rights
            ),
            post => _get_sub_check_right_assistant_aware(
                'do_context_on_site_content_add',
                $is_assistant_only, $assistant_rights
            ),
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource => 'fast_ban',
            delete   => 'do_fast_ban_delete',
            get      => 'fast_ban_view',
            post     => 'do_fast_ban_add',
        },
        {
            resource => 'context_on_site_natural',
            get      => _get_sub_check_right_assistant_aware(
                'context_on_site_natural_view',
                $is_assistant_only, $assistant_rights
            ),
            post => sub {
                FALSE
            },
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource => 'context_on_site_adblock',
            get      => sub {
                return
                     $has_role_from_group_dev
                  || shift->check_rights('context_on_site_adblock_view')
                  || $can_view_adblock->();
            },
            post => sub {
                my ($app) = @_;

                return $app->check_rights('do_context_on_site_adblock_add')
                  ? ($has_assistant_role_without_adblock_partner ? $can_edit_adblock->() : TRUE)
                  : FALSE;
            },
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },

        # Продукты РСЯ Поиск
        {
            resource => 'search_on_site_campaign',
            get      => sub {
                unless ($_[0]->check_rights('search_on_site_campaign_view')) {
                    return FALSE;
                }
                $has_permission_view_search_on_site_campaigns //=
                  $self->_has_permission_view_search_on_site_campaigns($has_role_from_group_dev, $cur_user_roles);
                return $has_permission_view_search_on_site_campaigns;
            },
            post       => 'do_search_on_site_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource => 'search_on_site_premium',
            get      => sub {
                unless ($_[0]->check_rights('search_on_site_premium_view')) {
                    return FALSE;
                }
                $has_permission_view_search_on_site_campaigns //=
                  $self->_has_permission_view_search_on_site_campaigns($has_role_from_group_dev, $cur_user_roles);
                return $has_permission_view_search_on_site_campaigns;
            },
            post => sub {
                my ($app) = @_;

                return $app->check_rights('do_search_on_site_premium_add')
                  ? ($has_assistant_role_without_an_partner ? $has_edit_search_campaign->() : TRUE)
                  : FALSE;
            },
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource => 'search_on_site_direct',
            get      => sub {
                unless ($_[0]->check_rights('search_on_site_direct_view_blocks')) {
                    return FALSE;
                }
                $has_permission_view_search_on_site_campaigns //=
                  $self->_has_permission_view_search_on_site_campaigns($has_role_from_group_dev, $cur_user_roles);
                return $has_permission_view_search_on_site_campaigns;
            },
            post => sub {
                my ($app) = @_;

                return $app->check_rights('do_search_on_site_direct_add')
                  ? ($has_assistant_role_without_an_partner ? $has_edit_search_campaign->() : TRUE)
                  : FALSE;
            },
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        # Продукты Видеосеть Сайт
        {
            resource   => 'video_an_site',
            get        => 'video_an_site_view',
            post       => 'do_video_an_site_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        # Продукты Видеосеть Видео
        {
            resource => 'video_an_site_instream',
            get      => 'video_an_site_instream_view',
            post     => sub {
                my ($app) = @_;

                return $app->check_rights('do_video_an_site_instream_add')
                  ? ($has_assistant_role_without_video_partner ? $has_edit_video_campaign->() : TRUE)
                  : FALSE;
            },
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource => 'video_an_site_inpage',
            get      => 'video_an_site_inpage_view',
            post     => sub {
                my ($app) = @_;

                return $app->check_rights('do_video_an_site_inpage_add')
                  ? ($has_assistant_role_without_video_partner ? $has_edit_video_campaign->() : TRUE)
                  : FALSE;
            },
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource => 'video_an_site_fullscreen',
            get      => 'video_an_site_fullscreen_view',
            post     => sub {
                my ($app) = @_;

                return $app->check_rights('do_video_an_site_fullscreen_add')
                  ? ($has_assistant_role_without_video_partner ? $has_edit_video_campaign->() : TRUE)
                  : FALSE;
            },
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },

        # Продукты Дистрибуция
        {
            resource   => 'distribution_campaign',
            get        => 'distribution_campaign_view',
            post       => 'do_distribution_campaign_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },

        # Продукты Сервисы Яндекса Контекст
        {
            resource   => 'internal_context_on_site_campaign',
            get        => 'internal_context_on_site_campaign_view',
            post       => 'do_internal_context_on_site_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource   => 'internal_context_on_site_content',
            get        => 'internal_context_on_site_content_view',
            post       => sub {FALSE},
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource   => 'internal_context_on_site_natural',
            get        => 'internal_context_on_site_natural_view',
            post       => sub {FALSE},
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource   => 'internal_context_on_site_rtb',
            get        => 'internal_context_on_site_rtb_view',
            post       => 'do_internal_context_on_site_rtb_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource   => 'internal_context_on_site_stripe',
            get        => 'internal_context_on_site_stripe_view',
            post       => sub {FALSE},
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        # Продукты Сервисы Яндекса Поиск
        {
            resource   => 'internal_search_on_site_campaign',
            get        => 'internal_search_on_site_campaign_view',
            post       => 'do_internal_search_on_site_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        # Продукты Сервисы Яндекса Приложения
        {
            resource   => 'internal_mobile_app',
            get        => 'internal_mobile_app_view',
            post       => 'do_internal_mobile_app_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource   => 'internal_mobile_app_rtb',
            get        => 'internal_mobile_app_rtb_view',
            post       => 'do_internal_mobile_app_rtb_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        # Мульти модели
        {
            resource => 'multimodel_all_pages',
            get      => sub {
                my ($app) = @_;

                foreach my $accessor ($app->multimodel_all_pages->get_multimodel_restapi_models()) {
                    return TRUE if $app->check_rights($accessor . '_view');
                }

                return FALSE;
            },
        },
        {
            resource => 'pages',
            get      => sub {
                my ($app) = @_;

                foreach my $accessor ($app->pages->get_multimodel_restapi_models()) {
                    return TRUE if $app->check_rights($accessor . '_view');
                }

                return FALSE;
            },
        },
        {
            resource => 'blocks',
            get      => sub {
                my ($app) = @_;

                foreach my $accessor ($app->blocks->get_multimodel_restapi_models()) {
                    return TRUE if $app->check_rights($accessor . '_view');
                }

                return FALSE;
            },
        },
        # Продукты Рекламные Платформы Подключения
        {
            resource => 'dsp',
            get      => sub {
                my ($app) = @_;
                # Проверяем либо доступ к своим сущностям, либо ко всем в базе, так как
                # GET запрос должен быть открыт у dsp_partner (видит только свои дспшки) и
                # у всяких менеджеров, которые видят и чужие дсп.
                return $app->check_rights('dsp_view') || $app->check_rights('dsp_view_all');
            },
            post       => 'do_dsp_action_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        # Продукты RTB-блоки ссылка на % за DSP
        {
            resource => 'yql_dsp_percent',
            get      => 'yql_dsp_percent',
        },
        # Разработчику Управление Simple notification
        {
            resource => 'simple_notification_tool',
            get      => 'simple_notification_tool',
        },
        {
            resource => 'notification_tool',
            get      => 'notification_tool',
        },
        {
            resource => 'moderation_reason_tool',
            get      => 'moderation_reason_tool',
        },

        {
            resource   => 'moderation_reason',
            get        => 'moderation_reason_view',
            post       => 'do_moderation_reason_action_add',
            properties => {has_action_log => 1},
        },
        # Разработчику Загрузка тэгов
        {
            resource => 'pages_tags',
            get      => 'pages_tags_view',
        },
        # Менеджер - задачи поддержки
        {
            resource => 'queue_support_task',
            get      => 'queue_support_task',
        },
        # Статистика - конструктор отчётов

        {
            resource => 'statistics_constructor',
            get      => sub {
                $_[0]->check_rights('statistics_constructor_view');
            },
        },
        # финансовые документы
        {
            resource => 'financial_documents',
            get      => 'financial_documents_view',
        },

        # Разработчику Информация Письма
        {
            resource => 'devel_view_mail',
            get      => 'mailer_view',
        },

        # Разработчику Информация Кроновые задачи
        {
            resource => 'devel_cron_jobs',
            get      => 'view_cron_job',
        },

        # Разработчику Баланс Создать клиента
        {
            resource => 'devel_create_client',
            get      => 'devel_create_client',
        },

        # Разработчику Баланс Отвязать клиента от пользователя
        {
            resource => 'devel_remove_user_client_association',
            get      => 'remove_user_client_association',
        },

        # Разработчику БК Сравнение данных ПИ и БК
        {
            resource => 'devel_bk_pi_comparison',
            get      => 'devel_bk_pi_comparison',
        },

        # Разработчику Бk Просмотр лога отправки в БК
        {
            resource => 'devel_bk_send_log',
            get      => 'bk_edit_page_view_bk_send_log',
        },

        # Разработчику Проверка Проверка полей
        {
            resource => 'devel_fields',
            get      => 'view_check_fields',
        },

        # Разработчику Проверка яндекс Денег
        {
            resource => 'devel_test_yamoney',
            get      => 'devel_test_yamoney',
        },

        # Разработчику Изменение опций пейджа
        {
            resource => 'devel_update_options',
            get      => 'devel_update_options',
        },

        # Статистика Перезабор статистики
        {
            resource => 'queue_statistics_intake',
            get      => 'queue_statistics_intake',
            post     => 'queue_add_statistics_intake',
        },

        # Разработчику Баланс Удалить пользователя
        {
            resource => 'devel_drop_user',
            get      => sub {shift->check_rights('devel_drop_user') && $is_dev_or_ts;},
        },

        # Разработчику Баланс Переключатель CreateOrUpdatePlace

        {
            resource => 'devel_create_or_update_place_switch',
            get      => sub {shift->check_rights('create_or_update_place_switch') && $is_dev_or_ts;},
        },

        # Разработчику Управление Форма для добавления дополнительного дохода
        {
            resource => 'devel_insert_additional_income_stat',
            get      => 'devel_insert_additional_income_stat',
        },

        {
            resource => 'test_stage_warning',
            get      => sub {$is_dev_or_ts;},
        },

        {
            resource => 'production_stage_warning',
            get      => sub {
                my ($app) = @_;
                return $app->get_option('stage', 'unknown') eq 'production'
                  && $app->check_rights('show_production_stage_warning');
            },
        },
        {
            resource => 'assessor_warning',
            get      => sub {
                my ($app) = @_;
                return $is_preprod;
            },
        },

        {
            resource => 'agreement_rsya_link',
            get      => sub {$has_role_from_group_dev_or_yan;},
        },

        {
            resource => 'feedback_rsya_link',
            get      => sub {$has_role_from_group_dev_or_yan;},
        },

        {
            resource => 'feedback_dsp_link',
            get      => sub {$has_role_from_group_dev_or_dsp;},
        },

        {
            resource => 'mailto_pi_link',
            get      => sub {$cur_user_is_internal;},
        },

        {
            resource => 'mailto_balance_link',
            get      => sub {$cur_user_is_internal;},
        },

        {
            resource => 'global_concurents',
            get      => sub {
                return exists($cur_user_roles->{$SITE_PARTNER_ROLE_ID})
                  || exists($cur_user_roles->{$MOBILE_PARTNER_ROLE_ID});
            },
        },
        # Телефоны глобальных конкурентов
        {
            resource => 'user_global_excluded_phones',
            get      => 'user_global_excluded_phones_view',
        },

        # Домены глобальных конкурентов
        {
            resource => 'user_global_excluded_domains',
            get      => 'user_global_excluded_domains_view',
        },

        # Мобильная медиация
        {
            resource => 'mobile_mediation',
            get      => sub {
                $cur_user->{has_mobile_mediation};
            },
        },
        {
            resource => 'common_offer',
            get      => sub {
                $can_common_offer
            },
        },
        {
            resource => 'bkdata',
            get      => sub {
                !in_array($self->get_option('stage', ''), [qw(preprod production)]);
            },
        },

        {
            resource   => 'block_presets',
            get        => 'block_presets_view',
            post       => 'do_block_presets_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        {
            resource => 'mds_avatars',
            get      => sub {TRUE},
            post     => sub {TRUE},
        },
        {
            resource => 'indoor_import',
            get      => 'indoor_import_view'
        },
        # Единые сценарии instream
        {
            resource   => 'video_scenaries',
            get        => 'video_scenaries_view',
            post       => 'do_video_scenaries_add',
            properties => {
                has_action_log => 1,
                has_archive    => 1,
            },
        },
        # Обновленная статистика Отчеты
        {
            resource => 'statistics_reports_mol',
            get      => 'statistics_reports_mol_view',
            post     => 'statistics_reports_mol_view',
        },
        # Обновленная статистика (МОЛ, расширенная статистика)
        {
            resource => 'statistics_mol',
            get      => 'statistics_mol_view',
        },
        # Обновленная биллинговая статистика
        {
            resource => 'statistics_billing',
            get      => 'statistics_billing_view',
        },
        {
            resource => 'settings_in_menu',
            get      => 'settings_in_menu_view',
        },
        {
            resource => 'widgets',
            get      => sub {TRUE},
            post     => sub {
                my ($app) = @_;
                my $is_ok = $app->check_rights('do_widgets_add');
                if ($is_ok) {
                    my $is_exists = $app->widgets->get_all(fields => [qw(is_exists)])->[0]->{is_exists};
                    $is_ok = $is_exists ? 0 : 1;
                }
                return $is_ok;
            },
            properties => {has_action_log => 1,},
        },
    );

    my @resources;

    foreach my $permision (@permisions) {

        my $resource = $permision->{'resource'};

        next if %need_resources && !$need_resources{$resource};

        my $show_in_menu = $self->_show_in_menu($resource, $cur_user, $has_role_from_group_dev);

        my @methods;
        foreach my $method (qw(get post delete)) {
            next if not defined($permision->{$method});
            my $check = $permision->{$method};

            if (ref($check)) {
                next unless $check->($self->app);
            } else {
                next unless $self->check_rights($check);
            }

            push @methods, uc($method);

            next if $method ne 'get';
            next unless $self->app->can($resource) and $self->app->$resource->can('api_can_edit');

            push @methods, 'PATCH' if $self->app->$resource->api_can_edit();
        }

        next unless @methods;

        push @resources,
          {
            resource     => $resource,
            methods      => \@methods,
            show_in_menu => $show_in_menu,
          };

        if ($permision->{properties}{has_action_log}) {
            my $action_log_resource = $self->app->$resource->action_log_accessor();
            my $right               = $self->app->$resource->get_right('view_action_log');

            if ($action_log_resource && $self->check_rights($right)) {
                push @resources,
                  {
                    resource     => $action_log_resource,
                    methods      => ['GET'],
                    show_in_menu => \0,
                  };
            }
        }

        push @resources,
          {
            resource     => $resource . '_deleted',
            methods      => ['GET'],
            show_in_menu => $show_in_menu,
          }
          if $permision->{properties}{has_archive};

    }

    $self->memcached->set(
        available_resources => $cache_key,
        \@resources, $self->app->get_option('resources_cache_time', 30 * 60)
    ) unless %need_resources;

    return \@resources;
}

=head2 get_get_resources

Возвращает HASHREF со списком всех ресурсов текущего пользователя, у которых
есть метод GET.

Пример возвращаемых данных:

    {
        context_on_site_campaign         => 1,
        context_on_site_campaign_deleted => 1,
        context_on_site_direct           => 1,
        context_on_site_direct_deleted   => 1,
        context_on_site_rtb              => 1,
        context_on_site_rtb_deleted      => 1,
        context_on_site_stripe           => 1,
        mobile_app_rtb                   => 1,
        mobile_app_settings              => 1,
        search_on_site_campaign          => 1,
        search_on_site_campaign_deleted  => 1,
        search_on_site_direct            => 1,
        search_on_site_direct_deleted    => 1,
        search_on_site_premium           => 1,
        search_on_site_premium_deleted   => 1,
        site                             => 1,
        statistics                       => 1,
        statistics_reports               => 1,
        test_stage_warning               => 1,
    }

=cut

sub get_get_resources {
    my ($self, $need_resources) = @_;

    my $all = $self->get_available_resources($need_resources);

    my $get = {map {$_->{resource} => 1} grep {in_array('GET', $_->{methods})} @$all};

    return $get;
}

sub _show_in_menu {
    my ($self, $resource, $cur_user, $has_role_from_group_dev) = @_;

    my $flag = TRUE;
    if (   $resource eq 'resources'
        || $resource eq 'common_offer'
        || $resource eq 'fast_ban'
        || $resource eq 'widgets'
        || $resource =~ /action_log$/)
    {
        $flag = FALSE;
    } elsif ($resource eq 'users') {
        $flag = $self->check_rights('users_view');
    } elsif ($resource eq 'site') {
        $flag = $self->check_rights('site_view');
    } elsif ($resource eq 'mobile_app') {
        $flag = $self->check_rights('mobile_app_view_all');
    } elsif ($resource eq 'statistics_reports') {
        $flag = ($has_role_from_group_dev || !$self->check_rights('statistics_reports_mol_view'));
    } elsif ($resource eq 'bk_statistics') {
        $flag = ($has_role_from_group_dev || !$self->check_rights('statistics_mol_view'));
    } elsif ($resource eq 'statistics_constructor') {
        $flag = ($has_role_from_group_dev || !$self->check_rights('statistics_mol_view'));
    } elsif ($resource eq 'dsp') {
        # в тикете PI-29527 есть ограничение, что вкладку меню могут видеть только роли, непосредственно меняющие
        # таблицу ДСП. На права dsp_view или dsp_view_all тут заложиться нельзя, так как они есть у менеджерских ролей
        # для получения списка ДСП при редактировании блока.
        $flag = $self->check_rights('do_dsp_action_add');
    }

    $flag = FALSE
      if !$cur_user->{has_rsya} && !$MOBILE_MEDIATION_RESOURCES->{$resource};

    return $flag ? \1 : \0;
}

sub _get_sub_check_right_assistant_aware {
    my ($right_name, $has_assistant_role_only, $rights_from_page_owners) = @_;
    return sub {
        return $has_assistant_role_only
          ? ($rights_from_page_owners->{$right_name} || $_[0]->rbac->cur_user_is_internal())
          : $_[0]->check_rights($right_name);
    };
}

1;
