package Application::Model::Role;

use strict;
use warnings;

use base 'Role::Tiny';
use QBit::Array;

use Carp qw(croak);
use Class::Method::Modifiers qw(before around install_modifier);

my $composition_method = {
    hashref => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, @args) = @_;
            my $hashref = $orig->($self, @args);

            no strict 'refs';
            %$hashref = (%{"${role}::${method}"->($self, @args)}, %$hashref);

            return $hashref;
        };
    },
    hashref_merge_keys => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, @args) = @_;
            my $hashref = $orig->($self, @args) // {};

            no strict 'refs';
            my $result = "${role}::${method}"->($self, @args);

            $hashref = _merge_hasref_keys($hashref, $result);
            return $hashref;
        };
    },
    hashref_replace => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, @args) = @_;
            my $hashref = $orig->($self, @args);

            no strict 'refs';
            %$hashref = (%$hashref, %{"${role}::${method}"->($self, @args)});

            return $hashref;
        };
    },
    filter => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, @args) = @_;
            my $filter = $orig->($self, @args);

            no strict 'refs';
            $filter->{fields} = {%{"${role}::${method}"->($self)->{fields}}, %{$filter->{fields} // {}}};

            return $filter;
        };
    },
    arrayref => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, @args) = @_;
            my $arrayref = $orig->($self, @args);

            no strict 'refs';
            push(@$arrayref, @{"${role}::${method}"->($self, @args)});

            return $arrayref;
        };
    },
    rights => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, @args) = @_;
            my $rights = $orig->($self, @args);

            no strict 'refs';
            $rights->[0]{rights} = {%{"${role}::${method}"->($self)->[0]{rights}}, %{$rights->[0]{rights}}};

            return $rights;
        };
    },
    set => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, @args) = @_;
            my $fields = $orig->($self, @args);

            no strict 'refs';
            $fields->{$_} = 1 foreach keys(%{"${role}::${method}"->($self)});

            return $fields;
        };
    },
    accum => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, $data) = @_;

            no strict 'refs';
            return "${role}::${method}"->($self, $orig->($self, $data), $data);
        };
    },
    accum2 => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, $object, $data) = @_;

            no strict 'refs';
            return "${role}::${method}"->($self, $object, $orig->($self, $object, $data));
        };
    },
    bk_data => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, $page, $data, @opts) = @_;

            no strict 'refs';
            return "${role}::$method"->($self, $page, {$orig->($self, $page, $data, @opts)});
        };
    },
    bk_block_data => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, $block, $data) = @_;

            no strict 'refs';
            return "${role}::$method"->($self, $block, $orig->($self, $block, $data));
        };
    },
    origin_method_first => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, @args) = @_;

            $orig->(@args);

            no strict 'refs';
            return "${role}::$method"->(@args);
        };
    },
    origin_method_last => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, @args) = @_;

            no strict 'refs';
            "${role}::$method"->(@args);

            return $orig->(@args);
        };
    },
    logical_and => sub {
        my ($role, $method) = @_;
        sub {
            my ($orig, $self, @args) = @_;

            no strict 'refs';
            return $orig->($self, @args) && "${role}::$method"->($self, @args);
        };
    },
};

my $structure_methods = {
    can_action_approve                       => 'logical_and',
    can_action_delete                        => 'logical_and',
    can_action_duplicate                     => 'logical_and',
    can_action_edit                          => 'logical_and',
    can_action_reject                        => 'logical_and',
    can_action_delete                        => 'logical_and',
    can_action_restore                       => 'logical_and',
    can_action_set_blocked                   => 'logical_and',
    can_action_set_need_approve              => 'logical_and',
    check_action                             => 'logical_and',
    collect_editable_fields                  => 'accum2',
    fix_template                             => 'origin_method_first',
    get_actions_depends                      => 'arrayref',
    get_add_fields                           => 'accum',
    get_available_fields                     => 'accum2',
    get_available_fields_depends             => 'arrayref',
    get_bk_block_data                        => 'bk_block_data',
    get_bk_data                              => 'bk_data',
    get_db_filter_simple_fields              => 'arrayref',
    get_dsp_type                             => 'arrayref',
    get_dsp_type_depends                     => 'hashref_replace',
    get_editable_fields_depends              => 'arrayref',
    get_fields_defaults                      => 'hashref_replace',
    get_need_update_in_bk_fields             => 'set',
    get_structure_model_accessors            => 'hashref',
    get_structure_model_fields               => 'hashref',
    get_structure_model_filter               => 'filter',
    get_structure_multistates_graph          => 'hashref_merge_keys',
    get_structure_rights_to_register         => 'rights',
    hook_fields_processing_before_validation => 'origin_method_first',
    hook_fields_validation                   => 'origin_method_first',
    hook_preparing_fields_to_save            => 'origin_method_last',
    hook_processing_after_insert             => 'origin_method_first',
    hook_processing_after_update             => 'origin_method_first',
    hook_save_fields_from_related_models     => 'origin_method_first',
    hook_set_initialize_settings             => 'origin_method_first',
    hook_stash_edit_fields                   => 'accum',
    on_action_approve                        => 'origin_method_first',
    on_action_delete                         => 'origin_method_first',
    on_action_reject                         => 'origin_method_first',
    on_action_reset_blocked                  => 'origin_method_first',
    on_action_restore                        => 'origin_method_first',
    on_action_return_to_moderation           => 'origin_method_first',
    on_action_set_blocked                    => 'origin_method_first',
    on_action_set_need_approve               => 'origin_method_first',
    on_action_start                          => 'origin_method_first',
    on_action_stop                           => 'origin_method_first',
    pre_process_fields                       => 'origin_method_first',
    related_models                           => 'hashref',
    make_fields_defaults                     => 'hashref_merge_keys',
    get_fields_depends                       => 'hashref_merge_keys',
};

before apply_roles_to_package => sub {
    my ($me, $to, @roles) = @_;

    for my $role (@roles) {
        Role::Tiny::_load_module($role);
        my $s = Role::Tiny::_getstash($role);
        my @refs = map {$s->{$_} ? (*{$s->{$_}}{CODE} // ()) : ()} (keys(%$structure_methods));
        @{$Role::Tiny::INFO{$role}{not_methods}}{@refs} = @refs;
    }
};

around _install_subs => sub {
    my ($orig, $self, $target) = @_;
    no strict 'refs';
    no warnings 'redefine';
    local *Role::Tiny::_getglob = sub {
        if ($_[0] eq "${target}::with") {
            # TODO: resolve conflict with QBit::Role::Consume?
            return \*{"${target}::_consume"};
        } else {
            return \*{$_[0]};
        }
    };
    $self->$orig($target);
};

sub role_application_steps {
    my ($self) = @_;
    return (
        (
            qw(
              _install_structure_methods
              )
        ),
        $self->SUPER::role_application_steps(),
    );
}

our %COMPOSED_STRUCTURE_METHODS;

sub _install_structure_methods {
    my ($me, $to, $role) = @_;

    return if $COMPOSED_STRUCTURE_METHODS{$role};

    for my $method (keys %$structure_methods) {

        my $role_method_ref = $role->can($method);

        if ($role_method_ref) {
            push @{$Role::Tiny::INFO{$role}{modifiers}},
              ['around', $method, $composition_method->{$structure_methods->{$method}}->($role, $method),];
        }
    }

    $COMPOSED_STRUCTURE_METHODS{$role} //= 1;
}

sub _merge_hasref_keys {
    my ($hashref, $result) = @_;
    my $duplicate_keys = arrays_intersection([keys %$hashref], [keys %$result]);
    foreach (@$duplicate_keys) {
        if (exists $result->{$_}) {
            if (ref $result->{$_} eq 'ARRAY') {
                push @{$hashref->{$_}}, @{$result->{$_}};
            } elsif (ref $result->{$_} eq 'HASH') {

                %{$hashref->{$_}} = %{_merge_hasref_keys($hashref->{$_}, $result->{$_})};
            }
        }
    }
    %$hashref = (%$result, %$hashref);
    return $hashref;
}

1;
