package Application::Model::Role::Block::Has::ContentBlock;

use qbit;

use Application::Model::Role;
use Exception::Validator::Fields;

use PiConstants qw(
  $M_RU_TEXT_USER_ID
  $INTERNAL_DSP_TAGS
  $DSP_TYPE_MEDIA
  $MIN_CPM_STRATEGY_ID
  $MAX_REVENUE_STRATEGY_ID
  $SEPARATE_CPM_STRATEGY_ID
  );

consume qw(
  Application::Model::Role::Has::CreateDate
  Application::Model::Role::Has::Brands
  Application::Model::Role::Has::Categories
  Application::Model::Role::Has::CustomBkData
  Application::Model::Role::Has::Geo
  Application::Model::Role::Has::Strategies
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::Block::DesignSettings
  );

sub storage_method_of_strategies {'opts'}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {api_http_news => 'Application::Model::API::Yandex::News'};
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            rights => {
                map {$self->get_description_right($_)}
                  qw(
                  edit_field__comment
                  view_field__comment
                  ),
            }
        }
    ];
}

sub get_actions_depends {
    [
        qw(
          id
          campaign_id
          multistate
          page
          is_protected
          )
    ];
}

sub get_available_fields_depends {
    [qw(multistate id)];
}

sub get_editable_fields_depends {
    [
        qw(
          page_id
          campaign_id
          id
          multistate
          is_protected
          page
          )
    ];
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        {name => $self->get_page_id_field_name(), label => gettext('Page ID')},
        {name => 'id',                            label => gettext('Block ID')},
        {name => 'caption',                       label => gettext('Block\'s caption')},
        {name => 'campaign.all_domain',           label => gettext('Domain and mirror')},
        {name => 'multistate',                    label => gettext('Status')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => d_pgettext('Block multistate', 'New'),
        multistates => [
            [deleted           => d_pgettext('Block multistate', 'Archived')],
            [working           => d_pgettext('Block multistate', 'Working')],
            [check_statistics  => d_pgettext('Block multistate', 'Check statistics'), deprecated => TRUE],
            [deleted_with_page => d_pgettext('Block multistate', 'Archived with page'), private => TRUE],
            [need_update       => d_pgettext('Block multistate', 'Need update')],
            [updating          => d_pgettext('Block multistate', 'Updating')],
        ],
        actions => {
            edit                   => d_pgettext('Block action', 'Edit'),
            delete                 => d_pgettext('Block action', 'Archive'),
            delete_with_page       => d_pgettext('Block action', 'Archive with page'),
            restore                => d_pgettext('Block action', 'Restore'),
            restore_with_page      => d_pgettext('Block action', 'Restore with page'),
            start                  => d_pgettext('Block action', 'Start'),
            stop                   => d_pgettext('Block action', 'Stop'),
            reset_check_statistics => d_pgettext('Block action', 'Reset "check_statistics"'),
            duplicate              => d_pgettext('Block action', 'Duplicate'),
        },
        right_name_prefix => $self->accessor . '_',
        right_actions     => {
            add => d_pgettext('Block action', 'Add'),
            set_need_update =>
              {label => d_pgettext('Block action', 'Set "need_update"'), dont_write_to_action_log => TRUE},
            start_update => {label => d_pgettext('Block action', 'Start update'), dont_write_to_action_log => TRUE},
            stop_update  => {label => d_pgettext('Block action', 'Stop update'),  dont_write_to_action_log => TRUE},
        },
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action    => 'delete_with_page',
                from      => 'not deleted',
                set_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action    => 'start',
                from      => 'not working and not deleted',
                set_flags => ['working'],
            },
            {
                action      => 'stop',
                from        => 'working',
                reset_flags => ['working'],
            },
            {
                action      => 'reset_check_statistics',
                from        => 'check_statistics',
                reset_flags => ['check_statistics'],
            },
            {
                action => 'duplicate',
                from   => 'not (deleted or deleted_with_page)',
            },
            {
                action    => 'set_need_update',
                from      => '__EMPTY__ or not __EMPTY__',
                set_flags => ['need_update'],
            },
            {
                action      => 'start_update',
                from        => 'need_update',
                reset_flags => ['need_update'],
                set_flags   => ['updating'],
            },
            {
                action      => 'stop_update',
                from        => 'updating',
                reset_flags => ['updating'],
            },
        ],
    };
}

sub get_strategies {
    return {
        $MIN_CPM_STRATEGY_ID     => {},
        $MAX_REVENUE_STRATEGY_ID => {},
    };
}

sub collect_editable_fields {
    my ($self, $data, $fields) = @_;

    $self->_get_common_add_edit_fields($fields, $data);

    return $fields;
}

sub get_available_fields {
    my ($self, $obj, $fields) = @_;

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        $fields,
        {
            # TODO: outdated?
            $accessor . '_view_field__%s' => [qw(comment)],
            $accessor . '_edit_field__%s' => [qw(comment)],
        }
    );

    return $fields;
}

sub get_add_fields {
    my ($self, $fields) = @_;

    $self->_get_common_add_edit_fields($fields);
    $fields->{page_id} = TRUE;

    return $fields;
}

sub _get_common_add_edit_fields {
    my ($self, $fields, $object) = @_;

    $self->get_fields_by_right(
        res_fields      => $fields,
        no_right_fields => ['caption'],
        right_fields    => {edit => ['comment']}
    );

    return $fields;
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    $self->update_in_bk($self->hook_stash->get('id')) if $self->need_update_in_bk($self->hook_stash->get('fields'));
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    $self->maybe_do_action($self->hook_stash->get('id'), 'start');
    $self->update_in_bk($self->hook_stash->get('id'));
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    my $campaign_model = $self->get_campaign_model_name;

    $self->partner_db->transaction(
        sub {
            $opts->{'id'} = $self->$campaign_model->get_next_block_id($opts->{'page_id'});
        }
    );

    $opts->{'campaign_id'} = $opts->{'page_id'};

}

sub query_filter {
    my ($self, $filter, $opts) = @_;

    $filter = $self->limit_filter_by_campaign($filter, $opts);

    return $filter;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    return {caption => TRUE};
}

sub get_bk_direct_design {
    my ($self, $block) = @_;
    return +{
        # TODO: content block v2
    };
}

sub get_bk_block_data_key {'rtb_blocks'}

sub get_bk_common_data {
    return ();
}

sub get_bk_block_data {
    my ($self, $block, $bk_data) = @_;

    $bk_data->{'BlockCaption'} = $block->{'caption'};
    $bk_data->{'MultiState'}   = $block->{'multistate'};

    $bk_data->{'DSPType'} = $DSP_TYPE_MEDIA;

    my $strategy = $block->{'strategy'};
    $bk_data->{'OptimizeType'} = $strategy == $SEPARATE_CPM_STRATEGY_ID ? 0 : $strategy;

    $self->get_ad_type_set($bk_data, $block, allow_media_performance => FALSE, media => 0);

    map {$bk_data->{'PageImpOptions'}{$_} = [] if !exists($bk_data->{'PageImpOptions'}{$_})} qw(Enable Disable);

    push @{$bk_data->{'PageImpOptions'}{'Enable'}}, 'widget';

    #PI-13940 default values
    $bk_data->{'BlindLevel'}  = 0;
    $bk_data->{'DirectLimit'} = 1;
    $bk_data->{'Height'}      = 250;
    $bk_data->{'Sizes'}       = [
        {
            'Height' => 0,
            'Width'  => 0
        }
    ];
    $bk_data->{'Width'} = 970;

    return $bk_data;
}

sub get_opts_schema_name {'content_block_opts'}

sub api_available_actions {
    my ($self) = @_;

    my @actions = qw(
      edit
      delete
      restore
      );

    return @actions;
}

sub api_can_edit {TRUE}

sub api_can_add {TRUE}

sub direct_blocks {
    my ($self, $block) = @_;

    my %direct_blocks = map {$_->{'id'} => $_} @{$self->get_types_direct_blocks($block)};

    return \%direct_blocks;
}

TRUE;
