package Application::Model::Role::Block::Has::ContentWidget;

use qbit;

use Application::Model::Role;
use Exception::Validator::Fields;

use PiConstants qw(
  $CONTENT_BLOCK_GRID_SIZE
  $CONTENT_SOURCE_MAX_ID
  );

consume qw(
  Application::Model::Role::Block::Has::ContentWidgetCommon
  );

my $NEWS_JSON_OPTION_ID = 48;    # new news API

my $GRID_MAX_ROWS    = 10;
my $GRID_MAX_COLUMNS = 10;

my $NEWS_PARAMS_DOC_SOURCE = [qw(web turbo)];
my $NEWS_PARAMS_MODEL_TYPE = [qw(ctr long_ctr)];

my @new_format_fields = qw(
  adv_positions
  need_pictures
  );
my @old_format_fields = qw(
  grid_position
  sources
  );

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            rights => {
                map {$self->get_description_right($_)}
                  map {("edit_field__$_", "view_field__$_")}
                  qw(
                  add_url_parameter
                  doc_source
                  ),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        sources => {
            from_opts  => 'from_hash',
            label      => d_gettext('Sources'),
            type       => 'array',
            sub_type   => 'number',
            need_check => {
                optional => TRUE,      # not always optional, see fix_template
                type     => 'array',
                all      => {
                    type => 'int_un',
                    max  => $CONTENT_SOURCE_MAX_ID
                },
                # TODO: метод будет использоваться, как появится ручка от поиска
                # check => sub {
                #     my ($qv, $sources) = @_;
                #     my $check = $qv->app->api_http_news->check_sources($sources);
                #     throw Exception::Validator::Fields $check->{message} unless $check->{result};
                #
                # },
            },
            api => 1,
        },
        grid_position => {
            from_opts  => 'from_hash',
            label      => d_gettext('Ad position'),
            type       => 'number',
            depends_on => [qw(grid_rows grid_columns)],
            need_check => {
                optional => TRUE,    # not always optional, see fix_template
                check    => sub {
                    my ($qv, $grid_position) = @_;

                    throw Exception::Validator::Fields gettext(
                        'Position cannot be greater than product of columns and rows')
                      unless ($grid_position > 0
                        && $grid_position <= $qv->data->{grid_rows} * $qv->data->{grid_columns})
                      || $grid_position == -1;
                },
            },
            api => 1,
        },
        # -- new recommendations widget
        need_pictures => {
            from_opts  => 'from_hash',
            label      => d_gettext('Show items with pictures only'),
            type       => 'boolean',
            need_check => {
                optional => TRUE,
                type     => 'boolean',
            },
            api => 1,
        },
        news_params => {
            api        => 1,
            from_opts  => 'from_hash',
            label      => d_gettext('News Params'),
            need_check => {
                fields => {
                    candidates => {type => 'candidates'},
                    days       => {
                        optional => TRUE,
                        min      => 1,
                        type     => 'int_un',
                    },
                    filter_clicked => {
                        optional => FALSE,
                        type     => 'boolean',
                    },
                    model_type => {
                        in       => $NEWS_PARAMS_MODEL_TYPE,
                        optional => FALSE,
                        type     => 'scalar',
                    },
                },
                must_exists => [qw(candidates filter_clicked model_type)],
                type        => 'hash',
                check       => sub {
                    my ($qv, $news_params) = @_;

                    if (   $qv->app->hook_stash->inited
                        && !$qv->app->check_short_rights('add_other_domains')
                        && $qv->app->DOES('Application::Model::Product::AN::ContextOnSite::Content'))
                    {

                        my $opts = $qv->app->hook_stash->get('opts');
                        return unless exists $opts->{news_params};

                        my $exist_domains = {
                            $qv->app->hook_stash->mode('edit')
                            ? map {$_->{host} => TRUE}
                              @{$qv->app->hook_stash->get('current')->{news_params}->{candidates}}
                            : ()
                        };

                        my @new_domains =
                          grep {!$exist_domains->{$_}} @{array_uniq(map {$_->{'host'}} @{$news_params->{candidates}})};
                        return unless @new_domains;
                        my $owner = $qv->app->context_on_site_campaign->get_all(
                            fields => ['owner_id'],
                            filter => {page_id => $qv->data->{page_id}}
                        )->[0]->{owner_id};

                        my $domains = {};
                        foreach (qw(context_on_site_campaign search_on_site_campaign video_an_site)) {
                            my $data =
                              $qv->app->$_->get_all(fields => ['domain', 'mirrors'], filter => {owner_id => $owner});
                            map {
                                $domains->{$_->{domain}} = TRUE;
                                map {$domains->{ref($_) eq 'HASH' ? $_->{domain} : $_} = TRUE} @{$_->{mirrors}}
                            } @$data;
                        }

                        my $subdomain_re = join ')|(', keys %{$domains};
                        my @bad_domains =
                          grep {!$domains->{$_} && not $_ =~ /\.($subdomain_re)$/} @new_domains;
                        throw Exception::Validator::Fields gettext(
                            "Domains %s not found in domain and mirrors of campaigns", join ',', @bad_domains)
                          if @bad_domains;
                    }
                  }
            },
            type => 'complex',
        },
        add_url_parameter => {
            api        => 1,
            from_opts  => 'from_hash',
            need_check => {
                optional => TRUE,
                type     => 'boolean',
            },
            type => 'boolean',
        },
        doc_source => {
            api        => 1,
            from_opts  => 'from_hash',
            need_check => {
                in       => $NEWS_PARAMS_DOC_SOURCE,
                optional => TRUE,
                type     => 'scalar',
            },
            type => 'string',
        },
        adv_positions => {
            from_opts  => 'from_hash',
            label      => d_gettext('Ad position'),
            type       => 'array',
            sub_type   => 'number',
            depends_on => [qw(grid_rows grid_columns)],
            need_check => {
                type     => 'array',
                optional => TRUE,
                check    => sub {
                    my ($qv, $adv_position_list) = @_;

                    if (0 == @$adv_position_list) {
                        throw Exception::Validator::Fields gettext('Position number list can not be empty');
                    }

                    my $positions_number = $qv->data->{grid_rows} * $qv->data->{grid_columns};
                    if ($positions_number < @$adv_position_list) {
                        throw Exception::Validator::Fields gettext(
                            'Position number cannot be greater than product of columns and rows');
                    }

                    if ($positions_number == @$adv_position_list) {
                        throw Exception::Validator::Fields gettext(
                            'Use Native blocks if you want to include only an ad');
                    }

                    foreach my $adv_position (@{$adv_position_list}) {
                        throw Exception::Validator::Fields gettext(
                            'Position cannot be greater than product of columns and rows')
                          unless ($adv_position > 0 && $adv_position <= $positions_number)
                          || $adv_position == -1;
                    }
                },
            },
            api => 1,
        },
    };
}

sub get_available_fields {
    my ($self, $obj, $fields) = @_;

    foreach (qw(add_url_parameter doc_source)) {
        if ($self->check_short_rights("view_field__$_")) {
            $fields->{$_} = TRUE;
        } else {
            delete $fields->{$_};
        }
    }

    return $fields;
}

sub get_add_fields {
    my ($self, $fields) = @_;

    return $self->_add_all_fields($fields);
}

sub collect_editable_fields {
    my ($self, $obj, $fields) = @_;

    return $self->_add_all_fields($fields);
}

sub get_widget_grid_size {
    return $CONTENT_BLOCK_GRID_SIZE;
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    return $self->_add_all_fields();
}

sub _set_defaults {
    my ($self, $news_params) = @_;

    foreach my $candidate (@{$news_params->{candidates}}) {
        $candidate->{$_} //= [] foreach (qw(tags tags_blacklist url_prefixes url_prefixes_blacklist));
    }
    if ($news_params->{days}) {
        $news_params->{days} += 0;
    } else {
        delete $news_params->{days};
    }
    $news_params->{filter_clicked} = 0 + ($news_params->{filter_clicked} ? 1 : 0);
    $news_params->{model_type} //= $NEWS_PARAMS_MODEL_TYPE->[0];

    return TRUE;
}

sub get_bk_block_data {
    my ($self, $block, $data) = @_;

    my $h = $data->{'CustomBlockData'}{WidgetInfo} //= {};

    # new content block recommendations format
    # https://st.yandex-team.ru/PI-15303
    if ($self->_is_new_format($block)) {
        # BK option; 33 for new news API
        $h->{'NewsJsonOptionId'} = $NEWS_JSON_OPTION_ID;
        # 0|1, default 1
        $h->{'NeedPictures'} = $block->{need_pictures} // 1;
        # [-1,1..25]
        $h->{'AdvPosition'} = [map {0 + $_} @{$block->{adv_positions}}];
        # DirectLimit - count of adv_positions
        $data->{'DirectLimit'} = 0 + scalar @{$block->{adv_positions}};
        # [ { host => 'drive.ru', tags => [ qw/tag1 tag2 tag3/ ] }, ]
        $h->{'NewsParams'} //= {};
        $h->{'NewsParams'} = clone($block->{news_params});
        $self->_set_defaults($h->{'NewsParams'});
        $h->{'NewsParams'}{add_url_parameter} = !defined($block->{add_url_parameter})
          || $block->{add_url_parameter} ? 1 : 0;
        $h->{'NewsParams'}{doc_source} = $block->{doc_source};
    } else {
        $h->{'Sources'} = [map {0 + $_} @{$block->{sources}}];
        $h->{'AdvPosition'} = 0 + $block->{grid_position};
    }

    return $data;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    $opts->{$_} = undef foreach (
          $self->_is_new_format($self->hook_stash->get('settings'))
        ? @old_format_fields
        : @new_format_fields
    );

    return TRUE;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->_set_defaults($opts->{news_params});
    $opts->{add_url_parameter} //= 1;
    $opts->{doc_source} //= $NEWS_PARAMS_DOC_SOURCE->[0];

    return TRUE;
}

sub hook_preparing_fields_to_save {
    my ($self, $opts) = @_;

    if ($opts->{news_params} && $opts->{news_params}{candidates}) {
        foreach my $candidate (@{$opts->{news_params}{candidates}}) {
            foreach my $field (qw(tags tags_blacklist url_prefixes url_prefixes_blacklist)) {
                if ($candidate->{$field}) {
                    $candidate->{$field} = [sort @{array_uniq($candidate->{$field})}];
                }
            }
        }
    }

    return TRUE;
}

sub fix_template {
    my ($self, $qv) = @_;

    my $fields = $qv->template->{'fields'};

    my $is_new_format = $self->_is_new_format($qv->data);

    # no old fields in new format; no new fields in old format
    foreach ($is_new_format ? @old_format_fields : @new_format_fields) {
        $fields->{$_} = {eq => undef, optional => TRUE};
    }
    # FIXME: this is needed only for check_json_schemas.t
    # new fields should be required if new format and vice versa
    foreach ($is_new_format ? @new_format_fields : @old_format_fields) {
        $fields->{$_}->{optional} = FALSE;
    }
}

sub _add_all_fields {
    my ($self, $fields) = @_;

    $fields //= {};

    $fields->{$_} = TRUE foreach (
        qw(
        adv_positions
        need_pictures
        news_params
        )
    );

    foreach (qw(add_url_parameter doc_source)) {
        $fields->{$_} = TRUE if $self->check_short_rights("edit_field__$_");
    }

    return $fields;
}

sub _is_new_format {
    my ($self, $obj) = @_;

    return TRUE
      if ( $self->hook_stash->inited()
        && ($self->hook_stash->mode('add') || $self->hook_stash->mode('edit'))
        && !$self->hook_stash->get('current')->{is_custom_bk_data}
        && !$obj->{is_custom_bk_data});

    return 0 < grep {exists $obj->{$_} && defined $obj->{$_}} @new_format_fields;
}

TRUE;
