package Application::Model::Role::Block::Has::UnmoderatedDSPs;

use qbit;

use Application::Model::Role;
use Exception::Validator::Fields;
use PiConstants qw($YAN_MANAGER_ROLE_ID);

consume qw(Application::Model::Role::Has::DSPS);

sub get_structure_model_accessors {
    return {
        block_dsps             => 'Application::Model::Block::DSPs',
        block_dsps_unmoderated => 'Application::Model::Block::DSPsUnmoderated',
        dsp                    => 'Application::Model::DSP',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            rights => {
                map {$self->get_description_right($_)}
                  map {("edit_field__$_", "view_field__$_")}
                  qw(
                  all_unmoderated_dsps
                  unmoderated_dsps
                  ),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        all_unmoderated_dsps => {
            api        => 1,
            depends_on => [qw(page_id id unmoderated_rtb_auction block_dsps.dsp_id)],
            get        => sub {
                my ($fields, $obj) = @_;
                my $self = $fields->model;
                my $app  = $self->app;

                # На пейджовой модели должен быть включен unmoderated_rtb_auction
                return undef unless $obj->{unmoderated_rtb_auction};

                my %block_dsps = map {$_->{dsp_id} => TRUE} @{$fields->{block_dsps}{$obj->{page_id}, $obj->{id}} // []};

                my $unmoderated_dsps = [grep {$block_dsps{$_}} @{$fields->{'__DSPS_UNMODERATED__'}}];

                return $unmoderated_dsps;
            },
            label    => d_gettext('All unmoderated DSP list'),
            sub_type => 'string',
            type     => 'array',
        },
        unmoderated_dsps => {
            api        => 1,
            depends_on => [qw(page_id id all_unmoderated_dsps block_dsps_unmoderated.dsp_id)],
            get        => sub {
                my ($fields, $obj) = @_;
                my $self = $fields->model;
                my $app  = $self->app;

                my $all_available_unmoderated_dsps = $obj->{all_unmoderated_dsps};
                return undef unless $all_available_unmoderated_dsps;

                my %block_dsps_unmoderated =
                  map {$_->{dsp_id} => TRUE} @{$fields->{block_dsps_unmoderated}{$obj->{page_id}, $obj->{id}} // []};

                # arrays_intersection нельзя так как all_unmoderated_dsps отсортирован
                my $unmoderated_dsps = [grep {$block_dsps_unmoderated{$_}} @$all_available_unmoderated_dsps];

                return $unmoderated_dsps;
            },
            label      => d_gettext('Unmoderated DSP list'),
            need_check => {
                optional => TRUE,
                type     => 'array',
                all      => {type => 'int_un',},
                check    => sub {
                    my ($qv, $unmoderated_dsps) = @_;
                    my $self = $qv->app;
                    my $app  = $self->app;

                    my $block_dsps = $qv->data->{dsps};
                    my $diff = arrays_difference($unmoderated_dsps, $block_dsps);

                    throw Exception::Validator::Fields
                      sprintf(gettext('Following DSPs are not available: %s'), join(', ', @$diff))
                      if @$diff;

                    # На пейджовой модели должен быть включен unmoderated_rtb_auction
                    throw Exception::Validator::Fields gettext('Page has not flag unmoderated')
                      unless $qv->data->{unmoderated_rtb_auction};

                    # На DSP должен быть флаг немодерируемости
                    my $dsps = $self->dsp->get_unmoderated_dsps($unmoderated_dsps);

                    throw Exception::Validator::Fields sprintf(
                        gettext('Following DSPs are not unmoderated: %s'),
                        join(', ', @{arrays_difference($unmoderated_dsps, $dsps)})
                    ) if (@$dsps != @$unmoderated_dsps);

                    # Не все могут включать галку
                    unless ($self->check_rights('can_turn_on_unmoderated_dsps')) {
                        my $diff =
                          arrays_difference($unmoderated_dsps, $self->hook_stash->get('current')->{unmoderated_dsps});
                        throw Exception::Validator::Fields gettext('Your can\'t turn on unmoderated dsps')
                          if @$diff;
                    }

                    return TRUE;
                },
            },
            sub_type => 'string',
            type     => 'array',
        },
        unmoderated_rtb_auction => {
            depends_on => [qw(page_id pages.unmoderated_rtb_auction)],
            get        => sub {
                my ($fields, $obj) = @_;

                return $fields->{pages}{$obj->{page_id}}{unmoderated_rtb_auction};
            },
            label      => d_gettext('Unmoderated RTB auction'),
            need_check => {
                optional => TRUE,
                skip     => TRUE,
            },
            type => 'boolean',
        },
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('all_unmoderated_dsps')) {
        $fields->{'__DSPS_UNMODERATED__'} = $self->dsp->get_unmoderated_dsps();
    }
}

sub get_available_fields_depends {
    return [qw(unmoderated_rtb_auction)];
}

sub get_editable_fields_depends {
    return [qw(unmoderated_rtb_auction)];
}

sub related_models {
    my ($self) = @_;

    return {
        block_dsps_unmoderated => {
            accessor => 'block_dsps_unmoderated',
            filter   => sub {
                my ($fields, $blocks) = @_;

                return {page_id => array_uniq(map {$_->{page_id}} @{$blocks}),};
            },
            key_fields => [qw(page_id block_id)],
            value_type => 'array',
        },
    };
}

sub get_available_fields {
    my ($self, $obj, $fields) = @_;

    foreach (qw(all_unmoderated_dsps unmoderated_dsps unmoderated_rtb_auction)) {
        if ($self->check_short_rights("view_field__$_") && $obj->{unmoderated_rtb_auction}) {
            $fields->{$_} = TRUE;
        } else {
            delete($fields->{$_});
        }
    }

    return $fields;
}

sub collect_editable_fields {
    my ($self, $obj, $fields) = @_;

    if ($self->check_short_rights('edit_field__unmoderated_dsps') && $obj->{unmoderated_rtb_auction}) {
        $fields->{unmoderated_dsps} = TRUE;
    } else {
        delete $fields->{unmoderated_dsps};
    }
    delete $fields->{qw(all_unmoderated_dsps unmoderated_rtb_auction )};

    return $fields;
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    if ($self->hook_stash->check('fields_from_related_models')
        and my $dsps = $self->hook_stash->get('fields_from_related_models')->{unmoderated_dsps})
    {
        my $page_id_field_name = $self->get_page_id_field_name;
        my %filter;
        @filter{'page_id', 'block_id'} = @{$self->hook_stash->get('id')}{$page_id_field_name, 'id'};

        my $tmp_rights = $self->app->add_tmp_rights('block_dsps_unmoderated_edit');
        $self->block_dsps_unmoderated->replace(\%filter, $dsps, ($self->hook_stash->mode('add') ? 'add' : 'edit'));
    }
}

sub get_need_update_in_bk_fields {
    return {unmoderated_dsps => TRUE};
}

sub get_bk_block_data {
    my ($self, $block, $bk_data) = @_;

    if (defined($block->{unmoderated_dsps}) && @{$block->{unmoderated_dsps}}) {
        my %unmoderated_dsps = map {$_ => TRUE} @{$block->{unmoderated_dsps}};

        foreach my $DSPInfo (@{$bk_data->{DSPInfo}}) {
            $DSPInfo->{PageDspOptions} //= {
                Disable => [],
                Enable  => [],
            };
            if ($unmoderated_dsps{$DSPInfo->{DSPID}}) {
                push @{$DSPInfo->{PageDspOptions}{Enable}}, 'unmoderated-rtb-auction';
            } else {
                push @{$DSPInfo->{PageDspOptions}{Disable}}, 'unmoderated-rtb-auction';
            }
        }

        $bk_data->{'DSPSettings'}{'Unmoderated'} = $block->{unmoderated_dsps};
    }

    return $bk_data;
}

TRUE;
