package Application::Model::Role::Has::Articles;

use qbit;

use Application::Model::Role;
use Partner::BlockPreset::Specification::Articles;

sub get_structure_model_accessors {
    return {
        tns_dict_article => 'Application::Model::TNSDict::Article',
        articles         => 'Application::Model::Product::Articles',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            rights => {
                map {$self->get_description_right($_)}
                  qw(
                  edit_field__articles
                  ),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $articles_specification = Partner::BlockPreset::Specification::Articles->new();
    my $validation_template    = $articles_specification->get_validation_template();

    return {
        articles => {
            depends_on => ['data_articles'],
            label      => d_gettext('Articles'),
            get        => sub {
                to_json($_[1]->{'data_articles'}, canonical => 1),;
            },
            type       => 'string',
            need_check => $validation_template->{'articles'},
            need_trim  => TRUE,
            api        => 1,
        },
        data_articles => {
            label      => d_gettext('Articles data'),
            depends_on => ['page_id', 'id', 'articles.article_sid', 'articles.cpm'],
            get        => sub {
                my ($fields, $row) = @_;
                return [map {+{id => $_->{article_sid} . "", cpm => $_->{cpm} + 0}}
                      @{$fields->{'articles'}{$row->{'page_id'}, $row->{'id'}}}];
            },
            type     => 'complex',
            fix_type => \&fix_type_for_articles,
            api      => 1,
        },
        has_articles_settings => {
            depends_on => ['data_articles'],
            get        => sub {
                return defined($_[1]->{'data_articles'}) && @{$_[1]->{'data_articles'}} > 0;
            },
            type => 'boolean',
            api  => 1,
        },
    };
}

sub get_available_fields {
    my ($self, $obj, $fields) = @_;

    $fields->{$_} = TRUE foreach (qw(articles data_articles has_articles_settings));

    return $fields;
}

sub get_add_fields {
    my ($self, $fields) = @_;

    $fields->{'articles'} = TRUE if $self->check_short_rights('edit_field__articles');

    return $fields;
}

sub get_need_update_in_bk_fields {
    return {articles => TRUE};
}

sub collect_editable_fields {
    my ($self, $obj, $fields) = @_;

    $fields->{'articles'} = TRUE if $self->check_short_rights('edit_field__articles');

    return $fields;
}

sub get_bk_block_data {
    my ($self, $block, $bk_data) = @_;

    $bk_data->{'Article'} = $self->tns_dict_article->get_complete_articles($block->{'data_articles'});

    return $bk_data;
}

sub fix_type_for_articles {
    my ($model, $value) = @_;

    for (@$value) {
        $_->{id} .= '';
        $_->{cpm} += 0;
    }

    return $value;
}

sub related_models {
    return {
        articles => {
            accessor => 'articles',
            filter   => sub {
                return {page_id => array_uniq(map {$_->{'page_id'} // ()} @{$_[1]})};
            },
            key_fields => ['page_id', 'block_id'],
            value_type => 'array',
        },
    };
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    if ($self->hook_stash->check('fields_from_related_models')
        and my $articles = $self->hook_stash->get('fields_from_related_models')->{'articles'})
    {
        my $page_id_field_name = $self->get_page_id_field_name;
        my %filter;
        @filter{'page_id', 'block_id'} = @{$self->hook_stash->get('id')}{$page_id_field_name, 'id'};

        my $tmp_rights = $self->app->add_tmp_rights('articles_edit');

        $self->articles->replace(\%filter, from_json($articles), ($self->hook_stash->mode('add') ? 'add' : 'edit'));
    }
}

TRUE;
