package Application::Model::Role::Has::Block::DesignTemplates;

use qbit;

use Application::Model::Role;
use Exception::Validator::Fields;
use Exception::API::FormatSystem;
use JSON::Schema::Fit;

requires qw(design_field get_bk_direct_design);

use PiConstants qw(
  $ADINSIDE_USER_ID
  $DEFAULT_DESIGN_TYPE
  $DESIGN_TYPES
  $FORMAT_SYSTEM
  $SITE_VERSIONS_TURBO
  );

my $DESIGN_TEMPLATES_LIMITS = {
    $DEFAULT_DESIGN_TYPE    => {min => 1, max => 20,},
    $DESIGN_TYPES->{NATIVE} => {min => 0, max => 20,},
    default                 => {min => 0, max => 5,},
};

my %AVAILABLE_DESIGNES_BY_SITE_VERSION = (
    desktop           => {map {$DESIGN_TYPES->{$_} => TRUE} qw(TGA NATIVE VIDEO MEDIA)},
    mobile            => {map {$DESIGN_TYPES->{$_} => TRUE} qw(TGA NATIVE VIDEO MEDIA)},
    turbo             => {map {$DESIGN_TYPES->{$_} => TRUE} qw(TGA NATIVE VIDEO MEDIA)},
    turbo_desktop     => {map {$DESIGN_TYPES->{$_} => TRUE} qw(TGA NATIVE VIDEO MEDIA)},
    mobile_fullscreen => {map {$DESIGN_TYPES->{$_} => TRUE} qw(TGA VIDEO MEDIA)},
    mobile_rewarded   => {map {$DESIGN_TYPES->{$_} => TRUE} qw(TGA NATIVE VIDEO MEDIA)},
    mobile_floorad    => {map {$DESIGN_TYPES->{$_} => TRUE} qw(TGA NATIVE VIDEO MEDIA)},
    amp               => {map {$DESIGN_TYPES->{$_} => TRUE} qw(TGA MEDIA)},
);

my %NATIVE_DESIGN_BY_FEATURE = (
    desktop       => 'design_auction_native',
    mobile        => 'design_auction_native',
    turbo         => 'design_auction_native_turbo',
    turbo_desktop => 'design_auction_native_turbo',
);

sub get_structure_model_accessors {
    my ($self) = @_;

    return {
        api_format_system => 'Application::Model::API::Yandex::FormatSystem',
        design_templates  => 'Application::Model::DesignTemplates',
        partner_db        => 'Application::Model::PartnerDB',
        product_manager   => 'Application::Model::ProductManager',
        user_features     => 'Application::Model::Users::Features'
    };
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        direct_block => {
            depends_on => ['design_templates'],
            get        => sub {
                my $template = $_[0]->model->get_design_with_min_id($_[1]->{'design_templates'});

                return exists($template->{'design_settings'}{'name'}) ? $template->{'design_settings'}{'name'} : '';
            },
            label => d_gettext('Direct block format'),
        },
        design_templates => {
            depends_on => [
                qw(page_id id horizontal_align),    'design_templates.id',
                'design_templates.design_settings', 'design_templates.caption',
                'design_templates.filter_tags',     'design_templates.is_custom_format_direct',
                'design_templates.type'
            ],
            label => d_gettext('Design templates'),
            get   => sub {
                return $_[0]->{'design_templates'}{$_[1]->{'page_id'}, $_[1]->{'id'}};
            },
            type       => 'array',
            sub_type   => 'design_templates',
            need_check => {
                type  => 'array',
                all   => {type => 'design_template'},
                check => sub {
                    my ($qv, $templates) = @_;

                    throw Exception::Validator::Fields gettext('Block should have at least one design')
                      unless @$templates;

                    my $features_and_options = $qv->get_stash('features_and_options');

                    my %template_by_type;
                    foreach (@$templates) {
                        push @{$template_by_type{$_->{type}}}, $_;
                    }

                    my $site_version = $qv->data->{site_version};
                    if (@{$template_by_type{$DESIGN_TYPES->{NATIVE}} // []}) {
                        if (
                               $site_version
                            && $AVAILABLE_DESIGNES_BY_SITE_VERSION{$site_version}->{$DESIGN_TYPES->{NATIVE}}
                            && (!exists($NATIVE_DESIGN_BY_FEATURE{$site_version})
                                || $features_and_options->{$NATIVE_DESIGN_BY_FEATURE{$site_version}})
                           )
                        {
                            unless (@{$template_by_type{$DEFAULT_DESIGN_TYPE} // []}) {
                                throw Exception::Validator::Fields gettext('Need set at least one %s design',
                                    $DEFAULT_DESIGN_TYPE)
                                  unless $features_and_options->{'allowed_design_auction_native_only'};
                            }
                        } else {
                            throw Exception::Validator::Fields gettext('You can not add native designs');
                        }
                    }
                    return unless $site_version;
                    my $msf_limits = {};
                    try {
                        $msf_limits = $qv->app->api_format_system->get_design_limits(site => $site_version);
                    }
                    catch Exception::API::FormatSystem with {
                        throw Exception::Validator::Fields gettext('Invalid site version: %s', $site_version);
                    };
                    foreach my $key (sort keys %$DESIGN_TYPES) {
                        my $type         = $DESIGN_TYPES->{$key};
                        my $design_count = scalar(@{$template_by_type{$type} // []});
                        my $limits       = $qv->app->_get_design_limits($type, $qv->{data}, $features_and_options);

                        throw Exception::Validator::Fields gettext("Need set at least one %s design", $type)
                          if $design_count < $limits->{min};
                        my $max = $msf_limits->{$type} // $limits->{max};
                        throw Exception::Validator::Fields gettext("Maximum count of %s designs is %s", $type, $max)
                          if $design_count > $max;
                    }
                },
            },
            api => 1,
        }
    };
}

sub get_add_fields {
    my ($self, $fields) = @_;

    $fields->{design_templates} = TRUE;

    return $fields;
}

sub collect_editable_fields {
    my ($self, $data, $fields) = @_;

    $fields->{design_templates} = TRUE;

    return $fields;
}

sub get_bk_block_data {
    my ($self, $block, $bk_data) = @_;

    return $bk_data if $block->{is_custom_bk_data};

    $bk_data->{$self->design_field} = {};

    my $is_interscroller;
    my $is_fullscreen;
    my $has_video;
    my $has_rewarded;
    my $only_rewarded = TRUE;
    foreach my $template (@{$block->{design_templates}}) {
        my $design = $template->{design_settings};
        $design->{blockId} = $self->public_id($block);

        if ($template->{type} eq $DESIGN_TYPES->{NATIVE}) {
            $design->{name} = 'nativeDesign';

            foreach (qw(grid_columns grid_rows)) {
                $design->{template} =~ s/\$\{$_\}/$design->{$_}/g;
            }
            if ($design->{template} =~ /<ya-units-slider/ && $design->{grid_rows} == 1) {
                $template->{type} = 'native-carousel';
            }

            $design->{limit} = (delete $design->{grid_columns}) * (delete $design->{grid_rows});

        }

        if ($template->{type} eq $DEFAULT_DESIGN_TYPE) {
            if ($design->{name} eq 'rewarded' or $design->{rewarded}) {
                $has_rewarded = TRUE;
            } else {
                $only_rewarded = FALSE;
            }
        }
        $bk_data->{$self->design_field}->{$template->{id}} = +{
            name   => $template->{caption},
            design => $design,
            type   => $template->{type},
        };

        $is_interscroller = 1 if !$is_interscroller && exists($design->{interscroller}) && $design->{interscroller};
        $is_fullscreen    = 1 if !$is_fullscreen    && exists($design->{fullscreen})    && $design->{fullscreen};
        $has_video = 1 if $template->{type} eq $DESIGN_TYPES->{VIDEO};
    }

    if ($has_video) {
        my $video = $bk_data->{Video} //= {};
        $video->{API}            = [];
        $video->{CategoryID}     = 0;
        $video->{CountPositions} = 1;
        $video->{Protocols}      = [];
        $video->{Type}           = 'inpage';
        push @{$bk_data->{'PageImpOptions'}{'Disable'}}, 'allow-multiple-dsp-ads';
        $bk_data->{'AdTypeSet'}{'video-motion'} = 0;
    }

    if ($has_rewarded) {
        for my $t (qw(video video-performance)) {
            $bk_data->{'AdTypeSet'}{$t} = 1;
        }
        if ($only_rewarded) {
            $bk_data->{'AdTypeSet'}{text} = 0;
        }
    }
    return $bk_data;
}

sub get_available_design_template_types {
    my ($self, $block) = @_;

    my @result = ($DESIGN_TYPES->{TGA}, $DESIGN_TYPES->{MEDIA});

    if ($block->{site_version} && $AVAILABLE_DESIGNES_BY_SITE_VERSION{$block->{site_version}}) {

        my $designes             = $AVAILABLE_DESIGNES_BY_SITE_VERSION{$block->{site_version}};
        my $page_id              = $block->{page_id} // $block->{$self->get_page_id_field_name()};
        my $features_and_options = $self->get_features_and_options_on_page($page_id);

        push @result, $DESIGN_TYPES->{VIDEO}
          if $designes->{$DESIGN_TYPES->{VIDEO}} && $block->{show_video};
        push @result, $DESIGN_TYPES->{NATIVE}
          if $designes->{$DESIGN_TYPES->{NATIVE}}
              && $features_and_options->{$NATIVE_DESIGN_BY_FEATURE{$block->{site_version}}};
    }

    return \@result;
}

sub get_fields_defaults {
    my ($self, $opts) = @_;

    my %need_fields = map {$_ => TRUE} @{$opts->{fields}};

    my $result = {};

    if ($need_fields{available_design_template_types}) {
        $result->{available_design_template_types} = $self->get_available_design_template_types($opts->{attributes});
    }

    return $result;
}

sub adjust_format_system_designs_before_validation {
    my ($self, $opts) = @_;

    my %designs;
    my $block_settings = $self->hook_stash->get('settings');

    if (exists($opts->{design_templates})
        && defined($opts->{design_templates}))
    {

        foreach my $template (@{$opts->{design_templates}}) {
            next unless $template->{type} eq $DEFAULT_DESIGN_TYPE;
            push(@{$designs{$template->{design_settings}{name}}}, $template);
        }
    }

    foreach my $design_name (keys %designs) {
        my $format;
        if (
            eval {
                $format = $self->api_format_system->formats(
                    format => $design_name,
                    role   => 'manager',
                    site   => $FORMAT_SYSTEM->{site_versions}->{$block_settings->{site_version}}
                      // $block_settings->{site_version},
                );
            }
           )
        {

            my $json_schema = {
                type       => 'object',
                properties => {},
            };

            for my $field (@{$format->{settings}}) {
                $json_schema->{properties}{$field->{name}} = {type => $field->{jsType}};
            }

            foreach my $design (@{$designs{$design_name}}) {
                $design->{design_settings} =
                  JSON::Schema::Fit->new()->get_adjusted($design->{design_settings}, $json_schema);
            }
        }
    }
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    my $settings = $self->hook_stash->get('settings');

    return if $settings->{is_custom_bk_data};

    map {$_->{type} //= $DEFAULT_DESIGN_TYPE} @{$opts->{'design_templates'}} if defined $opts->{'design_templates'};
    my $current = $self->hook_stash->get('current');

    if (defined $opts->{show_video}) {
        if (!$opts->{show_video} && $current->{show_video}) {
            @{$opts->{'design_templates'}} =
              grep {$_->{type} ne $DESIGN_TYPES->{VIDEO}} @{$settings->{'design_templates'}};
        } elsif (
            $opts->{show_video} && !$current->{show_video} && !(
                grep {
                    $_->{type} eq $DESIGN_TYPES->{VIDEO}
                } @{$settings->{'design_templates'}}
            )
          )
        {
            $opts->{'design_templates'} =
              [@{$settings->{'design_templates'}}, $self->get_default_video_design($settings)];
        }

    }

    if (defined $opts->{'design_templates'} && grep {$_->{type} eq $DEFAULT_DESIGN_TYPE} @{$opts->{'design_templates'}})
    {
        $self->adjust_format_system_designs_before_validation($opts);
    }

    if (   $self->hook_stash->mode('add')
        && $opts->{show_video})
    {
        my @video_designs = grep {$_->{type} eq $DESIGN_TYPES->{VIDEO}} @{$opts->{design_templates}};
        push @{$opts->{design_templates}}, $self->get_default_video_design($opts)
          unless (@video_designs);
    }
}

sub get_bk_data_design_ids {
    my ($self, $bk_data_structure) = @_;

    my $rtb_design = $bk_data_structure->{$self->design_field()};

    return () unless (defined $rtb_design
        && ref($rtb_design) eq 'HASH');

    return grep {$_ ne '0'} keys %$rtb_design;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    #При размножении приходит массив дизайнов с id от другого блока. Приходится их затирать
    foreach my $template (@{$opts->{design_templates}}) {
        delete $template->{'id'}
          if defined($template->{'block_id'}) && $template->{'block_id'} > 0;
    }
}

sub related_models {
    my ($self) = @_;

    return {
        design_templates => {
            accessor => 'design_templates',
            filter   => sub {
                +{page_id => array_uniq(map {$_->{'page_id'}} @{$_[1]}), multistate => 'not deleted'};
            },
            key_fields => ['page_id', 'block_id'],
            value_type => 'array',
        },
    };
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    return {design_templates => TRUE};
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    if ($self->hook_stash->check('fields_from_related_models')
        and my $design_templates = $self->hook_stash->get('fields_from_related_models')->{'design_templates'})
    {
        my @saved_templates_ids;
        my $page_id_field_name = $self->get_page_id_field_name;
        my $id                 = $self->hook_stash->get('id');

        my $settings = {%{$self->hook_stash->get('settings')}, page_id => $id->{$page_id_field_name}};

        my $current = $self->hook_stash->get('current');
        foreach my $template (@{$design_templates}) {
            $template->{block_id} = $id->{'id'};
            $template->{page_id}  = $id->{$page_id_field_name};

            my $design_template = clone $template;
            delete $design_template->{id};
            foreach (qw(is_custom_format_direct filter_tags)) {
                delete $design_template->{$_} unless defined $design_template->{$_};
            }
            if (defined $template->{id}) {
                push @saved_templates_ids, $template->{id};
                my $tmp_rights = $self->app->add_tmp_rights('do_design_templates_edit');
                $self->design_templates->do_action($template->{id}, 'edit', %$design_template);
            } else {
                my $tmp_rights = $self->app->add_tmp_rights('do_design_templates_add');
                my $templ_id   = $self->design_templates->add(%$design_template);
                push @saved_templates_ids, $templ_id;
            }
        }
        my $old_templates = $self->design_templates->get_all(
            fields => [qw(id)],
            filter => [
                'AND' => [
                    [page_id    => '='      => $id->{$page_id_field_name}],
                    [block_id   => '='      => $id->{id}],
                    [multistate => '='      => 'not deleted'],
                    [id         => 'NOT IN' => \@saved_templates_ids]
                ]
            ]
        );
        my $tmp_rights = $self->app->add_tmp_rights('do_design_templates_delete');
        foreach my $templ (@$old_templates) {
            $self->design_templates->do_action($templ, "delete");
        }
    }

}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    # T.к. bk_data отсутствует в add_fields при добавлении сразу выходим,
    # а при дублировании меняем id дизайнов в bk_data
    return unless $self->DOES('Application::Model::Role::Has::CustomBkData') && defined($opts->{bk_data});

    my @bk_design_ids = $self->get_bk_data_design_ids(from_json($opts->{bk_data}));

    my $map_bk_designs = {map {$_ => $_} @bk_design_ids};

    my $id = $self->hook_stash->get('id');

    my $page_id_field_name = $self->get_page_id_field_name;
    my $templates          = $self->design_templates->get_all(
        fields => [qw(id)],
        filter => [
            'AND' => [
                [page_id    => '=' => $id->{$page_id_field_name}],
                [block_id   => '=' => $id->{id}],
                [multistate => '=' => 'not deleted'],
            ]
        ],
        order_by => [qw(id)]
    );

    foreach (sort {$a <=> $b} keys %$map_bk_designs) {
        my $design = shift @$templates;
        $map_bk_designs->{$_} = $design->{id} // $map_bk_designs->{$_};
    }
    my $bk_data = from_json($opts->{'bk_data'});
    foreach my $key (keys %{$map_bk_designs}) {
        $bk_data->{'RtbDesign'}->{$map_bk_designs->{$key}} = delete $bk_data->{'RtbDesign'}->{$key}
          if defined $map_bk_designs->{$key};
    }
    $opts->{'bk_data'} = to_json($bk_data, pretty => TRUE);
    $self->partner_db_table()->edit($id, {bk_data => $opts->{'bk_data'}});

}

sub _get_design_limits {
    my ($self, $type, $opts, $features_and_options) = @_;

    if ($type eq $DESIGN_TYPES->{VIDEO}) {
        return $opts->{show_video} ? {min => 1, max => 1} : {min => 0, max => 0};
    }
    if ($type eq $DESIGN_TYPES->{MEDIA}) {
        return {min => 1, max => 1};
    }

    my $limits = $DESIGN_TEMPLATES_LIMITS->{$type} // $DESIGN_TEMPLATES_LIMITS->{default};
    if ($type eq $DEFAULT_DESIGN_TYPE && $features_and_options->{'allowed_design_auction_native_only'}) {
        $limits = clone($limits);
        $limits->{min} = 0;
    }

    return $limits;
}

sub fix_design_template {
    my ($self, $qv, $fields) = @_;

    $fields->{'horizontal_align'} = {optional => TRUE, type => 'boolean'};
}

sub get_default_video_design {
    my ($self, $opts) = @_;
    return {
        caption         => gettext('default video design'),
        type            => $DESIGN_TYPES->{VIDEO},
        page_id         => $opts->{page_id},
        design_settings => {name => 'inpage',}
    };
}

sub get_design_with_min_id {
    my ($self, $templates) = @_;

    my $template = undef;
    foreach (grep {$_->{type} eq $DEFAULT_DESIGN_TYPE} @$templates) {
        if (   !defined($template)
            || (!exists($template->{id}) && exists($_->{id}))
            || (defined($template->{id}) && defined($_->{id}) && $template->{id} > $_->{id}))
        {
            $template = $_;
        }
    }

    return $template;
}

sub fix_template {
    my ($self, $qv) = @_;

    $qv->set_stash('features_and_options', $self->get_features_and_options_on_page($qv->data->{'page_id'}));
}

TRUE;
