package Application::Model::Role::Has::CustomBKOptions;

use qbit;

use Application::Model::Role;
use Exception::Validator::Fields;

my $FIELD_NAME    = 'custom_bk_options';
my $EDIT_RIGHT    = 'edit_field__' . $FIELD_NAME;
my $VIEW_RIGHT    = 'view_field__' . $FIELD_NAME;
my $BK_NAME       = 'CustomBlockData';
my $DEFAULT_VALUE = [];

sub get_structure_model_accessors {
    return {custom_bk_options => 'Application::Model::CustomBKOptions',};
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [{rights => {map {$self->get_description_right($_)} $EDIT_RIGHT, $VIEW_RIGHT,}}];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        $FIELD_NAME => {
            api        => 1,
            from_opts  => 'from_hash',
            sub_type   => 'number',
            type       => 'array',
            need_check => {
                all   => {type => 'int_un'},
                check => sub {
                    my ($qv, $value) = @_;

                    my $model   = $qv->app;
                    my $options = $model->custom_bk_options->get_all(
                        fields => [qw(id bk_name)],
                        filter => {
                            id         => $value,
                            multistate => 'not deleted',
                        },

                    );
                    throw Exception::Validator::Fields gettext('Some options are unavailable')
                      if scalar(@$options) != scalar(@$value);

                    my @names = sort {length($a) <=> length($b)} map {$_->{bk_name} . '.'} @$options;
                    for (my $i = 0; $i < $#names; $i++) {
                        for (my $j = $i + 1; $j < scalar @names; $j++) {
                            throw Exception::Validator::Fields gettext('Some options have conflict changes')
                              if $names[$i] eq substr($names[$j], 0, length($names[$i]));
                        }
                    }

                },
                optional => TRUE,
                type     => 'array',
            },
        },
    };
}

sub get_available_fields {
    my ($self, $obj, $fields) = @_;

    $self->app->delete_field_by_rights($fields, {$self->get_right($VIEW_RIGHT) => $FIELD_NAME,},);

    return $fields;
}

sub get_add_fields {
    my ($self, $fields) = @_;

    $fields->{$FIELD_NAME} = TRUE if $self->check_short_rights($EDIT_RIGHT);

    return $fields;
}

sub collect_editable_fields {
    my ($self, $data, $fields) = @_;

    $fields->{$FIELD_NAME} = TRUE if $self->check_short_rights($EDIT_RIGHT);

    return $fields;
}

sub get_need_update_in_bk_fields {
    return {$FIELD_NAME => TRUE};
}

sub get_bk_block_data {
    my ($self, $block, $data) = @_;

    my $h;
    my $options = $self->custom_bk_options->get_all(
        fields => [qw(bk_name bk_value)],
        filter => {
            id         => $block->{$FIELD_NAME},
            multistate => 'not deleted',
        },
    );
    $h = $data->{$BK_NAME} //= {} if @$options;
    foreach my $opt (@$options) {
        my $value = $opt->{bk_value};
        my $json_value = eval {from_json $value};
        $value = $json_value unless $@;

        my $path  = $opt->{bk_name};
        my @path  = split /\./, $path;
        my $point = $h;
        for (my $i = 0; $i < $#path; $i++) {
            $point = $point->{$path[$i]} //= {};
        }

        $point->{$path[-1]} = $value;
    }

    return $data;
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $opts->{$FIELD_NAME} //= $DEFAULT_VALUE;
}

sub get_fields_defaults {
    my ($self, $opts) = @_;

    my %need_fields = map {$_ => TRUE} @{$opts->{fields}};

    my $result = {};

    if ($need_fields{$FIELD_NAME}) {
        my $options = $self->custom_bk_options->get_all(
            fields => [qw(id name description)],
            filter => {multistate => 'not deleted',},
        );
        $result->{$FIELD_NAME} = $options;
    }

    return $result;
}

1;
