package Application::Model::Role::Has::DSPSWithoutMode;

use qbit;

use Application::Model::Role;

use PiConstants qw(@VIDEO_AWAPS_DSPS $MIN_CPM_STRATEGY_ID $SEPARATE_CPM_STRATEGY_ID);

use Exception::Validator::Fields;

# TODO: get rid of this right and sub
requires qw(get_dsp_type get_dsp_rules);

sub generate_links_from_dsp_list {
    my ($self, $dsp_list) = @_;

    return $dsp_list;
}

sub get_structure_model_accessors {
    return {
        #TODO: accessor for dsps
        dsp                    => 'Application::Model::DSP',
        block_dsps             => 'Application::Model::Block::DSPs',
        block_dsps_unmoderated => 'Application::Model::Block::DSPsUnmoderated',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            rights => {
                map {$self->get_description_right($_)}
                  qw(
                  edit_field__dsps
                  view_field__dsps
                  ),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {dsps => $self->get_dsps_field_definition()};
}

sub related_models {
    my ($self) = @_;

    return {
        block_dsps => {
            accessor => 'block_dsps',
            filter   => sub {
                +{page_id => array_uniq(map {$_->{'page_id'}} @{$_[1]}), is_deleted => 0};
            },
            key_fields => ['page_id', 'block_id'],
            value_type => 'array',
        },
    };
}

sub get_available_fields {
    my ($self, $obj, $fields) = @_;

    if ($self->check_short_rights('view_field__dsps')) {
        $fields->{'dsps'} = TRUE;
    } else {
        delete($fields->{'dsps'});
    }

    return $fields;
}

sub get_add_fields {
    my ($self, $fields) = @_;

    if ($self->check_short_rights('edit_field__dsps')) {
        $fields->{'dsps'} = TRUE;
    }

    return $fields;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    my $rules = $self->get_dsp_rule_set();

    if ($self->hook_stash->mode('add')) {
        my $deps = $rules->build_dependencies('default_dsps');

        # Do we need these?
        my %defaults = (
            adfox_block => 0,
            show_video  => 0,
        );

        my %dsps_opts = map {$_ => ($opts->{$_} // $defaults{$_})} @{$deps->{block}};

        unless ($self->check_short_rights('edit_field__dsps') && exists($opts->{dsps}) && defined($opts->{dsps})) {
            $opts->{'dsps'} =
              $self->generate_links_from_dsp_list([map {$_->{'id'}} @{$self->get_default_dsps(\%dsps_opts)}]);
        }
    }

    if ($self->hook_stash->mode('edit')) {
        my $current = $self->hook_stash->get('current');

        $rules->edit_dsp_list($current, $opts);
    }
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    if ($self->hook_stash->check('fields_from_related_models')
        and my $dsps = $self->hook_stash->get('fields_from_related_models')->{'dsps'})
    {
        my $page_id_field_name = $self->get_page_id_field_name;
        my %filter;
        @filter{'page_id', 'block_id'} = @{$self->hook_stash->get('id')}{$page_id_field_name, 'id'};

        my $tmp_rights = $self->app->add_tmp_rights('block_dsps_edit');

        $self->block_dsps->replace(\%filter, $dsps, ($self->hook_stash->mode('add') ? 'add' : 'edit'));
    }
}

sub get_need_update_in_bk_fields {
    return {dsps => TRUE};
}

sub get_bk_block_data {
    my ($self, $block, $bk_data) = @_;

    my $mincpm = $block->{'mincpm'} || 0;
    $mincpm *= 1000 if grep {$_ eq $block->{strategy}} ($MIN_CPM_STRATEGY_ID, $SEPARATE_CPM_STRATEGY_ID);

    $bk_data->{'DSPInfo'} = [map {{DSPID => $_, CPM => $mincpm + 0}} @{$block->{'dsps'}}];

    return $bk_data;
}

sub collect_editable_fields {
    my ($self, $obj, $fields) = @_;

    if ($self->check_short_rights('edit_field__dsps')) {
        $fields->{'dsps'} = TRUE;
    }

    return $fields;
}

sub get_dsp_rule_set {
    my ($self) = @_;

    return Partner::DSP::RuleSet->new(rules => $self->get_dsp_rules());
}

sub _get_dsps {
    my ($self, $block, $method, $rules) = @_;

    $block = clone($block);

    my $tmp_rights = $self->app->add_tmp_rights(
        qw(
          dsp_view_all
          dsp_view_field__id
          dsp_view_field__short_caption
          dsp_view_field__unmoderated_rtb_auction
          )
    );

    my $page_id = $block->{page_id};

    my $deps = $rules->build_dependencies($method);

    my @missing_fields = grep {!exists($block->{$_}) || !defined($block->{$_})} @{$deps->{block} // []};
    throw Exception::Validation::BadArguments gettext('Expected "%s"', join(', ', @missing_fields))
      if @missing_fields;

    my $page;
    if ($page_id && @{$deps->{page} // []}) {
        $page = $self->page->get_all(
            filter => {'page_id' => $page_id},
            fields => ['page_id', @{$deps->{page}}],
        )->[0] // throw Exception::Validation::BadArguments gettext("Page with page_id %s not found", $page_id);
    }

    my $filter = $rules->build_filter($method, $block, $page);

    return [] if $filter->is_zero();

    my $res = $self->dsp->get_all(
        fields => [qw(id short_caption types formats unmoderated_rtb_auction)],
        $filter->is_identity() ? () : (filter => $filter->to_model_filter()),
        order_by => ['short_caption']
    );

    foreach my $dsp (@$res) {
        delete $dsp->{unmoderated_rtb_auction} unless $dsp->{unmoderated_rtb_auction};
    }

    return $res;
}

sub get_available_dsps {
    my ($self, $block, $rules) = @_;

    $rules //= $self->get_dsp_rule_set();

    return $self->_get_dsps($block, 'available_dsps', $rules);
}

sub get_default_dsps {
    my ($self, $block, $rules) = @_;

    $rules //= $self->get_dsp_rule_set();

    return $self->_get_dsps($block, 'default_dsps', $rules);
}

sub validate_dsp_list {
    my ($self, $block, $dsps, $rules) = @_;

    my $page_id_field_name = $self->get_page_id_field_name();
    $block = clone($block);
    $block->{page_id} //= $block->{$page_id_field_name};

    $rules //= $self->get_dsp_rule_set();

    my %dsp_list =
      map {$_->{'id'} => TRUE} @{$self->get_available_dsps($block, $rules)};

    my %duplicate = ();
    my @not_found = ();
    foreach my $dsp_id (@$dsps) {
        $duplicate{$dsp_id}++;

        push(@not_found, $dsp_id) unless $dsp_list{$dsp_id};
    }

    my @duplicate = grep {$duplicate{$_} > 1} keys(%duplicate);

    throw Exception::Validator::Fields gettext('DSP\'s with ID %s duplicate', join(', ', @duplicate))
      if @duplicate;

    throw Exception::Validator::Fields gettext('DSP\'s with ID %s not found', join(', ', @not_found))
      if @not_found;

    $rules->validate_dsp_list($block, $dsps);
}

sub turn_on_dsp {
    my ($self, $dsp) = @_;

    my $table = $self->block_dsps->db_table_name;
    my %fields = map {$_->{name} => \undef} @{$self->partner_db->$table->fields};
    $fields{block_id}   = 'id';
    $fields{dsp_id}     = \$dsp->{id};
    $fields{is_deleted} = \0;

    my $page_id_name = $self->get_page_id_field_name;
    $fields{page_id} = $page_id_name;

    my $rules = $self->get_dsp_rule_set();
    my $filter = $rules->build_filter('turn_on_dsp', $dsp);

    return if $filter->is_zero();

    my $query = $self->partner_db->query->select(
        table => $self->query(
            fields => $self->_get_fields_obj([$page_id_name, 'id']),
            $filter->is_identity() ? () : (filter => $self->get_db_filter($filter->to_model_filter())),
        ),
        fields => \%fields
    );

    $self->block_dsps->partner_db_table->add_multi([$query], replace => TRUE);

    my @need_update =
      map {$_->{page_id}}
      @{$self->block_dsps->get_all(fields => ['page_id'], filter => {dsp_id => $dsp->{id}}, distinct => TRUE)};

    $self->all_pages->mark_pages_for_async_update(page_ids => array_uniq(@need_update), use_separate_queue => TRUE)
      if @need_update;
}

sub stop_dsp_on_mobile_block {
    my ($self, $dsp, $block_type) = @_;

    my $page_id_name = $self->get_page_id_field_name;

    my $filter = $self->partner_db->filter(
        [
            AND => [
                ['dsp_id', '=', \$dsp->{id}],
                [
                    {'', ['page_id', 'block_id']},
                    'IN',
                    $self->query(
                        fields => $self->_get_fields_obj([$page_id_name, 'id']),
                        filter => [block_type => 'IN' => \$block_type],
                    )
                ]
            ]
        ]
    );
    my @need_update =
      map {$_->{page_id}} @{$self->block_dsps->get_all(fields => ['page_id'], filter => $filter, distinct => TRUE)};
    $self->block_dsps->partner_db_table()->delete($filter);
    $self->block_dsps_unmoderated->partner_db_table()->delete($filter);

    $self->all_pages->mark_pages_for_async_update(page_ids => \@need_update, use_separate_queue => TRUE)
      if @need_update;
}

sub set_dsps_fields_defaults {
    my ($self, $object, $defaults) = @_;

    my $rules = $self->get_dsp_rule_set();
    $defaults->{'dsps_available'} = $self->get_available_dsps($object, $rules);
    $defaults->{'dsps_default'} = $self->get_default_dsps($object, $rules);
}

sub set_dsps_fields_depends {
    my ($self, $depends) = @_;

    my $rules    = $self->get_dsp_rule_set();
    my @required = @{
        array_uniq(
            @{$rules->build_dependencies('default_dsps')->{block}   // []},
            @{$rules->build_dependencies('available_dsps')->{block} // []},
        )
      };
    $depends->{required}{dsps} = \@required;
    for my $required_field (@required) {
        $depends->{depends}{$required_field} //= [];
        push @{$depends->{depends}{$required_field}}, 'dsps';
    }

}

sub get_dsps_field_definition {
    my ($self) = @_;

    return {
        depends_on => [qw(page_id id), 'block_dsps.dsp_id'],
        label      => d_gettext('DSP list'),
        get        => sub {
            [map {$_->{'dsp_id'}} @{$_[0]->{'block_dsps'}{$_[1]->{'page_id'}, $_[1]->{'id'}} // []}];
        },
        type       => 'array',
        sub_type   => 'string',
        need_check => {
            type  => 'array',
            all   => {type => 'int_un',},
            check => sub {
                my ($qv, $dsps) = @_;

                $qv->app->validate_dsp_list($qv->data, $dsps);
            },
        },
        api         => 1,
        adjust_type => 'array_str',
    };
}

TRUE;
