package Application::Model::Role::Has::FieldsModeration;

use qbit;

use Application::Model::Role;
use Utils::Logger qw (WARN);

use PiConstants qw(
  $MYSQL_MIN_DATETIME
  $MYSQL_MAX_DATETIME
  );

sub get_field_name_approved {
    my ($self, $field_name) = @_;

    return $field_name . '_approved';
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            rights => {
                map {$self->get_description_right("edit_field__$_")} qw(moderation waiting_moderation),
                map {$self->get_field_name_approved($_)} keys %{$self->get_fields_moderated()}
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $moderated_fields = $self->get_fields_moderated();

    return {
        waiting_moderation => {
            db    => TRUE,
            type  => 'date',
            label => d_gettext('Waiting moderation'),
        },
        moderation => {
            from_opts  => 'from_hash',
            type       => 'complex',
            need_check => {
                optional => TRUE,
                type     => 'hash',
                fields   => {map {$_ => {type => 'hash', extra => 1,},} keys %$moderated_fields},
            },
        },
        (
            map {
                $self->get_field_name_approved($_) => {
                    from_opts  => 'from_hash',
                    need_check => {skip => TRUE, optional => TRUE,},
                    (
                        $moderated_fields->{$_} eq 'ARRAY'
                        ? (
                            sub_type => 'number',
                            type     => 'array',
                          )
                        : (type => 'string')
                    ),
                  },
              } keys %$moderated_fields
        ),
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {fields => {waiting_moderation => {type => 'date', label => d_gettext('Waiting moderation')},},};
}

sub get_actions_depends {
    my ($self) = @_;
    return [(keys %{$self->get_fields_moderated()}), 'moderation'];
}

sub get_add_fields {
    my ($self, $fields) = @_;

    $fields->{moderation} = TRUE if $self->check_short_rights('edit_field__moderation');

    return $fields;
}

sub collect_editable_fields {
    my ($self, $obj, $fields) = @_;

    foreach (qw(moderation waiting_moderation),
        map {$_, $self->get_field_name_approved($_)} keys %{$self->get_fields_moderated()})
    {
        $fields->{$_} = TRUE if $self->check_short_rights("edit_field__$_");
    }

    return $fields;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    my $current          = $self->hook_stash->get('current');
    my $moderated_fields = $self->get_fields_moderated();
    if ($self->hook_stash->mode('add')) {
        # При добавлении для всех моерируемых полей заводим структуру
        my $moderation = $opts->{moderation} = {};
        foreach my $field_name (keys %$moderated_fields) {
            my @values_list = (
                $current->{$field_name}
                ? (
                    # Если модерируемое поле массив
                    $moderated_fields->{$field_name} eq 'ARRAY'
                      # ... используем все его значения
                    ? @{$current->{$field_name}}
                      # ... иначе одно значение поля
                    : $current->{$field_name}
                  )
                : ()
            );
            $moderation->{$field_name} = {
                map {
                    # нулевое значение verdict - требуется модерация
                    # пустое значение request_id - требуется отправка на модерацию
                    $_ => {request_id => '', verdict => 0}
                  } @values_list
            };
        }
    } elsif ($self->hook_stash->mode('edit')) {
        # При редактировании в структуре обнуляем модерацию только для изменившихся полей
        my $moderation = $opts->{moderation} // $current->{moderation};
        my $set_need_approve = FALSE;    # Флаг того, что что-то изменилось
        foreach my $field_name (keys %$moderated_fields) {
            my $field               = $current->{$field_name};
            my $field_name_approved = $self->get_field_name_approved($field_name);
            # Проверяем только если поле пришло на редактирование
            if (exists $opts->{$field_name}) {
                my $old_moderation = $moderation->{$field_name};
                my $approved_values;
                my $new_values;
                if ($moderated_fields->{$field_name} eq 'ARRAY') {
                    $approved_values = $current->{$field_name_approved} // [];
                    $new_values = $opts->{$field_name};
                } else {
                    $approved_values = (
                        $current->{$field_name_approved}
                        ? [$current->{$field_name_approved}]
                        : []
                    );
                    $new_values = [$opts->{$field_name}];
                }
                my $new_moderation = $moderation->{$field_name} = {
                    map {
                        # Сохраняем результаты последней успешной модерации
                        # А если её по какой-то причине нет, то генерим фиктивную положительную
                        $_ => $old_moderation->{$_} // {request_id => '', verdict => 1}
                      } @{$approved_values}
                };
                foreach (@{$new_values}) {
                    unless ($new_moderation->{$_}) {
                        if ($old_moderation->{$_}) {
                            # Если значение уже было учтено, используем эти данные
                            $new_moderation->{$_} = $old_moderation->{$_};
                        } else {
                            # Для тех значений, для которых ещё нет заявок или результатов
                            $new_moderation->{$_} = {request_id => '', verdict => 0};
                            $set_need_approve = TRUE;
                        }
                    }
                }
            }
        }

        # обновляем инфу о модерации
        $opts->{moderation} = $moderation;

        # При необходимости делаем пометку, что потом надо переотправить на модерацию
        $self->hook_stash->set(set_need_approve => $set_need_approve);
    }
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    # a case when we can approve/reject without sending to moderation
    # may happen with indoor block, where only photos are moderated and photos are not mandatory
    if (exists $opts->{moderation}) {
        $self->_try_moderate_in_place($opts->{moderation});
    }
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    if ($self->hook_stash->get('set_need_approve')) {
        my $obj = $self->_get_object_fields($self->hook_stash->get('current'), ['multistate']);
        my $multistate = $obj->{multistate};
        unless ($self->check_multistate_flag($multistate, 'testing')) {
            my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('set_need_approve'));
            $self->do_action($self->hook_stash->get('id'), 'set_need_approve');
        }
    } elsif (exists $opts->{moderation}) {
        # a case when we can approve/reject without sending to moderation
        # happens when partner change field to a value that has been moderated before and has a verdict
        $self->_try_moderate_in_place($opts->{moderation});
    }
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $opts->{waiting_moderation} = $MYSQL_MIN_DATETIME;

    # this hook is called on add/duplicate
    # FIXME: there is no mode='duplicate', only 'add';
    # this is needed when block is duplicated, but when block is added this is pointless
    for my $field_name (keys %{$opts->{'moderation'}}) {
        for my $request_verdict (values %{$opts->{'moderation'}{$field_name}}) {
            # reset request_id on values waiting moderation
            # so they will be send separately with own request_id
            if (0 == $request_verdict->{verdict}) {
                $request_verdict->{request_id} = '';
            }
            # dont touch -1|1 verdicts: they were already moderated, no need to send again
        }
    }
}

sub hook_stash_edit_fields {
    my ($self, @data) = @_;

    pop @data;
    my %data       = @data;
    my $field_name = 'set_need_approve';
    push @{$data{free}}, $field_name unless grep {$_ eq $field_name} @{$data{free} // []};

    return %data;
}

sub _aggregate_moderation_verdicts {
    my ($self, $obj) = @_;

    my $moderated_fields = $self->get_fields_moderated();
    my $fields           = $self->_get_object_fields($obj, [qw(moderation), keys %{$moderated_fields}]);
    my $moderation       = $fields->{moderation};
    my @result = (0, 0, 0);    # need_approve, approved, rejected

    foreach my $field_name (keys %$moderated_fields) {
        my $field_moderation = $moderation->{$field_name};
        my $field_values =
          ($moderated_fields->{$field_name} eq 'ARRAY' ? $fields->{$field_name} : [$fields->{$field_name}]);
        for (my $i = 0; $i < @$field_values; $i++) {
            my $value_moderation = $field_moderation->{$field_values->[$i]};
            $result[$value_moderation->{verdict} // 0]++;
        }
    }

    return \@result;
}

sub can_action_approve {
    my ($self, $obj) = @_;

    my $moderation = $self->_aggregate_moderation_verdicts($obj);

    # Нет незавершённых модераций и нет отклонённых
    return !$moderation->[0] && !$moderation->[-1] ? TRUE : FALSE;
}

sub on_action_approve {
    my ($self, $obj, %opts) = @_;

    $self->maybe_do_action($obj, 'start', %opts);
    $self->partner_db_table()->edit($obj, {'waiting_moderation' => $MYSQL_MAX_DATETIME});
    my $tmp_rights = $self->app->add_tmp_rights($self->get_rights_by_actions('set_need_update'));
    $self->do_action($obj, 'set_need_update');

    return TRUE;
}

sub can_action_reject {
    my ($self, $obj) = @_;

    my $moderation = $self->_aggregate_moderation_verdicts($obj);

    # Нет незавершённых модераций
    return !$moderation->[0];
}

sub on_action_reject {
    my ($self, $obj, %opts) = @_;

    # Останавливать и отправлять в БК не надо, так как блок ещё ни разу не был заппрувлен
    $self->partner_db_table()->edit($obj, {'waiting_moderation' => $MYSQL_MAX_DATETIME});

    return TRUE;
}

sub on_action_set_blocked {
    my ($self, $obj, %opts) = @_;

    $self->maybe_do_action($obj, 'stop', %opts);    # остановить, если ещё не остановлен
}

sub on_action_set_need_approve {
    my ($self, $obj, %opts) = @_;

    $self->partner_db_table()->edit($obj, {waiting_moderation => $MYSQL_MIN_DATETIME});

    return TRUE;
}

sub on_moderation_request_sent {
    my ($self, $obj_list, $converted_data) = @_;

    my %objects;
    foreach my $obj (@$obj_list) {
        $objects{$obj->{page_id}, $obj->{id}} = $obj;
    }
    my $moderated_fields = $self->get_fields_moderated();
    foreach my $r (@$converted_data) {
        my ($page_id, $block_id, $fields, $position) = @{$r->{extra}}{qw(page_id id fields position)};
        my $request_id = $r->{meta}{request_id};
        my $obj = $objects{$page_id, $block_id};
        foreach my $field_name (@$fields) {
            my $field_value =
              ($moderated_fields->{$field_name} eq 'ARRAY' ? $obj->{$field_name}[$position] : $obj->{$field_name});
            $obj->{moderation}{$field_name}{$field_value} = {
                request_id => $request_id,
                verdict    => 0,
            };
        }
    }
    my $curdate = curdate(oformat => 'db_time');
    foreach my $obj (@$obj_list) {
        $self->do_action($obj, 'edit', moderation => $obj->{moderation}, waiting_moderation => $curdate,);
    }

    return TRUE;
}

sub _apply_verdict {
    my ($self, $obj, $moderation_data) = @_;

    my $moderated_fields = $self->get_fields_moderated();
    my $moderation       = $obj->{moderation};
    my $request_id       = $moderation_data->{request_id};
    my $verdict          = $moderation_data->{verdict};
    unless (defined($request_id) && defined($verdict)) {
        WARN {
            message => 'Incorrect moderation data',
            extra   => {
                block           => $obj,
                moderation_data => $moderation_data,
            },
        };
        return {is_verdict_found => FALSE,};
    }

    my $is_verdict_found = FALSE;
    my %to_save;
    my @verdicts = (0, 0, 0);    # need_approve, approved, rejected
    foreach my $field_name (keys %$moderated_fields) {
        my $field_name_approved = $self->get_field_name_approved($field_name);
        my $field_moderation    = $moderation->{$field_name};
        my @to_check            = (
            defined $obj->{$field_name}
            ? (
                $moderated_fields->{$field_name} eq 'ARRAY'
                ? @{$obj->{$field_name}}
                : $obj->{$field_name}
              )
            : ()
        );

        my %rejected;
        my %approved;
        my @field_verdicts = (0, 0, 0);    # need_approve, approved, rejected
        foreach my $field_value (@to_check) {
            my $field_moderation_data = $field_moderation->{$field_value};
            if ($field_moderation_data) {
                if ($field_moderation_data->{request_id} eq $request_id) {
                    $is_verdict_found = TRUE;
                    if ($field_moderation_data->{verdict}) {
                        # Вердикт уже есть
                        WARN {
                            message => 'Duplicate moderation verdict',
                            extra   => {
                                block           => $obj,
                                moderation_data => $moderation_data,
                            },
                        };
                        next;
                    } elsif ($verdict) {
                        $field_moderation_data->{verdict} = 1;
                        $approved{$field_value} = TRUE;
                    } else {
                        $field_moderation_data->{verdict} = -1;
                        $rejected{$field_value} = TRUE;
                    }
                }
                $field_verdicts[$field_moderation_data->{verdict}]++;
                $verdicts[$field_moderation_data->{verdict}]++;
            }
        }
        if (scalar keys %rejected) {
            if ($moderated_fields->{$field_name} eq 'ARRAY') {
                my @data = grep {!exists $rejected{$_}} @{$obj->{$field_name}};
                if ($field_verdicts[0]) {
                    # ещё остались непромодерированные, просто применим текущее изменение
                    $to_save{$field_name} = \@data;
                } else {    # всё промодерировали
                    if (scalar @data) {
                        # есть успешно промодерированные, их применяем
                        $to_save{$field_name}          = \@data;
                        $to_save{$field_name_approved} = \@data;
                    } elsif (defined $obj->{$field_name_approved})
                    {       # ничего не осталось, возвращаем старое значение
                        $to_save{$field_name} = $obj->{$field_name_approved};
                    } else {
                        $to_save{$field_name} = [];
                    }
                }
            } else {
                $to_save{$field_name} = $obj->{$field_name_approved} if $obj->{$field_name_approved};
            }
        }
        if (scalar keys %approved) {
            if ($moderated_fields->{$field_name} eq 'ARRAY') {
                unless ($field_verdicts[0]) {    # всё промодерировали
                    $to_save{$field_name_approved} = $obj->{$field_name};
                }
            } else {
                $to_save{$field_name_approved} = $obj->{$field_name};
            }
        }
    }
    $to_save{moderation} = $moderation if $is_verdict_found;

    return {
        is_verdict_found => $is_verdict_found,
        to_save          => \%to_save,
        verdicts         => \@verdicts,
    };
}

sub on_verdict_received {
    my ($self, $block, $moderation_data) = @_;

    my $moderated_fields = $self->get_fields_moderated();
    my $obj =
      $self->_get_object_fields($block,
        [qw(moderation opts), map {$_, $self->get_field_name_approved($_)} keys %$moderated_fields]);

    my $result = $self->_apply_verdict($obj, $moderation_data);

    if ($result->{is_verdict_found}) {
        # Сохранить результаты модерации
        $self->do_action($block, 'edit', %{$result->{to_save}});

        # Возможно, уже можно поставить вердикт целиком
        $self->maybe_do_action($block, ($result->{verdicts}[-1] ? 'reject' : 'approve')) unless $result->{verdicts}[0];
    } else {
        # Если пришёл какой-то старый перевердикт или какой-то левый вердикт
        # Это не повод останавливать всю модерацию, из-за этого только сообщение
        WARN {
            message => 'There is no element to apply moderation verdict',
            extra   => {
                block           => $obj,
                moderation_data => $moderation_data,
            },
        };
    }

    return TRUE;
}

sub _try_moderate_in_place {
    my ($self, $moderation, $obj) = @_;

    my $obj_id = (exists $obj->{page_id}) ? $obj : $self->hook_stash->get('id');

    # no unmoderated values; no verdict=0
    if (0 == (grep {0 == $_->{verdict}} map {values %$_} values %{$moderation})) {
        my $can_be_approved = TRUE;
        for (keys %$moderation) {
            # no verdicts at all (photos is not mandatory) || there is at least one positive verdict
            $can_be_approved &&= (0 == keys %{$moderation->{$_}})
              || (0 < grep {1 == $_->{verdict}} values %{$moderation->{$_}});
        }
        my $tmp = $self->app->add_tmp_rights($self->get_rights_by_actions(qw(approve reject)));
        $self->maybe_do_action($obj_id, $can_be_approved ? 'approve' : 'reject');
    }
}

TRUE;
