package Application::Model::Role::Has::MidRoll;

use qbit;

use Application::Model::Role;

use PiConstants qw(
  $MIDROLL
  $MIDROLL_MIN_START_TIME
  $INROLL
  $S2SROLL
  $OVERLAY
  $INROLL_OVERLAY
  $OVERLAY_TYPES

  $MIN_CPM_STRATEGY_ID
  $MAX_REVENUE_STRATEGY_ID
  $DSP_DIRECT_ID

  $PREROLL
  $POSTROLL
  $INTERSTITIAL
  $PAUSE
  $INPAGE

  );

use Application::Model::Block::Video::MidRoll::MIDROLL;
use Application::Model::Block::Video::MidRoll::INROLL;
use Application::Model::Block::Video::MidRoll::S2SROLL;
use Application::Model::Block::Video::MidRoll::OVERLAY;
use Application::Model::Block::Video::MidRoll::INROLL_OVERLAY;
use Application::Model::Block::Video::MidRoll::Default;

my %MIDROLL_TYPES = (
    $MIDROLL        => 'MIDROLL',
    $INROLL         => 'INROLL',
    $S2SROLL        => 'S2SROLL',
    $OVERLAY        => 'OVERLAY',
    $INROLL_OVERLAY => 'INROLL_OVERLAY',
);

my %valid_type = (
    %MIDROLL_TYPES,

    $PREROLL      => 'Default',
    $POSTROLL     => 'Default',
    $INTERSTITIAL => 'Default',
    $PAUSE        => 'Default',
    $INPAGE       => 'Default',
);

sub call_method_by_type {
    my ($type, $method, @args) = @_;

    die "unsupported type '$type'" unless $valid_type{$type};

    my $class = 'Application::Model::Block::Video::MidRoll::' . $valid_type{$type};
    my $sub   = $class->can($method);

    die "unsupported method '$method' from '$class' for '$type'" unless $sub;

    return $sub->(@args);
}

our $REPEAT_IDS = {
    # 0 - 'Без повтора', 1 - 'Повтор по времени', 2 - 'Строго по метке'
    0 => d_gettext('No repeat'),
    1 => d_gettext('Time repeat'),
    2 => d_gettext('Label repeat'),
};

our $BROADCAST_REPLACE_IDS = {
    # 0 - 'Раздвижение', 1 - 'Замена'
    0 => d_gettext('Move'),
    1 => d_gettext('Change'),
};

our $SERVER_SIDE_IDS = {
    # 0 - 'Клиентский', 1 - 'Серверный'
    0 => d_gettext('Client'),
    1 => d_gettext('Server'),
};

my $MIN_MAX_DURATION = {
    $MIDROLL => [5, 600],
    map {$_ => [1, 30]} keys %$OVERLAY_TYPES,
};

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        # Способ вызова
        repeat => {
            db         => TRUE,
            depends_on => [qw(type)],
            type       => 'number',
            need_check => {
                type => 'int_un',
                in   => [sort keys(%$REPEAT_IDS)],
                msg  => d_gettext("Unknown repeat"),
            },
            api => 1
        },
        # Поведение односительно контента
        broadcast_replace => {
            db         => TRUE,
            depends_on => [qw(type)],
            type       => 'number',
            need_check => {
                type => 'int_un',
                in   => [sort keys(%$BROADCAST_REPLACE_IDS)],
                msg  => d_gettext("Unknown broadcast_replace"),
            },
            api => 1
        },
        # Picture in picture
        pip => {
            db         => TRUE,
            depends_on => [qw(type)],
            type       => 'boolean',
            need_check => {type => 'boolean',},
            api        => 1
        },
        # Время старта блока относительно просмотра
        start_time => {
            db         => TRUE,
            label      => d_gettext('Start time'),
            type       => 'number',
            need_check => {
                type => 'int_un',
                min  => $MIDROLL_MIN_START_TIME,
                max  => 65000,
                msg  => gettext(
                    'Field "start_time" required and must be in a range [%s, %s]', $MIDROLL_MIN_START_TIME, 65000
                ),
            },
            api => 1
        },
        # Повтор через NN
        repeat_after => {
            db         => TRUE,
            type       => 'number',
            need_check => {
                type => 'int_un',
                min  => 1,
                max  => 65000,
                msg  => gettext('Field "repeat_after" required and must be in a range [%s, %s]', 1, 65000),
            },
            api => 1
        },
        # Количество повторов
        max_repeat_count => {
            db         => TRUE,
            type       => 'number',
            need_check => {
                type => 'int_un',
                min  => 0,
                max  => 1000,
                msg  => gettext('Field "max_repeat_count" required and must be in a range [%s, %s]', 0, 1000),
            },
            api => 1
        },
        # Тип запроса
        server_side => {
            db         => TRUE,
            depends_on => [qw(type)],
            type       => 'number',
            need_check => {
                type => 'int_un',
                in   => [sort keys(%$SERVER_SIDE_IDS)],
                msg  => d_gettext("Unknown server_side"),
            },
            api => 1
        },
        # Клиентский вызов после серверного
        try_client => {
            db         => TRUE,
            depends_on => [qw(type)],
            type       => 'boolean',
            need_check => {type => 'boolean',},
            api        => 1
        },
    };
}

sub get_add_fields {
    my ($self, $fields) = @_;

    $fields->{$_} = TRUE foreach get_midroll_like_specific_fields();

    return $fields;
}

sub collect_editable_fields {
    my ($self, $obj, $fields) = @_;

    call_method_by_type($obj->{'type'}, 'collect_editable_fields', @_);
}

sub get_need_update_in_bk_fields {
    my ($self) = @_;

    my $fields = {};

    $fields->{$_} = TRUE foreach get_midroll_like_specific_fields();

    return $fields;
}

sub get_bk_block_data {
    my ($self, $block, $data) = @_;

    # "Способ вызова" = "Строго по метке" (2)
    $data->{Video}->{MetadataAdEnabled} = 1 if defined($block->{'repeat'}) && $block->{'repeat'} == 2;

    # "Способ вызова" = "Без повтора" (0) или "Повтор по времени" (1)
    $data->{Video}->{MetadataAdEnabled} = 0
      if defined($block->{'repeat'}) && ($block->{'repeat'} == 0 || $block->{'repeat'} == 1);

    my %partner2bk = (
        broadcast_replace => 'BroadcastReplace',
        repeat_after      => 'RepeatAfter',
        max_repeat_count  => 'MaxRepeatCount',
        pip               => 'Pip',
        server_side       => 'ServerSide',
        try_client        => 'TryClient',
    );

    foreach my $field (keys %partner2bk) {
        if (defined($block->{$field})) {
            $data->{Video}->{$partner2bk{$field}} = $block->{$field};
        }
    }

    call_method_by_type($block->{type}, 'get_bk_block_data', @_);

    return $data;
}

sub get_editable_fields_depends {
    [
        qw(
          repeat
          broadcast_replace
          server_side
          )
    ];
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    my $old_settings = $self->hook_stash->get('current');
    my $new_settings = $self->hook_stash->get('settings');

    if (is_type_midroll_like($new_settings->{'type'})) {

        call_method_by_type(
            $new_settings->{'type'},
            'hook_fields_processing_before_validation',
            $old_settings, $new_settings
        );

        if ($new_settings->{repeat} == 0) {
            $new_settings->{$_} = undef foreach (
                qw(
                repeat_after
                max_repeat_count
                server_side
                try_client
                )
            );
        } elsif ($new_settings->{repeat} == 1) {
            $new_settings->{$_} = undef foreach (
                qw(
                server_side
                try_client
                )
            );
        } elsif ($new_settings->{repeat} == 2) {
            $new_settings->{$_} = undef foreach (
                qw(
                start_time
                repeat_after
                max_repeat_count
                )
            );

            if (defined($new_settings->{server_side}) && $new_settings->{server_side} == 0) {
                $new_settings->{try_client} = undef;
            }
        }

        if (!defined($new_settings->{broadcast_replace}) || $new_settings->{broadcast_replace} == 0) {
            $new_settings->{$_} = undef foreach (
                qw(
                pip
                )
            );
        }

    } else {
        # Эти поля есть только у midroll
        $new_settings->{$_} = undef foreach get_midroll_like_specific_fields();
    }

    my @keys = (keys(%{$old_settings}), keys(%{$new_settings}));
    foreach my $key (@{array_uniq(@keys)}) {
        if (defined($old_settings->{$key}) && defined($new_settings->{$key})) {
            $opts->{$key} = $new_settings->{$key} if $old_settings->{$key} ne $new_settings->{$key};
        } else {
            $opts->{$key} = $new_settings->{$key};
        }
    }

    return $new_settings;
}

sub fix_template {
    my ($self, $qv) = @_;

    my $template = $qv->template;

    my %not_applicable_fields;

    throw Exception::Validation::BadArguments gettext('Missed required field "%s"', 'type')
      unless defined $qv->data->{'type'};

    if (is_type_midroll_like($qv->data->{'type'})) {

        if (!defined($qv->data->{'broadcast_replace'}) || $qv->data->{'broadcast_replace'} != 1) {
            # Не равен 1 - 'Замена'"
            # Picture in picture доступен только если 'Поведение относительно контента' = 'Замена'",
            $not_applicable_fields{'pip'} = 1;
        }

        if ($qv->data->{'repeat'} == 0) {
            # 'Без повтора'

            $not_applicable_fields{$_} = 1 foreach (
                qw(
                repeat_after
                max_repeat_count
                server_side
                try_client
                )
            );

        } elsif ($qv->data->{'repeat'} == 1) {
            # 'Повтор по времени'

            $not_applicable_fields{$_} = 1 foreach (
                qw(
                server_side
                try_client
                )
            );

        } elsif ($qv->data->{'repeat'} == 2) {
            # 'Строго по метке'

            $not_applicable_fields{$_} = 1 foreach (
                qw(
                start_time
                repeat_after
                max_repeat_count
                )
            );

            if (!defined($qv->data->{'server_side'}) || $qv->data->{'server_side'} != 1) {
                # 'Клиентский вызов после серверного' доступен только если 'Тип запроса' = 'Серверный'
                $not_applicable_fields{'try_client'} = 1;
            }
        }

        my ($MIN_DURATION, $MAX_DURATION) = _get_minmax_duration($qv->data->{'type'});
        $template->{fields}{max_duration}{check} = sub {
            my ($qv, $max_duration) = @_;

            throw Exception::Validator::Fields gettext('"max_duration" must be of a range: %s - %s sec', $MIN_DURATION,
                $MAX_DURATION)
              unless $max_duration >= $MIN_DURATION && $max_duration <= $MAX_DURATION;
        };

        call_method_by_type($qv->data->{'type'}, 'fix_template', $template);

    } else {
        # Эти поля есть только у midroll
        $not_applicable_fields{$_} = 1 foreach $self->get_midroll_like_specific_fields();
    }

    foreach my $field (keys(%not_applicable_fields)) {
        $template->{'fields'}{$field} = {eq => undef};
    }

    $qv->template($template);
}

sub get_fields_defaults {
    my ($self, $opts) = @_;

    my $result = {};
    return $result unless defined $opts->{attributes}{'type'};

    if (is_type_midroll_like($opts->{attributes}{'type'})) {
        $result->{'max_duration'} = [];
    }

    return $result;
}

sub get_midroll_dsp_rule {
    return Partner::DSP::Rule->new(
        dsps_field_validation => Partner::DSP::Rule::Part->new(
            sub => sub {
                my ($block, $dsps) = @_;

                return unless $block->{type};
                call_method_by_type($block->{type}, 'get_midroll_dsp_rule', $block, $dsps);
            },
        ),
    );
}

sub get_midroll_like_specific_fields {
    return keys %{get_structure_model_fields()};
}

sub is_type_midroll_like {
    my ($type) = @_;
    return exists $MIDROLL_TYPES{$type};
}

sub _get_minmax_duration {
    my ($block_type) = @_;
    return
      exists $MIN_MAX_DURATION->{$block_type} ? @{$MIN_MAX_DURATION->{$block_type}} : @{$MIN_MAX_DURATION->{$MIDROLL}};
}

TRUE;
