package Application::Model::Role::Has::ModerationReason;

use qbit;

use Application::Model::Role;

my $FIELD_NAME      = 'moderation_reason';
my $FIELD_LABLE     = d_gettext('Moderation reason');
my $TEXT_FIELD_NAME = $FIELD_NAME . '_name';
my $EDIT_RIGHT      = 'edit_field__' . $FIELD_NAME;
my $VIEW_RIGHT      = 'view_field__' . $FIELD_NAME;

sub get_structure_model_accessors {
    return {moderation_reason => 'Application::Model::ModerationReason',};
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [{rights => {map {$self->get_description_right($_)} $EDIT_RIGHT, $VIEW_RIGHT}}];
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        $FIELD_NAME => {
            api        => 1,
            from_opts  => 'from_hash',
            label      => $FIELD_LABLE,
            need_check => {
                all      => {type => 'int_un'},
                optional => TRUE,
                type     => 'array',
            },
            sub_type => 'string',    # См. hook_preparing_fields_to_save ниже
            type     => 'array',
        },
        $TEXT_FIELD_NAME => {
            api        => 1,
            depends_on => [$FIELD_NAME],
            get        => sub {
                $_[0]->model->get_moderation_reason_name($_[1]->{$FIELD_NAME});
            },
            label => d_gettext('Moderation reason'),
            type  => 'string',
        },
        moderation_timeout => {
            api        => 1,
            depends_on => [$FIELD_NAME],
            get        => sub {
                $_[0]->model->get_moderation_timeout($_[1]->{$FIELD_NAME});
            },
            label => d_gettext('Moderation timeout'),
            type  => 'number',
        },
    };
}

sub get_structure_model_filter {
    return {
        fields => {
            $FIELD_NAME => {
                type       => 'json_dict',
                value_type => 'dictionary',
                label      => gettext('Moderation reason'),
                values     => sub {
                    my ($self) = @_;

                    return $self->get_moderation_reason_dict();
                },
                db_filter => sub {
                    my ($self, $filter, $field, %opts) = @_;

                    my $reasons = [
                        map {$_->{id}}
                          grep {
                            !ref $filter->[2]
                              ? ($filter->[2] eq $_->{group_id} or $filter->[2] eq $_->{id})
                              : (in_array($_->{id}, $filter->[2]) or in_array($_->{group_id}, $filter->[2]))
                          } @{$self->moderation_reason->get_all(fields => [qw(id group_id reason_txt)])}
                    ];
                    $filter->[2] = $reasons;
                    return $self->QBit::Application::Model::DBManager::Filter::json_dict::as_filter($filter, $field,
                        %opts);
                },
            },
        },
    };
}

sub get_db_filter_simple_fields {
    return [{name => $FIELD_NAME, label => $FIELD_LABLE->(),}];
}

sub get_moderation_reason_name {
    my ($self, $moderation_reason) = @_;

    return '' unless $moderation_reason;
    $moderation_reason = [$moderation_reason] unless ref $moderation_reason;

    my $reasons = $self->moderation_reason->get_all(
        fields => ['reason_txt'],
        filter => [AND => [{multistate => 'not deleted'}, [id => IN => $moderation_reason],]]
    );

    my @reasons = sort @{array_uniq(grep {$_} map {$_->{reason_txt}} @$reasons)};

    return join "\n", @reasons;
}

sub get_moderation_timeout {
    my ($self, $moderation_reason) = @_;

    return 0 unless $moderation_reason;
    return 0 if $self->check_rights('can_set_need_approve_without_timeout');

    $moderation_reason = [$moderation_reason] unless ref $moderation_reason;

    my $timeouts = $self->moderation_reason->get_all(
        fields   => ['timeout'],
        filter   => [AND => [{multistate => 'not deleted'}, [id => IN => $moderation_reason],]],
        order_by => [['timeout']],                                                                 # ASC
    );

    return (
        scalar @$timeouts
        ? (
            $timeouts->[0]{timeout} == -1
            ? -1
            : $timeouts->[-1]{timeout}
          )
        : 0
    );
}

sub get_moderation_reason_dict {
    my ($self) = @_;

    my $all_reasons = $self->moderation_reason->get_all(
        fields => [qw(id group_id reason_txt)],
        filter => {multistate => 'not deleted'},
    );

    my %dict =
      map {$_->{$self->check_rights('moderation_reason_manager_txt_view') ? 'id' : 'group_id'} => $_} @$all_reasons;
    return [map {{id => $_, label => $dict{$_}->{reason_txt}}} keys %dict];
}

sub get_available_fields {
    my ($self, $obj, $fields) = @_;

    $self->app->delete_field_by_rights($fields, {$self->get_right($VIEW_RIGHT) => $FIELD_NAME,},);

    return $fields;
}

sub collect_editable_fields {
    my ($self, $data, $fields) = @_;

    $fields->{$FIELD_NAME} = TRUE if $self->check_short_rights($EDIT_RIGHT);

    return $fields;
}

sub get_fields_defaults {
    my ($self, $opts) = @_;

    my %need_fields = map {$_ => TRUE} @{$opts->{fields}};

    my $result = {};

    if ($need_fields{$FIELD_NAME}) {
        $result->{$FIELD_NAME} = $self->get_moderation_reason_dict();
    }

    return $result;
}

sub get_actions_depends {
    [qw(moderation_timeout)];
}

sub hook_preparing_fields_to_save {
    my ($self, $opts) = @_;

    # Принудительная стрингификация
    # Так как фильтры по json-массивам работают с учётом типа
    # А скаляры всегда проходят обязательную процедуру квотирования
    if ($opts->{$FIELD_NAME}) {
        foreach (@{$opts->{$FIELD_NAME}}) {
            $_ = "$_";
        }
    }

    return TRUE;
}

sub _on_positive_action {
    my ($self, $obj, %opts) = @_;

    $self->partner_db_table()->edit($obj, {opts => {json_set => ['opts', \"\$.$FIELD_NAME", \undef,]}});
}

sub _can_negative_action {
    my ($self, $obj, %opts) = @_;

    $self->app->validator->check(
        data     => \%opts,
        app      => $self,
        throw    => TRUE,
        template => {
            type   => 'hash',
            fields => {$FIELD_NAME => $self->get_model_fields()->{$FIELD_NAME}{need_check},}
        },
    );

    return TRUE;
}

sub _on_negative_action {
    my ($self, $obj, %opts) = @_;

    $self->partner_db_table()
      ->edit($obj,
        {opts => {json_set => ['opts', \"\$.$FIELD_NAME", {json_array => [map {\$_} @{$opts{$FIELD_NAME}}]},]}})
      if exists $opts{$FIELD_NAME} && ref($opts{$FIELD_NAME}) eq 'ARRAY';
}

TRUE;
