package Application::Model::Role::Has::RTB;

use qbit;

use Application::Model::Role;

requires qw(get_filter_applicable_to_by_strategy get_ad_type_list);

consume qw(
  Application::Model::Role::Has::Brands
  Application::Model::Role::Has::CustomBkData
  Application::Model::Role::Has::Geo
  );

use PiConstants qw(
  $BLIND_VALUES
  $MAX_REVENUE_STRATEGY_ID
  $MIN_CPM_STRATEGY_ID
  $SEPARATE_CPM_STRATEGY_ID
  );

use Exception::Validator::Fields;

sub get_structure_model_fields {
    my ($self) = @_;

    my $page_field_name = $self->get_page_id_field_name();

    return {
        dsp_blocks => {
            depends_on => [qw(page_id id)],
            label      => d_gettext('DSP blocks format'),
            get        => sub {
                $_[0]->{'dsp_blocks'}{$_[1]->{'page_id'}, $_[1]->{'id'}} // [];
            },
            type       => 'array',
            sub_type   => 'string',
            need_check => {
                type  => 'array',
                all   => {len_max => 12,},    # as in db media_sizes.type
                check => sub {
                    my ($qv, $dsp_blocks) = @_;

                    throw Exception::Validator::Fields gettext('Invalid DSP block types')
                      if ref($dsp_blocks) ne 'ARRAY'
                          || @$dsp_blocks == 0;

                    my $cnt = {};
                    my @duplicates = grep {++$cnt->{$_} == 2} @$dsp_blocks;
                    throw Exception::Validator::Fields gettext('Got duplicates in "dsp_blocks": %s',
                        join(', ', @duplicates))
                      if @duplicates;

                    my %dsp_blocks = map {$_->{'id'} => $_} @{$qv->app->get_types_dsp_blocks($qv->data)};

                    my @bad_blocks = grep {!$dsp_blocks{$_}} @$dsp_blocks;

                    throw Exception::Validator::Fields gettext('Invalid DSP block types: %s', join(', ', @bad_blocks))
                      if @bad_blocks;
                },
            },
            api => 1,
        },
        alternative_code => {
            db         => TRUE,
            label      => d_gettext('Alternative code'),
            type       => 'string',
            need_check => {optional => TRUE,},
            api        => 1
        },
        blind =>
          {db => TRUE, label => d_gettext('Blind'), type => 'number', need_check => {in => $BLIND_VALUES}, api => 1},
        horizontal_align => {
            db         => TRUE,
            label      => d_gettext('Horizontal align'),
            type       => 'boolean',
            need_check => {type => 'boolean'},
            api        => 1,
        },
        dsps => {
            depends_on => [qw(page_id id), 'block_dsps.dsp_id'],
            label      => d_gettext('DSP list'),
            get        => sub {
                [map {$_->{'dsp_id'}} @{$_[0]->{'block_dsps'}{$_[1]->{'page_id'}, $_[1]->{'id'}} // []}];
            },
            type       => 'array',
            sub_type   => 'string',
            need_check => {
                optional => TRUE,
                type     => 'array',
                all      => {type => 'int_un'},
                check    => sub {
                    my ($qv, $dsps) = @_;

                    $qv->app->validate_dsp_list($qv->data, $dsps);
                },
            },
            api => 1,
        },
        media_formats_list => {
            depends_on => [qw(media_block dsp_blocks)],
            label      => d_gettext('Media formats list'),
            get        => sub {
                return join(
                    ', ',
                    sort {
                        my ($aw, $ah) = $a =~ /^(\d+)x(\d+)/;
                        my ($bw, $bh) = $b =~ /^(\d+)x(\d+)/;
                        $aw && $bw ? $aw <=> $bw || $ah <=> $bh : $a cmp $b;
                      } @{array_uniq($_[1]->{'media_block'}, $_[1]->{'dsp_blocks'})}
                );
            },
        },
    };
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $opts->{horizontal_align} //= TRUE unless ($opts->{block_type} && $opts->{block_type} eq 'native');
}

sub get_strategies {
    my ($self, %opts) = @_;

    my $strategy_separate_cpm_fields = $self->get_ad_type_list(%opts);
    my $strategies_available         = $self->get_strategies_available(%opts);
    my $strategies                   = {
        $MIN_CPM_STRATEGY_ID     => {$self->get_filter_applicable_to_by_strategy($MIN_CPM_STRATEGY_ID)},
        $MAX_REVENUE_STRATEGY_ID => {$self->get_filter_applicable_to_by_strategy($MAX_REVENUE_STRATEGY_ID)},
        (
            exists $strategies_available->{$SEPARATE_CPM_STRATEGY_ID}
            ? (
                $SEPARATE_CPM_STRATEGY_ID => {
                    $self->get_filter_applicable_to_by_strategy($SEPARATE_CPM_STRATEGY_ID),
                    (fields => $strategy_separate_cpm_fields)
                },
              )
            : ()
        )
    };

    return $strategies;
}

sub storage_method_of_strategies {'db'}

sub _rtb_make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{'page_id'}) {
        $result{'page_id'} = $self->get_all_campaigns_for_adding(
            ($opts->{page_filter} ? (filter   => $opts->{page_filter}) : ()),
            ($opts->{page_limit}  ? (limit    => $opts->{page_limit})  : ()),
            ($opts->{page_sort}   ? (order_by => $opts->{page_sort})   : ()),
        );
    }

    if ($need_fields->{'media_block'}
        && !in_array($self->accessor, [qw(context_on_site_rtb internal_context_on_site_rtb)]))
    {
        $result{'media_block'} = $self->get_types_media_blocks($opts->{attributes});
    }

    if ($need_fields->{'dsp_blocks'}) {
        $result{'dsp_blocks'} = $self->get_types_dsp_blocks($opts->{attributes}, filter_by_site_version => TRUE);
    }

    if ($need_fields->{'dsps'}) {
        my $rules = $self->get_dsp_rule_set();
        $result{'dsps_available'} = $self->get_available_dsps($opts->{attributes}, $rules);
        $result{'dsps_default'} = $self->get_default_dsps($opts->{attributes}, $rules);

        unless ($self->check_short_rights('view_field__dsps')) {
            my $page = $self->page->get_all(
                filter => {page_id => $opts->{attributes}{page_id}},
                fields => [qw(unmoderated_rtb_auction)]
            )->[0];
            if ($page && $page->{unmoderated_rtb_auction}) {
                $result{'dsps_available'} =
                  [grep {exists $_->{unmoderated_rtb_auction} && $_->{unmoderated_rtb_auction}}
                      @{$result{'dsps_available'}}];
                $result{'dsps_default'} =
                  [grep {exists $_->{unmoderated_rtb_auction} && $_->{unmoderated_rtb_auction}}
                      @{$result{'dsps_default'}}];
            } else {
                $result{'dsps_available'} = [];
                $result{'dsps_default'}   = [];
            }
        }
    }

    if ($need_fields->{'site_version'}) {
        $result{'site_version'} = $self->get_block_site_versions($opts->{attributes});
    }

    if ($need_fields->{'strategy'}) {
        my $is_video_applicable = $opts->{attributes}{show_video}
          && $self->is_show_video_applicable($opts->{attributes})->{applicable};
        $result{'strategy'} =
          $self->get_strategies_specification(is_video_applicable => $is_video_applicable)
          ->get_strategies_types($self, $opts->{attributes});
    }

    return \%result;
}

TRUE;
