package Application::Model::Role::Has::SiteVersion;

use qbit;

use Application::Model::Role;

use PiConstants qw($SITE_VERSIONS $SITE_VERSIONS_TURBO);

my @SITE_VERSIONS_DESKTOP = qw(desktop turbo_desktop);

requires qw(available_versions default_site_version turbo_desktop_available site_versions_that_depend_on_feature);

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            rights => {
                map {$self->get_description_right($_)}
                  qw(
                  edit_field__site_version
                  ),
            }
        }
    ];
}

sub get_structure_model_fields {
    return {
        site_version => {
            db         => TRUE,
            type       => 'string',
            need_check => {
                check => sub {
                    my ($qv, $site_version) = @_;

                    if ($qv->app->check_multistate_flag($qv->data->{'multistate'}, 'deleted')) {
                        my $site_versions = $qv->app->get_site_versions();
                        throw Exception::Validator::Fields gettext('Invalid site version: %s', $site_version)
                          unless $site_versions->{$site_version};
                    } else {
                        my $block_site_versions = $qv->app->get_block_site_versions(
                            {page_id => $qv->data->{'page_id'}, adfox_block => $qv->data->{'adfox_block'}});

                        throw Exception::Validator::Fields gettext('Unavailable site version: %s', $site_version)
                          unless grep {$site_version eq $_->{'id'}} @$block_site_versions;
                    }
                  }
            },
            api => 1
        }
    };
}

sub get_structure_model_filter {
    return {
        fields => {
            site_version => {
                type   => 'dictionary',
                label  => gettext('Site version'),
                values => sub {
                    my ($model) = @_;

                    my $site_versions = $model->get_site_versions();

                    return [
                        sort {$a->{'label'} cmp $b->{'label'}}
                        map {{id => $_, label => $site_versions->{$_}}} keys(%$site_versions)
                    ];
                  }
            }
        }
    };
}

sub get_db_filter_simple_fields {
    return [{name => 'site_version', label => gettext('Site version')}];
}

sub get_add_fields {
    my ($self, $fields) = @_;

    $fields->{'site_version'} = TRUE;

    return $fields;
}

sub collect_editable_fields {
    my ($self, $obj, $fields) = @_;

    if ($self->check_short_rights('edit_field__site_version')) {
        $fields->{'site_version'} = 1;
    } else {
        delete $fields->{'site_version'};
    }

    return $fields;
}

sub get_bk_block_data {
    my ($self, $block, $data) = @_;

    my $site_versions = $self->get_site_versions();

    #PI-9826 PI-15176
    foreach my $id (sort keys(%$site_versions)) {
        next unless exists($SITE_VERSIONS->{$id}{'page_imp_option'});

        if ($block->{'site_version'} eq $id) {
            push(@{$data->{'PageImpOptions'}{'Enable'}}, $SITE_VERSIONS->{$id}{'page_imp_option'});
        } else {
            push(@{$data->{'PageImpOptions'}{'Disable'}}, $SITE_VERSIONS->{$id}{'page_imp_option'});
        }
    }

    return $data;
}

sub get_actions_depends {
    ['site_version'];
}

sub get_site_versions {
    my ($self) = @_;

    my %available_versions = map {$_ => TRUE} $self->available_versions();

    my $res = {};
    foreach my $id (sort keys(%$SITE_VERSIONS)) {
        next unless $available_versions{$id};

        $res->{$id} = $SITE_VERSIONS->{$id}{'label'}->();
    }

    return $res;
}

sub get_block_site_versions {
    my ($self, $object) = @_;

    my $page_id = $object->{'page_id'} // throw Exception::Validation::BadArguments gettext('Expected "%s"', 'page_id');

    my $site_versions = $self->get_site_versions();

    my $need_check_features = scalar @{$self->site_versions_that_depend_on_feature};

    my $pages = $self->page->get_all(
        fields => [
            qw(allowed_turbo),
            ($self->page->accessor eq 'context_on_site_campaign' ? 'allowed_amp' : ()),
            ($need_check_features ? 'owner' : ())
        ],
        filter => {page_id => $page_id}
    );

    my $page = $pages->[0] // throw Exception::Validation::BadArguments gettext('Page "%s" not found', $page_id);

    my @res = ();

    my $hidden_version = $self->get_versions_to_hide($page_id);

    foreach my $id (sort keys(%$site_versions)) {
        next if $id eq 'amp' && !$page->{'allowed_amp'};
        next if $SITE_VERSIONS_TURBO->{$id} && !$page->{'allowed_turbo'};
        next if $id eq 'turbo_desktop' && !$self->turbo_desktop_available($page);
        next if $hidden_version->{$id};

        push(@res, {id => $id, label => $site_versions->{$id}});
    }

    return \@res;
}

sub get_versions_to_hide {
    my ($self, $page_id) = @_;
    my %hide;
    my $optional       = $self->site_versions_that_depend_on_feature();
    my $owner_features = {};
    if (@$optional) {
        $owner_features = $self->get_features_and_options_on_page($page_id);
    }
    for my $version (@$optional) {
        $hide{$version} = TRUE unless $owner_features->{"${version}_available"};
    }
    return \%hide;
}

sub is_desktop_site_version {
    my ($self, $site_version) = @_;
    return in_array($site_version, \@SITE_VERSIONS_DESKTOP);
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    my $hook_stash = $self->hook_stash();

    if ($hook_stash->inited()) {
        my $site_version = $hook_stash->get('settings')->{'site_version'} // $self->default_site_version();

        $hook_stash->get('opts')->{'site_version'} //= $site_version;
    }
}

TRUE;
