package Application::Model::Role::JavaJsonApiProxy;

use qbit;

use Utils::Logger qw(
  ERROR
  INFO
  INFOF
  WARN
  );

use Application::Model::Role;

use PiConstants qw(
  $DO_ACTION_USE_QUEUE_KEY
  $MAX_PAGE_SIZE
  $USE_JAVA_JSONAPI_KEY
  );

use Utils::JSON qw(clean_decoded_json_from_boolean_objects_in_place);

sub can_use_java_for_do_action {
    my ($self) = @_;

    my $use_queue = from_json($self->app->kv_store->get($DO_ACTION_USE_QUEUE_KEY) // '{}');

    return $use_queue->{$self->accessor()};
}

sub multi_do_action {
    my ($self, $page_id, $action, %opts) = @_;

    unless ($self->can_use_java_for_do_action()) {
        return $self->Application::Model::Block::multi_do_action($page_id, $action, %opts);
    }

    INFOF('Start multi_do_action for accessor: %s, page_id: %d, action: %s', $self->accessor(), $page_id, $action);

    my $vmap_id;
    if (defined($opts{filter})) {
        throw Exception::Validation::BadArguments gettext('Expected HASH') if ref($opts{filter}) ne 'HASH';

        my @fields = keys(%{$opts{filter}});
        throw Exception::Validation::BadArguments gettext('Expected only one key "vmap_id"')
          if @fields != 1 || $fields[0] ne 'vmap_id';

        $vmap_id = $opts{filter}->{'vmap_id'};
    }

    my $tmp_rights = $self->app->add_tmp_rights(qw(queue_add_do_action));
    my $id         = $self->app->queue->add(
        method_name => 'do_action',
        group_id    => $page_id,
        params      => {
            userId     => $self->get_option('cur_user')->{'id'},
            modelName  => $self->accessor(),
            actionName => $action,
            pageId     => $page_id,
            (defined($vmap_id) ? (vmapId => $vmap_id) : ()),
        },
    );

    INFOF('Added queue with id: %d', $id);
}

sub can_use_java_for_get_all {
    my ($self) = @_;

    my $use_java_jsonapi = from_json($self->app->kv_store->get($USE_JAVA_JSONAPI_KEY) // '{}');

    return $use_java_jsonapi->{$self->accessor()};
}

sub get_all {
    my ($self, %opts) = @_;

    my $use_java = $self->can_use_java_for_get_all() && ($opts{use_java_json_api} // TRUE);

    my @data = ();
    if ($use_java) {
        my $class = ref($self);

        my $caller    = [caller(0)];
        my $call_code = $caller->[0] . " ($caller->[1] line $caller->[2])";

        my %field_map = ($self->get_page_id_field_name() => 'page_id',);

        my @not_allowed_options = grep {$opts{$_}} qw(for_update all_locales distinct group_by);

        $opts{'fields'} //= [];
        $opts{'fields'} = [$opts{'fields'}] if ref($opts{'fields'}) ne 'ARRAY';

        my @available_fields = $self->api_available_fields();
        push(@available_fields, 'public_id', keys(%field_map));

        my %api_fields = map {$_ => TRUE} @available_fields;

        my @not_available_fields = grep {!$api_fields{$_}} @{$opts{'fields'} // []};
        if (!@{$opts{'fields'}}) {
            $use_java = FALSE;
            WARN {
                message => $self->_get_message($call_code, \%opts, "$class\::get_all got empty field set"),
                fingerprint => ['JavaJsonApiProxy', $self->accessor(), $call_code, 'get_all']
            };
        } elsif (@not_available_fields) {
            # передали поля которых нет в АПИ
            $use_java = FALSE;
            my $message = "$class\::get_all got not available fields: " . join(' ', @not_available_fields);
            WARN {
                message => $self->_get_message($call_code, \@not_available_fields, $message),
                fingerprint => ['JavaJsonApiProxy', $self->accessor(), $call_code, 'get_all']
            };
        } elsif (exists($opts{'filter'}) && !in_array(ref($opts{'filter'}), ['ARRAY', 'HASH'])) {
            # передали фильтр от базы
            $use_java = FALSE;

            my $opts_to_json = {%opts, filter => 'PartnerDB::Filter'};

            INFO $self->_get_message($call_code, $opts_to_json, "$class\::get_all got filter for DB");
        } elsif (@not_allowed_options) {
            # такие запросы нужно переделывать
            $use_java = FALSE;

            my $message = "$class\::get_all";
            foreach (@not_allowed_options) {
                $message .= " got $_ = 1";
            }

            WARN {
                message => $self->_get_message($call_code, \%opts, $message),
                fingerprint => ['JavaJsonApiProxy', $self->accessor(), $call_code, 'get_all']
            };
        } elsif (exists($opts{'limit'}) && $opts{'limit'} > $MAX_PAGE_SIZE) {
            # limit указан больше чем может отдать за раз jsonapi
            # можно конечно сделать несклько запросов, но лучше такие места переписать
            $use_java = FALSE;

            WARN $self->_get_message($call_code, \%opts, "$class\::get_all got limit more than $MAX_PAGE_SIZE");
        }

        if ($use_java) {
            my $json_opts = {
                resource  => $self->accessor(),
                user_id   => $self->get_option('cur_user', {})->{'id'},
                page_size => $opts{'limit'},
                (defined($opts{'filter'}) ? (filter => $opts{'filter'}) : ()),
            };

            if ($opts{'calc_rows'}) {
                $json_opts->{'meta'} = 'total';
            }

            if (exists($opts{'order_by'})) {
                $json_opts->{'order_by'} = [map {[ref($_) ? ($_->[0], $_->[1]) : ($_, 0)]} @{$opts{'order_by'}}];
            }

            if (exists($opts{'limit'}) && $opts{'offset'}) {
                my $page_number = ($opts{'offset'} / $opts{'limit'}) + 1;
                $json_opts->{'page_number'} = $page_number;
            } elsif (!exists($opts{'limit'})) {
                # выставляем максимальный дефолт
                $json_opts->{'page_size'} = $MAX_PAGE_SIZE;
            }

            my @fields_need_map_back = ();
            my $need_public_id       = FALSE;
            my $need_actions         = FALSE;

            $json_opts->{fields} = [];
            foreach (@{$opts{'fields'}}) {
                if (exists($field_map{$_})) {
                    push(@fields_need_map_back,   $_);
                    push(@{$json_opts->{fields}}, $field_map{$_});
                } else {
                    push(@{$json_opts->{fields}}, $_);
                }

                $need_public_id = TRUE if $_ eq 'public_id';
                $need_actions   = TRUE if $_ eq 'actions';
            }

            $json_opts->{fields} = [sort @{array_uniq(@{$json_opts->{fields}})}];

            my $need_map_back = @fields_need_map_back || $need_actions;

            try {
                my $response = $self->app->api_java_jsonapi->get_all($json_opts);

                $self->{'__LAST_FIELDS__'} = {map {$_ => TRUE} @{$response->{'meta'}{'fields'}}};

                if (exists($response->{'meta'}{'found_rows'})) {
                    $self->{'__FOUND_ROWS__'} = $response->{'meta'}{'found_rows'};
                }

                my @response_data = map {
                    $_->{attributes}{public_id} = $_->{id} if $need_public_id;
                    $_->{'attributes'}
                } @{$response->{'data'}};

                push(@data, @response_data);

                if (!exists($opts{'limit'}) && @response_data == $MAX_PAGE_SIZE) {
                    # исходный запрос без лимита а сущности за один запрос не достать

                    WARN $self->_get_message($call_code, \%opts,
                        "$class\::get_all called without limit, but response has $MAX_PAGE_SIZE items");

                    # удаляем meta она нам больше не нужна
                    delete($json_opts->{'meta'});

                    $json_opts->{'page_number'} = 2;
                    while (@response_data == $MAX_PAGE_SIZE) {
                        $response = $self->app->api_java_jsonapi->get_all($json_opts);

                        @response_data = map {
                            $_->{attributes}{public_id} = $_->{id} if $need_public_id;
                            $_->{'attributes'}
                        } @{$response->{'data'}};

                        push(@data, @response_data);

                        $json_opts->{'page_number'}++;

                        ERROR {
                            message => $self->_get_message(
                                $call_code, \%opts,
                                "$class\::get_all called without limit, and got more than 5 pages"
                            ),
                            fingerprint => ['JavaJsonApiProxy', $self->accessor(), $call_code, 'get_all']
                          }
                          if $json_opts->{'page_number'} == 5;
                    }
                }

                if ($need_map_back) {
                    foreach my $row (@data) {
                        foreach my $field_name (@fields_need_map_back) {
                            $row->{$field_name} = delete($row->{$field_map{$field_name}});
                        }

                        if ($need_actions && exists($row->{'actions'}{'archive'})) {
                            $row->{'actions'}{'delete'} = delete $row->{'actions'}{'archive'};
                        }
                    }
                }
            }
            catch {
                my ($exception) = @_;

                # Выводим ошибку везде, кроме прогона тестов в тимсити
                ERROR {
                    exception   => $exception,
                    message     => 'Could not get items from java API',
                    fingerprint => ['JavaJsonApiProxy', $self->accessor(), $call_code, 'get_all']
                  }
                  unless $ENV{TEAMCITY_BUILD_ID};

                $use_java = FALSE;
            };
        }
    }

    if ($use_java) {
        #        ldump({java=>\@data,perl=>$self->_get_all(\%opts)});
        return clean_decoded_json_from_boolean_objects_in_place(\@data);
    } else {
        return $self->_get_all(\%opts);
    }
}

sub do_action {
    my ($self, $obj, $action, %params) = @_;

    my $use_java = $self->can_use_java_for_do_action();

    if ($use_java) {
        my $class = ref($self);

        my $caller    = [caller(0)];
        my $call_code = $caller->[0] . " ($caller->[1] line $caller->[2])";
        if ($action eq 'edit') {
            # в java очередь не поддерживает параметры поэтому edit нельзя выполнять через очередь
            # и такого кода не должно быть
            $use_java = FALSE;
            my $message = $self->_get_message(
                $call_code,
                {obj => $obj, action => $action, params => \%params},
                "$class\::do_action called for action 'edit'"
            );

            if ($call_code =~ /BlockPresets|CPMCurrency/) {
                INFO($message);
            } else {
                ERROR {
                    message     => $message,
                    fingerprint => ['JavaJsonApi', $action, $call_code],
                };
            }
        }

        if (ref($obj) && !$self->can('public_id')) {
            # Очередь умеет работать только с unique_id или public_id
            $use_java = FALSE;

            ERROR {
                message => $self->_get_message(
                    $call_code,
                    {obj => $obj, action => $action, params => \%params},
                    "$class\::do_action called with composite primary keys"
                ),
                fingerprint => ['JavaJsonApi', $action, $call_code],
            };
        }
    }

    if ($use_java) {
        my $group_id;

        my $public_id = $obj;
        if ($self->can('_split_id')) {
            $obj = $self->_split_id($obj);
        }

        if (ref($obj)) {
            $public_id = $self->public_id($obj);
            INFOF("Converting from '%s' into '%s'", to_json($obj, canonical => TRUE), $public_id);

            if ($self->can('get_page_id_field_name')) {
                $group_id = $obj->{$self->get_page_id_field_name()};
                INFOF("Will be using group_id: %d", $group_id);
            }
        }

        my $tmp_rights = $self->app->add_tmp_rights(qw(queue_add_do_action));
        my $id         = $self->app->queue->add(
            method_name => 'do_action',
            (defined($group_id) ? (group_id => $group_id) : ()),
            params => {
                userId     => $self->get_option('cur_user')->{'id'},
                modelName  => $self->accessor(),
                actionName => $action,
                modelIds   => [$public_id],
            },
        );

        INFOF('Added queue with id: %d', $id);
    } else {
        return $self->Application::Model::Product::do_action($obj, $action, %params);
    }
}

sub do_action_with_result {
    my $self = shift(@_);

    return $self->_do_action_with_recording_in_error_log('do_action_with_result', @_);
}

sub maybe_do_action {
    my $self = shift(@_);

    return $self->_do_action_with_recording_in_error_log('maybe_do_action', @_);
}

sub force_do_action {
    my $self = shift(@_);

    return $self->_do_action_with_recording_in_error_log('force_do_action', @_);
}

sub _do_action_with_recording_in_error_log {
    my ($self, $method, @params) = @_;

    my $class = ref($self);

    my $caller    = [caller(1)];
    my $call_code = $caller->[0] . " ($caller->[1] line $caller->[2])";

    ERROR {
        message     => $self->_get_message($call_code, \@params, "$class\::$method"),
        fingerprint => ['JavaJsonApi',                 $method,  $call_code],
    };
    my $full_method = "Application::Model::Product::$method";

    return $self->$full_method(@params);
}

sub _get_all {
    my ($self, $opts) = @_;

    return $self->QBit::Application::Model::DBManager::get_all(%$opts);
}

sub _get_message {
    my ($self, $call_code, $opts, $message) = @_;

    return sprintf(
        '%sCalled from: %s with params %s',
        (defined($message) ? "$message. " : ''),
        $call_code, to_json({accessor => $self->accessor(), opts => $opts}, canonical => TRUE)
    );
}

TRUE;
