package Application::Model::Role::Page::Has::AppMarketFields;

use qbit;

use Application::Model::Role;
use PiConstants qw($APP_STORE);

requires qw(check_simple_inapp_feature);

=head
Поля из маркета приложений
=cut

sub get_structure_model_accessors {
    my ($class) = @_;

    return {api_rmp => 'Application::Model::API::Yandex::RMP',};
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        store_url => {
            default    => TRUE,
            db         => TRUE,
            label      => d_gettext('Store URL'),
            need_check => {
                len_max => 255,
                check   => sub {
                    my ($qv, $value) = @_;
                    throw Exception::Validator::Fields gettext('Invalid Store URL "%s"', $value)
                      unless get_domain($value);
                },
            },
            type => 'string',
            api  => TRUE,
        },
        apple_store_id => {
            depends_on => ['type'],
            from_opts  => 'from_hash',
            label      => d_gettext('Apple Store ID'),
            need_check => {
                optional => TRUE,
                len_max  => 255,
                check    => sub {
                    my ($qv, $value) = @_;
                    if (exists $qv->data->{type} && defined $qv->data->{type}) {
                        throw Exception::Validator::Fields gettext('Invalid Apple Store ID "%s"', $value)
                          unless $qv->data->{type} eq '2';
                        throw Exception::Validator::Fields gettext('Invalid Apple Store ID "%s"', $value)
                          unless $value =~ /^id\d+$/;
                    }
                },
            },
            type => 'string',
            api  => TRUE,
        },
        icon => {
            # for example:
            # https://avatars.mds.yandex.net/get-google-play-app-icon/1997237/092bb5f9937128fdd125c88e43170582/icon-xld-retina
            # https://avatars.mds.yandex.net/get-itunes-icon/28548/91faddc38a1cdbf262247e5b9929cb94/icon-xld-retina
            from_opts  => 'from_hash',
            label      => d_gettext('App icon'),
            need_check => {
                optional => TRUE,
                len_max  => 255,
            },
            type => 'string',
            api  => TRUE,
        },
        developer => {
            from_opts  => 'from_hash',
            label      => d_gettext('App developer'),
            need_check => {
                len_max  => 255,
                optional => TRUE,
            },
            type => 'string',
            api  => TRUE,
        },
    };
}

sub collect_editable_fields {
    my ($self, $obj, $fields) = @_;

    if ($self->check_simple_inapp_feature()) {

        if ($self->hook_stash->inited()) {
            if (!$self->hook_stash->get('current')->{store_url}) {
                $fields->{$_} = TRUE for (qw(store_url));
            }
        } else {
            if (!$obj->{store_url}) {
                $fields->{$_} = TRUE for (qw(store_url));
            }
        }
    } else {
        $fields->{store_url} = TRUE;
    }

    return $fields;
}

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    if ($self->check_simple_inapp_feature()) {
        $opts->{store_url}      = undef unless $opts->{store_url};
        $opts->{apple_store_id} = undef unless $opts->{apple_store_id};

        if (exists $opts->{store_url} && $opts->{store_url}) {
            # here we go to endpoint to get icon/developer/apple_store_id
            my $store_type = $self->hook_stash->get('current')->{type};
            my $app        = $self->api_rmp->get_app_info(
                type      => $store_type,
                store_url => $opts->{store_url}
            );
            $opts->{$_} //= $app->{$_} for (qw(icon developer apple_store_id store_id));

            if ($store_type ne $APP_STORE) {
                $opts->{apple_store_id} = undef;
            }

        } else {
            delete $opts->{store_id};
            delete $opts->{store_url};
        }
    }
}

return TRUE;
