package Application::Model::SendMail;

=encoding UTF-8

=head1 Название

Application::Model::SendMail - расширенная отправка писем

Модуль расширяет QBit::Application::Model::SendMail.

В базовом варианте в поля from, to, cc, bcc (т.е. в полях, которые
принимают email адрес) могли было передавать либо элемент, либо массив
элементов.

В базовом варианте "элемент" это скаляр с email адересом или hashref вида

    { 'my@mail.ru' => 'My Name' }

Благодаря этому расширению стало возможно в качестве "элементов" использовать
две дополнительные вещи:

=over

=item * 'default' - email по умолчанияю

=item * { user_id => 123 } - uid пользователя (из базы достанется имя)

=back

Т.е. благодаря этому дополнению можно написать:

    $app->mailer->send(
        from    => 'default',
        to      => [ { user_id => $uid }, 'dsp@yandex-team.ru' ],
        subject => 'test subject',
        body    => "some message",
    );

=cut

use qbit;
use MIME::Base64;
use Utils::Logger;
use Encode;
use PiConstants qw($PARTNERS_MAIL  $PARTNERS_TEST_MAIL  $PARTNER2_CRON_MAIL);

use base qw(QBit::Application::Model::SendMail Application::Model::DBManager::Base);

use Exception::Denied;
use Exception::SendMail;
use Exception::SendMail::BadAddress;

sub accessor      {'mailer'}
sub db_table_name {'email'}

__PACKAGE__->model_accessors(
    users           => 'Application::Model::Users',
    partner_logs_db => 'Application::Model::PartnerLogsDB',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'mailer',
            description => d_gettext('Right to manage mailer'),
            rights      => {mailer_view => d_gettext('Right to view info about mails'),}
        }
    ]
);

__PACKAGE__->model_fields(
    id                   => {default => TRUE, db => TRUE, pk => TRUE,},
    dt                   => {default => TRUE, db => TRUE,},
    subject              => {default => TRUE, db => TRUE,},
    last_send_successful => {db      => TRUE,},
    from                 => {db      => TRUE},
    to                   => {db      => TRUE},
    cc                   => {db      => TRUE,},
    bcc                  => {db      => TRUE},
    email_data           => {
        depends_on => ['id'],
        get        => sub {
            $_[0]->{'__EMAIL_DATA__'}{$_[1]->{'id'}} || {};
        },
    },
    body => {
        depends_on => ['email_data'],
        get        => sub {
            $_[1]->{'email_data'}{'body'};
        },
    },
    headers => {
        depends_on => ['email_data'],
        get        => sub {
            $_[1]->{'email_data'}{'headers'};
        },
    },
    attachments => {
        depends_on => ['id'],
        get        => sub {
            $_[0]->{'__ATTACHMENTS__'}{$_[1]->{'id'}} || {};
        },
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_logs_db',
    fields      => {
        id      => {type => 'number', label => d_gettext('ID'),},
        dt      => {type => 'text',   label => d_gettext('Date'),},
        subject => {type => 'text',   label => d_gettext('Subject'),},
        from    => {type => 'text',   label => d_gettext('From'),},
        to      => {type => 'text',   label => d_gettext('To'),},
        cc      => {type => 'text',   label => d_gettext('CC'),},
        bcc     => {type => 'text',   label => d_gettext('BCC'),},
    },
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        [
            {name => 'id',      label => gettext('ID'),},
            {name => 'dt',      label => gettext('Date'), hint => gettext('YYYY-MM-DD hh:mm:ss'),},
            {name => 'subject', label => gettext('Subject'),},
            {name => 'from',    label => gettext('From'),},
        ],
        [
            {name => 'to',  label => gettext('To'),},
            {name => 'cc',  label => gettext('CC'),},
            {name => 'bcc', label => gettext('BCC'),},
        ]
    ];
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('email_data')) {
        $fields->{'__EMAIL_DATA__'} = {
            map {$_->{'email_id'} => $_} @{
                $self->partner_logs_db->query->select(
                    table  => $self->partner_logs_db->email_data,
                    fields => [qw(email_id headers body)],
                    filter => ['email_id' => '=' => \array_uniq(map {$_->{'id'}} @$result)],
                  )->get_all
              }
        };
    }

    if ($fields->need('attachments')) {
        my $attachments = $self->partner_logs_db->query->select(
            table  => $self->partner_logs_db->email_attachment,
            fields => [qw(email_id id filename content_type)],
            filter => ['email_id' => '=' => \array_uniq(map {$_->{'id'}} @$result)],
        )->get_all();

        $fields->{'__ATTACHMENTS__'}{$_->{'email_id'}}{$_->{'id'}} = $_ foreach @$attachments;
    }
}

sub query_filter {
    my ($self, $filter) = @_;

    throw Exception::Denied unless $self->check_short_rights('view');

    return $filter;
}

# TODO: вынести в общий модуль
sub get_fields_cnt {
    my ($self, %opts) = @_;

    my $data = $self->partner_logs_db->query->select(
        table  => $self->partner_db_table(),
        fields => {map {("cnt_$_") => {count => [{distinct => [$_]}]}} @{$opts{'fields'}}},
        filter => [
            id => ' = ANY' => $self->query(
                fields => $self->_get_fields_obj(['id']),
                filter => $self->get_db_filter($opts{'filter'})
            )
        ]
    )->get_all();

    return $data->[0];
}

$QBit::Application::Model::SendMail::FIELD_TYPE{'email'}->{'in'}{'hash'}[0] = sub {
    shift->get_emails_hash(@_);
};

sub get_emails_hash {
    my ($self, $data, $opt, $data_between_steps, $name) = @_;

    $name ||= '';
    my $out = [];

    if ($self->_is_to_one_user_only($data)) {
        my $tmp_rights = $self->app->add_tmp_rights('users_view_all');
        my $user = $self->users->get([%$data]->[1], fields => [qw(email full_name)])
          // throw Exception::SendMail::BadAddress gettext('Invalid user_id');

        push(@$out, {email => unset_utf($user->{'email'}), name => $user->{'full_name'}});
    } elsif (!ref($data) && $data eq 'default') {
        push(@$out, {email => $PARTNERS_MAIL, name => gettext('Yandex partner interface')});
    } elsif (ref($data) eq 'HASH') {
        while (my ($email, $name) = each(%$data)) {
            push(@$out, {name => $name, email => $email});
        }
    } elsif (!ref($data) && $data) {
        push(@$out, {name => '', email => $data});
    } elsif (ref($data) eq 'ARRAY') {

        foreach my $d (@$data) {
            foreach my $sub (@{$QBit::Application::Model::SendMail::FIELD_TYPE{email}->{in}->{hash}}) {
                my $res;
                $res = $sub->($self, $d, $opt, $data_between_steps, $name) if ref($sub) eq 'CODE' && defined($d);
                if (defined $res) {
                    push(@$out, @$res);
                    last;
                }
            }
        }

    }

    my @recipients = ();
    foreach my $elem (@$out) {
        $elem->{name} =~ s/(^\s*|\s*$)//g;

        my @emails = split(/\s*,\s*/, $elem->{email});

        throw Exception::SendMail::BadAddress gettext('Bad email [%s]', $elem->{email})
          if grep {!check_email($_)} @emails;

        if (@emails > 1) {

            $elem->{email} = shift(@emails);

            push(@recipients, map {{name => $elem->{name}, email => $_}} @emails);
        }
    }

    $out = [@$out, @recipients];

    return (($name eq 'from' || $name eq 'reply_to') && @$out ? [$out->[0]] : $out);
}

sub _after_send {
    my ($self, $msg) = @_;

    my $from    = decode_utf8($self->_decode($msg->get('from')));
    my $to      = decode_utf8($self->_decode($msg->get('to')));
    my $subject = decode_utf8($self->_decode($msg->get('subject')));

    INFO "Mail send. From: '$from' To: '$to' Subject: '$subject'";

    my $email_id = $self->partner_db_table()->add(
        {
            dt                   => curdate(oformat => 'db_time'),
            from                 => $self->_decode($msg->get('from')),
            to                   => $self->_decode($msg->get('to')),
            cc                   => $self->_decode($msg->get('cc')),
            bcc                  => $self->_decode($msg->get('bcc')),
            subject              => $self->_decode($msg->get('subject')),
            last_send_successful => $msg->last_send_successful(),
        }
    );

    # Иначе оч. много данных, на dev-partner не хватает места на даиске (#PI-14911)
    unless ($to eq $PARTNER2_CRON_MAIL) {

        my @parts = $msg->parts();
        my $body = (shift @parts || $msg)->data();

        $self->partner_logs_db->email_data->add(
            {
                email_id => $email_id,
                headers  => $msg->fields_as_string($msg->fields()),
                body     => $body,
            }
        );

        if (@parts) {
            $self->partner_logs_db->email_attachment->add_multi(
                [
                    map {
                        {
                            email_id     => $email_id,
                            filename     => $self->_decode($_->filename()),
                            content_type => $_->attr('content-type'),
                            data         => $_->data(),
                        }
                      } @parts
                ]
            );
        }
    }
}

sub _decode {
    my ($self, $string) = @_;
    return undef unless defined($string);
    return $string =~ /^\Q=?UTF-8?B?\E(.+)\Q?=\E(.*)/ ? decode_base64($1) . $2 : $string;
}

sub _do_d_gettext {
    my ($self, $structure) = @_;

    # Aliases $_[x] are aliases of hadled structure.
    my %sub_for = (
        'HASH' => sub {
            $self->_do_d_gettext($_) foreach values %{$_[0]};
        },
        'ARRAY' => sub {
            $self->_do_d_gettext($_) foreach @{$_[0]};
        },
        'REF' => sub {
            $self->_do_d_gettext(${$_[0]});
        },
        'CODE' => sub {
            # The original structure change.
            $_[0] = $_[0]->();
        },
    );
    my $structure_type = ref($structure);
    if (exists($sub_for{$structure_type})) {
        $sub_for{$structure_type}->($_[1]);
    }
}

sub _is_to_one_user_only {
    my ($self, $to) = @_;
    return (ref($to) eq 'HASH' && keys(%$to) == 1 && [%$to]->[0] eq 'user_id' && [%$to]->[1] =~ /^\d+$/);
}

sub _mail_create {
    my ($self, $hash) = @_;

    $hash = clone($hash);

    my $tmp_app_locale;
    $tmp_app_locale = $self->app->set_tmp_app_locale($self->app->get_user_lang($hash->{to}->[0]->{user_id}))
      if ( ref($hash->{to}) eq 'ARRAY'
        && @{$hash->{to}} == 1
        && $self->_is_to_one_user_only($hash->{to}->[0]));

    $self->_do_d_gettext($hash);

    my $msg = $self->SUPER::_mail_create($hash);

    unless ($self->get_option('real_send')) {
        my %old_values = map {$_ => $msg->get($_)} grep {$msg->get($_)} qw(to cc bcc);

        $msg->replace(to => $PARTNERS_TEST_MAIL);
        $msg->replace($_ => '') foreach (qw(cc bcc));

        my $part_with_body = ($msg->parts() ? ($msg->parts())[0] : $msg);
        $part_with_body->data(($part_with_body->data // '')
            . "\n\n-- \nTest letter\nOriginal recipients:\n  "
              . join("\n ", map {uc($_) . ': ' . $self->_decode($old_values{$_})} keys %old_values));
    }

    my $msg_id = time() . '_' . sprintf("%09d", rand(999999999));
    $msg->replace('Message-ID' => "<$msg_id.support\@partner.yandex.ru>");

    return $msg;
}

sub _message_struct_in {
    my ($self, $data, $media, $struct) = @_;

    local $QBit::Application::Model::SendMail::TEMPLATERS{'TT2'} = sub {
        my $d = shift;
        use Template;

        my $wself = $self;
        weaken($wself);

        my $template = Template->new(
            ENCODING     => 'utf8',
            INCLUDE_PATH => [$self->get_option('ApplicationPath') . 'mail_templates',],
            COMPILE_EXT  => '.ttc',
            COMPILE_DIR  => $self->app->get_option('TemplateCachePath')
              || '/tmp/tt_cache-mail' . (
                # Cron currently has no request class, Rosetta has an unblessed hash as a request
                $self->app->can('request')
                  && blessed($self->app->request)
                  && $self->app->request->isa('QBit::WebInterface::Request')
                ? '_' . $self->app->request->server_name() . '-' . $self->app->request->server_port()
                : ''
              )
              . "_$<",
            EVAL_PERL => 1,
            MINIMIZE  => $self->get_option('MinimizeTemplate'),
            (
                defined($d->{'content_type'}) && $d->{'content_type'} ne 'text/plain'
                ? (
                    PRE_CHOMP  => 2,
                    POST_CHOMP => 2,
                    TRIM       => 2,
                  )
                : ()
            ),
            RECURSION  => 1,
            PRE_DEFINE => {

                check_rights => sub {$wself->check_rights(@_)},

                gettext  => sub {return gettext shift,  @_},
                ngettext => sub {return ngettext shift, shift, shift, @_},
                pgettext  => sub {return pgettext shift,  shift, @_},
                npgettext => sub {return npgettext shift, shift, shift, shift, @_},

                dumper => sub {
                    local $Data::Dumper::Terse = 1;
                    my $dump_text = Data::Dumper::Dumper(@_);
                    for ($dump_text) {
                        s/&/&amp;/g;
                        s/ /&nbsp;/g;
                        s/</&lt;/g;
                        s/>/&gt;/g;
                        s/\n/<br>\n/g;
                    }
                    return $dump_text;
                },
                to_json => sub {
                    return to_json(shift);
                },
                format_date => sub {
                    return format_date(shift, shift, %{shift || {}});
                },
                format_number => sub {
                    return format_number(shift, %{shift || {}});
                },
                cmd_link => sub {
                    my ($new_cmd, $new_path, $params) = @_;
                    return $self->app->make_cmd($new_cmd, $new_path, %{$params || {}});
                },
            },
        );

        my $out = '';
        $template->process($d->{template}, $d->{vars}, \$out)
          || throw Exception::SendMail gettext('Template process error [%s]', $template->error());

        return $out;
    };

    return $self->SUPER::_message_struct_in($data, $media, $struct);
}

TRUE;
