package Application::Model::Sentry;

use qbit;

use base qw(QBit::Application::Model);

use Sentry::Raven;
use File::Slurp qw(read_file);
use List::Util qw(min max);

sub accessor {'sentry'}

sub _logger_name {
    my ($self) = @_;

    if (defined($self->app->get_option('restapi_controller'))) {
        return 'RestApi';
    } else {
        return blessed($self->app);
    }

}

sub _prune_at_depth {
    my ($ref, $depth, $current_depth) = @_;
    $current_depth //= 0;

    if (!defined($ref)) {
        return 'undef';
    } elsif (
        (!defined($depth) || $current_depth < $depth) && grep {
            $_ eq ref($ref)
        } (qw(ARRAY HASH))
      )
    {
        if (ref($ref) eq 'ARRAY') {
            my @arr;
            for my $i (0 .. $#$ref) {
                $arr[$i] = _prune_at_depth($ref->[$i], $depth, $current_depth + 1);
            }
            return \@arr;
        } elsif (ref($ref) eq 'HASH') {
            my %hash;
            for my $key (keys(%$ref)) {
                $hash{$key} = _prune_at_depth($ref->{$key}, $depth, $current_depth + 1);
            }
            return \%hash;
        }
    } else {
        $ref //= 'undef';
        return qq[$ref];
    }

    return $ref;
}

sub _set_source_code_context {
    my ($sentry_frame, $qbit_frame) = @_;

    my $source_code_context_range = 10;

    if (-f $qbit_frame->{filename}) {
        my @lines = read_file($qbit_frame->{filename}, {binmode => ':utf8'});
        $sentry_frame->{context_line} = $lines[$qbit_frame->{line} - 1];
        $sentry_frame->{pre_context} =
          [@lines[max(1, $qbit_frame->{line} - $source_code_context_range) - 1 .. $qbit_frame->{line} - 2]];
        $sentry_frame->{post_context} =
          [@lines[$qbit_frame->{line} .. min(scalar(@lines), $qbit_frame->{line} + $source_code_context_range) - 1]];
    }
}

sub _convert_stacktrace {
    my ($exception) = @_;

    my $cs = $exception->{callstack};
    my @st;
    for my $frame (@$cs) {
        my $sentry_frame = {
            filename => $frame->{filename},
            lineno   => $frame->{line},
            module   => $frame->{package},
            function => $frame->{subroutine},
            vars     => {'@_' => _prune_at_depth($frame->{args})},
        };

        _set_source_code_context($sentry_frame, $frame);

        unshift @st, $sentry_frame;
    }

    my $caller_frame = {
        filename => $exception->{filename},
        lineno   => $exception->{line},
        module   => $exception->{package},
    };

    _set_source_code_context($caller_frame, $exception);

    push @st, $caller_frame;

    return \@st;
}

sub _filter_oauth_token {
    my ($str) = @_;
    $str =~ s/oauth_token=[^&]+(&|$)/oauth_token=FILTERED$1/ if defined $str;
    return $str;
}

sub _filtered_env {
    my %env = %ENV;
    delete $env{HTTP_COOKIE};

    for ($env{QUERY_STRING}, $env{REQUEST_URI}) {
        $_ = _filter_oauth_token($_);
    }

    my @secrets = grep {m/SECRET|PASSWORD/} keys(%env);
    for (@secrets) {
        $env{$_} = 'FILTERED';
    }

    return \%env;
}

sub context {
    my ($self, %opts) = @_;
    my %context;
    my $app_type = blessed($self->app);

    if (grep {$_ eq $app_type} (qw(WebInterface API IntAPI))) {
        my @available_headers = (
            [host              => 'Host'],
            [referer           => 'Referer'],
            ['user-agent'      => 'User agent'],
            ['remote-addr'     => 'Remote address'],
            [accept            => 'Accept'],
            ['accept-encoding' => 'Accept encoding'],
            ['accept-language' => 'Accept languages'],
            [cookie            => 'Cookie']
        );

        my $headers = {map {$_->[1] => ($self->app->request->http_header($_->[0]) || '')} @available_headers};
        $headers->{Cookie} =~ s/^.*(yandex_login=[^;]+);.*$/$1/ if defined($headers->{'Cookie'});

        my $url = $self->app->request->url;
        $url = _filter_oauth_token($url);

        %context = (
            culprit => $url,
            Sentry::Raven->request_context(
                $url,
                method  => $self->app->request->method,
                headers => $headers,
                env     => $self->_filtered_env(),
            ),
        );
    } elsif ($app_type eq 'Rosetta') {
        my $rosetta_request = $self->app->request // ['UNDEF'];

        my $culprit;
        if ($rosetta_request->[0] eq 'call') {
            my $accessor = $rosetta_request->[1]{model};
            my $method   = $rosetta_request->[1]{method};
            $culprit = join('::', blessed($self->app->$accessor), $method);
        } else {
            $culprit = join('::', $app_type, $rosetta_request->[0]);
        }

        my $headers = $self->app->get_option('http_headers');
        $headers->{cookie} =~ s/^.*(yandex_login=[^;]+);.*$/$1/ if defined($headers->{'Cookie'});

        %context = (
            culprit => $culprit,
            extra   => {request => $rosetta_request},
            Sentry::Raven->request_context(
                $headers->{referer} // 'unknown',
                headers => $headers,
                env     => $self->_filtered_env(),
            ),
        );
    } elsif ($app_type eq 'Cron') {
        my $path   = $self->app->get_option('cron_path');
        my $method = $self->app->get_option('cron_method');
        %context = (
            tags => {
                cron_path   => $path,
                cron_method => $method,
            },
            culprit => qq[$app_type\->new->do("$path", "$method")]
        );
    } elsif (my $controller = $self->get_option('restapi_controller')) {
        my $req     = $controller->req;
        my $headers = $req->headers->to_hash;
        $headers->{'Cookie'} =~ s/^.*(yandex_login=[^;]+);.*$/$1/ if defined($headers->{'Cookie'});
        my $env = $req->env;
        delete $env->{HTTP_COOKIE};
        my $url = $controller->get_current_url();
        if ($env->{'HTTP_X_REQUEST_ORIGIN'} // '' eq 'Frontend') {
            my $mojo_url = Mojo::URL->new($url);
            $url = $mojo_url->path(Mojo::Path->new('/restapi/')->merge($mojo_url->path->leading_slash(0)))->to_string;
        }
        %context = (
            culprit => join(' ', $req->method, $req->url->path->to_string),
            Sentry::Raven->request_context(
                $url,
                method  => $req->method,
                headers => $headers,
                env     => _prune_at_depth($env, 1),
            )
        );
    }

    $context{extra}{time_of_process} = change_time_format($self->app->get_time());
    $context{extra}{PID}             = $$;
    $context{level}                  = $opts{level};
    return %context;
}

sub send_exception {
    my ($self, $exception, %opts) = @_;

    my $raven = Sentry::Raven->new(
        sentry_dsn  => $self->get_option('dsn'),
        servername  => $self->get_option('hostname'),
        environment => $self->get_option('stage'),
        timeout     => 1,
        release     => $self->get_option('version'),
    );

    my $cur_user = $self->get_option('cur_user');
    my %context  = $self->context(%opts);
    if (ref($context{extra}) eq 'HASH') {
        %{$context{extra}} = (%{$context{extra}}, %{$exception->{sentry}{extra} // {}});
    } else {
        $context{extra} = $exception->{sentry}{extra};
    }

    my %sentry_data = (
        type        => blessed($exception),
        logger      => $self->_logger_name(),
        fingerprint => $exception->{sentry}{fingerprint},
        Sentry::Raven->user_context(id => $cur_user->{id}, username => $cur_user->{login}),
        Sentry::Raven->stacktrace_context(_convert_stacktrace($exception)),
        %context,
    );

    $self->app->errorbooster->convert_and_send($exception, \%sentry_data);

    delete $exception->{sentry};

    return TRUE;
}

1;
