package Application::Model::SimpleNotification;

use Exception::Form;
use Exception::Validator::Fields;

use PiConstants qw($SIMPLE_NOTIFICATION_TYPES $SELFEMPLOYED_STATUS_AVAILABLE $SELFEMPLOYED_STATUS_NOT);
use qbit;
use base qw(
  Application::Model::Common
  RestApi::MultistateModel
  Application::Model::ValidatableMixin
  );

use Utils::JSON qw(fix_type_for_complex);

sub accessor      {'simple_notification'}
sub db_table_name {'simple_notification'}

__PACKAGE__->register_rights(
    [
        {
            name        => 'simple_notification',
            description => d_gettext('Rights to use simple notification'),
            rights      => {
                simple_notification_view_action_log         => d_gettext('Right to view %s', 'simple_notification log'),
                simple_notification_view_field__roles       => d_gettext('Right to view %s', 'roles'),
                simple_notification_edit_field__roles       => d_gettext('Right to edit %s', 'roles'),
                simple_notification_edit_field__message     => d_gettext('Right to edit %s', 'message'),
                simple_notification_edit_field__ntype       => d_gettext('Right to edit %s', 'ntype'),
                simple_notification_edit_field__icon        => d_gettext('Right to edit %s', 'icon'),
                simple_notification_edit_field__link        => d_gettext('Right to edit %s', 'link'),
                simple_notification_edit_field__short_name  => d_gettext('Right to edit %s', 'short_name'),
                simple_notification_edit_field__title       => d_gettext('Right to edit %s', 'title'),
                simple_notification_edit_field__button_text => d_gettext('Right to edit %s', 'button_text'),
            }
        }
    ]
);

__PACKAGE__->model_accessors(
    rbac       => 'Application::Model::RBAC',
    partner_db => 'Application::Model::PartnerDB',
);

sub get_product_name {
    gettext('simple_notification');
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        id      => {default => TRUE, db => TRUE, pk => TRUE, type => 'number'},
        message => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                type => 'hash',
                fields =>
                  {map {$_ => {len_max => 1024, len_min => 1,},} sort keys %{$self->app->get_option('locales')}},
            },
        },
        ntype => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {in => $SIMPLE_NOTIFICATION_TYPES,},
        },
        icon => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {optional => TRUE, len_max => 1024},
        },
        link => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {optional => TRUE, len_max => 1024},
        },
        short_name => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {optional => TRUE, len_max => 255},
        },
        title => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'hash',
                fields   => {map {$_ => {len_max => 1024,},} sort keys %{$self->app->get_option('locales')}},
            },
        },
        button_text => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'hash',
                fields   => {map {$_ => {len_max => 1024,},} sort keys %{$self->app->get_option('locales')}},
            },
        },
        params => {
            depends_on => ['id', 'ntype'],
            default    => TRUE,
            type       => 'complex',
            get        => sub {
                my ($fields, $row) = @_;

                my $cur_user = $fields->{'__CUR_USER__'};
                my $p;
                if ($row->{ntype} eq 'common_offer') {
                    if (defined $cur_user->{common_offer}) {
                        $p->{deadline} = $cur_user->{common_offer};
                    }
                }
                if ($row->{ntype} eq 'custom_link') {
                    $p->{link} = $fields->{'__CUSTOM_LINK__'}{$row->{id}};
                }
                return $p;
            },
            api => 1,
        },
        roles => {
            forced_depends_on => ['id'],
            get               => sub {
                my ($fields, $row) = @_;

                $fields->{'__SIMPLE_NOTIFICATION_ROLES__'}->{$row->{'id'}} // [];
            },
            type       => 'array',
            sub_type   => 'number',
            api        => 1,
            need_check => {
                type  => 'array',
                check => sub {
                    my ($qv, $value) = @_;
                    my %valid_roles = map {$_->{id} => TRUE} @{$qv->app->rbac->get_roles()};
                    for (@$value) {
                        throw Exception::Validator::Fields gettext('Bad role')
                          unless $valid_roles{$_};
                    }
                },
            },
        },
        fields_depends => {
            depends_on => ['id'],
            get        => sub {
                {};
            },
            type => 'complex'
        },
        show_message => {
            default    => TRUE,
            depends_on => ['id', 'multistate', 'ntype', 'roles', 'params', 'short_name'],
            get        => sub {
                my ($fields, $row) = @_;

                my $roles_cur_user = $fields->{'__CUR_USER_ROLES__'};

                if ($row->{multistate} & $fields->{'__STATE_DELETED__'}) {
                    return 0;
                }

                if ($row->{ntype} eq 'common_offer') {
                    return exists $row->{params}{deadline} ? 1 : 0;
                }

                if ($row->{ntype} eq 'custom_link') {
                    return exists $fields->{'__CUSTOM_LINK__'}{$row->{id}} ? 1 : 0;
                }

                if ($row->{ntype} eq 'action_banner' && defined $row->{short_name}) {
                    if (in_array($row->{short_name}, ['fill_requisites', 'fill_requisites_games'])) {
                        return FALSE if $fields->{'__CUR_USER__'}{'is_form_done'};
                        return TRUE
                          if ($row->{short_name} eq 'fill_requisites' && !$fields->{'__CUR_USER__'}{'is_games'});
                        return TRUE
                          if ($row->{short_name} eq 'fill_requisites_games' && $fields->{'__CUR_USER__'}{'is_games'});
                    } elsif ($row->{short_name} eq 'welcome_web') {
                        return TRUE
                          unless scalar @{
                                  $fields->model->app->context_on_site_campaign->get_all(
                                      fields => ['public_id'],
                                      filter => {multistate => 'not deleted'},
                                      limit  => 1,
                                  )
                              };
                    } elsif ($row->{short_name} eq 'welcome_mobile') {
                        return TRUE
                          unless scalar @{
                                  $fields->model->app->mobile_app_settings->get_all(
                                      fields => ['public_id'],
                                      filter => {multistate => 'not deleted'},
                                      limit  => 1,
                                  )
                              };
                    } elsif ($row->{short_name} eq 'become_self_employed') {
                        return 1
                          if defined($fields->{'__CUR_USER__'}{'self_employed'})
                              && in_array($fields->{'__CUR_USER__'}{'self_employed'},
                                  [$SELFEMPLOYED_STATUS_AVAILABLE, $SELFEMPLOYED_STATUS_NOT]);
                    } elsif ($row->{short_name} eq 'goto_payoneer') {
                        return 1
                          if ( defined($fields->{'__CUR_USER__'}{'payoneer'})
                            && defined($fields->{'__CUR_USER__'}{'payoneer_step'})
                            && ($fields->{'__CUR_USER__'}{'payoneer_step'} < 8));
                    } elsif ($row->{short_name} =~ /^goto_game_offer_/) {
                        unless (defined $fields->{'__CAN_GAME_OFFER__'}) {
                            my $user =
                              $fields->model->app->users->check_game_offer({id => $fields->{'__CUR_USER__'}{id}});
                            if (!$user || $user->{has_game_offer}) {
                                $fields->{'__CAN_GAME_OFFER__'} = {can => FALSE};
                            } else {
                                $fields->{'__CAN_GAME_OFFER__'} = {
                                    can   => TRUE,
                                    rez   => $fields->model->app->documents->is_resident($user->{active_contract}),
                                    is_by => in_array(149, $fields->{'__CUR_USER__'}{country_id}),
                                };
                            }
                        }
                        if ($fields->{'__CAN_GAME_OFFER__'}{can}) {
                            if ($fields->{'__CAN_GAME_OFFER__'}{rez}) {
                                if ($row->{short_name} eq 'goto_game_offer_rez') {
                                    return TRUE;
                                }
                            } else {
                                if ($row->{short_name} eq 'goto_game_offer_notrez') {
                                    return $fields->{'__CAN_GAME_OFFER__'}{is_by} ? FALSE : TRUE;
                                } elsif ($row->{short_name} eq 'goto_game_offer_belarus') {
                                    return $fields->{'__CAN_GAME_OFFER__'}{is_by} ? TRUE : FALSE;
                                }
                            }
                        }
                    } elsif ($row->{short_name} eq 'ua_payment_banner') {
                        return $fields->model->app->users->has_feature($fields->{'__CUR_USER__'}->{id},
                            'ua_payment_banner');
                    }
                    return 0;
                }

                if (grep {$roles_cur_user->{$_}} @{$row->{roles}}) {
                    return 1;
                }
                return 0;
            },
            type => 'boolean',
            api  => 1
        },
        multistate      => {default => TRUE, db => TRUE, type => 'number', api => 1},
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                my ($fields, $row) = @_;

                $fields->model->get_multistate_name($row->{'multistate'});
            },
            type => 'string',
            api  => 1
        },
        public_id => {
            db      => TRUE,
            db_expr => 'id',
            type    => 'string',
        },
        status => {
            depends_on => ['id'],
            get        => sub {
                'sinc'
            },
            type => 'string',
            api  => 1
        },
        editable_fields => {
            forced_depends_on => [qw(id)],
            get               => sub {
                my ($fields, $row) = @_;

                return $fields->model->get_editable_fields();
            },
            type     => 'complex',
            fix_type => \&fix_type_for_complex,
            api      => 1
        },
        available_fields => {
            depends_on => [qw(id ntype multistate)],
            label      => d_gettext('Available fields'),
            get        => sub {
                my ($fields, $row) = @_;

                return $fields->model->get_available_fields($row);
            },
            type     => 'complex',
            api      => 1,
            fix_type => \&fix_type_for_complex,
        },
        actions => {
            label      => d_gettext('Actions'),
            depends_on => [qw(id multistate)],
            get        => sub {
                my ($fields, $row) = @_;

                $fields->model->get_actions($row);
            },
            type     => 'complex',
            fix_type => \&fix_type_for_complex,
            api      => 1
        },
    };
}

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id         => {type => 'number',     label => d_gettext('ID')},
        message    => {type => 'text',       label => d_gettext('Short caption')},
        multistate => {type => 'multistate', label => d_gettext('Status')},
        ntype      => {
            type   => 'dictionary',
            label  => gettext('Type'),
            values => [map {{id => $_, label => $_}} @$SIMPLE_NOTIFICATION_TYPES],
        }
    },
);

__PACKAGE__->multistates_graph(
    empty_name    => d_gettext('Working'),
    multistates   => [[deleted => d_gettext('Deleted')]],
    right_actions => {
        delete => d_gettext('Delete'),
        add    => d_gettext('Add'),
        edit   => d_gettext('Edit'),
    },
    right_group        => [simple_notification_action => d_gettext('Right to manage notification')],
    right_name_prefix  => 'simple_notification_action_',
    multistate_actions => [
        {
            action    => 'delete',
            from      => 'not deleted',
            set_flags => ['deleted']
        },
        {
            action => 'add',
            from   => '__EMPTY__',
        },
        {
            action => 'edit',
            from   => 'not deleted',
        },
    ],
);

#API

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    my @ids;
    my $has_custom_link;
    for my $row (@$result) {
        push @ids, $row->{id};
        if ($row->{ntype} && $row->{ntype} eq 'custom_link') {
            $has_custom_link = $row->{id};
        }
    }

    if ($fields->need('roles')) {
        map {push(@{$fields->{'__SIMPLE_NOTIFICATION_ROLES__'}{$_->{'notification_id'}}}, $_->{'role_id'})} @{
            $self->partner_db->simple_notification_roles->get_all(
                fields => [qw(notification_id role_id)],
                filter => {notification_id => [sort {$a <=> $b} @{array_uniq(@ids)}]},
            )
          };
    }

    if ($fields->need('params') || $fields->need('show_message')) {
        $fields->{'__CUR_USER__'} = $self->get_option('cur_user', {});

        # получаем недостаюшие поля текущего пользователя
        my $additional_fields = $self->app->users->get($fields->{'__CUR_USER__'}{id},
            fields => ['self_employed', 'payoneer', 'payoneer_step', 'is_form_done', 'is_games']);
        $fields->{'__CUR_USER__'}{$_} = $additional_fields->{$_} for keys %$additional_fields;

        # получаем данные формы
        my $form_data = $self->partner_db->form_data->get_all(
            fields => ['country_id'],
            filter => [user_id => '=' => \$fields->{'__CUR_USER__'}{id}],
        );
        $fields->{'__CUR_USER__'}{country_id} = [map $_->{country_id}, @$form_data];

        if ($has_custom_link) {
            my $list = $self->partner_db->simple_notification_custom_link->get_all(
                fields => [qw(link)],
                filter => {login => $fields->{'__CUR_USER__'}{login}},
            );
            if ($list && $list->[0]) {
                $fields->{'__CUSTOM_LINK__'}{$has_custom_link} = $list->[0]{link};
            }
        }
    }

    if ($fields->need('show_message')) {
        $fields->{'__CUR_USER_ROLES__'} = {map {$_ => 1} sort keys %{$fields->{'__CUR_USER__'}{roles}}};
        $fields->{'__STATE_DELETED__'} = $self->get_multistate_by_name('deleted');
    }
}

sub api_can_add {TRUE}

sub get_available_fields {
    my ($self, $obj) = @_;
    if ($obj && $self->check_multistate_flag($obj->{'multistate'}, 'deleted')) {
        return {};
    }

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    my $accessor = $self->accessor();
    $self->app->delete_field_by_rights(\%fields, {$accessor . '_view_field__%s' => [qw(roles)]});

    return \%fields;
}

sub get_add_fields {
    my ($self, %opts) = @_;

    my %fields;
    foreach my $field (qw(message roles ntype icon link short_name title button_text)) {
        $fields{$field} = TRUE if $self->check_short_rights("edit_field__$field");
    }

    return \%fields;
}

sub get_editable_fields {
    my ($self) = @_;
    my %res;

    foreach (qw(message roles ntype icon link short_name title button_text)) {
        $res{$_} = TRUE if $self->check_short_rights('edit_field__' . $_);
    }

    return \%res;
}

sub add {
    my ($self, %opts) = @_;
    throw Exception::Denied gettext('Access denied')
      unless $self->check_rights($self->get_rights_by_actions('add'));

    my $add_fields = $self->get_add_fields();
    $self->check_bad_fields_in_add(\%opts, $add_fields);
    $self->app->validator->check(
        data  => \%opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => [keys(%$add_fields)]),
    );

    my $roles = delete $opts{roles};

    my $id;
    $self->partner_db->transaction(
        sub {
            $id = $self->partner_db_table()->add(\%opts);
            $self->partner_db->simple_notification_roles->add_multi(
                [map {{notification_id => $id, role_id => $_}} @$roles]);
            $self->do_action($id, 'add');
        }
    );
    return $id;
}

sub on_action_edit {
    my ($self, $obj, %opts) = @_;

    my $edit_fields = $self->check_bad_fields($obj, \%opts);
    $self->app->validator->check(
        data  => \%opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => [keys(%$edit_fields)]),
    );

    my $roles = delete $opts{roles};

    $self->partner_db->transaction(
        sub {
            $self->partner_db->simple_notification_roles->delete(
                $self->partner_db->filter({notification_id => $obj->{id}}));
            $self->partner_db_table()->edit($obj, \%opts);
            $self->partner_db->simple_notification_roles->add_multi(
                [map {{notification_id => $obj->{id}, role_id => $_}} @$roles]);
        }
    );

    return FALSE;
}

sub query_filter {
    my ($self, $filter) = @_;

    my @roles_cur_user = sort {$a <=> $b} keys(%{$self->get_option('cur_user', {})->{roles}});
    $filter->and(
        [
            id => '= ANY' => $self->partner_db->query->select(
                table  => $self->partner_db->simple_notification_roles,
                fields => {'notification_id' => ""},
                filter => [role_id => '=' => \\@roles_cur_user],
            )
        ]
    ) unless $self->check_rights('simple_notification_view_all');

    return $filter;
}

sub api_available_actions {
    return qw(delete edit);
}

1;
