package Application::Model::Source;

use qbit;

use base qw(Application::Model::Product);

use PiConstants qw($PARTNER2_CRON_MAIL);

__PACKAGE__->abstract_methods(qw(get_page_model_names));

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    $self->register_rights($self->get_structure_rights_to_register());
    $self->model_fields($self->get_structure_model_fields());
    $self->model_filter($self->get_structure_model_filter());
    $self->multistates_graph($self->get_structure_multistates_graph());
}

sub get_structure_model_accessors {
    my ($class) = @_;

    return {
        partner_db         => 'Application::Model::PartnerDB',
        users              => 'Application::Model::Users',
        rbac               => 'Application::Model::RBAC',
        mail_notification  => 'Application::Model::MailNotification',
        text_template      => 'Application::Model::TextTemplate',
        user_notifications => 'Application::Model::UserNotifications',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    return [
        {
            name        => $self->accessor(),
            description => d_gettext('Right to manage %s', $self->accessor()),
            rights      => {
                map {$self->get_description_right($_)}
                  qw(
                  view
                  view_all
                  add_other
                  view_action_log
                  ),
            }
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $FIELDS_DEPENDS;

    return {
        id        => {default => TRUE, db => TRUE, pk => TRUE, label => d_gettext('ID'), type => 'number', api => TRUE},
        public_id => {
            db      => TRUE,
            db_expr => 'id',
            type    => 'string',
            api     => 1,
        },
        create_date => {db => TRUE, label => d_gettext('Create date'), type => 'string'},
        multistate  => {db => TRUE, label => d_gettext('Status'),      type => 'number', api => TRUE},
        multistate_name => {
            depends_on => ['multistate'],
            label      => d_gettext('Multistate name'),
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => TRUE,
        },
        is_approved => {
            depends_on => [qw(multistate)],
            get        => sub {
                return $_[0]->model->check_multistate_flag($_[1]->{'multistate'}, 'approved');
            },
            type => 'boolean'
        },
        product_count => {
            depends_on => [qw(id)],
            get        => sub {
                $_[0]->{'__PRODUCT_COUNT__'}{$_[1]->{'id'}} // {};
            },
            type => 'complex',
            api  => TRUE,
        },
        fields_depends => {
            get => sub {
                $FIELDS_DEPENDS //= $_[0]->model->get_fields_depends();

                return $FIELDS_DEPENDS;
            },
            type => 'complex',
            api  => 1,
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            id         => {type => 'number',     label => d_gettext('ID')},
            multistate => {type => 'multistate', label => d_gettext('Status')},
        },
    };
}

sub pre_process_fields {
    my ($self, $fields, $result, %opts) = @_;

    if ($fields->need('product_count')) {
        $fields->{'__PRODUCT_COUNT__'} = {};

        my $source_ids = array_uniq(map {$_->{'id'}} @$result);

        my @page_model_names = $self->get_page_model_names;

        my %block_count_fields = ();
        foreach my $model (@page_model_names) {
            $block_count_fields{$model} = [map {$_ . '_count'} @{$self->$model->get_block_model_names()}];
        }

        foreach my $source_id (@$source_ids) {
            $fields->{'__PRODUCT_COUNT__'}{$source_id} = {
                (map {$_ . '_count' => 0} @page_model_names),
                map {$_ => 0} map {@{$block_count_fields{$_}}} keys(%block_count_fields)
            };
        }

        my $tmp_rights = $self->app->add_tmp_rights(
            qw(context_on_site_campaign_view_field__domain_id search_on_site_campaign_view_field__domain_id));

        foreach my $model (keys(%block_count_fields)) {
            my $pages = $self->$model->get_all(
                fields => ['source_id', @{$block_count_fields{$model}}],
                filter => {source_id => $source_ids, multistate => 'not deleted'}
            );

            foreach my $row (@$pages) {
                my $count = $fields->{'__PRODUCT_COUNT__'}{$row->{'source_id'}};

                $count->{$model . '_count'}++;

                foreach (@{$block_count_fields{$model}}) {
                    $count->{$_} += $row->{$_} || 0;
                }
            }
        }
    }
}

sub send_letter_after_reject {
    my ($self, %opts) = @_;

    $opts{'reason'} = [$opts{'reason'}] if ($opts{'reason'} && !ref $opts{'reason'});
    $self->app->validator->check(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                user_id   => {type => 'int_un'},
                full_name => {},
                resource  => {},
                reason => {optional => TRUE, type => 'array'},
            }
        },
        throw => TRUE,
    );

    my @reasons;
    if ($opts{'reason'}) {
        my $reasons = $self->moderation_reason->get_all(
            distinct => TRUE,
            fields   => ['partner_txt'],
            filter   => [AND => [{multistate => 'not deleted'}, [id => IN => $opts{'reason'}],]],
            order_by => [qw(group_id)]
        );

        @reasons = map {$_->{partner_txt}} @$reasons;

    }

    $self->mail_notification->add_when_resource_rejected_by_moderation(
        {
            user_id   => $opts{'user_id'},
            resource  => $opts{'resource'},
            full_name => $opts{'full_name'},
            reasons   => \@reasons,
        }
    );

    my $mail_notification = $self->_get_reject_mail_notification();
    $self->mail_notification->$mail_notification(
        {user_id => $opts{'user_id'}, resource => $opts{resource}, reasons => \@reasons});

    my $tmp = $self->app->add_tmp_rights('do_user_notifications_add');
    $self->user_notifications->add(
        notification_id => 3,
        user_id         => $opts{'user_id'},
        accessor        => $self->accessor,
        custom_data     => {resource => $opts{'resource'}, reason => \@reasons},
    );
}

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $opts->{'create_date'} //= curdate(oformat => 'db_time');
}

TRUE;
