package Application::Model::StatDownloadData;

=encoding UTF-8

=cut

use qbit;

use base qw(Application::Model::DBManager::Base);

use Digest::CRC;
use File::Copy qw(move);
use Fcntl;
use File::ReadBackwards;
use Utils::Logger qw(INFO WARN);

use Exception::StatDownloadData;

sub accessor      {'stat_download_data'}
sub db_table_name {'stat_download_data'}

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB');

__PACKAGE__->model_fields(
    id          => {db      => TRUE, pk => TRUE},
    method      => {default => TRUE, db => TRUE},
    stat_date   => {default => TRUE, db => TRUE},
    hostname    => {default => TRUE, db => TRUE},
    checksum    => {db      => TRUE},
    update_time => {db      => TRUE},
    filename    => {db      => TRUE},
    sync        => {default => TRUE, db => TRUE},
    filepath    => {
        depends_on => [qw(filename)],
        get        => sub {
            return $_[0]->model->get_option('local_stat_cache_dir') . $_[1]->{filename};
          }
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id          => {type => 'number',  label => 'id'},
        method      => {type => 'text',    label => 'stat product'},
        stat_date   => {type => 'date',    label => 'stat_date'},
        hostname    => {type => 'text',    label => 'hostname'},
        checksum    => {type => 'text',    label => 'source_hashes'},
        update_time => {type => 'date',    label => 'update_time'},
        sync        => {type => 'boolean', label => 'sync'},
    }
);

sub cache_http_data {
    my ($self, %opts) = @_;
    # %opts = (
    #    method   => 'get_publisher_stat',
    #    date     => '2017-04-01',
    #    path     => $tmp_path,
    #    remove_end_marker => "#END\n",
    # );

    my ($method, $date, $src_path) = @opts{qw( method date path )};

    my $update_time = curdate(oformat => 'db_time');

    my $dst_name = $self->_get_cache_file_name($method, $date, $update_time);
    my $dst_path = $self->_get_cache_file_path($dst_name);

    move $src_path, $dst_path or throw Exception::StatDownloadData "Failed to move $src_path to $dst_path: $!";

    if ($opts{remove_end_marker}) {
        my $bw = File::ReadBackwards->new($dst_path)
          or throw Exception::StatDownloadData "Failed to read $dst_path: $!";
        my $last_line = $bw->readline();
        throw Exception::StatDownloadData "Last line of $dst_path is $last_line, expected $opts{remove_end_marker}"
          unless $last_line eq $opts{remove_end_marker};
        my $truncate_pos = $bw->tell();
        $bw->close();
        sysopen(my $fh, $dst_path, O_RDWR) or throw Exception::StatDownloadData "Failed to sysopen $dst_path: $!";
        $fh->truncate($truncate_pos) or throw Exception::StatDownloadData "Failed to truncate $dst_path: $!";
    }

    INFO "Moved $src_path to $dst_path";

    open(my $fh, '<', $dst_path) or throw Exception::StatDownloadData "Failed to open $dst_path: $!";
    my $crc = Digest::CRC->new(type => 'crc32')->addfile($fh)->digest;
    close($fh) or throw Exception::StatDownloadData "Failed to close $dst_path: $!";

    $self->partner_db->stat_download_data->add(
        {
            method      => $method,
            stat_date   => $date,
            hostname    => $self->get_option('hostname'),
            checksum    => $crc,
            update_time => $update_time,
            filename    => $dst_name,
            sync        => 0,
        }
    );

    return 1;
}

sub get_all_most_recent {
    my ($self, %opts) = @_;

    my $fields = $self->_get_fields_obj($opts{'fields'}, $opts{'all_locales'});

    my $query = $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => $fields->get_db_fields(),
        filter => $self->get_db_filter($opts{'filter'}),
        alias  => 'all_files',
      )->join(
        table => $self->partner_db->query->select(
            table  => $self->partner_db_table(),
            fields => {
                method    => 'method',
                stat_date => 'stat_date',
                max_ut    => {MAX => ['update_time']},
            },
          )->group_by('method', 'stat_date'),
        fields  => [qw()],
        alias   => 'most_recent_files',
        join_on => [
            AND => [
                [{'method'    => 'all_files'}         => '=' => 'method'],
                [{'stat_date' => 'all_files'}         => '=' => 'stat_date'],
                [{'max_ut'    => 'most_recent_files'} => '=' => {'update_time' => 'all_files'}]
            ]
        ],
      );

    my $result = $query->get_all();

    if (@$result) {
        $self->pre_process_fields($fields, $result);
        $result = $fields->process_data($result);
    }

    return $result;
}

sub _get_cache_file_name {
    my ($self, $method, $date, $update_time) = @_;
    $update_time =~ s/\s/_/g;
    $update_time =~ tr/:/-/;
    return sprintf('%s__%s__%s.dump', $method, $date, $update_time);
}

sub _get_cache_file_path {
    my ($self, $filename) = @_;
    return $self->get_option('local_stat_cache_dir') . $filename;
}

sub call_rsync {
    my (@args) = @_;
    my $exit_code = -1;
    my @cmd = ('rsync', @args);
    foreach (1 .. 3) {
        $exit_code = system(@cmd);
        last if $exit_code == 0;
    }
    INFO '`' . join(' ', @cmd) . '` exit_code = ' . $exit_code;
    return $exit_code;
}

sub pull_changes_from_other_host {
    my ($self) = @_;

    my $local_dir  = $self->get_option('local_stat_cache_dir');
    my $remote_dir = $self->get_option('remote_stat_cache_dir');

    for ($local_dir, $remote_dir) {
        $_ .= '/' unless substr($_, -1) eq '/';
    }

    $self->partner_db->transaction(
        sub {
            my $unsynced = $self->get_all(
                fields     => [qw(id filename filepath hostname)],
                filter     => [AND => [[sync => '=' => 0], [hostname => '<>' => $self->get_option('hostname')],]],
                for_update => 1,
            );

            my @created;
            my @deleted;

            for my $file (@$unsynced) {
                my $exit_code =
                  call_rsync('--dirs', '--delete', "--include=$file->{filename}", '--exclude=*',
                    $file->{hostname} . $remote_dir, $local_dir);
                if ($exit_code == 0) {
                    if (-e $file->{filepath}) {
                        INFO "Downloaded $file->{filename}";
                        push @created, $file;
                    } else {
                        INFO "Deleted $file->{filename}";
                        push @deleted, $file;
                    }
                } else {
                    INFO "rsync failed with exit_code = $exit_code";
                }
            }

            $self->partner_db_table()->edit(\@created, {sync => 1}) if @created;
            $self->partner_db_table()->delete(\@deleted) if @deleted;
        }
    );
}

sub delete_cached_data {
    my ($self, $filter) = @_;

    $self->partner_db->transaction(
        sub {
            my $to_be_deleted = $self->get_all(
                fields     => [qw(id filename filepath sync)],
                filter     => [AND => [$filter, [hostname => '=' => $self->get_option('hostname')]]],
                for_update => 1,
            );

            my @unsynced;
            my @deleted;

            for my $file (@$to_be_deleted) {
                if (unlink($file->{filepath})) {
                    if ($file->{sync}) {
                        push @unsynced, $file;
                    } else {
                        push @deleted, $file;
                    }
                    INFO "Deleted $file->{filepath}";
                } else {
                    WARN "Failed to unlink $file->{filepath}: $!";
                }
            }

            $self->partner_db_table()->edit(\@unsynced, {sync => 0}) if @unsynced;
            $self->partner_db_table()->delete(\@deleted) if @deleted;
        }
    );
}

TRUE;
