package Application::Model::Statistics;

=encoding UTF-8

=cut

use qbit;

use base qw(QBit::Application::Model Application::Model::Statistics::_Utils::Money);

use List::Util qw(shuffle);

use Application::Model::Statistics::_Utils::Query;

use QBit::Application::Model::DB::clickhouse::String;

use QBit::Validator;

use Exception::Statistics::InvalidLevel;
use Exception::Validation::BadArguments;
use Exception::DB;

use PiConstants qw($CH_MONEY_POST_SCALE);

use Utils::Logger qw(WARN);

our $TAGS_LIMIT = 50;

sub accessor {'statistics'}

__PACKAGE__->model_accessors(
    bk_statistics                => 'Application::Model::BKStatistics',
    clickhouse_db                => 'Application::Model::ClickhouseDB',
    cur_user                     => 'Application::Model::CurUser',
    currency                     => 'Application::Model::Currency',
    geo_base                     => 'Application::Model::GeoBase',
    memcached                    => 'QBit::Application::Model::Memcached',
    partner_db                   => 'Application::Model::PartnerDB',
    product_manager              => 'Application::Model::ProductManager',
    rbac                         => 'Application::Model::RBAC',
    stat_report_params_digest    => 'Application::Model::Statistics::ReportParamsDigest',
    statistics_additional_income => 'Application::Model::Statistics::AdditionalIncome',
    statistics_charging          => 'Application::Model::Statistics::Charging',
    users                        => 'Application::Model::Users',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'statistics',
            description => d_gettext('Rights for statistics'),
            rights      => {
                statistics_view__partner_share_text => d_gettext('View "Partner share" text'),
                statistics_update_view              => d_gettext('Right to view statistic update tasks'),
                statistics_view_all                 => d_gettext('Right to view all statistics'),
            }
        }
    ]
);

sub get_levels_for_monitoring {
    my ($self) = @_;

    return sort @{$self->product_manager->get_statistics_products()};
}

sub get_available_level {
    my ($self, $level) = @_;

    throw Exception::Statistics::InvalidLevel gettext('Invalid level')
      unless ref($level) eq 'HASH' && exists($level->{'id'});

    my $stat_level = $self->get_level($level->{'id'});

    throw Exception::Statistics::InvalidLevel gettext('Invalid level name "%s"', $level->{'id'})
      unless defined($stat_level) && $stat_level->is_available();

    return $stat_level;
}

sub get_level {
    my ($self, $name) = @_;

    return $self->{'__TREE_ENTITIES_HS__'}{$name} // '';
}

sub get_level_by_product_accessor {
    my ($self, $product_accessor) = @_;

    throw Exception gettext('Can not support product accessor "users"') if $product_accessor eq 'users';

    unless (exists($self->{'__LEVEL_BY_PRODUCT_ACCESSOR__'}{$product_accessor})) {
        foreach (values(%{$self->{'__TREE_ENTITIES_HS__'}})) {
            my $level_product_accessor = $_->get_product->accessor();

            $self->{'__LEVEL_BY_PRODUCT_ACCESSOR__'}{$product_accessor} = $_
              if $level_product_accessor eq $product_accessor;
        }

        throw Exception gettext('Level not found by product accessor "%s"', $product_accessor)
          unless exists($self->{'__LEVEL_BY_PRODUCT_ACCESSOR__'}{$product_accessor});
    }

    return $self->{'__LEVEL_BY_PRODUCT_ACCESSOR__'}{$product_accessor};
}

sub get_level_full_title {
    my ($self, $name) = @_;

    my $level = $self->get_level($name)
      // throw Exception::Validation::BadArguments gettext('Unknown statistics level "%s"', $name);

    return join(' / ', (map {$self->get_level($_)->title()} @{$level->{'path'}}), $level->title());
}

sub get_level_ids_for_reloading {
    my ($self) = @_;

    return map {$self->app->$_->id}
      grep {$self->app->$_->isa('Application::Model::Statistics::Update') && $self->app->$_->can_be_reloaded()}
      keys(%{$self->app->get_models()});
}

sub get_level_sort_priority {
    my ($self, $name) = @_;

    my $level = $self->get_level($name)
      // throw Exception::Validation::BadArguments gettext('Unknown statistics level "%s"', $name);

    return $level->can('sort_priority') ? $level->sort_priority() : 0;
}

sub get_tree {
    my ($self) = @_;

    my $cur_user = $self->get_option('cur_user', {});
    my $user_id = $cur_user->{'id'};

    my $cur_user_roles = $cur_user->{'roles'} // $self->rbac->get_cur_user_roles() // {};
    my @cur_user_role_ids = sort {$a <=> $b} keys(%$cur_user_roles);

    my $cache_key = sprintf(
        'user_id: %s, lang: %s, roles: [%s]',
        $user_id,
        $self->get_option('locale', 'ru'),
        join(',', @cur_user_role_ids)
    );

    my $tree = $self->memcached->get(statistics_tree => $cache_key);

    unless ($tree) {
        $tree = $self->_get_tree();
        $self->memcached->set(statistics_tree => $cache_key, $tree, 30 * 60);
    }

    return $tree;
}

sub _find_stat_level_in_tree {
    my ($self, $from, $level_id) = @_;

    foreach my $level (@$from) {
        return $level if $level->{'id'} eq $level_id;
        if ($level->{children}) {
            my $found = $self->_find_stat_level_in_tree($level->{children}, $level_id);
            return $found if $found;
        }
    }
    return undef;
}

sub get_tree2 {
    my ($self) = @_;

    my $tree = $self->bk_statistics->get_tree2();

    return $tree;
}

sub init {
    my ($self) = @_;

    $self->SUPER::init();

    $self->{'__TREE_ENTITIES_HS__'} = {};
    _entities2hash([$self->statistics_additional_income, $self->statistics_charging,], $self->{'__TREE_ENTITIES_HS__'});
}

sub _collect_dimension_fields {
    my ($self, $data, $dimension_fields, %opts) = @_;

    my %field_values;

    foreach my $dimension_field (@$dimension_fields) {
        next unless $dimension_field->{'get'} eq 'dictionary';

        my $dim_id         = $dimension_field->{'id'};
        my $dim_model      = $dimension_field->{'model'};
        my $dim_model_only = $dimension_field->{'select_model_only'};

        my $dim_db_fields = $dimension_field->{'db_fields'};
        my $dim_db_first  = $dim_db_fields->[0];
        my $dim_db_many   = @$dim_db_fields > 1;

        my $dim_prefix = $dimension_field->{'default_prefix'};
        $dim_prefix = $dim_prefix->() if $dim_prefix && ref($dim_prefix) eq 'CODE';

        my $dim_caption_sub = $dimension_field->{'default_caption'};

        my %dim_model_values;
        %dim_model_values = map {$_->{'id'} => $_->{'label'}} @{$self->get_dimension_field_values($dimension_field)}
          if $dim_model;

        my $have_model_data = %dim_model_values ? TRUE : undef;    # keep undef here

        my %unique_data;

        my ($id, $label);

        foreach my $row (@$data) {
            $id = $dim_db_many ? join('_', map {$row->{$_}} @$dim_db_fields) : $row->{$dim_db_first};
            $label = $have_model_data
              && $dim_model_values{$id} // (
                 !$dim_model_only
                ? $dim_prefix && ($dim_prefix . $row->{$dim_db_first})
                  // $dim_caption_sub && $dim_caption_sub->($dim_db_fields, $row) // $row->{$dim_db_first}
                : undef
              );

            $unique_data{$id} = $label if defined($label) && !defined($unique_data{$id});
        }

        $field_values{$dimension_field->{'id'}} = \%unique_data;
    }
    return \%field_values;
}

sub _join_dimension_fields {
    my ($self, $data, $dimension_fields, %opts) = @_;

    foreach my $dimension_field (@$dimension_fields) {
        next unless $dimension_field->{'get'} eq 'dictionary';

        my $dim_id    = $dimension_field->{'id'};
        my $dim_model = $dimension_field->{'model'};

        my $dim_db_fields = $dimension_field->{'db_fields'};
        my $dim_db_first  = $dim_db_fields->[0];
        my $dim_db_many   = @$dim_db_fields > 1;

        my $dim_prefix = $dimension_field->{'default_prefix'};
        $dim_prefix = $dim_prefix->() if $dim_prefix && ref($dim_prefix) eq 'CODE';

        my $dim_caption_sub = $dimension_field->{'default_caption'};
        my $dim_value_sub   = $dimension_field->{'value'};

        next if $dim_id eq 'currency_id';    # don't convert currency_id to code in statistics

        if (!($dim_model || $dim_db_many || $dim_prefix || $dim_caption_sub || $dim_value_sub)) {
            next if $dim_id eq $dim_db_first;    # like tag_id

            foreach my $row (@$data) {
                $row->{$dim_id} = $row->{$dim_db_first};    # like dsp_id_name
            }
            next;
        }

        my %dim_model_values;
        %dim_model_values = map {$_->{'id'} => $_->{'label'}} @{$self->get_dimension_field_values($dimension_field)}
          if $dim_model;

        if ($dim_value_sub) {
            foreach my $row (@$data) {
                $row->{$dim_id} = $dim_value_sub->(\%dim_model_values, $dim_db_fields, $row);
            }
        } else {
            if (%dim_model_values) {
                foreach my $row (@$data) {
                    $row->{$dim_id} = $dim_model_values{
                        $dim_db_many
                        ? join('_', map {$row->{$_}} @$dim_db_fields)
                        : $row->{$dim_db_first}
                      } // $dim_prefix
                      && ($dim_prefix . $row->{$dim_db_first}) // $dim_caption_sub
                      && $dim_caption_sub->($dim_db_fields, $row) // $row->{$dim_db_first};
                }
            } else {
                foreach my $row (@$data) {
                    $row->{$dim_id} = $dim_prefix && ($dim_prefix . $row->{$dim_db_first})
                      // $dim_caption_sub && $dim_caption_sub->($dim_db_fields, $row) // $row->{$dim_db_first};
                }
            }
        }
    }
}

my %get_dimension_field_values_cache;

sub get_available_statistics_levels {
    my ($self, %opts) = @_;

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                only_products => {type => 'boolean', optional => TRUE},
                output        => {in   => [qw(levels accessors)]}
            },
        },
        throw => TRUE,
    );

    my $stat_levels;
    if ($opts{'only_products'}) {
        $stat_levels = $self->product_manager->get_statistics_products();
    } else {
        $stat_levels = $self->product_manager->get_statistics_accessors();
    }

    my @result = ();

    foreach (@$stat_levels) {
        next unless $self->app->$_->is_available();

        if ($opts{'output'} eq 'accessors') {
            push(@result, $_);
        } elsif ($opts{'output'} eq 'levels') {
            push(@result, $self->app->{$_}->id);
        } else {
            throw Exception::Validation::BadArguments gettext('Uknown output format "%s"', $opts{'output'});
        }
    }

    return [sort @result];
}

sub get_dimension_field_values {
    my ($self, $dimension_field, %opts) = @_;

    local $Data::Dumper::Terse    = TRUE;
    local $Data::Dumper::Sortkeys = TRUE;
    local $Data::Dumper::Indent   = 0;
    my $cache_key = Dumper($dimension_field, \%opts);

    $get_dimension_field_values_cache{$cache_key} //= $self->_get_dimension_field_values($dimension_field, %opts);

    return $get_dimension_field_values_cache{$cache_key};
}

sub get_filter_values {
    my ($self, %opts) = @_;

    my $query = Application::Model::Statistics::_Utils::Query->new($self, %opts);

    my $stat_level = $query->stat_level;

    my $get_data;
    if ($stat_level->support_clickhouse) {
        $query->_parse_period('dt', 0);

        my $filter = $self->clickhouse_db->filter();

        my $period_filter = $query->period_filter();
        $filter->and($period_filter) if defined($period_filter);

        my $product_ids = $stat_level->add_authorization_filter($filter);

        my $clickhouse_expressions = $stat_level->get_clickhouse_expressions();

        $self->_model_filter_to_clickhouse(
            $filter, $stat_level,
            level_filter           => $query->level_filter(),
            clickhouse_expressions => $clickhouse_expressions,
            product_ids            => $product_ids,
        );

        $get_data = sub {
            my ($dimension_field) = @_;

            my $db_fields = $self->_get_fields(
                $stat_level,
                dimension_fields => {map {$_ => $dimension_field} @{$dimension_field->{'db_fields'}}},
                clickhouse_expressions => $clickhouse_expressions,
                product_ids            => $product_ids
            );

            my $ch_query = $self->get_query_for_clickhouse(
                $stat_level,
                db_fields => $db_fields,
                filter    => $filter,
                distinct  => TRUE,
            );

            $ch_query->format('JSONCompact');

            return $ch_query->get_all();
        };
    } else {
        $get_data = sub {
            my ($dimension_field) = @_;

            return $stat_level->get_data(
                stat_db_fields       => {map {$_ => ''} @{$dimension_field->{'db_fields'}}},
                stat_db_group_fields => {},
                filter               => $query->period_filter,
                entity_filter        => $query->level_filter,
                distinct             => TRUE,
            );
        };
    }

    my %filter_values;
    foreach my $dimension_field (grep {$_->{'ajax'}} values %{$query->dimension_fields}) {
        my $data = $get_data->($dimension_field);

        unless (@$data) {
            $filter_values{$dimension_field->{'id'}} = [];
            next;
        }

        my $field_values = $self->_collect_dimension_fields($data, [$dimension_field]);
        my $unique_data = $field_values->{$dimension_field->{'id'}};

        $filter_values{$dimension_field->{'id'}} =
          _sort_by_type($dimension_field,
            [map {{id => $_, key => "id$_", label => $unique_data->{$_}}} keys(%$unique_data)]);
    }

    return \%filter_values;
}

sub get_product_parent {
    my ($self, $product, $all_classes, $top_product) = @_;

    my $parent = $product;
    while ($parent eq $product || !exists $all_classes->{$parent}) {
        ($parent) = $parent =~ /^(\S+)::\S+$/;
        throw gettext('Not statistics child %s', $parent) if length($parent) < length($top_product);
    }

    return $parent;
}

sub _normalize_period {
    my ($period) = @_;

    if ($period && !(ref($period) eq 'ARRAY')) {
        try {
            $period = [name2dates($period, '', '', iformat => 'db', oformat => 'db')];
        }
        catch Exception with {
            my $e = shift;
            throw Exception::Validation::Statistics::InvalidPeriod $e;
        }
    }

    return $period;
}

sub _transform_currencies_in_stat_data {
    my ($stat) = @_;

    if (exists $stat->{'currencies'}) {
        my %currencies_by_id = map {$_->{'id'} => $_->{'code'}} @{$stat->{'currencies'}};

        $_->{'currency_id'} = $currencies_by_id{$_->{'currency_id'}} foreach @{$stat->{'data'}};
    }

    return $stat;
}

sub _get_date_grouping_from_dimension_fields {
    my ($fields) = @_;

    my $cmp_date_group = undef;

    foreach (@{$fields}) {
        if ($_ =~ /^date\|(\w*)$/) {
            $cmp_date_group = $1 || 'day';
            last;
        }
    }

    return $cmp_date_group;
}

sub _get_interval_map_for_period {
    my ($start, $end) = @_;

    $start = trdate(db_time => sec => $start . ' 00:00:00');
    $end   = trdate(db_time => sec => $end . ' 23:59:59');

    my %interval_map = map {
        (
            $_ => {
                start => trdate($_ => sec => trdate(sec => $_ => $start)),
                end   => trdate($_ => sec => trdate(sec => $_ => $end))
            }
        )
    } qw(week month year);

    $interval_map{day}{start} = $start;
    $interval_map{day}{end}   = $end;

    return \%interval_map;
}

sub get_statistics_cmp_periods {
    my ($self, %opts) = @_;

    my $period = $opts{'period'};

    throw Exception::Validation::BadArguments gettext("Wrong period")
      unless $period
          && (ref($period) eq 'ARRAY')
          && @$period == 2
          && (ref($period->[0]) eq 'ARRAY')
          && (ref($period->[1]) eq 'ARRAY');

    $period->[0] = _normalize_period($period->[0]);
    $period->[1] = _normalize_period($period->[1]);

    foreach (@{$period->[0]}, @{$period->[1]}) {
        throw Exception::Validation::BadArguments gettext('Wrong date format: "%s", expected format "yyyy-mm-dd"', $_)
          unless $_ && check_date($_, iformat => 'db');
    }

    throw Exception::Validation::BadArguments gettext('Wrong period starts in "%s" and end in "%s"', $period->[0][0],
        $period->[0][1])
      unless $period->[0][0] le $period->[0][1];

    throw Exception::Validation::BadArguments gettext('Wrong period starts in "%s" and end in "%s"', $period->[1][0],
        $period->[1][1])
      unless $period->[1][0] le $period->[1][1];

    my $date_group = _get_date_grouping_from_dimension_fields($opts{'dimension_fields'});

    my $fst_interval_map  = _get_interval_map_for_period(@{$period->[0]});
    my $snd_interval_map  = _get_interval_map_for_period(@{$period->[1]});
    my $fst_period_length = $fst_interval_map->{'day'}{'end'} - $fst_interval_map->{'day'}{'start'};
    my $snd_period_length = $snd_interval_map->{'day'}{'end'} - $snd_interval_map->{'day'}{'start'};

    throw Exception::Validation::BadArguments gettext("Can't compare periods of different lengths")
      unless $fst_period_length == $snd_period_length;

    my $ordered_fields     = $self->_get_sorted_fields_from_stat_request(\%opts);
    my %fields             = map {($_->{'field'} => $_)} @$ordered_fields;
    my $group_fields       = $self->_get_sorted_group_fields_from_request(\%opts, \%fields);
    my @group_fields_names = map {$_->{'field'}} @$group_fields;
    my @sort_by_fields;

    my $sort_by;
    my $sort_order = 'asc';
    if ($opts{'order_by'} && @{$opts{'order_by'}}) {
        # TODO: process all possible arguments
        $sort_by    = $opts{'order_by'}[0]{'field'};
        $sort_order = $opts{'order_by'}[0]{'dir'};
    }

    if ($sort_by) {
        throw Exception "sort_by field '$sort_by' is missing in request data"
          unless exists($fields{$sort_by});
        @sort_by_fields = ($sort_by, map {$_->{'field'} ne $sort_by ? $_->{'field'} : ()} @$ordered_fields);
    } else {
        @sort_by_fields = map {$_->{'field'}} @$ordered_fields;
    }

    throw Exception "sort_order '$sort_order' is unknown (possible: 'asc', 'desc')"
      unless $sort_order eq 'asc' || $sort_order eq 'desc';

    my $period_opts = clone(\%opts);
    $period_opts->{'period'} = $period->[0];
    my $stat1 = $self->get_statistics(%$period_opts);
    $period_opts->{'period'} = $period->[1];
    my $stat2 = $self->get_statistics(%$period_opts);

    my @data = $self->_process_data_for_cmp_periods(
        $stat1->{'data'},  $stat2->{'data'}, \%fields,         \@group_fields_names, $fst_interval_map,
        $snd_interval_map, $date_group,      \@sort_by_fields, $sort_order,          TRUE
    );

    $stat1->{'data'} = \@data;
    if (exists($stat1->{'total'})) {
        $stat1->{'total'} = [$stat1->{'total'}, $stat2->{'total'}];
    }
    if (exists($stat1->{'report_title'})) {
        $stat1->{'report_title'} =
          gettext('Report for period %s', join(' - ', @{$period->[0]}) . ' / ' . join(' - ', @{$period->[1]}));
    }

    return $stat1;
}

=head2 get_statistics

=cut

sub get_statistics {
    my ($self, %opts) = @_;

    # Incoming data format for offset and limit
    #
    #order_by => [
    #    {field => 'PageName',   dir => 'asc'}, # desc
    #    {field => 'RegionName', dir => 'asc'}
    #],
    #
    #limits => {
    #    offset => 0, # по-умолчанию - 0
    #    limit => 100 # по-умолчанию - отсутствует
    #},
    #
    #row_ids => [],
    #top_keys => 5,
    #
    #total => 0 # 0/1

    my $top_keys        = delete($opts{'top_keys'}) // 0;
    my $only_query      = $opts{'only_query'}       // FALSE;
    my $only_sql        = $opts{'only_sql'}         // FALSE;
    my $force_ch_master = $opts{'force_ch_master'}  // FALSE;
    my $fields_order    = $opts{'fields'};

    my $query = Application::Model::Statistics::_Utils::Query->new($self, %opts);
    my $stat_level = $query->stat_level;

    my $support_clickhouse = $stat_level->support_clickhouse();

    throw Exception::Validation::BadArguments gettext('Option "only_query" available with storage "clickhouse" only')
      if $only_query && !$support_clickhouse;

    my $order_by = $query->order_by();

    my ($data, $result, $data_summary);
    if ($support_clickhouse) {
        $query->_parse_period('dt', 0);

        $self->clickhouse_db->close_dbh();
        if ($force_ch_master) {
            delete($ENV{'CLICKHOUSE_SPECIFIC_PORT'});
        } else {
            $ENV{'CLICKHOUSE_SPECIFIC_PORT'} = $self->clickhouse_db->get_option('balancer_port');
        }

        # если не задан возвращается пустой массив
        my $dimension_filter = $query->dimension_filter();

        my %query_opts = (
            fields             => $query->fields(),
            entity_fields      => $query->entity_fields(),
            dimension_fields   => $query->dimension_fields(),
            dimension_filter   => $dimension_filter,
            period_filter      => $query->period_filter(),
            level_filter       => $query->level_filter(),
            group_by           => $query->group_by(),
            is_currency_needed => $query->is_currency_needed(),
            money_format       => !$query->no_money_format(),
        );

        my $filter = $self->clickhouse_db->filter();
        $filter->and($query_opts{'period_filter'}) if defined($query_opts{'period_filter'});

        my $group_fields = $self->_get_group_fields($stat_level, %query_opts);

        my $product_ids = $stat_level->add_authorization_filter($filter, $group_fields, %query_opts);

        my $clickhouse_expressions = $stat_level->get_clickhouse_expressions();

        my $db_fields = $self->_get_fields(
            $stat_level, %query_opts,
            clickhouse_expressions => $clickhouse_expressions,
            product_ids            => $product_ids
        );

        $self->_model_filter_to_clickhouse(
            $filter, $stat_level, %query_opts,
            clickhouse_expressions => $clickhouse_expressions,
            product_ids            => $product_ids,
        );

        if ($top_keys) {
            my $sort_field = $order_by->[0]{'field'};
            undef $order_by;

            throw Exception sprintf('You must set "order_by"')
              unless defined($sort_field);

            my @group_fields_wo_date = grep {$_ ne 'date'} keys(%$group_fields);

            my $top_key = {cityHash64 => [map {$db_fields->{$_} || $_} @group_fields_wo_date]};

            my $top_keys_query = $self->get_query_for_clickhouse(
                $stat_level,
                db_fields => {top_key => $top_key, $sort_field => $db_fields->{$sort_field}},
                filter    => $filter,
                group_fields => {top_key => TRUE},
                order_by => [{field => $sort_field, dir => 'desc'}],
                offset   => 0,
                limit    => $top_keys,
            );

            my $top_result = $top_keys_query->get_all();

            if (@$top_result) {
                $filter->and([$top_key, 'IN', \[map {$_->{'top_key'}} @$top_result]]);
            } else {
                $top_keys = 0;
            }
        }

        my $ch_query = $self->get_query_for_clickhouse(
            $stat_level,
            db_fields    => $db_fields,
            filter       => $filter,
            group_fields => $group_fields,
            order_by     => $order_by,
            offset       => $query->offset(),
            limit        => $query->limit(),
        );

        return $ch_query if $only_query;
        return [$ch_query->get_sql_with_data()] if $only_sql;

        $ch_query->format('JSONCompact');

        $result = $ch_query->get_all();

        return {data => []} unless @$result;

        $data_summary = $self->_summarize_data_total_for_ch($ch_query, $top_keys)
          if $query->need_summary;
    } else {
        my %fields = map {$_ => TRUE} keys(%{$query->fields});
        @$order_by = grep {$fields{$_->{'field'}}} @$order_by;

        my $need_to_limit_tags = grep {$_ eq 'tag_id' || $_ eq 'tag_name'} @{$query->group_by_fields()};
        my $additional_filter;

        if ($need_to_limit_tags) {
            my $product_accessor = $stat_level->product->accessor();
            my $page_id_field_name =
              $product_accessor eq 'context_on_site_adblock'
              ? 'campaign_id'
              : $stat_level->product->get_page_id_field_name();
            my $tags = $self->_get_top_tags(%opts, page_id_field_name => $page_id_field_name);
            $additional_filter = $self->_get_filter_from_tags($tags, $page_id_field_name);
        }

        my $opts = {
            stat_db_fields       => $query->stat_db_fields,
            stat_db_group_fields => $query->stat_db_group_fields,
            filter               => [
                AND => [$query->period_filter, $query->dimension_filter, ($additional_filter ? $additional_filter : ())]
            ],
            entity_fields => [$query->entity_field_names],
            entity_filter => $query->level_filter,
            group_by      => $query->db_group_by_fields,
            offset        => $query->offset(),
            limit         => $query->limit(),
        };

        if ($only_sql) {
            my @queries = map {$_->get_sql_with_data()} $stat_level->data_query(%$opts);
            return \@queries;
        }

        $data = $stat_level->get_data(%$opts);

        return {data => []} unless @$data;

        $data = $self->_obtain_uniform_data($query, $data);

        $data = $self->_process_data_entity_fields($query, $data);

        my $nodate_keys;
        if ($top_keys && $order_by && @$order_by) {
            my $nodate_data = $self->_group_without_date($query, $data, order_by => $order_by);
            $nodate_data = $self->_sort_by_fields($query, $nodate_data, order_by => $order_by);
            $nodate_keys = $self->_get_nodate_keys($query, $nodate_data, offset => 0, limit => $top_keys);
        }

        $data = $self->_group_by_data($query, $data, $nodate_keys);

        $result = $self->_keep_requested_fields_only($query, $data, %opts);

        $data_summary = $self->_summarize_data_total($query, $data, %opts)
          if $query->need_summary;
    }

    my $measures = {};

    if ((ref($result) eq 'ARRAY') && (@$result > 0)) {
        $measures = $self->_get_measures(keys => $fields_order);
    }

    return {
        data     => $result,
        measures => $measures,
        (keys(%$data_summary) ? (total => $data_summary) : ()),
        ($query->is_currency_needed ? (currencies => $self->currency->get_all(fields => [qw(id code)])) : ()),
        report_title => gettext('Report for period %s', $query->period_description),
    };
}

sub get_query_for_clickhouse {
    my ($self, $stat_level, %opts) = @_;

    my ($db_fields, $filter, $group_fields, $order_by, $offset, $limit, $distinct) =
      @opts{qw(db_fields filter group_fields order_by offset limit distinct)};

    my %fields_to_result_sql             = ();
    my @fields_for_filter_not_empty_rows = ();
    foreach my $field_name (keys(%$db_fields)) {
        my $expression = delete($db_fields->{$field_name});
        $db_fields->{"_$field_name"} = ref($expression) eq '' && $expression eq '' ? $field_name : $expression;

        $fields_to_result_sql{$field_name} = "_$field_name";

        unless ($group_fields->{$field_name}) {
            push(@fields_for_filter_not_empty_rows, {if => [{isNaN => [$field_name]}, \0, $field_name]});
        }
    }

    my $table_name = $stat_level->clickhouse_table_name;

    my $sql_query = $self->clickhouse_db->query(dont_check_fields => {}, without_table_alias => TRUE)->select(
        table  => $self->clickhouse_db->$table_name,
        fields => $db_fields,
        filter => $filter,
    );

    $sql_query->group_by(map {"_$_"} sort keys(%$group_fields)) if %$group_fields;

    $sql_query->format('NOFormat');

    my $result_query = $self->clickhouse_db->query(dont_check_fields => {}, without_table_alias => TRUE)->select(
        table  => $sql_query,
        alias  => 'statistics',
        fields => \%fields_to_result_sql,
        (
                 @fields_for_filter_not_empty_rows
              && %$group_fields ? (filter => [['+' => \@fields_for_filter_not_empty_rows], '>', \0]) : ()
        ),
    );

    if ($order_by && @$order_by) {
        $result_query->order_by(map {[$_->{'field'}, $_->{'dir'} eq 'asc' ? 0 : 1]} @$order_by);
    }

    $result_query->limit($offset, $limit) if $limit;

    $result_query->distinct($distinct) if $distinct;

    return $result_query;
}

sub get_fields_for_block_levels_only {
    return qw(
      public_id
      tag_id
      tag_caption
      adfox_block
      type
      mobile_block_type
      mobile_block_type_label
      );
}

sub _model_filter_to_clickhouse {
    my ($self, $filter_obj, $stat_level, %opts) = @_;

    my ($clickhouse_expressions, $product_ids, $level_filter, $dimension_filter) =
      @opts{qw(clickhouse_expressions product_ids level_filter dimension_filter)};

    if (defined($level_filter)) {
        my $product = $stat_level->get_product;

        my $db_filter            = $product->get_db_filter($level_filter);
        my $db_filter_expression = $db_filter->expression;

        _fix_filter_from_mysql_to_clickhouse($db_filter_expression, $product_ids, $stat_level, $clickhouse_expressions);

        $filter_obj->and($db_filter_expression);
    }

    if (defined($dimension_filter)) {
        _fix_filter_from_mysql_to_clickhouse($dimension_filter, $product_ids, $stat_level, $clickhouse_expressions);

        $filter_obj->and($dimension_filter);
    }

    return $filter_obj;
}

sub _fix_filter_from_mysql_to_clickhouse {
    # ($filter, $product_ids, $stat_level, $clickhouse_expressions) = @_;

    if (ref($_[0]) eq '' && exists($_[3]->{$_[0]})) {
        # 'campaign_id'
        $_[0] = _get_clickhouse_expression($_[3]->{$_[0]}, $_[1], $_[2]);
    } elsif (ref($_[0]) eq 'ARRAY' && @{$_[0]} == 3) {
        # [<FIELD>, <OPR>, <VALUE>]
        my ($field_name, $expr, $value) = @{$_[0]};
        my $res = [];
        if (ref($field_name) eq 'HASH') {
            # {'CONCAT' => [\'R-I-', 'campaign_id', \'-', 'id']}
            # функция приходит после обработки фильтра моделью
            my $mysql_func_name = [sort keys(%$field_name)]->[0];

            if (uc($mysql_func_name) eq 'CONCAT') {
                foreach (@{$field_name->{$mysql_func_name}}) {
                    my $item = $_;

                    if (ref($_) eq '') {
                        # Заменяем поля на функции из CH
                        _fix_filter_from_mysql_to_clickhouse($item, $_[1], $_[2], $_[3]);

                        # Делаем приведение типа
                        $_ = {toString => [$item]};
                    }
                }
            } elsif (uc($mysql_func_name) eq 'JSON_UNQUOTE') {
                my $app   = $_[2]->app;
                my $model = $_[2]->get_product->accessor();

                my $fields = $app->$model->isa('Application::Model::Page') ? ['page_id'] : ['page_id', 'id'];

                my $id_list = $app->$model->get_all(
                    fields => $fields,
                    filter => $app->partner_db->filter($_[0])
                );

                my $ch_fields = {id => 'block_id',};

                $res->[0] //= {'' => [map {$ch_fields->{$_} // $_} @$fields]};
                $res->[1] //= 'IN';
                # results in ((1,11),(2,22),...) + convert to number
                my @res2;
                my @nan;
                for my $id (@$id_list) {
                    if (grep {!looks_like_number($_)} values %$id) {
                        push @nan, $id;
                        push @res2, {'' => [map {\$_} @$id{@$fields}]};
                    } else {
                        push @res2, {'' => [map {\($_ + 0)} @$id{@$fields}]};
                    }
                }
                $res->[2] //= {'' => @res2 ? \@res2 : [map {\undef} @$fields]};

                if (@nan) {
                    WARN 'Received not numeric ids: ', to_json(\@nan, pretty => 1);
                }

                $_[0] = $res;
                return;

            } else {
                throw Exception gettext("Can't process mysql function: %s (level: %s)", $mysql_func_name, $_[2]->id);
            }
        } elsif (exists($_[3]->{$field_name})) {
            $res->[0] = _get_clickhouse_expression($_[3]->{$field_name}, $_[1], $_[2]);
        }

        $expr = 'IN' if $expr eq '= ANY';

        if (blessed($value) && $value->isa('QBit::Application::Model::DB::Query')) {
            my $key_name = (%{$value->{'__TABLES__'}[0]{'fields'}})[0];

            $res->[1] = 'IN';
            $res->[2] = \[map {$_->{$key_name}} @{$value->get_all()}];
        }

        $res->[0] //= $field_name;
        $res->[1] //= $expr;
        $res->[2] //= $value;

        my $expression_type = _get_type_by_expression($_[2], $res->[0]);
        $res->[2] = _get_value_by_expression_type($expression_type, $res->[2]);

        $_[0] = $res;
    } elsif (ref($_[0]) eq 'ARRAY' && @{$_[0]} == 2 && @{$_[0]->[1]} == 1 && ref($_[0]->[1][0]) eq '') {
        # boolean
        # true ['AND', [<FIELD>]]

        my $field_name = $_[0]->[1][0];
        if (exists($_[3]->{$field_name})) {
            $field_name = _get_clickhouse_expression($_[3]->{$field_name}, $_[1], $_[2]);
        }

        $_[0] = ['AND', [$field_name]];
    } elsif (ref($_[0]) eq 'ARRAY'
        && @{$_[0]} == 2
        && ref($_[0]->[1][0]) eq 'HASH'
        && [%{$_[0]->[1][0]}]->[0] eq 'NOT')
    {
        # boolean
        # false ['AND',[{'NOT' => [<FIELD>]}]]

        my $field_name = $_[0]->[1][0]{'NOT'}[0];
        if (exists($_[3]->{$field_name})) {
            $field_name = _get_clickhouse_expression($_[3]->{$field_name}, $_[1], $_[2]);
        }

        $_[0] = ['AND', [{'NOT' => [$field_name]}]];
    }

    if (ref($_[0]) eq 'ARRAY' && ($_[0]->[0] eq 'AND' || $_[0]->[0] eq 'OR')) {
        _fix_filter_from_mysql_to_clickhouse($_, $_[1], $_[2], $_[3]) foreach @{$_[0]->[1]};
    }
}

sub _get_type_by_expression {
    my ($stat_level, $expression) = @_;

    my $ch_type;
    if (ref($expression) eq 'HASH') {
        my $ch_func    = [keys(%$expression)]->[0];
        my $uc_ch_func = uc($ch_func);

        if ($uc_ch_func eq 'IF') {
            return _get_type_by_expression($stat_level, $expression->{$ch_func}[2]);
        } elsif ($uc_ch_func eq 'CONCAT' || $ch_func eq 'TRANSFORM') {
            return 'string';
        }

        ($ch_type) = $ch_func =~ /^dictGet(.+?)(?:OrDefault)?\z/;
    } elsif (ref($expression) eq 'SCALAR') {
        return looks_like_number($$expression) ? 'number' : 'string';
    } else {
        my $ch_table = $stat_level->clickhouse_table_name();
        my %table_fields = map {$_->name => $_->type} @{$stat_level->clickhouse_db->$ch_table->fields()};

        $ch_type = $table_fields{$expression}
          // throw Exception gettext('Level "%s" has not field: %s', $stat_level->id, $expression);
    }

    return
      $ch_type eq 'String' ? 'string'
      : (
        $ch_type =~ /^(?:U?Int)|(?:Float)/ ? 'number'
        : throw Exception gettext('Unknown type: %s (level: %s)', $ch_type, $stat_level->id)
      );
}

sub _get_value_by_expression_type {
    my ($field_type, $value) = @_;

    my $data = $$value;

    if ($field_type eq 'string') {
        if (ref($data) eq 'ARRAY') {
            return \[map {QBit::Application::Model::DB::clickhouse::String->new($_)} @$data];
        } else {
            return QBit::Application::Model::DB::clickhouse::String->new($data);
        }
    } else {
        if (ref($data) ne 'ARRAY') {
            $data = [$data];
        }

        my @not_numbers = grep {defined($_) && !looks_like_number($_)} @$data;

        if (@not_numbers) {
            throw Exception::Validation::BadArguments gettext('Expected type "number", but got: %s',
                join(', ', map {"'$_'"} @not_numbers));
        }

        return $value;
    }
}

sub _get_fields {
    my ($self, $stat_level, %opts) = @_;

    my ($clickhouse_expressions, $product_ids, $money_format, $fields, $entity_fields, $dimension_fields, $group_by,
        $is_currency_needed)
      = @opts{
        qw(clickhouse_expressions product_ids money_format fields entity_fields dimension_fields group_by is_currency_needed)
      };

    my @fields_definitions = grep {defined($_)} $group_by, $dimension_fields, $entity_fields, $fields;

    my $sql_fields = {};
    foreach my $fields_definition (@fields_definitions) {
        foreach my $field_name (keys(%$fields_definition)) {
            if (exists($clickhouse_expressions->{$field_name})) {
                $sql_fields->{$field_name} = _get_clickhouse_expression(
                    $clickhouse_expressions->{$field_name},
                    $product_ids, $stat_level,
                    $fields_definition->{$field_name},
                    money_format => $money_format
                );
            } else {
                $sql_fields->{$field_name} = '';
            }

            if (   $money_format
                && $fields_definition->{$field_name}{'unit'}
                && $fields_definition->{$field_name}{'unit'} eq 'money')
            {
                $sql_fields->{$field_name} =
                  {round => [['/' => [$sql_fields->{$field_name}, \$CH_MONEY_POST_SCALE]], \2]};
            }
        }
    }

    $sql_fields->{'currency_id'} = '' if $is_currency_needed;

    return $sql_fields;
}

sub _get_clickhouse_expression {
    my ($clickhouse_expression, $product_ids, $stat_level, $field, %opts) = @_;

    return ref($clickhouse_expression) eq 'CODE'
      ? $clickhouse_expression->($field, $product_ids, $stat_level, %opts)
      : $clickhouse_expression;
}

sub _get_group_fields {
    my ($self, $stat_level, %opts) = @_;

    my ($entity_fields, $dimension_fields, $group_by, $is_currency_needed) =
      @opts{qw(entity_fields dimension_fields group_by is_currency_needed)};

    my %uniq_fields = map {$_ => TRUE} map {keys(%$_)} grep {defined($_)} $group_by, $dimension_fields, $entity_fields;
    $uniq_fields{'currency_id'} = TRUE if $is_currency_needed;

    return \%uniq_fields;
}

# Формирование данных для двух периодов происходит следующим образом:
# 1. Для данных из каждого периода дата заменяется цифрой смещения (в секундах) от
#    периода;
# 2. Для данных из каждого периода вычисляется уникальный ключ из группирующих полей;
# 3. По вычисленным ключам проверяется, в каком из периодов данные есть, а в
#    каком - отсутствуют;
# 4. Данные, отсутствующие во втором периоде - сразу заполняются прочерками;
# 5. Данные, отсутствующие в первом периоде заполняются фиктивными значениями
#    из второго (нужно для следующего шага);
# 6. Выполняется сортировка по первому периоду (по полю sort_by, всегда в порядке
#    sort_order) и остальным в соответствии в порядком полей;
# 7. Данные по периодам записываются в два выходных массива в соответствии с сортировкой
#    по первому периоду;
# 8. Даты восстанавливаются из смещений, а фиктивные значения из первого периода
#    заменяются прочерками.
sub _process_data_for_cmp_periods {
    my (
        $self,         $stat_data_fst,    $stat_data_snd,    $fields,
        $group_fields, $fst_interval_map, $snd_interval_map, $cmp_date_group,
        $sort_by,      $sort_order,       $need_missing_date
       ) = @_;

    my %dates_weights;
    my %date_by_weight_for_fst_period = ('-' => '-');
    my %date_by_weight_for_snd_period = ('-' => '-');
    my @first_period;
    my $format_date;
    my $process_stat_data = sub {
        my ($stat_data, $interval_map, $date_by_weight_for_period, $is_fst) = @_;
        my %period_keys;

        foreach (@$stat_data) {
            if ($cmp_date_group) {
                # (1) Для того, чтобы выровнять данные в периодах относительно друх друга
                # реальная дата подменяется значением смещения от начала периода
                my $date = $_->{'date'};
                $format_date //= $date =~ /^\d{4}-\d{2}-\d{2}$/ ? 'db' : $cmp_date_group;
                my $sec = trdate($format_date => sec => $date);
                #TODO: вернуть после переезда на CH
                #my $sec = trdate('db' => sec => $date);

                my $sec_start = $interval_map->{$cmp_date_group}{'start'};

                $dates_weights{$date} = $sec - $sec_start;
                $date_by_weight_for_period->{$dates_weights{$date}} = trdate(sec => 'db' => $sec);
                #TODO: вернуть после переезда на CH
                #$date_by_weight_for_period->{$dates_weights{$date}} = $date;
                $_->{'date'} = $dates_weights{$date};
            }

            my $key = join('-', @{$_}{@$group_fields});
            $_->{'_uniq-period-key'} = $key;
            $period_keys{$key} = $_;
            push @first_period, $_ if $is_fst;
        }

        return \%period_keys;
    };
    # (2) Получение уникальных ключей для проверки отсутствующих данных
    my $first_period_keys =
      $process_stat_data->($stat_data_fst, $fst_interval_map, \%date_by_weight_for_fst_period, TRUE);
    my $second_period_keys =
      $process_stat_data->($stat_data_snd, $snd_interval_map, \%date_by_weight_for_snd_period, FALSE);

    # (3) Отсутствующие данные в первом периоде
    my @missing_data_in_second_period =
      sort map {exists($second_period_keys->{$_}) ? () : $_} keys(%$first_period_keys);
    # (3) Отсутствующие данные во втором периоде
    my @missing_data_in_first_period =
      sort map {exists($first_period_keys->{$_}) ? () : $_} keys(%$second_period_keys);

    # Прочерки для заполнения отсутствующих данных
    my $dashed_stub_data = {map {$_ => '-'} keys(%$fields)};
    # Вставляет реальное смещение в дату отсутствующих данных
    my $prepare_stub_data;
    unless ($cmp_date_group && $need_missing_date) {
        $prepare_stub_data = sub {$dashed_stub_data};
    } else {
        $prepare_stub_data = sub {
            my ($key, $is_fst) = @_;
            my $offset;
            # Если stub для первого периода, то берем смещение из второго и
            # вычисляем абсолютную дату для восстановления даты.
            # Наоборот, если из второго.
            if ($is_fst) {
                $offset = $second_period_keys->{$key}{'date'};
                $date_by_weight_for_fst_period{$offset} =
                  trdate(sec => 'db' => ($fst_interval_map->{$cmp_date_group}{'start'} + $offset));
            } else {
                $offset = $first_period_keys->{$key}{'date'};
                $date_by_weight_for_snd_period{$offset} =
                  trdate(sec => 'db' => ($snd_interval_map->{$cmp_date_group}{'start'} + $offset));
            }
            my $stub = clone($dashed_stub_data);
            $stub->{'date'} = $offset;
            return $stub;
        };
    }
    # (4) По ключам, отсутствующим во втором периоде добавляются прочерки
    $second_period_keys->{$_} = $prepare_stub_data->($_, 0) foreach @missing_data_in_second_period;

    # (5) Сортировка выполняется по первом периоду
    # Для этого, если данных в первом периоде нет, нужно добавить туда полноценные значения
    # и пометить этот период ключом stub, для того чтобы после сортировки подменить
    # реальные значения прочерками
    foreach my $key (@missing_data_in_first_period) {
        my $stub_data = clone($second_period_keys->{$key});
        $stub_data->{'_stub'} = TRUE;
        $first_period_keys->{$key} = TRUE;
        push @first_period, $stub_data;
    }

    # (6) Так как дата подменяется цифровым значением смещения от начала периода,
    # сортировать по полю дата необходимо как number
    $fields->{'date'}{'sort_as'} = 'number' if $cmp_date_group;
    my $sorted_first_period = $self->_sort_stat_data(\@first_period, $fields, $sort_by, $sort_order);
    $fields->{'date'}{'sort_as'} = 'text' if $cmp_date_group;

    my @fst_period;
    my @snd_period;
    # (7) Формирование значений периодов в соответствии с сортировкой по первому из них
    foreach (@$sorted_first_period) {
        my $key = delete($_->{'_uniq-period-key'});
        delete($first_period_keys->{$key});
        # (8) Если это были фиктивные значения для отсутствующих данных в первом периоде
        if ($_->{'_stub'}) {
            push @fst_period, $prepare_stub_data->($key, 1);
        } else {
            push @fst_period, $_;
        }

        my $second_period_data = delete($second_period_keys->{$key});
        delete($second_period_data->{'_uniq-period-key'});
        push @snd_period, $second_period_data;
    }
    # (8) Восстановление значения даты
    if ($cmp_date_group) {
        $_->{'date'} = $date_by_weight_for_fst_period{$_->{'date'}} foreach @fst_period;
        $_->{'date'} = $date_by_weight_for_snd_period{$_->{'date'}} foreach @snd_period;
    }

    return (\@fst_period, \@snd_period);
}

sub _sort_stat_data {
    my ($self, $stat_data, $fields, $sort_by, $sort_order) = @_;

    my $number_sorting_key = sub {!defined($_[0]) || $_[0] eq '-' ? -1 : $_[0]};
    my $string_sorting_key = sub {defined($_[0]) ? $_[0] : '-'};

    my %sort_by_cmp;
    foreach my $column (@$sort_by) {
        if ($fields->{$column}{'sort_as'} eq 'number') {
            $sort_by_cmp{$column} =
              ($sort_order eq 'desc')
              ? sub {$number_sorting_key->($_[1]) <=> $number_sorting_key->($_[0])}
              : sub {$number_sorting_key->($_[0]) <=> $number_sorting_key->($_[1])};
        } else {
            $sort_by_cmp{$column} =
              ($sort_order eq 'desc')
              ? sub {$string_sorting_key->($_[1]) cmp $string_sorting_key->($_[0])}
              : sub {$string_sorting_key->($_[0]) cmp $string_sorting_key->($_[1])};
        }
    }

    my $sort_sub = sub {
        my $res = 0;
        foreach my $column (@$sort_by) {
            $res = $sort_by_cmp{$column}->($a->{$column}, $b->{$column});
            last if $res;
        }
        return $res;
    };

    my @sorted_stat_data = sort $sort_sub @$stat_data;

    return \@sorted_stat_data;
}

sub _get_sorted_group_fields_from_request {
    my ($self, $stat_query, $all_fields) = @_;

    my @fields_names = map {@$_} @{$stat_query}{qw(dimension_fields entity_fields)};
    # fix date
    foreach (@fields_names) {
        if (m/date/) {
            $_ = 'date';
            last;
        }
    }
    my @fields = sort {$a->{'field_order'} <=> $b->{'field_order'}}
      map {$all_fields->{$_}} @fields_names;

    return \@fields;
}

sub _get_sorted_fields_from_stat_request {
    my ($self, $stat_query) = @_;

    my $tree     = $self->get_tree();
    my $level_id = $stat_query->{'levels'}[0]{'id'};
    my $level    = $self->_find_stat_level_in_tree($tree, $level_id);

    throw Exception::Denied gettext("You have no access to this report") unless $level;

    my $all_fields = $self->_get_all_fields_for_stat_level($level);
    my @fields_names = map {@$_} @{$stat_query}{qw(dimension_fields entity_fields fields)};
    # PI-12190 - grep {defined($_)} - filter out fields that are present in $stat_query but unavailable for user
    my @fields =
      sort {$a->{'field_order'} <=> $b->{'field_order'}} grep {defined($_)} map {$all_fields->{$_}} @fields_names;

    return \@fields;
}

sub _get_all_fields_for_stat_level {
    my ($self, $level) = @_;

    my %all_fields;
    my $order = 0;

    map {
        $all_fields{$_->{'id'}} = {
            field       => $_->{'id'},
            name        => $_->{'title'} // $_->{'label'},
            type        => $_->{'type'},
            sort_as     => _sorting_type($_),
            field_order => $order++,
          }
      } map {
        @{$level->{$_}}
      } qw(dimension_fields entity_fields fields);

    if (exists($all_fields{'date'})) {
        $all_fields{'date|'}      = $all_fields{'date'};
        $all_fields{'date|day'}   = $all_fields{'date'};
        $all_fields{'date|week'}  = $all_fields{'date'};
        $all_fields{'date|month'} = $all_fields{'date'};
        $all_fields{'date|year'}  = $all_fields{'date'};
    }

    return \%all_fields;
}

sub _sorting_type {
    my ($field) = @_;

    my $default_sorting_type = 'text';

    my %type_to_sort = (
        money   => 'number',
        percent => 'number',
        number  => 'number',
        page_id => 'number',
        unknown => $default_sorting_type,
    );

    return $field->{'sort_as'} // $type_to_sort{$field->{'type'} // 'unknown'} // $default_sorting_type;
}

=head2 get_statistics2

=cut

sub get_statistics2 {
    my ($self, %opts) = @_;

    my $answer = $self->bk_statistics->get_statistics2(%opts);

    return $answer;
}

=head2 get_data_from_hash2

На входе - id сохраненного статистического отчета расширенной статистики.

    my $data = $app->statistics->get_data_from_hash2(report_hash => 'ad7ba44d0f906b4e0d34e6414f17988ff281ce98');

Возвращает структуру данных вида

    {
        fields => [
            {
                field => 'date',
                name => 'Дата',
            },
            {
                field => 'rtb_block_hits_unsold',
                name => 'Непроданные запросы RTB-блоков',
            },
            {
                field => 'rtb_block_shows_own_adv',
                name => 'Показы своей рекламы в RTB-блоках',
            },
        ],
        sorted_stat_data => [
            {
                date => 2017-07-11,
                rtb_block_hits_unsold => 2058552071,
                rtb_block_shows_own_adv => 51071266,
            },
        ],
    }

Ответ этого метода предполагается использовать для формирования excel файла с помощью:

    my $xls_binary = xls_with_fields_and_names($data->{'sorted_stat_data'}, $data->{'fields'});

Так что структура fields и sorted_stat_data - это то с чем работает саба xls_with_fields_and_names()

=cut

sub get_data_from_hash2 {
    my ($self, %opts) = @_;

    my ($report_hash, $query, $rows_limit) = @opts{qw( report_hash  query  rows_limit )};

    if ($report_hash) {
        my $report = $self->stat_report_params_digest->get($report_hash, fields => ['params']);
        throw Exception::Validation::BadArguments gettext('Non-existent report') unless $report;
        $self->stat_report_params_digest->update($report_hash);

        my $report_params = from_json($report->{'params'});

        my $level_name = defined($report_params->{'tableLevel'}) ? 'tableLevel' : 'chartLevel';

        $report_params->{$level_name}[0]{'order_by'}[0] = {
            'field' => $report_params->{'tableViewModelData'}{'sortBy'},
            'dir'   => $report_params->{'tableViewModelData'}{'sortOrder'}
        };
        $query = $report_params->{$level_name}[0];
    }

    my $tmp_rights = $self->app->add_tmp_rights(qw(bk_statistics_allow_any_limit));

    if ($rows_limit) {
        $query->{limits} = {
            offset => 0,
            limit  => $rows_limit,
        };
    }

    my $stat = $self->app->statistics->get_statistics2(%$query);

    my @sorted_stat_data;

    for my $row (@{$stat->{'points'}}) {
        $row->{'dimensions'}{'date'} = $row->{'dimensions'}{'date'}[0] if ($row->{'dimensions'}{'date'});
        $row->{'dimensions'}{$_} = $row->{'measures'}[0]{$_} for (keys %{$row->{'measures'}[0]});
        push(@sorted_stat_data, $row->{'dimensions'});
    }

    my %tmp_fields;
    foreach my $type (qw(dimensions measures)) {
        foreach my $id (keys(%{$stat->{$type}})) {
            my $field = $stat->{$type}->{$id};

            $tmp_fields{$field->{index}} = {
                field => $id,
                name  => (ref($field->{title}) eq 'CODE' ? $field->{title}->() : $field->{title}),
            };
        }
    }

    my @fields = map {$tmp_fields{$_}} sort {$a <=> $b} keys(%tmp_fields);

    return {
        sorted_stat_data => \@sorted_stat_data,
        fields           => \@fields
    };
}

sub is_available {
    my ($self) = @_;

    foreach my $product (@{$self->product_manager->get_statistics_products()}) {
        return TRUE if $self->app->$product->is_available();
    }

    return FALSE;
}

sub pre_run {
    my ($self) = @_;

    %get_dimension_field_values_cache = ();
}

sub _entities2hash {
    my ($entities, $hs, $path) = @_;

    $path //= [];
    foreach my $entity (@$entities) {
        $entity->{'path'} = $path;

        throw Exception gettext('Duplicate statistic entity ID "%s"', $entity->id) if exists($hs->{$entity->id});

        $hs->{$entity->id} = $entity;

        _entities2hash([$entity->children], $hs, [@$path, $entity->id])
          if $entity->isa('Application::Model::Statistics::Hierarchy');
    }
}

sub _get_all_required_fields {
    my ($field_name, $rec, $query) = @_;

    my $need_fields = $query->need_fields;

    foreach (@{$need_fields->{$field_name}{'depends_on'} // []}) {
        _get_all_required_fields($_, $rec, $query);
    }

    $rec->{$field_name} //=
      $need_fields->{$field_name}{'get'} eq 'db'
      ? ($rec->{$field_name} || 0) + 0
      : $need_fields->{$field_name}{'get'}($rec);
}

sub _get_dimension_field_values {
    my ($self, $dimension_field, %opts) = @_;

    my $id      = $dimension_field->{'id_field'}      || 'id';
    my $caption = $dimension_field->{'caption_field'} || 'caption';

    my @result = ();

    my $models = $dimension_field->{'model'};
    $models = [$models] unless ref($models) eq 'ARRAY';

    foreach my $model (@$models) {
        my $fields = $self->app->$model->get_model_fields();
        my @rights = ();

        foreach ($id, $caption) {
            push(@rights, Application::Model::Multistate::DBManager::_get_rights($fields->{$_}{'check_rights'}))
              if exists($fields->{$_}{'check_rights'});
        }

        my $tmp_rights =
          $self->app->add_tmp_rights('dsp_view_all', 'internal_context_on_site_campaign_view_all', @rights);

        my $data = $self->app->$model->get_all(
            fields => [$id, $caption],
            (
                $opts{'data'} && @{$opts{'data'}}
                ? (
                    filter => [
                        $id => '=' => array_uniq(
                            map {
                                my $row = $_;
                                join('_', map {$row->{$_}} @{$dimension_field->{'db_fields'}})
                              } @{$opts{'data'}}
                        )
                    ]
                  )
                : (filter => [$id => 'IS NOT' => undef])
            ),
            order_by => [$caption],
        );

        push(@result, map {+{label => $_->{$caption}, id => $_->{$id}, key => "id$_->{$id}"}} @$data);
    }

    return \@result;
}

=begin comment _get_filter_from_tags

На входе - список тегов в формате

    [
        {
            campaign_id => 149863,
            tag_id => 7,
        },
        {
            campaign_id => 123194,
            tag_id => 16,
        },
        ...
    ]

и название статистикеского поля в котором содержится page_id

На выходе структура данных, которая соответствует такому sql условию:

    ( `campaign_id` = '149863' AND `tag_id` = '7')
    OR ( `campaign_id` = '123194' AND `tag_id` = '16')
    ...

=end comment

=cut

sub _get_filter_from_tags {
    my ($self, $tags, $page_id_field_name) = @_;

    if ($tags && @$tags) {
        my $tags_filter;

        foreach my $tag (@$tags) {
            push @$tags_filter,
              [
                'AND',
                [[$page_id_field_name, '=', \$tag->{$page_id_field_name},], ['tag_id', '=', \$tag->{'tag_id'},],],
              ],
              ;
        }

        return [OR => $tags_filter];
    } else {
        return [\0 => '=' => \1];
    }

}

sub _get_measures {
    my ($self, %opts) = @_;

    my $keys = delete($opts{keys});

    my $measures = {};

    my $index = 1;
    foreach my $k (@$keys) {
        if (exists($Application::Model::Statistics::Product::FIELD_TYPES{$k})) {
            my $tmp = $Application::Model::Statistics::Product::FIELD_TYPES{$k};
            $measures->{$k} = {
                index => $index,
                title => $tmp->{'title'}->(),
                type  => $tmp->{'type'},
                unit  => $tmp->{'unit'},
            };
            $index++;
        }
    }

    return $measures;
}

sub _get_nodate_keys {
    my ($self, $query, $data, %opts) = @_;

    return [map {$_->{'__KEY__'}} splice(@$data, $opts{'offset'}, $opts{'limit'})];
}

=begin comment _get_top_tags

Метод принимает точно такие же параметры как и get_statistics().

Ответ этого метода - это ARRAYREF со списоком срезов, на которых было
больше всего денег.

Пример ответа:

    [
        {
            campaign_id => 149863,
            tag_id => 7,
        },
        {
            campaign_id => 123194,
            tag_id => 16,
        },
        ...
    ]

=end comment

=cut

sub _get_top_tags {
    my ($self, %opts) = @_;

    my $query = Application::Model::Statistics::_Utils::Query->new($self, %opts);
    my $stat_level = $query->stat_level;

    my %get_data_opts = (
        stat_db_group_fields => {
            'tag_id'                    => '',
            $opts{'page_id_field_name'} => '',
            'all_w_nds'                 => {SUM => ['all_w_nds']},
        },
        filter        => [AND => [$query->period_filter, $query->dimension_filter]],
        entity_fields => [$query->entity_field_names],
        entity_filter => $query->level_filter,
        group_by => ['tag_id',    $opts{'page_id_field_name'}],
        order_by => ['all_w_nds', TRUE],
        limit    => $TAGS_LIMIT,
    );

    my $tags = $stat_level->get_data(%get_data_opts);

    return $tags;
}

sub _get_tree {
    my ($self) = @_;

    # Quick and dirty hack to speed up "user" subfilter in entity_filter_fields()
    # TODO: somehow cache whole subfilters called from lower level branches
    my $roles = $self->rbac->get_roles();
    no warnings 'redefine';
    local *Application::Model::RBAC::get_roles = sub {$roles};

    my %products =
      map {$_ => $self->_get_tree_node_data($_)}
      ($self->statistics_additional_income->is_available ? 'statistics_additional_income' : ()),
      @{$self->get_available_statistics_levels(only_products => TRUE, output => 'accessors')};

    my %roots;
    while (my $product =
        (sort {$products{$b}->{'depth'} <=> $products{$a}->{'depth'}} grep {!$roots{$_}} keys(%products))[0])
    {
        my $parent_accessor = $self->app->$product->parent_accessor();

        if (defined($parent_accessor)) {
            my @children =
              map  {delete($products{$_->accessor})}
              sort {$a->sort_priority() <=> $b->sort_priority()}
              grep {$products{$_->accessor}} $self->app->$parent_accessor->children;

            $products{$parent_accessor} = $self->_get_tree_node_data($parent_accessor, \@children);
        } else {
            $roots{$product} = TRUE;
        }
    }

    my $tree =
      [sort {$self->get_level_sort_priority($a->{'id'}) <=> $self->get_level_sort_priority($b->{'id'})}
          @products{keys(%roots)}];

    $self->_tree_remove_excess_fields($tree);

    return $tree;
}

sub _get_tree_node_data {
    my ($self, $accessor, $children) = @_;

    my $level = $self->app->$accessor;

    my $fields = Application::Model::Statistics::Hierarchy::fields_union(map {$_->{'fields'}} @$children);

    my $depth = () = ref($level) =~ /::/g;

    return {
        depth => $depth,
        id    => $level->id(),
        title => $level->title,
        (@$children ? (children => $children) : ()),
        fields => $level->get_fields((@$children ? (child_fields => $fields) : (with_shared => TRUE)), no_nds => 1),
        dimension_fields => $level->_can('dimension_fields')
        ? $level->get_dimension_fields()
        : Application::Model::Statistics::Hierarchy::fields_intersection(map {$_->{'dimension_fields'}} @$children),
        entity_fields               => $level->get_entity_fields(),
        conflict_fields             => $level->get_conflict_fields(@$children),
        entity_filter_fields        => $level->entity_filter_fields(),
        entity_filter_simple_fields => $level->entity_filter_simple_fields(),
        has_raw_stat                => $level->has_raw_stat(),
        has_product                 => !!$level->can('product'),
    };
}

sub _group_by_data {
    my ($self, $query, $data, $nodate_keys) = @_;

    # Additional group_by in Perl
    # possibly using calculated entity fields and added dimension field values.

    my @group_by_fields        = @{$query->group_by_fields};
    my @nodate_group_by_fields = grep {$_ ne 'date'} @group_by_fields;
    my @fields                 = $query->need_db_field_names;

    my %nodate_keys;
    %nodate_keys = map {$_ => TRUE} @$nodate_keys if $nodate_keys;

    my $do_filtering = %nodate_keys ? 1 : 0;

    my %group;

    my ($key);

    foreach my $rec (@$data) {
        next if $do_filtering && !$nodate_keys{$rec->{'__KEY__'}};

        $key = join($;, map {$rec->{$_} // chr(2)} @group_by_fields);

        if (exists($group{$key})) {
            $group{$key}{$_} += ($rec->{$_} // 0) foreach @fields;
        } else {
            $group{$key} = $rec;    # DESTRUCTIVE to source $data !!!  because this is an alias
        }

        $group{$key}{$_} //= '-' foreach @group_by_fields;
    }

    return [values(%group)];
}

sub _group_without_date {
    my ($self, $query, $data, %opts) = @_;

    my @order_by_fields = map {$_->{'field'}} @{$opts{'order_by'}};
    my %order_by_fields = map {$_ => TRUE} @order_by_fields;

    my @nodate_group_by_fields = grep {$_ ne 'date'} @{$query->group_by_fields};

    my @fields = grep {$order_by_fields{$_->{'id'}}} $query->get_fields_in_right_order;

    my %group;

    my ($key, $new_rec, $fld);
    my $tmp_rec;
    foreach my $rec (@$data) {
        $key = join($;, map {$rec->{$_} // chr(2)} @nodate_group_by_fields);

        $group{$key}{'__KEY__'} = $key if !exists($group{$key});
        $new_rec = $group{$key};

        foreach my $field (@fields) {
            if ($field->{'get'} eq 'db') {
                $new_rec->{$field->{'id'}} += ($rec->{$field->{'id'}} // 0);
            } else {
                $tmp_rec = clone($rec) unless defined($tmp_rec);
                _get_all_required_fields($field->{'id'}, $tmp_rec, $query);    # TODO: refactor _get_all_required_fields
                $fld = $field->{'get'}($tmp_rec);
                $new_rec->{$field->{'id'}} += ($fld eq '-' ? 0 : $fld);
            }
        }
        $rec->{'__KEY__'} = $key;
        $tmp_rec = undef;
    }

    return [values(%group)];
}

sub _keep_requested_fields_only {
    my ($self, $query, $data, %opts) = @_;

    my @group_by_fields    = @{$query->group_by_fields};
    my @fields             = $query->get_fields_in_right_order;
    my %fields_declaration = (%{$query->dimension_fields}, %{$query->entity_fields});

    my @result;
    foreach my $rec (@$data) {
        my %new_rec;

        foreach my $field (@fields) {
            _get_all_required_fields($field->{'id'}, $rec, $query);

            $new_rec{$field->{'id'}} =
              $field->{'get'} eq 'db' ? ($rec->{$field->{'id'}} || 0) + 0 : $field->{'get'}($rec);

            my $type = $field->{'type'} || '';

            if ($type eq 'boolean') {
                $new_rec{$field->{'id'}} = $new_rec{$field->{'id'}} ? 1 : 0;
            } elsif (!$opts{'no_money_format'} && $type eq 'money') {
                $new_rec{$field->{'id'}} = $self->money2float($new_rec{$field->{'id'}});
            }

            $rec->{$field->{'id'}} //= $new_rec{$field->{'id'}};
        }

        foreach my $field_name (@group_by_fields) {
            if (($fields_declaration{$field_name}->{'type'} // '') eq 'boolean') {
                $new_rec{$field_name} = $rec->{$field_name} ? 1 : 0;
            } else {
                $new_rec{$field_name} = $rec->{$field_name};
            }
        }

        push(@result, \%new_rec);
    }

    return \@result;
}

sub _obtain_uniform_data {
    my ($self, $query, $data) = @_;

    my @nocurrency_group_by_fields = grep {$_ ne 'currency_id'} @{$query->group_by_fields};
    my @fields                     = $query->need_db_field_names;
    my $is_currency_needed         = $query->is_currency_needed;

    if ($is_currency_needed) {
        my $no_currency;

        foreach my $row (@$data) {
            $row->{$_} //= 0 foreach @fields;
            $no_currency++ unless $row->{'currency_id'};
        }
        if ($no_currency) {
            #
            # TODO: refactor this part!
            #
            # Refactored code should be tested by requesting a page-level field like "hits"
            # together with some money field in a page-level report.
            #
            my @sort_data =
              sort {exists($a->{'currency_id'}) ? (exists($b->{'currency_id'}) ? 0 : -1) : 1} @$data;

            my %hash_currency_id;

            my $key;
            foreach my $row (@sort_data) {
                $key = join($;, map {$row->{$_} // chr(2)} @nocurrency_group_by_fields);

                if ($key && $row->{'currency_id'}) {
                    $hash_currency_id{$key}{$row->{'currency_id'}} = TRUE
                      unless $hash_currency_id{$key}{$row->{'currency_id'}};
                } elsif ($key) {
                    my $currency_id = (sort {$a <=> $b} keys(%{$hash_currency_id{$key}}))[0]
                      unless $hash_currency_id{$key}{2};

                    $row->{'currency_id'} = $currency_id || '2';
                } else {
                    $row->{'currency_id'} //= '2';
                }
            }
            $data = \@sort_data;
        }
    } else {
        foreach my $row (@$data) {
            $row->{$_} //= 0 foreach @fields;
        }
    }

    return $data;
}

sub _process_data_entity_fields {
    my ($self, $query, $data, %opts) = @_;

    my @entity_field_names = $query->entity_field_names;

    my $fields_obj;
    $fields_obj = $query->stat_level->get_entity_fields_obj([@entity_field_names]) if @entity_field_names;

    $fields_obj->model->pre_process_fields($fields_obj, $data) if $fields_obj && defined($fields_obj->model);

    $self->_join_dimension_fields($data, [values(%{$query->dimension_fields})]);

    $fields_obj->process_data($data) if $fields_obj;

    return $data;
}

sub _sort_by_fields {
    my ($self, $query, $data, %opts) = @_;

    # TEMP: 1 field only
    my $sort_fld = $opts{'order_by'}[0]{'field'};

    # reverse sort
    my @res_data = sort {$b->{$sort_fld} <=> $a->{$sort_fld}} @$data;

    return \@res_data;
}

sub _sort_by_type {
    my ($dimension_field, $data) = @_;

    my $type = $dimension_field->{'sort_as'} || 'text';

    my @result;

    if ($type eq 'text') {
        @result = sort {$a->{'label'} cmp $b->{'label'}} @$data;
    } elsif ($type eq 'number') {
        @result = sort {$a->{'label'} <=> $b->{'label'}} @$data;
    }

    return \@result;
}

sub _summarize_data_total {
    my ($self, $query, $data, %opts) = @_;

    my %total_db_fields;
    foreach my $rec (@$data) {
        $total_db_fields{$rec->{'currency_id'} // ''}->{$_} += $rec->{$_} foreach $query->need_db_field_names;
    }

    my %data_summary = ();
    if (%total_db_fields) {
        foreach my $currency (keys(%total_db_fields)) {
            foreach my $field (values(%{$query->fields})) {
                _get_all_required_fields($field->{'id'}, $total_db_fields{$currency}, $query);

                $data_summary{$currency}->{$field->{'id'}} =
                  $field->{'get'} eq 'db'
                  ? ($total_db_fields{$currency}->{$field->{'id'}} || 0) + 0
                  : $field->{'get'}($total_db_fields{$currency});

                $data_summary{$currency}->{$field->{'id'}} =
                  $self->money2float($data_summary{$currency}->{$field->{'id'}})
                  if !$opts{'no_money_format'} && ($field->{'type'} || '') eq 'money';
            }
        }
    }

    return \%data_summary;
}

sub _summarize_data_total_for_ch {
    my ($self, $ch_query, $top_keys) = @_;

    my $sub_query = $ch_query->{'__TABLES__'}[0]{'table'}{'query'};

    if ($top_keys) {
        my $filter = $sub_query->{'__TABLES__'}[0]->{'filter'}->expression;

        delete($filter->[1][-1]);

        $sub_query->{'__TABLES__'}[0]->{'filter'} = $self->clickhouse_db->filter($filter);
    }

    my $group_by = delete($sub_query->{'__GROUP_BY__'}) // [];

    my $currency_id_exists = in_array('_currency_id', $group_by);
    if ($currency_id_exists) {
        $group_by = [grep {$_ ne '_currency_id'} @$group_by];
    }

    foreach my $field (@$group_by) {
        delete($sub_query->{'__TABLES__'}[0]{'fields'}{$field});
        $field =~ s/^_//;
        delete($ch_query->{'__TABLES__'}[0]{'fields'}{$field});
    }

    if ($currency_id_exists) {
        $sub_query->group_by(qw(_currency_id));
    }

    delete($ch_query->{'__ORDER_BY__'});
    delete($ch_query->{'__LIMIT__'});

    my $data = $ch_query->get_all();

    return {map {my $currency_id = delete($_->{'currency_id'}) // ''; ($currency_id => $_)} @$data};
}

sub _tree_remove_excess_fields {
    my ($self, $subtree) = @_;

    foreach my $level (@$subtree) {
        delete($level->{'depth'});
        if (exists($level->{'children'})) {
            $self->_tree_remove_excess_fields($level->{'children'});
        } else {
            $level->{'fields'} = [$self->get_level($level->{'id'})->remove_shared_fields(@{$level->{'fields'}})];
        }
    }
}

TRUE;
