package Application::Model::Statistics::Charging::DSP;

use qbit;

use base qw(Application::Model::Statistics::Product);
use PiConstants qw( $VAT_COEFFICIENT );
use Math::Round;

use Exception::Validation::BadArguments;

sub accessor        {'statistics_dsp'}
sub db_table_name   {'statistics_dsp'}
sub parent_accessor {'statistics_charging'}

__PACKAGE__->model_accessors(
    partner_db => 'Application::Model::PartnerDB',
    dsp        => 'Application::Model::DSP',
    currency   => 'Application::Model::Currency'
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'can_be_statistics_reloaded',
            description => d_gettext('Rights for reloading statistics'),
            rights      => {
                statistics_dsp_can_be_reloaded => d_gettext('Right to reloading statistics by DSP'),
                statistics_dsp_view_field__internal_context_campaign_id =>
                  d_gettext('Right to view field "internal_context_campaign_id"'),
                statistics_dsp_view_field__campaign => d_gettext('Right to view field "campaign"'),
                statistics_dsp_view_field__is_internal_campaign =>
                  d_gettext('Right to view field "is_internal_campaign"'),
                statistics_dsp_view_field__login          => d_gettext('Right to view field "login"'),
                statistics_dsp_always_view                => d_gettext('Right to view level dsp always'),
                statistics_dsp_view_field__all_hits       => d_gettext('Right to view field "all_hits"'),
                statistics_dsp_view_field__all_real_price => d_gettext('Right to view field "all_real_price"'),
                statistics_dsp_view_field__average_rate   => d_gettext('Right to view field "average_rate"'),
            }
        }
    ]
);

sub query {
    my ($self, %opts) = @_;

    my $filter = $self->partner_db->filter(clone($opts{'filter'}));

    if (!$self->check_rights('dsp_view_all')) {
        my @pages = map {$_->{'id'}} @{$self->dsp->query(fields => $self->dsp->_get_fields_obj(['id']))->get_all()};
        $filter->and([dsp_id => '=' => \\@pages]);
    }

    my $res = $self->partner_db->query->select(
        table  => $self->_get_stat_table(),
        alias  => 's_dsp',
        fields => $opts{'fields'},
        filter => $filter,
    );
}

sub can_be_reloaded {
    my ($self) = @_;

    return $self->check_short_rights('can_be_reloaded');
}

sub dimension_fields {
    my ($self) = @_;

    return {
        date                 => {},
        currency_id          => {},
        page_id              => {check_rights => 'statistics_dsp_view_field__internal_context_campaign_id'},
        campaign             => {check_rights => 'statistics_dsp_view_field__campaign'},
        is_internal_campaign => {check_rights => 'statistics_dsp_view_field__is_internal_campaign'},
    };
}

sub entity_fields {
    my ($self) = @_;

    return [
        (
            $self->check_rights('dsp_view_all')
            ? {id => 'id', label => gettext('DSP&nbsp;ID'), type => 'straight_number'}
            : ()
        ),
        {id => 'short_caption', label => gettext('Connection'), verbatim => 1},
        (
            $self->check_short_rights('view_field__login')
            ? {
                id       => 'login',
                label    => gettext('Login'),
                verbatim => 1,
                ($self->check_rights('settings_cmd_ucard') ? (type => 'login') : ())
              }
            : ()
        )
    ];
}

sub fields {
    my ($self) = @_;

    return {
        dsp_shows                          => {},
        dsp_bids                           => {},
        dsp_cpm_charging_w_nds             => {},
        dsp_cpm_charging_wo_nds            => {},
        dsp_charging_w_nds                 => {},
        dsp_charging_wo_nds                => {},
        charging_w_nds                     => {},
        charging_wo_nds                    => {},
        all_hits                           => {check_rights => 'statistics_dsp_view_field__all_hits'},
        all_real_price_w_nds               => {check_rights => 'statistics_dsp_view_field__all_real_price'},
        all_real_price_wo_nds              => {check_rights => 'statistics_dsp_view_field__all_real_price'},
        average_rate_all_real_price_w_nds  => {check_rights => 'statistics_dsp_view_field__average_rate'},
        average_rate_all_real_price_wo_nds => {check_rights => 'statistics_dsp_view_field__average_rate'},
    };
}

# Ф-ция вовращает список полей по которым можо фильтровать СОБЫТИЯ в соответствии с текущим уровнем
sub get_event_fields {
    my ($self) = @_;
    return [{name => 'block_public_id', field => 'id'}];
}

sub join_condition_data {
    my ($self) = @_;
    return (
        dsp   => {condition => [id => '=' => {dsp_id => $self->_get_stat_table()}],},
        users => {
            condition  => [id => '=' => {owner_id => $self->partner_db->dsp}],
            depends_on => ['dsp'],
        }
    );
}

sub prefix {'dsp_'}

sub product {$_[0]->dsp}

sub sort_priority {270;}

sub support_clickhouse {FALSE}

sub title {gettext('Advertising platforms')}

sub _fix_delete_filter {
    my ($self, $filter, %opts) = @_;

    my @dsp_ids = $self->_filter_opts2dsp_ids(%opts);

    $filter->and({dsp_id => \@dsp_ids}) if @dsp_ids;
}

sub _filter_opts2dsp_ids {
    my ($self, %opts) = @_;

    return () unless %opts;

    my $unknown_opts = arrays_difference([sort keys(%opts)], [qw(logins)]);

    throw Exception::Validation::BadArguments ngettext(
        'Unknown filter option "%s"',
        'Unknown filter options "%s"',
        scalar(@$unknown_opts),
        join(', ', @$unknown_opts)
    ) if @$unknown_opts;

    my @res = @{$opts{'dsp_ids'} // []};

    if ($opts{'logins'}) {
        my $dsps = $self->product->get_all(
            fields => ['id'],
            filter => [owner => 'MATCH' => [login => '=' => $opts{'logins'}]]
        );

        # добавляем 0, чтобы не удалить всю статистику, если нет пейджей для логинов
        push(@res, @$dsps ? (map {$_->{'id'}} @$dsps) : 0);
    }

    return @res;
}

sub _filter_stat {
    my ($self, $data, %opts) = @_;

    my @ids = $self->_filter_opts2dsp_ids(%opts);

    return $data unless @ids;

    my %dsp_ids = map {$_ => TRUE} @ids;

    return [grep {$dsp_ids{$_->{'dsp_id'}}} @$data];
}

sub _get_last_update_day {
    return 3;
}

sub _get_stat_from_balance {
    my ($self, %opts) = @_;

    throw gettext("Expecting 'from'") unless defined $opts{'from'};
    throw gettext("Expecting 'to'")   unless defined $opts{'to'};

    throw gettext('DSPs statistics cannot be reload before 1 June 2013') if $opts{'from'} lt '2013-06-01';

    my %known_dsp = map {$_->{'id'} => TRUE} @{$self->product->get_all(fields => ['id'])};

    my %currency = map {$_->{'code'} => $_->{'id'}} @{$self->currency->get_all()};
    # Хак.
    # Ручка баланса Balance.GetDistributionSerpHits нам передает в качестве
    # кода валюты RUR, а мы ожидаем увидеть код RUB.
    # RUB является правильным кодом, поэтому мы ждем исправления, которое
    # будет сделано в BALANCE-15191
    # До тех пор пока баланс это не исправил для решения тикета PI-645
    # и реализован этот хак: и RUR преобразуется в код валюты для RUB
    $currency{'RUR'} = 2;

    my @balance_stat = map {
        {
            dt          => trdate('db_time', 'db', $_->{'DT'}),
            dsp_id      => $_->{'DSP_ID'},
            campaign_id => $_->{'PAGE_ID'},
            currency_id => $currency{$_->{'CURRENCY'}} // throw gettext('Unknown currency %s', $_->{'CURRENCY'}),
            dsp_shows   => $_->{'SHOWS'},
            dsp_bids    => $_->{'HITS'},
            dsp_charging_w_nds  => $self->_get_int($_->{'DSP'}),
            dsp_charging_wo_nds => $self->_get_int($_->{'DSPWITHOUTNDS'}),
            charging_w_nds      => $self->_get_int($_->{'DSP'}),
            charging_wo_nds     => $self->_get_int($_->{'DSPWITHOUTNDS'}),
        }
      } grep {
        $known_dsp{$_->{'DSP_ID'}}
      } @{$self->app->api_balance->get_dsp_money_completition_with_page_id(from => $opts{'from'}, to => $opts{'to'})};

    my @bk_data =
      grep {$known_dsp{$_->{'DSPID'}}} @{
        $self->app->api_http_bk->get_dsp_page_stat(
            starttime => $opts{'from'},
            stoptime  => $opts{'to'},
        )
      };

    my %dsp_page_stat = ();
    foreach my $row (@bk_data) {
        my $key = join($;, 2, map {$row->{$_} // chr(2)} qw(Date PageID DSPID));

        if (exists($dsp_page_stat{$key})) {
            $dsp_page_stat{$key}->{'all_hits'}              += $row->{'AllHits'}      || 0;
            $dsp_page_stat{$key}->{'all_real_price_wo_nds'} += $row->{'AllRealPrice'} || 0;
        } else {
            $dsp_page_stat{$key} = {
                hash_transform(
                    $row,
                    [],
                    {
                        Date         => 'dt',
                        PageID       => 'campaign_id',
                        DSPID        => 'dsp_id',
                        AllHits      => 'all_hits',
                        AllRealPrice => 'all_real_price_wo_nds',
                    }
                )
            };
        }
    }

    foreach my $row (@balance_stat) {
        my $key = join($;, map {$row->{$_} // chr(2)} qw(currency_id dt campaign_id dsp_id));

        if (exists($dsp_page_stat{$key})) {
            $row->{'all_hits'}              = $dsp_page_stat{$key}->{'all_hits'};
            $row->{'all_real_price_wo_nds'} = $self->_get_int($dsp_page_stat{$key}->{'all_real_price_wo_nds'}, 4);
            $row->{'all_real_price_w_nds'}  = round($row->{'all_real_price_wo_nds'} * $VAT_COEFFICIENT) + 0;
            delete($dsp_page_stat{$key});
        } else {
            $row->{'all_hits'}              = 0;
            $row->{'all_real_price_wo_nds'} = 0;
            $row->{'all_real_price_w_nds'}  = 0;
        }
    }

    push(
        @balance_stat,
        map {
            {
                %$_,
                  all_real_price_w_nds => round($self->_get_int($_->{'all_real_price_wo_nds'}, 4) * $VAT_COEFFICIENT) +
                  0,
                  all_real_price_wo_nds => $self->_get_int($_->{'all_real_price_wo_nds'}, 4),
                  currency_id           => 2,
                  dsp_shows             => 0,
                  dsp_bids              => 0,
                  dsp_charging_w_nds    => 0,
                  dsp_charging_wo_nds   => 0,
                  charging_w_nds        => 0,
                  charging_wo_nds       => 0,
            }
          } values(%dsp_page_stat)
        ) if %dsp_page_stat;

    return \@balance_stat;
}

TRUE;
