package Application::Model::Statistics::Fields::ClickhouseExpressions;

use qbit;

use base qw(Exporter);

use Application::Model::Statistics::Fields::Types qw(@FIELD_TYPES);

our @EXPORT = qw($CLIKHOUSE_EXPRESSIONS);

my $category_id = {
    if => [
        ['product_id', '=', \'video_an_site'],
        'category_id',
        {
            dictGetUInt32 => [
                \'blocks_dict', \'category_id',
                {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
            ]
        }
    ]
};

our $CLIKHOUSE_EXPRESSIONS = {
    # Metrics
    (map {$_->{'id'} => $_->{'clickhouse_expression'}} @FIELD_TYPES),

    #
    # Common
    #

    date => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $value = $field->{'value'} || 'day';

        if ($value eq 'day' || $value eq 'wo') {
            return 'dt';
        } elsif ($value eq 'week') {
            return {toMonday => ['dt'],};
        } elsif ($value eq 'month') {
            return {toStartOfMonth => ['dt'],};
        } elsif ($value eq 'year') {
            return {toStartOfYear => ['dt'],};
        }
    },

    multistate => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $product  = $stat_level->get_product;
        my $accessor = $product->accessor;

        if ($accessor eq 'site') {
            return {
                dictGetUInt64 => [
                    \'sources_dict', \'multistate',
                    {dictGetUInt64 => [\'pages_dict', \'source_id', {toUInt64 => ['page_id']}]}
                ]
            };
        } elsif ($product->isa('Application::Model::Page')) {
            return {dictGetUInt64 => [\'pages_dict', \'multistate', {toUInt64 => ['page_id']}]};
        } elsif ($product->isa('Application::Model::Block')) {
            return {
                dictGetUInt64 => [
                    \'blocks_dict', \'multistate',
                    {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
                ]
            };
        } else {
            throw gettext('Unknown case');
        }
    },

    #
    # Sites
    #

    # Domains
    bundle_id => {
        dictGetString =>
          [\'sources_dict', \'domain', {dictGetUInt64 => [\'pages_dict', \'source_id', {toUInt64 => ['page_id']}]}]
    },
    domain => {
        dictGetString =>
          [\'sources_dict', \'domain', {dictGetUInt64 => [\'pages_dict', \'source_id', {toUInt64 => ['page_id']}]}]
    },
    site_domain => {
        dictGetString =>
          [\'sources_dict', \'domain', {dictGetUInt64 => [\'pages_dict', \'source_id', {toUInt64 => ['page_id']}]}]
    },
    store_id => {
        dictGetString =>
          [\'sources_dict', \'domain', {dictGetUInt64 => [\'pages_dict', \'source_id', {toUInt64 => ['page_id']}]}]
    },

    is_graysite => {
        dictGetUInt8 => [
            \'sources_dict', \'is_graysite', {dictGetUInt64 => [\'pages_dict', \'source_id', {toUInt64 => ['page_id']}]}
        ]
    },

    #
    # Users
    #
    user_id  => {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]},
    owner_id => {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]},
    name     => sub {
        my ($field, $product_ids, $stat_level) = @_;

        if ($stat_level->get_product->accessor eq 'ssp_seller') {
            return {
                dictGetString => [
                    \'sellers_dict', \'name', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}
                ]
            };
        } else {
            return {dictGetString =>
                  [\'users_dict', \'name', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}]
            };
        }
    },
    lastname => {
        dictGetString =>
          [\'users_dict', \'lastname', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}]
    },
    login => {
        dictGetString =>
          [\'users_dict', \'login', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}]
    },
    client_id => {
        dictGetUInt32 =>
          [\'users_dict', \'client_id', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}]
    },
    is_tutby => {
        dictGetUInt8 =>
          [\'users_dict', \'is_tutby', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}]
    },
    is_mobile_mediation => {
        dictGetUInt8 => [
            \'users_dict', \'is_mobile_mediation',
            {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}
        ]
    },
    is_adfox_partner => {
        dictGetUInt8 => [
            \'users_dict', \'is_adfox_partner',
            {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}
        ]
    },
    is_video_blogger => {
        dictGetUInt8 => [
            \'users_dict', \'is_video_blogger',
            {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}
        ]
    },
    is_games => {
        dictGetUInt8 =>
          [\'users_dict', \'is_games', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}]
    },
    is_efir_blogger => {
        dictGetUInt8 => [
            \'users_dict', \'is_efir_blogger',
            {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}
        ]
    },
    email => {
        dictGetString =>
          [\'users_dict', \'email', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}]
    },
    phone => {
        dictGetString =>
          [\'users_dict', \'phone', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}]
    },

    #
    # Sellers
    #

    seller_id => {
        dictGetUInt32 =>
          [\'users_dict', \'client_id', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}]
    },
    seller_name => {
        dictGetString =>
          [\'sellers_dict', \'name', {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}]
    },

    #
    # Pages
    #

    campaign_id     => 'page_id',
    context_page_id => 'page_id',

    # используется в фильтре all_pages
    #`model` IN ('context_on_site_campaign', 'search_on_site_campaign')
    model => \'context_on_site_campaign',

    # Captions

    an_campaign_caption      => {dictGetString => [\'pages_dict', \'caption', {toUInt64 => ['page_id']}]},
    context_campaign_caption => {dictGetString => [\'pages_dict', \'caption', {toUInt64 => ['page_id']}]},
    internal_mobile_caption  => {dictGetString => [\'pages_dict', \'caption', {toUInt64 => ['page_id']}]},
    mobile_caption           => {dictGetString => [\'pages_dict', \'caption', {toUInt64 => ['page_id']}]},
    search_campaign_caption  => {dictGetString => [\'pages_dict', \'caption', {toUInt64 => ['page_id']}]},
    video_caption            => {dictGetString => [\'pages_dict', \'caption', {toUInt64 => ['page_id']}]},
    caption                  => sub {
        my ($field, $product_ids, $stat_level) = @_;

        # caption has different meaning on different levels
        return $stat_level->isa('Application::Model::Statistics::Hierarchy')
          ? {dictGetString => [\'pages_dict', \'caption', {toUInt64 => ['page_id']}]}
          : {
            dictGetString => [
                \'blocks_dict', \'caption',
                {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
            ]
          };
    },

    # Domain ids
    application_id => sub {
        my ($field, $product_ids, $stat_level) = @_;

        if ($stat_level->get_product->accessor eq 'internal_mobile_app_rtb') {
            return 'page_id';
        } else {
            return {dictGetUInt32 => [\'pages_dict', \'domain_id', {toUInt64 => ['page_id']}]};
        }
    },
    domain_id => {dictGetUInt32 => [\'pages_dict', \'domain_id', {toUInt64 => ['page_id']}]},
    site_id   => {dictGetUInt32 => [\'pages_dict', \'domain_id', {toUInt64 => ['page_id']}]},
    source_id => {dictGetUInt32 => [\'pages_dict', \'domain_id', {toUInt64 => ['page_id']}]},

    # ID
    id => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $product          = $stat_level->get_product;
        my $product_accessor = $product->accessor;

        if ($product_accessor eq 'ssp_seller') {
            return {
                dictGetUInt32 => [
                    \'users_dict', \'client_id',
                    {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]}
                ]
            };
        } elsif ($product_accessor eq 'internal_site') {
            return {dictGetUInt32 => [\'pages_dict', \'domain_id', {toUInt64 => ['page_id']}]};
        } elsif ($product->isa('Application::Model::Users')) {
            return {dictGetUInt64 => [\'pages_dict', \'owner_id', {toUInt64 => ['page_id']}]};
        } elsif ($product->isa('Application::Model::Page')) {
            return {dictGetUInt32 => [\'pages_dict', \'id', {toUInt64 => ['page_id']}]};
        } elsif ($product->isa('Application::Model::Block')) {
            return 'block_id';
        }
    },
    page_public_id => {dictGetUInt32 => [\'pages_dict', \'id', {toUInt64 => ['page_id']}]},

    # Product id
    is_internal_campaign => {
        #TODO: added all internal products?
        if => [['product_id', '=', \'internal_context_on_site_campaign'], \1, \0]
    },

    category_id => $category_id,
    categories  => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $locale = $stat_level->app->get_option('locale', 'ru');

        return {
            dictGetStringOrDefault => [
                \'video_categories_dict', \"name_$locale",
                {tuple => [$category_id, 'page_id']}, {toString => [$category_id]}
            ]
        };
    },

    #
    # Blocks
    #

    # Adfox block
    adfox_block => {
        dictGetUInt8 => [
            \'blocks_dict', \'adfox_block',
            {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
        ]
    },

    internal_campaign_id => {dictGetUInt32 => [\'pages_dict', \'id', {toUInt64 => ['page_id']}]},

    # Mobile block type
    block_type => {
        dictGetString => [
            \'blocks_dict', \'mobile_block_type',
            {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
        ]
    },
    internal_mobile_block_type_label => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $block_types = $stat_level->app->internal_mobile_app->get_block_types();

        my @values = ();
        my @labels = ();

        foreach (@$block_types) {
            push(@values, \$_->{'id'});
            push(@labels, \$_->{'label'});
        }

        return {
            transform => [
                {
                    dictGetString => [
                        \'blocks_dict',
                        \'mobile_block_type',
                        {
                            tuple =>
                              [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]
                        }
                    ]
                },
                {-array => \@values},
                {-array => \@labels},
            ]
        };
    },
    mobile_block_type_label => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $block_types = $stat_level->app->mobile_app_settings->get_block_types();

        my @values = ();
        my @labels = ();

        foreach (@$block_types) {
            push(@values, \$_->{'id'});
            push(@labels, \$_->{'label'});
        }

        return {
            transform => [
                {
                    dictGetString => [
                        \'blocks_dict',
                        \'mobile_block_type',
                        {
                            tuple =>
                              [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]
                        }
                    ]
                },
                {-array => \@values},
                {-array => \@labels},
            ]
        };
    },
    block_type_label => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $block_types = $stat_level->get_product->page->get_block_types();

        my @values = ();
        my @labels = ();

        foreach (@$block_types) {
            push(@values, \$_->{'id'});
            push(@labels, \$_->{'label'});
        }

        return {
            transform => [
                {
                    dictGetString => [
                        \'blocks_dict',
                        \'mobile_block_type',
                        {
                            tuple =>
                              [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]
                        }
                    ]
                },
                {-array => \@values},
                {-array => \@labels},
            ]
        };
    },

    # Direct block
    direct_block => {
        dictGetString => [
            \'blocks_dict', \'direct_block',
            {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
        ]
    },

    type => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $product = $stat_level->get_product;
        if ($product->isa('Application::Model::Block::Video')) {
            #Тип видео блока
            return {
                dictGetUInt8 => [
                    \'blocks_dict', \'video_block_type',
                    {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
                ]
            };
        } elsif ($product->accessor eq 'internal_mobile_app') {
            #Тип мобильного приложения ios/android
            return {
                dictGetUInt8 => [
                    \'sources_dict', \'type',
                    {dictGetUInt64 => [\'pages_dict', \'source_id', {toUInt64 => ['page_id']}]}
                ]
            };
        } else {
            #формат директа
            return {
                dictGetString => [
                    \'blocks_dict', \'direct_block',
                    {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
                ]
            };
        }
    },

    site_version => {
        dictGetString => [
            \'blocks_dict', \'site_version',
            {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
        ]
    },

    # Disposition id
    disposition_name => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $disposition_ids = $stat_level->get_product->get_disposition_ids();

        my @values = ();
        my @labels = ();

        foreach (sort keys(%$disposition_ids)) {
            push(@values, \$_);
            push(@labels, \$disposition_ids->{$_});
        }

        return {
            transform => [
                {
                    dictGetUInt8 => [
                        \'blocks_dict',
                        \'disposition_id',
                        {
                            tuple =>
                              [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]
                        }
                    ]
                },
                {-array => \@values},
                {-array => \@labels},
                $labels[0]
            ]
        };
    },

    # Media formats list
    media_formats_list => {
        dictGetString => [
            \'blocks_dict', \'media_formats_list',
            {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
        ]
    },

    # Public ID
    #TODO: если группировка по public_id на уровне пейджа, не учитывать пейджовый product_id
    public_id => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $app = $stat_level->app;

        my @values = ();
        my @labels = ();

        my %product_ids_hash = map {$_ => TRUE} @$product_ids;

        foreach my $block_accessor (sort @{$app->product_manager->get_block_model_accessors()}) {
            next unless $product_ids_hash{$block_accessor};

            push(@values, \$block_accessor);
            push(@labels, \$app->$block_accessor->public_id_prefix());
        }

        return {
            concat => [
                {transform => [{toString => ['product_id']}, {-array => \@values}, {-array => \@labels}]},
                {toString  => ['page_id']},
                \'-', {toString => ['block_id']}
            ]
        };
    },

    # Caption
    title => {
        dictGetString => [
            \'blocks_dict', \'caption',
            {tuple => [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]}
        ]
    },

    type_label => sub {
        my ($field, $product_ids, $stat_level) = @_;

        my $types = $stat_level->get_product->get_types();

        my @values = ();
        my @labels = ();

        foreach (@$types) {
            push(@values, \$_->{'id'});
            push(@labels, \$_->{'label'});
        }

        return {
            transform => [
                {
                    dictGetUInt8 => [
                        \'blocks_dict',
                        \'video_block_type',
                        {
                            tuple =>
                              [{toString => ['product_id']}, {toUInt32 => ['page_id']}, {toUInt32 => ['block_id']}]
                        }
                    ]
                },
                {-array => \@values},
                {-array => \@labels},
                $labels[0]
            ]
        };
    },

    dsp_id_name => 'dsp_id',

    monetizer_caption => sub {
        my $default_prefix = gettext('Monetizer #');

        return {
            dictGetStringOrDefault => [
                \'monetizers_dict', \'caption',
                {toUInt64 => ['monetizer_id']}, {concat => [\$default_prefix, {toString => ['monetizer_id']}]}
            ]
        };
    },

    #
    # DSPS
    #

    dsp_caption => sub {
        my $default_prefix = gettext('DSP #');

        return {
            dictGetStringOrDefault => [
                \'dsps_dict', \'short_caption',
                {toUInt64 => ['dsp_id']}, {concat => [\$default_prefix, {toString => ['dsp_id']}]}
            ]
        };
    },

    #
    # Tags
    #

    tag_caption => sub {
        my $default_prefix = gettext('Slice #');

        return {
            dictGetStringOrDefault => [
                \'block_tags_dict', \'caption',
                {tuple  => [{toUInt32         => ['page_id']}, {toUInt64 => ['tag_id']}]},
                {concat => [\$default_prefix, {toString        => ['tag_id']}]}
            ]
        };
    },
    geo_name => sub {
        my $default_name = gettext('All world');

        return {if => [['geo_id' => '=', \0], \$default_name, {regionToName => [{toUInt32 => ['geo_id']}]}]};
    },
};
