package Application::Model::Statistics::Fields::Metrics;

use qbit;

use base qw(Exporter);

our @EXPORT_OK = qw(
  get_db_field
  get_field
  get_field_all_metrics
  get_field_an_cover_hits
  get_field_an_fraud
  get_field_average_rate
  get_field_bad_metric_percent
  get_field_clicks
  get_field_conversion_percent
  get_field_cover_ratio
  get_field_cpc
  get_field_cpm
  get_field_cpmh
  get_field_ctr
  get_field_direct_ad_metrics
  get_field_direct_ads
  get_field_dsp_bids
  get_field_hits
  get_field_instream_block_metrics
  get_field_money
  get_field_money_with_currency
  get_field_page_ad_shows
  get_field_percent_money
  get_field_rpm
  get_field_shows
  get_field_video_sum_money
  get_field_visibility
  get_field_winrate
  );
our @EXPORT = @EXPORT_OK;

use Application::Model::Statistics::Fields::Dictionary qw(
  $VAT_HASH
  $ACCESSORS_BY_SHORT_PRODUCT
  $PRODUCTS_WITHOUT_POSTFIX_BLOCK
  $PAGE_ACCESSORS_BY_SHORT_PRODUCT
  get_rtb_products
  get_video_products
  );

use Application::Model::Statistics::Fields::Titles;
use Application::Model::Statistics::Fields::ShortTitles;
use Application::Model::Statistics::Fields::Hints;
use Application::Model::Statistics::Fields::AdditionalText;

use PiConstants qw(
  $CATEGORY_CALCULATED
  $CATEGORY_CALCULATED_WITH_MONEY
  $CATEGORY_CASH
  $CATEGORY_COUNTER
  $CATEGORY_PAGE_FIELDS
  $CH_MONEY_POST_SCALE
  $CH_MONEY_PRE_SCALE
  $VIDEO_BLOCK_TYPES
  );

my $RTB_PRODUCTS   = get_rtb_products();
my $VIDEO_PRODUCTS = get_video_products();

my $SHORT_PRODUCTS_REGEXP = '(' . join('|', map {"(?:$_)"} keys(%$ACCESSORS_BY_SHORT_PRODUCT)) . ')';

my $PRODUCTS_WITHOUT_POSTFIX_BLOCK_REGEXP =
  '(' . join('|', map {"(?:$_)"} keys(%$PRODUCTS_WITHOUT_POSTFIX_BLOCK)) . ')';

sub get_field {
    my ($id, %opts) = @_;

    my $unit = _get_unit($id, %opts);

    return {
        id    => $id,
        title => $opts{'title'} // $TITLES->{$id} // throw "Title not found for field '$id'",
        hint  => $opts{'hint'} // $HINTS->{$id} // throw "Hint not found for field '$id'",
        ($unit ? (unit => $unit) : ()),
        %opts,
    };
}

sub _get_unit {
    my ($id, %opts) = @_;

    if ($id =~ /conversion_percent|_ctr/) {
        return 'ctr_percent';
    } elsif ($id =~ /_percent|percent_|_winrate|_visibility|_ratio|_reach|_coverage/) {
        return 'percent';
    } elsif (
        $id =~ /_shows|_hits|_clicks|_senthits|dsp_bids|instream_block_view|instream_block_open_player|_impressions/)
    {
        return 'count';
    } elsif ($id =~ /_ads/) {
        return 'ads_count';
    } elsif ($id =~ /(?:direct|adblock|stripe|premium|dsp).*cpm/) {
        return 'cpm';
    } elsif ($id =~ /(?:instream|inpage|fullscreen|indoor|outdoor).*cpmh/) {
        return 'rpm';
    } elsif ($id =~ /(?:mcb|market|instream|inpage|fullscreen|indoor|outdoor).*cpm/) {
        return 'cpmv';
    } elsif ($id =~ /(?:content|rtb|mobile|natural).*cpmh/) {
        return 'ecpm';
    } elsif ($id =~ /(?:content|rtb|mobile|natural).*cpm/) {
        return 'cpmv';
    } elsif ($id =~ /(?:direct|stripe|premium|market).*cpc/) {
        return 'cpm';
    } elsif ($id =~ /rpm/) {
        return 'rpm';
    } elsif ($id =~ /_w_nds|_wo_nds|calculated_revenue_original/) {
        return 'money';
    } elsif ($id =~ /calculated_revenue/) {
        return {
            type => 'currency',
            id   => $opts{currency_id},
        };
    } else {
        return undef;
    }
}

sub _get_titles {
    my ($id, $money_type, $vat) = @_;

    my $title = $TITLES->{$id} // throw "Title not found for id '$id'";

    my $short_title = $SHORT_TITLES->{$id} // throw "Short title not found for id '$id'";

    my $additional_text = $ADDITIONAL_TEXT->{$money_type} // $TITLES->{$money_type}
      // throw "Additional text not found for id '$id'";

    return (
        title => sub {
            sprintf("%s&nbsp;— %s (%s)", $title->(), lc($additional_text->()), $VAT_HASH->{$vat}->());
        },
        short_title => sub {
            sprintf("%s&nbsp;— %s", $short_title->(), lc($additional_text->()));
        }
    );
}

sub _get_product_filter {
    my ($short_product, $product_ids, %opts) = @_;

    my %products;
    if ($opts{'products'}) {
        %products = map {$_ => TRUE} @{$opts{'products'}};
    } else {
        %products =
          map {$_ => TRUE} $opts{'page_accessors'}
          ? @{$PAGE_ACCESSORS_BY_SHORT_PRODUCT->{$short_product}}
          : @{$ACCESSORS_BY_SHORT_PRODUCT->{$short_product}};
    }

    my @used_product_ids = grep {$products{$_}} @$product_ids;

    if (@used_product_ids > 1) {
        return ['product_id', 'IN', \[@used_product_ids]];
    } elsif (@used_product_ids == 1) {
        return ['product_id', '=', \$used_product_ids[0]];
    } else {
        return (\0);
    }
}

sub get_field_dsp_bids {
    my ($id) = @_;

    if ($id ne 'dsp_bids') {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_dsp_bids');
    }

    return get_field(
        $id,
        get      => 'db',
        type     => 'number',
        category => $CATEGORY_COUNTER,
        section  => 'default',
        #TODO: add clickhouse_expression
    );
}

sub get_field_an_fraud {
    my ($id) = @_;

    unless ($id =~ /^an_fraud_(?:(?:shows)|(?:clicks)|(?:hits))\z/) {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_an_fraud');
    }

    return get_field(
        $id,
        get                   => 'db',
        type                  => 'number',
        category              => $CATEGORY_COUNTER,
        conflicts             => {dimension_fields => ['public_id', 'adfox_block']},
        clickhouse_expression => sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => [
                    $id,
                    _get_product_filter(
                        undef,
                        $product_ids,
                        products => [
                            qw(
                              ssp_context_on_site_campaign
                              ssp_video_an_site
                              video_an_site
                              indoor
                              outdoor
                              context_on_site_campaign
                              internal_context_on_site_campaign
                              )
                        ]
                    )
                ]
            };
        },
    );
}

sub get_field_page_ad_shows {
    my ($id) = @_;

    my ($short_product, $field_name);
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_page_ad_shows\z/) {
        $short_product = $1;

        $field_name = $short_product;
        $field_name =~ s/_search|_context//;
        $field_name .= '_page_ad_shows';
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_page_ad_shows');
    }

    return get_field(
        $id,
        get                   => 'db',
        type                  => 'number',
        category              => $CATEGORY_COUNTER,
        clickhouse_expression => sub {
            my ($field, $product_ids) = @_;

            return {sumIf => [$field_name, _get_product_filter($short_product, $product_ids, page_accessors => TRUE)]};
        },
        conflicts => {dimension_fields => ['public_id', 'adfox_block']},
    );
}

sub get_field_all_metrics {
    my ($id) = @_;

    my ($depends, $has_section, $clickhouse_expression, $has_dsp_conflicts);
    my $has_forced = TRUE;
    if ($id eq 'video_an_site_sum_all_hits') {
        $depends = [qw(instream_block_all_hits inpage_block_all_hits fullscreen_block_all_hits)];

        my %ch_depends = map {$_ => get_field_all_metrics($_)} @$depends;

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return ['+' => [map {$ch_depends{$_}->{'clickhouse_expression'}->($ch_depends{$_}, $product_ids)} @$depends]
            ];
        };
    } elsif ($id =~ /^video_an_site_all_(.+)\z/) {
        #video_an_site_all_hits_own_adv

        my $field = $1;

        $depends = [map {$_ . "_$field"} qw(instream_block inpage_block fullscreen_block)];

        my %ch_depends = ();
        if ($field =~ /shows/) {
            $ch_depends{$_} = get_field_shows($_) foreach @$depends;
        } else {
            $ch_depends{$_} = get_field_hits($_) foreach @$depends;
        }

        $has_section = $field eq 'shows' || $field eq 'hits';

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return ['+' => [map {$ch_depends{$_}->{'clickhouse_expression'}->($ch_depends{$_}, $product_ids)} @$depends]
            ];
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_all_hits\z/) {
        #instream_block_all_hits

        my $short_product = $1;

        # rtb blocks

        my $is_video = in_array($short_product, $VIDEO_PRODUCTS);
        if ($is_video) {
            $has_dsp_conflicts = TRUE;
            $depends = [map {"${short_product}_$_"} qw(win_hits win_hits_own_adv win_hits_unsold)];
        } else {
            $depends = [map {"${short_product}_$_"} qw(hits hits_own_adv hits_unsold)];
        }

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf =>
                  [$is_video ? 'win_max_positions_count' : 'hits', _get_product_filter($short_product, $product_ids)]
            };
        };

        my %has_not_forced = map {$_ => TRUE} qw(
          natural_block
          rtb_block
          mobile_rtb_block
          ssp_mobile_rtb_block
          adblock_block
          );

        $has_forced = !$has_not_forced{$short_product};
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_all_metrics');
    }

    return get_field(
        $id,
        depends_on => $depends,
        type       => 'number',
        get        => sub {
            my ($stat) = @_;
            my $sum = 0;

            foreach my $field (@$depends) {
                $sum += $stat->{$field};
            }

            return $sum;
        },
        category => $CATEGORY_COUNTER,
        ($has_dsp_conflicts ? (conflicts => {dimension_fields => ['dsp_id_name', 'dsp_caption']}) : ()),
        ($has_forced            ? (forced                => TRUE)                   : ()),
        ($has_section           ? (section               => 'default',)             : ()),
        ($clickhouse_expression ? (clickhouse_expression => $clickhouse_expression) : ())
    );
}

sub safe_div($$) {!defined($_[1]) || $_[1] == 0 ? '-' : $_[0] / $_[1]}

sub get_field_bad_metric_percent {
    my ($id) = @_;

    my $depends;
    my %ch_depends;
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_bad_win_hits_percent\z/) {
        #instream_block_bad_win_hits_percent
        $depends = [map {"$1_$_"} qw(hits bad_win_hits)];

        $ch_depends{$_} = get_field_hits($_) foreach @$depends;
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_bad_shows_percent\z/) {
        #instream_block_bad_shows_percent
        $depends = [map {"$1_$_"} qw(shows bad_shows)];

        $ch_depends{$_} = get_field_shows($_) foreach @$depends;
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_bad_win_hits_percent');
    }

    my $ch_count     = $ch_depends{$depends->[0]};
    my $ch_bad_count = $ch_depends{$depends->[1]};

    return get_field(
        $id,
        depends_on => $depends,
        type       => 'percent',
        get        => sub {
            my ($stat) = @_;

            safe_div($stat->{$depends->[1]} * 100, $stat->{$depends->[0]} + $stat->{$depends->[1]});
        },
        category              => $CATEGORY_CALCULATED,
        clickhouse_expression => sub {
            my ($field, $product_ids) = @_;

            my $bad_count_expr = $ch_bad_count->{'clickhouse_expression'}->($ch_bad_count, $product_ids);

            return [
                '/' => [
                    ['*' => [$bad_count_expr, \100]],
                    ['+' => [$ch_count->{'clickhouse_expression'}->($ch_count, $product_ids), $bad_count_expr]]
                ]
            ];
        },
    );
}

sub get_field_winrate {
    my ($id) = @_;

    my ($depends, %ch_fields, $short_product);
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_winrate\z/) {
        #instream_block_winrate

        $short_product = $1;
        $depends = [map {"${short_product}_$_"} qw(hits hits_own_adv hits_unsold)];

        $ch_fields{$_} = get_field_hits($_) foreach @$depends;
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_winrate');
    }

    my $is_video = in_array($short_product, $VIDEO_PRODUCTS);

    return get_field(
        $id,
        depends_on => $depends,
        type       => 'percent',
        get        => sub {
            my ($stat) = @_;

            safe_div($stat->{$depends->[0]} * 100,
                $stat->{$depends->[0]} + $stat->{$depends->[1]} + $stat->{$depends->[2]});
        },
        category              => $CATEGORY_CALCULATED,
        clickhouse_expression => sub {
            my ($field, $product_ids) = @_;

            my $ch_field =
              $ch_fields{$depends->[0]}->{'clickhouse_expression'}->($ch_fields{$depends->[0]}, $product_ids);

            #TODO: можно улучшить если будет тормазить
            # сумма это просто hits
            return [
                '/' => [
                    ['*' => [$ch_field, \100]],
                    [
                        '+' => [
                            $ch_field,
                            $ch_fields{$depends->[1]}->{'clickhouse_expression'}
                              ->($ch_fields{$depends->[1]}, $product_ids),
                            $ch_fields{$depends->[2]}->{'clickhouse_expression'}
                              ->($ch_fields{$depends->[2]}, $product_ids)
                        ]
                    ]
                ]
            ];
        },
        ($is_video ? (forced => TRUE) : ()),
        ($is_video ? (conflicts => {dimension_fields => ['dsp_id_name', 'dsp_caption']}) : ()),
    );
}

sub get_field_visibility {
    my ($id) = @_;

    my ($depends, $ch_shows, $ch_hits, $has_forced);
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_visibility\z/) {
        #instream_block_visibility

        if (in_array($1, $VIDEO_PRODUCTS)) {
            $depends = [map {"$1_$_"} qw(shows hits)];
            $has_forced = TRUE;
        } elsif ($1 eq 'mobile_mediation_block') {
            $depends = [map {"$1_$_"} qw(impressions hits)];
        } else {
            $depends = [map {"$1_$_"} qw(shows hits)];
        }

        $ch_shows = get_field_shows($depends->[0]);
        $ch_hits  = get_field_hits($depends->[1]);
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_visibility');
    }

    my $has_default = $id =~ /^(content|rtb|natural)_block_/;

    return get_field(
        $id,
        depends_on => $depends,
        type       => 'percent',
        get        => sub {
            my ($stat) = @_;

            safe_div($stat->{$depends->[0]} * 100, $stat->{$depends->[1]});
        },
        category              => $CATEGORY_CALCULATED,
        section               => 'default',
        clickhouse_expression => sub {
            my ($field, $product_ids) = @_;

            return [
                '/' => [
                    ['*' => [$ch_shows->{'clickhouse_expression'}->($ch_shows, $product_ids), \100]],
                    $ch_hits->{'clickhouse_expression'}->($ch_hits, $product_ids)
                ]
            ];
        },
        ($has_default ? (default => TRUE) : ()),
        ($has_forced  ? (forced  => TRUE) : ())
    );
}

sub get_field_cover_ratio {
    my ($id) = @_;

    my ($depends, $has_forced, $ch_shows, $ch_all_hits, $has_dsp_conflicts);
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_cover_ratio\z/) {
        #instream_block_cover_ratio

        my $short_product = $1;
        $depends = [map {"${short_product}_$_"} qw(shows all_hits)];

        $ch_shows    = get_field_shows($depends->[0]);
        $ch_all_hits = get_field_all_metrics($depends->[1]);

        $has_dsp_conflicts = $has_forced = in_array($short_product, $VIDEO_PRODUCTS);
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_((?:preroll)|(?:postroll)|(?:midroll)|(?:pauseroll))_cover_ratio\z/) {
        #instream_block_preroll_cover_ratio

        $depends = [map {"$1_$2_$_"} qw(shows all_hits)];

        $ch_shows    = get_field_instream_block_metrics($depends->[0]);
        $ch_all_hits = get_field_instream_block_metrics($depends->[1]);
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_cover_ratio');
    }

    return get_field(
        $id,
        depends_on => $depends,
        type       => 'percent',
        get        => sub {
            my ($stat) = @_;

            safe_div($stat->{$depends->[0]} * 100, $stat->{$depends->[1]});
        },
        category              => $CATEGORY_CALCULATED,
        clickhouse_expression => sub {
            my ($field, $product_ids) = @_;

            return [
                '/' => [
                    ['*' => [$ch_shows->{'clickhouse_expression'}->($ch_shows, $product_ids), \100]],
                    $ch_all_hits->{'clickhouse_expression'}->($ch_all_hits, $product_ids),
                ]
            ];
        },
        ($has_dsp_conflicts ? (conflicts => {dimension_fields => ['dsp_id_name', 'dsp_caption']}) : ()),
        ($has_forced ? (forced => TRUE) : ())
    );
}

sub get_field_conversion_percent {
    my ($id) = @_;

    my $depends;
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_((?:preroll)|(?:postroll)|(?:midroll)|(?:pauseroll))_conversion_percent\z/) {
        #instream_block_preroll_conversion_percent

        $depends = ["$1_$2_hits", "$1_open_player"];
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_cover_ratio');
    }

    my $ch_hits        = get_field_instream_block_metrics($depends->[0]);
    my $ch_open_player = get_field_instream_block_metrics($depends->[1]);

    return get_field(
        $id,
        depends_on => $depends,
        type       => 'percent',
        get        => sub {
            my ($stat) = @_;
            safe_div($stat->{$depends->[0]} * 100, $stat->{$depends->[1]});
        },
        category              => $CATEGORY_CALCULATED,
        clickhouse_expression => sub {
            my ($field, $product_ids) = @_;

            return [
                '/' => [
                    ['*' => [$ch_hits->{'clickhouse_expression'}->($ch_hits, $product_ids), \100]],
                    $ch_open_player->{'clickhouse_expression'}->($ch_open_player, $product_ids)
                ]
            ];
        },
    );
}

sub get_field_ctr {
    my ($id) = @_;

    my ($depends, $has_section, %ch_fields);
    if ($id =~ /^instream_block_ctr\z/) {
        $depends = [qw(instream_block_open_player instream_block_view)];

        $ch_fields{$_} = get_field_instream_block_metrics($_) foreach @$depends;
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_ctr\z/) {
        #premium_ctr

        $depends = [map {"$1_$_"} qw(clicks shows)];
        $has_section = TRUE;

        $ch_fields{$depends->[0]} = get_field_clicks($depends->[0]);
        $ch_fields{$depends->[1]} = get_field_shows($depends->[1]);
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_direct_ctr\z/) {
        #instream_block_direct_ctr

        $depends = [map {"$1_$_"} qw(direct_clicks direct_shows)];

        $ch_fields{$depends->[0]} = get_field_clicks($depends->[0]);
        $ch_fields{$depends->[1]} = get_field_shows($depends->[1]);
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_ctr');
    }

    return get_field(
        $id,
        depends_on => $depends,
        type       => 'float',
        get        => sub {
            my ($stat) = @_;

            safe_div(($stat->{$depends->[0]} // 0) * 100, $stat->{$depends->[1]});
        },
        category              => $CATEGORY_CALCULATED,
        clickhouse_expression => sub {
            my ($field, $product_ids) = @_;

            return [
                '/' => [
                    [
                        '*' => [
                            $ch_fields{$depends->[0]}->{'clickhouse_expression'}
                              ->($ch_fields{$depends->[0]}, $product_ids),
                            \100
                        ]
                    ],
                    $ch_fields{$depends->[1]}->{'clickhouse_expression'}->($ch_fields{$depends->[1]}, $product_ids),
                ]
            ];
        },
        ($has_section ? (section => 'default') : ())
    );
}

sub get_field_percent_money {
    my ($id) = @_;

    my ($depends, $money_type, $vat, $ch_bad_money, $ch_money);
    if ($id =~ /^percent_((.+?)_bad_win(?:_partner)?_price)_(\2_.+?)_(wo?_nds)\z/) {
        #percent_rtb_bad_win_price_rtb_all_w_nds
        #percent_mobile_rtb_bad_win_partner_price_mobile_rtb_partner_wo_nds

        $money_type = $3;
        $vat        = $4;

        $depends = ["$1_$vat", "$3_$vat"];

        $ch_bad_money = get_field_money($depends->[0]);
        $ch_money     = get_field_money($depends->[1]);
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_percent_money');
    }

    return get_field(
        $id,
        _get_titles($id, $money_type, $vat),
        depends_on => $depends,
        get        => sub {
            my ($stat) = @_;

            safe_div($stat->{$depends->[0]} * 100, $stat->{$depends->[1]} + $stat->{$depends->[0]});
        },
        type                  => 'percent',
        category              => $CATEGORY_CALCULATED_WITH_MONEY,
        clickhouse_expression => sub {
            my ($field, $product_ids, $stat_level) = @_;

            my $bad_money_expr =
              $ch_bad_money->{'clickhouse_expression'}
              ->($ch_bad_money, $product_ids, $stat_level, money_format => TRUE);

            return [
                '/' => [
                    ['*' => [$bad_money_expr, \100]],
                    [
                        '+' => [
                            $ch_money->{'clickhouse_expression'}
                              ->($ch_money, $product_ids, $stat_level, money_format => TRUE),
                            $bad_money_expr
                        ]
                    ]
                ]
            ];
        },
    );
}

sub get_field_cpmh {
    my ($id) = @_;

    my ($depends, $short_product, $money_type, $vat, $ch_hits, $ch_money, $has_forced);
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_cpmh_(.+?)_(wo?_nds)\z/) {
        #premium_cpmh_partner_w_nds
        #direct_search_cpmh_direct_search_all_w_nds
        $short_product = $1;
        $money_type    = $2;
        $vat           = $3;

        my %money_type_incorrect = map {$_ => TRUE} qw(
          premium
          stripe
          ), @{$RTB_PRODUCTS};

        if ($money_type_incorrect{$short_product}) {
            #instream, inpage, ...
            my $block_type = $short_product;
            $block_type =~ s/_block//;

            $depends = ["${short_product}_hits", "${block_type}_${money_type}_$vat"];
            $has_forced = in_array($short_product, $VIDEO_PRODUCTS);
        } else {
            $depends = ["${short_product}_hits", "${money_type}_$vat"];
        }

        $ch_hits  = get_field_hits($depends->[0]);
        $ch_money = get_field_money($depends->[1]);
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_cpmh');
    }

    my %has_conflict = map {$_ => TRUE} qw(
      premium
      direct_search
      direct_context
      );

    my %has_section = map {$_ => TRUE} qw(
      premium
      direct_search
      stripe
      );

    return get_field(
        $id,
        _get_titles($id, $money_type, $vat),
        depends_on => $depends,
        get        => sub {
            my ($stat) = @_;

            safe_div(($stat->{$depends->[1]} // 0) * 1000, $stat->{$depends->[0]});
        },
        type                  => 'money',
        category              => $CATEGORY_CALCULATED_WITH_MONEY,
        clickhouse_expression => sub {
            my ($field, $product_ids, $stat_level) = @_;

            return {
                round => [
                    [
                        '/' => [
                            [
                                '*' => [
                                    [
                                        '/' => [
                                            $ch_money->{'clickhouse_expression'}
                                              ->($ch_money, $product_ids, $stat_level, money_format => TRUE),
                                            \$CH_MONEY_POST_SCALE
                                        ]
                                    ],
                                    \1000
                                ]
                            ],
                            $ch_hits->{'clickhouse_expression'}->($ch_hits, $product_ids)
                        ]
                    ],
                    \2
                ]
            };
        },
        ($has_conflict{$short_product} ? (conflicts => {dimension_fields => ['public_id', 'adfox_block']}) : ()),
        ($has_section{$short_product} ? ($vat eq 'wo_nds' ? (section => 'default') : ()) : ()),
        ($has_forced ? (forced => TRUE) : ())
    );
}

sub get_field_rpm {
    my ($id) = @_;

    my ($depends, $short_product, $money_type, $vat, $ch_all_hits, $ch_money, $has_dsp_conflicts);
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_rpm_(.+?)_(wo?_nds)\z/) {
        #instream_block_rpm_all_w_nds
        $short_product = $1;
        $money_type    = $2;
        $vat           = $3;

        $has_dsp_conflicts = in_array($short_product, $VIDEO_PRODUCTS);
        #instream, inpage, ...
        my $block_type = $short_product;
        $block_type =~ s/_block//;

        $depends = ["${short_product}_all_hits", "${block_type}_${money_type}_$vat"];

        $ch_all_hits = get_field_all_metrics($depends->[0]);
        $ch_money    = get_field_money($depends->[1]);
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_rpm');
    }

    return get_field(
        $id,
        _get_titles($id, $money_type, $vat),
        depends_on => $depends,
        get        => sub {
            my ($stat) = @_;

            safe_div(($stat->{$depends->[1]} // 0) * 1000, $stat->{$depends->[0]});
        },
        type                  => 'money',
        category              => $CATEGORY_CALCULATED_WITH_MONEY,
        clickhouse_expression => sub {
            my ($field, $product_ids, $stat_level) = @_;

            return {
                round => [
                    [
                        '/' => [
                            [
                                '*' => [
                                    [
                                        '/' => [
                                            $ch_money->{'clickhouse_expression'}
                                              ->($ch_money, $product_ids, $stat_level, money_format => TRUE),
                                            \$CH_MONEY_POST_SCALE
                                        ]
                                    ],
                                    \1000
                                ]
                            ],
                            $ch_all_hits->{'clickhouse_expression'}->($ch_all_hits, $product_ids)
                        ]
                    ],
                    \2
                ]
            };
        },
        ($has_dsp_conflicts ? (conflicts => {dimension_fields => ['dsp_id_name', 'dsp_caption']}) : ()),
    );
}

sub get_field_cpm {
    my ($id) = @_;

    my ($depends, $short_product, $money_type, $vat, $ch_shows, $ch_money);
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_cpm_(.+?)_(wo?_nds)\z/) {
        #instream_block_cpm_all_w_nds
        #direct_context_cpm_direct_context_all_w_nds
        $short_product = $1;
        $money_type    = $2;
        $vat           = $3;

        my %money_type_incorrect = map {$_ => TRUE} qw(
          premium
          stripe
          dsp
          ), @{$RTB_PRODUCTS};

        if ($money_type_incorrect{$short_product}) {
            #instream, inpage, ...
            my $block_type = $short_product;
            $block_type =~ s/_block//;

            $depends = ["${short_product}_shows", "${block_type}_${money_type}_$vat"];
        } else {
            $depends = ["${short_product}_shows", "${money_type}_$vat"];
        }

        $ch_shows = get_field_shows($depends->[0]);
        $ch_money = get_field_money($depends->[1]);
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_cpm');
    }

    my %has_default = map {$_ => TRUE} qw(
      natural_block
      content_block
      rtb_block
      );

    my %has_not_section = map {$_ => TRUE} qw(
      direct_context
      );

    return get_field(
        $id,
        _get_titles($id, $money_type, $vat),
        depends_on => $depends,
        get        => sub {
            my ($stat) = @_;

            safe_div(($stat->{$depends->[1]} // 0) * 1000, $stat->{$depends->[0]});
        },
        type                  => 'money',
        category              => $CATEGORY_CALCULATED_WITH_MONEY,
        clickhouse_expression => sub {
            my ($field, $product_ids, $stat_level) = @_;

            return {
                round => [
                    [
                        '/' => [
                            [
                                '*' => [
                                    [
                                        '/' => [
                                            $ch_money->{'clickhouse_expression'}
                                              ->($ch_money, $product_ids, $stat_level, money_format => TRUE),
                                            \$CH_MONEY_POST_SCALE
                                        ]
                                    ],
                                    \1000
                                ]
                            ],
                            $ch_shows->{'clickhouse_expression'}->($ch_shows, $product_ids)
                          ]

                    ],
                    \2
                ]
            };
        },
        ($has_default{$short_product} ? (default => TRUE) : ()),
        ($has_not_section{$short_product} ? () : ($vat eq 'wo_nds' ? (section => 'default') : ())),
    );
}

sub get_field_cpc {
    my ($id) = @_;

    my ($depends, $short_product, $money_type, $vat, $ch_clicks, $ch_money);
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_cpc_(.+?)_(wo?_nds)\z/) {
        #instream_block_cpm_all_w_nds
        #direct_context_cpm_direct_context_all_w_nds
        $short_product = $1;
        $money_type    = $2;
        $vat           = $3;

        my %money_type_incorrect = map {$_ => TRUE} qw(
          premium
          stripe
          );

        if ($money_type_incorrect{$short_product}) {
            #instream, inpage, ...
            my $block_type = $short_product;
            $block_type =~ s/_block//;

            $depends = ["${short_product}_clicks", "${block_type}_${money_type}_$vat"];
        } else {
            $depends = ["${short_product}_clicks", "${money_type}_$vat"];
        }

        $ch_clicks = get_field_clicks($depends->[0]);
        $ch_money  = get_field_money($depends->[1]);
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_cpc');
    }

    my %has_not_section = map {$_ => TRUE} qw(
      market_context
      market_search
      );

    return get_field(
        $id,
        _get_titles($id, $money_type, $vat),
        depends_on => $depends,
        get        => sub {
            my ($stat) = @_;

            safe_div($stat->{$depends->[1]} // 0, $stat->{$depends->[0]});
        },
        type                  => 'money',
        category              => $CATEGORY_CALCULATED_WITH_MONEY,
        clickhouse_expression => sub {
            my ($field, $product_ids, $stat_level) = @_;

            return {
                round => [
                    [
                        '/' => [
                            [
                                '/' => [
                                    $ch_money->{'clickhouse_expression'}
                                      ->($ch_money, $product_ids, $stat_level, money_format => TRUE),
                                    \$CH_MONEY_POST_SCALE
                                ]
                            ],
                            $ch_clicks->{'clickhouse_expression'}->($ch_clicks, $product_ids)
                        ]
                    ],
                    \2
                ]
            };
        },
        ($has_not_section{$short_product} ? () : ($vat eq 'wo_nds' ? (section => 'default') : ())),
    );
}

sub get_field_average_rate {
    my ($id) = @_;

    my ($depends, $money_type, $vat);
    if ($id =~ /^average_rate_(.+?)_(wo?_nds)\z/) {
        #average_rate_all_real_price_w_nds
        $money_type = $1;
        $vat        = $2;

        $depends = ["all_hits", "${money_type}_$vat"];
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_average_rate');
    }

    my $ch_all_hits = get_field_hits($depends->[0]);
    my $ch_money    = get_field_money($depends->[1]);

    return get_field(
        $id,
        _get_titles($id, $money_type, $vat),
        depends_on => $depends,
        get        => sub {
            my ($stat) = @_;

            safe_div($stat->{$depends->[1]} // 0, $stat->{$depends->[0]});
        },
        type                  => 'money',
        category              => $CATEGORY_CALCULATED_WITH_MONEY,
        clickhouse_expression => sub {
            my ($field, $product_ids, $stat_level, %opts) = @_;

            if ($opts{'money_format'}) {
                return [
                    '/' => [
                        $ch_money->{'clickhouse_expression'}
                          ->($ch_money, $product_ids, $stat_level, money_format => TRUE),
                        $ch_all_hits->{'clickhouse_expression'}->($ch_all_hits, $product_ids)
                    ]
                ];
            } else {
                [
                    '/' => [
                        [
                            '/' => [
                                $ch_money->{'clickhouse_expression'}
                                  ->($ch_money, $product_ids, $stat_level, money_format => TRUE),
                                \$CH_MONEY_POST_SCALE
                            ]
                        ],
                        $ch_all_hits->{'clickhouse_expression'}->($ch_all_hits, $product_ids)
                    ]
                ];
            }
        },
        ($vat eq 'wo_nds' ? (section => 'default') : ()),
    );
}

sub get_field_money {
    my ($id) = @_;

    my ($money_id, $vat, $short_product, $field_name, $api_id_partner, $has_shared, $has_section);
    if ($id =~ /(.+)_(wo?_nds)/) {
        $money_id = $1;
        $vat      = $2;
    } elsif ($id =~ /^mobile_mediation_block_calculated_revenue(?:_original)?\z/) {
        $money_id = $id;
    }

    if ($money_id eq 'additional_income_partner') {
        $short_product = '';
        $field_name    = 'partner';

        $api_id_partner = TRUE;
        $has_section    = TRUE;
    } elsif ($money_id =~ /^${SHORT_PRODUCTS_REGEXP}_((?:partner)|(?:all))\z/) {
        $short_product = $1;
        $field_name    = $2;

        $api_id_partner = $field_name eq 'partner';
        $has_section = TRUE;
    } elsif ($money_id =~ /^${PRODUCTS_WITHOUT_POSTFIX_BLOCK_REGEXP}_((?:partner)|(?:all))\z/) {
        $short_product = $PRODUCTS_WITHOUT_POSTFIX_BLOCK->{$1};
        $field_name    = $2;

        $api_id_partner = $field_name eq 'partner';
        $has_section = TRUE;
    } elsif ($money_id =~ /^${PRODUCTS_WITHOUT_POSTFIX_BLOCK_REGEXP}_(bad_win(?:_partner)?_price)\z/) {
        $short_product = $PRODUCTS_WITHOUT_POSTFIX_BLOCK->{$1};
        $field_name    = $2;
    } elsif (
        in_array(
            $money_id,
            [
                qw(
                  all
                  all_real_price
                  bad_win_partner_price
                  bad_win_price
                  bk_partner_price
                  bk_price
                  charging
                  dsp_charging
                  partner
                  rs_all
                  )
            ]
        )
      )
    {
        $short_product = '';
        $field_name    = $money_id;

        $has_shared = grep {$money_id eq $_} qw(partner bad_win_partner_price all charging bk_price bad_win_price);

        $has_section = !grep {$money_id eq $_} qw(rs_all bk_partner_price bk_price bad_win_price bad_win_partner_price);
    } elsif ($money_id =~ /mobile_mediation_block_(calculated_revenue_original)/) {
        $field_name    = $1;
        $short_product = 'mobile_mediation_block';
        $has_section   = TRUE;
    } else {
        throw sprintf('Incorrect money id "%s" for sub "%s"', $money_id, 'get_field_money');
    }

    my $title = $TITLES->{$money_id} // throw "Title not found for money type '$money_id'";
    my $hint  = $HINTS->{$money_id}  // throw "Hint not found for money type '$money_id'";

    my $vat_suffix = defined $vat ? "_$vat" : "";
    return get_field(
        $id,
        title => sub {defined $vat ? sprintf("%s (%s)", $title->(), $VAT_HASH->{$vat}->()) : $title->()},
        short_title           => $title,
        get                   => 'db',
        type                  => 'money',
        hint                  => $hint,
        category              => $CATEGORY_CASH,
        clickhouse_expression => sub {
            my ($field, $product_ids, $stat_level, %opts) = @_;

            if ($opts{'money_format'}) {
                return $short_product
                  ? {
                    'sumIf' => [
                        ['/' => ["${field_name}$vat_suffix", \$CH_MONEY_PRE_SCALE]],
                        _get_product_filter($short_product, $product_ids)
                    ]
                  }
                  : {SUM => [['/' => ["${field_name}$vat_suffix", \$CH_MONEY_PRE_SCALE]]]};
            } else {
                return $short_product
                  ? {sumIf => ["${field_name}$vat_suffix", _get_product_filter($short_product, $product_ids)]}
                  : {SUM => ["${field_name}$vat_suffix"]};
            }
        },
        api_id => $api_id_partner ? "partner$vat_suffix" : "${money_id}$vat_suffix",
        ($has_shared ? (shared => TRUE) : ()),
        ($has_section && (!defined($vat) || $vat_suffix eq "" || $vat eq 'wo_nds') ? (section => 'default') : ()),
    );
}

sub get_field_money_with_currency {
    my ($id) = @_;

    my ($money_id, $vat, $short_product, $field_name, $has_section, $currency_id);
    if ($id =~ /^mobile_mediation_block_calculated_revenue\z/) {
        $money_id = $id;
    }

    if ($money_id =~ /mobile_mediation_block_(calculated_revenue)/) {
        $field_name    = $1;
        $short_product = 'mobile_mediation_block';
        $has_section   = TRUE;
        $currency_id   = 2;                          # RUB
    } else {
        throw sprintf('Incorrect money id "%s" for sub "%s"', $money_id, 'get_field_money_with_currency');
    }

    my $title = $TITLES->{$money_id} // throw "Title not found for money type '$money_id'";
    my $hint  = $HINTS->{$money_id}  // throw "Hint not found for money type '$money_id'";

    return get_field(
        $id,
        title                 => sub {$title->()},
        short_title           => $title,
        get                   => 'db',
        type                  => 'money_with_currency',
        hint                  => $hint,
        category              => $CATEGORY_CASH,
        clickhouse_expression => sub {
            my ($field, $product_ids, $stat_level, %opts) = @_;

            if ($opts{'money_format'}) {
                my $scale = $CH_MONEY_PRE_SCALE * $CH_MONEY_POST_SCALE;
                return $short_product
                  ? {
                    round => [
                        {
                            'sumIf' =>
                              [['/' => [$field_name, \$scale]], _get_product_filter($short_product, $product_ids)]
                        },
                        \2
                    ]
                  }
                  : {SUM => [['/' => [$field_name, \$scale]]]};
            } else {
                return $short_product
                  ? {sumIf => [$field_name, _get_product_filter($short_product, $product_ids)]}
                  : {SUM => [$field_name]};
            }
        },
        api_id      => $money_id,
        section     => 'default',
        currency_id => $currency_id,
    );
}

sub get_field_video_sum_money {
    my ($id) = @_;

    my ($depends, $money_type, $vat, %ch_fields);
    if ($id =~ /^video_an_site_sum_(.+?)_(wo?_nds)\z/) {
        $money_type = $1;
        $vat        = $2;

        $depends = ["instream_${money_type}_$vat", "inpage_${money_type}_$vat", "fullscreen_${money_type}_$vat"];

        $ch_fields{$_} = get_field_money($_) foreach @$depends;
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_video_sum_money');
    }

    my $title = $TITLES->{$id} // throw "Title not found for id '$id'";

    my $short_title = $SHORT_TITLES->{$id} // throw "Short title not found for id '$id'";

    return get_field(
        $id,
        title       => sub {sprintf("%s (%s)", $title->(), $VAT_HASH->{$vat}->())},
        short_title => $short_title,
        depends_on  => $depends,
        forced      => TRUE,
        get         => sub {
            my ($stat) = @_;

            my $sum = 0;
            map {$sum += $stat->{$_}} @$depends;

            return $sum;
        },
        type                  => 'money',
        shared                => TRUE,
        category              => $CATEGORY_CASH,
        clickhouse_expression => sub {
            my ($field, $product_ids, $stat_level, %opts) = @_;

            #TODO: можно улучшить если будет тормозить
            # тут сумма одного и того же поля на разных уровнях
            # {sumIf => [<FIELD>, ['product_id', 'IN', \[@PRODUCTS]]]}
            return [
                '+' => [map {$_->{'clickhouse_expression'}->($_, $product_ids, $stat_level, %opts)} values(%ch_fields)]
            ];
        },
        ($vat eq 'wo_nds' ? (section => 'default') : ()),
    );
}

sub get_field_an_cover_hits {
    my ($id) = @_;

    my ($short_product, $conflicts, $clickhouse_expression);
    if ($id eq 'an_cover_hits') {
        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => [
                    'an_cover_hits',
                    _get_product_filter(
                        $short_product,
                        $product_ids,
                        products => [
                            qw(
                              ssp_context_on_site_campaign
                              ssp_video_an_site
                              context_on_site_campaign
                              internal_context_on_site_campaign
                              )
                        ]
                    )
                ]
            };
        };
    } elsif ($id eq 'an_cover_senthits') {
        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => [
                    'an_cover_senthits',
                    _get_product_filter(
                        $short_product,
                        $product_ids,
                        products => [
                            qw(
                              ssp_context_on_site_campaign
                              ssp_video_an_site
                              context_on_site_campaign
                              internal_context_on_site_campaign
                              )
                        ]
                    )
                ]
            };
        };
    } elsif ($id =~ /^an_(ssp_mobile_rtb|mobile_rtb|rtb|content|natural)_cover_((?:direct_)?hits)\z/) {
        #an_mobile_rtb_cover_hits

        $short_product = $1;
        my $field_name = $2;

        my $products = $PAGE_ACCESSORS_BY_SHORT_PRODUCT->{$short_product};

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;
            return {
                sumIf => [
                    "an_rtb_cover_$field_name",
                    _get_product_filter($short_product, $product_ids, products => $products,)
                ]
            };
        };
    } elsif ($id =~ /^an_(ssp_mobile_rtb|mobile_rtb|rtb|content|natural)_cover_senthits\z/) {
        #an_rtb_cover_senthits

        $short_product = $1;

        my $products = $PAGE_ACCESSORS_BY_SHORT_PRODUCT->{$short_product};

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;
            return {sumIf =>
                  ['an_rtb_cover_senthits', _get_product_filter($short_product, $product_ids, products => $products,)]
            };
        };
    } elsif ($id =~ /^an_cover_((?:direct)|(?:market)|(?:mcb))_hits\z/) {
        #an_cover_direct_hits

        $short_product = $1;

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => [
                    $id,
                    _get_product_filter(
                        $short_product,
                        $product_ids,
                        products => [
                            qw(
                              ssp_context_on_site_campaign
                              ssp_video_an_site
                              context_on_site_campaign
                              internal_context_on_site_campaign
                              )
                        ]
                    )
                ]
            };
        };
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_an_cover_hits');
    }

    my @conflict_fields = qw(public_id adfox_block);

    if ($short_product && $short_product eq 'mobile_rtb') {
        push(@conflict_fields, 'mobile_block_type', 'mobile_block_type_label');
    }

    $conflicts = {dimension_fields => \@conflict_fields};

    return get_field(
        $id,
        get      => 'db',
        type     => 'number',
        category => $CATEGORY_COUNTER,
        ($conflicts             ? (conflicts             => $conflicts)             : ()),
        ($clickhouse_expression ? (clickhouse_expression => $clickhouse_expression) : ())
    );
}

sub get_field_hits {
    my ($id) = @_;

    my ($short_product, $has_section, $clickhouse_expression, $has_default, $has_shared);
    my $conflicts = {dimension_fields => []};

    if ($id eq 'video_bk_all_hits') {
        #TODO: удалить эту ветку вместе с кодом про паблишеров (publisher)
    } elsif ($id eq 'all_hits') {
        $short_product = 'dsp';

        $has_section = TRUE;

        #TODO: подумать норм ли складывать это поле в hits
        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf => ['hits', _get_product_filter($short_product, $product_ids)]};
        };
    } elsif ($id =~ /^video_bk_hits(?:(?:_own_adv)|(?:_unsold))?\z/) {
        #TODO: удалить эту ветку вместе с кодом про паблишеров (publisher)
    } elsif ($id =~ /^mobile_mediation_block_hits\z/) {
        $short_product = 'mobile_mediation_block';

        $has_section = TRUE;

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf => ['hits', _get_product_filter($short_product, $product_ids)]};
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_hits\z/) {
        $short_product = $1;

        if (in_array($short_product, $RTB_PRODUCTS)) {
            $has_section = TRUE;

            $has_default = $short_product eq 'rtb_block';

            $clickhouse_expression = sub {
                my ($field, $product_ids) = @_;

                return {
                    sumIf => [
                        'hits',
                        ['AND', [['dsp_id', 'NOT IN', \[5, 10]], _get_product_filter($short_product, $product_ids)]]
                    ]
                };
            };
        } elsif ($short_product eq 'stripe') {
            $clickhouse_expression = sub {
                my ($field, $product_ids) = @_;

                return {sumIf => ['hits', _get_product_filter($short_product, $product_ids)]};
            };
        } else {
            # direct blocks

            push @{$conflicts->{dimension_fields}}, qw(public_id adfox_block);

            my $field_name = $id;
            $field_name =~ s/(?:_context)|(?:_search)//;

            $clickhouse_expression = sub {
                my ($field, $product_ids) = @_;

                return {
                    sumIf => [$field_name, _get_product_filter($short_product, $product_ids, page_accessors => TRUE)]
                };
            };
        }
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_win_hits\z/) {
        $short_product = $1;

        if (in_array($short_product, $VIDEO_PRODUCTS)) {
            $has_shared = TRUE;

            $clickhouse_expression = sub {
                my ($field, $product_ids) = @_;

                return {
                    sumIf => [
                        'win_max_positions_count',
                        ['AND', [['dsp_id', 'NOT IN', \[5, 10]], _get_product_filter($short_product, $product_ids)]]
                    ]
                };
            };
        }
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_hits_own_adv\z/) {
        #instream_block_hits_own_adv

        $short_product = $1;

        #rtb blocks

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf => ['hits', ['AND', [['dsp_id', '=', \5], _get_product_filter($short_product, $product_ids)]]]
            };
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_win_hits_own_adv\z/) {
        #instream_block_win_hits_own_adv

        $short_product = $1;
        $has_shared    = TRUE;

        #rtb blocks
        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => [
                    'win_max_positions_count',
                    ['AND', [['dsp_id', '=', \5], _get_product_filter($short_product, $product_ids)]]
                ]
            };
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_hits_unsold\z/) {
        $short_product = $1;
        if (in_array($short_product, $VIDEO_PRODUCTS)) {

            my ($depends, $has_forced, $ch_all_hits, $ch_hits, $ch_hits_own_adv, %ch_fields);

            $depends         = [map {"${short_product}_$_"} qw(all_hits hits hits_own_adv)];
            $ch_all_hits     = get_field_all_metrics($depends->[0]);
            $ch_hits         = get_field_hits($depends->[1]);
            $ch_hits_own_adv = get_field_hits($depends->[2]);

            return get_field(
                $id,
                depends_on => $depends,
                type       => 'number',
                get        => sub {
                    my ($stat) = @_;

                    $stat->{$depends->[0]} - $stat->{$depends->[1]} - $stat->{$depends->[2]};
                },
                category              => $CATEGORY_COUNTER,
                clickhouse_expression => sub {
                    my ($field, $product_ids) = @_;

                    return [
                        '-',
                        [
                            $ch_all_hits->{'clickhouse_expression'}->($ch_all_hits, $product_ids),
                            [
                                '+',
                                [
                                    $ch_hits->{'clickhouse_expression'}->($ch_hits, $product_ids),
                                    $ch_hits_own_adv->{'clickhouse_expression'}->($ch_hits_own_adv, $product_ids),
                                ]
                            ],
                        ]
                    ];
                },
                conflicts => {dimension_fields => [qw(dsp_id_name dsp_caption)]},
                forced    => TRUE
            );
        } else {

            #rtb blocks
            $clickhouse_expression = sub {
                my ($field, $product_ids) = @_;

                return {sumIf =>
                      ['hits', ['AND', [['dsp_id', '=', \10], _get_product_filter($short_product, $product_ids)]]]
                };
            };
        }
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_win_hits_unsold\z/) {
        $short_product = $1;
        $has_shared    = TRUE;

        #rtb blocks
        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => [
                    'win_max_positions_count',
                    ['AND', [['dsp_id', '=', \10], _get_product_filter($short_product, $product_ids)]]
                ]
            };
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_bad_win_hits\z/) {
        $short_product = $1;

        $has_section = $short_product eq 'ssp_mobile_rtb_block';

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => [
                    'bad_hits',
                    ['AND', [['dsp_id', 'NOT IN', \[5, 10]], _get_product_filter($short_product, $product_ids)]]
                ]
            };
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_bad_win_hits_unsold\z/) {
        $short_product = $1;

        $has_section = $short_product eq 'ssp_mobile_rtb_block';
        push @{$conflicts->{dimension_fields}}, qw(dsp_id_name dsp_caption)
          if (in_array($short_product, $VIDEO_PRODUCTS));

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf =>
                  ['bad_hits', ['AND', [['dsp_id', '=', \10], _get_product_filter($short_product, $product_ids)]]]
            };
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_bad_win_hits_own_adv\z/) {
        $short_product = $1;

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => ['bad_hits', ['AND', [['dsp_id', '=', \5], _get_product_filter($short_product, $product_ids)]]]
            };
        };
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_hits');
    }

    return get_field(
        $id,
        get      => 'db',
        type     => 'number',
        category => $CATEGORY_COUNTER,
        (@{$conflicts->{dimension_fields}} ? (conflicts             => $conflicts)             : ()),
        ($clickhouse_expression            ? (clickhouse_expression => $clickhouse_expression) : ()),
        ($has_section                      ? (section               => 'default')              : ()),
        ($has_default                      ? (default               => TRUE)                   : ()),
        ($has_shared                       ? (shared                => TRUE)                   : ()),
    );
}

sub get_field_shows {
    my ($id) = @_;

    my ($short_product, $has_conflicts, $has_section, $clickhouse_expression, $has_default);
    if ($id =~ /^video_bk_shows(?:_own_adv)?\z/) {
        # video_bk_shows, video_bk_shows_own_adv
        #TODO: удалить эту ветку вместе с кодом про паблишеров (publisher)
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_impressions\z/) {
        #mobile_mediation_block_impressions

        $short_product = $1;

        $has_section = TRUE;

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf => ['impressions', _get_product_filter($short_product, $product_ids)]};
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_shows\z/) {
        #instream_block_shows

        $short_product = $1;

        $has_section = TRUE;

        if (in_array($short_product, $RTB_PRODUCTS)) {
            $has_default = $short_product eq 'rtb_block';

            $clickhouse_expression = sub {
                my ($field, $product_ids) = @_;

                return {sumIf =>
                      ['shows', ['AND', [['dsp_id', '<>', \5], _get_product_filter($short_product, $product_ids)]]]
                };
            };
        } else {
            #direct blocks and dsp_shows
            $clickhouse_expression = sub {
                my ($field, $product_ids) = @_;

                return {sumIf => ['shows', _get_product_filter($short_product, $product_ids)]};
            };
        }
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_shows_own_adv\z/) {
        $short_product = $1;

        #rtb blocks
        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => ['shows', ['AND', [['dsp_id', '=', \5], _get_product_filter($short_product, $product_ids)]]]
            };
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_direct_shows\z/) {
        $short_product = $1;

        #rtb blocks

        $has_section = TRUE;

        $has_default = $short_product eq 'rtb_block';

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf => ['direct_shows', _get_product_filter($short_product, $product_ids)]};
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_bad_shows\z/) {
        $short_product = $1;

        #rtb blocks

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf =>
                  ['bad_shows', ['AND', [['dsp_id', '<>', \5], _get_product_filter($short_product, $product_ids)]]]
            };
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_bad_shows_own_adv\z/) {
        $short_product = $1;

        #rtb blocks

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf =>
                  ['bad_shows', ['AND', [['dsp_id', '=', \5], _get_product_filter($short_product, $product_ids)]]]
            };
        };
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_hits');
    }

    return get_field(
        $id,
        get      => 'db',
        type     => 'number',
        category => $CATEGORY_COUNTER,
        ($has_conflicts ? (conflicts => {dimension_fields => [qw(public_id adfox_block)]}) : ()),
        ($clickhouse_expression ? (clickhouse_expression => $clickhouse_expression) : ()),
        ($has_section           ? (section               => 'default')              : ()),
        ($has_default           ? (default               => TRUE)                   : ()),
    );
}

sub get_field_instream_block_metrics {
    my ($id) = @_;

    my ($short_product, $type, $depends, $conflicts, $clickhouse_expression);
    if ($id eq 'instream_block_view') {
        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => ['view', _get_product_filter($short_product, $product_ids, products => [qw(video_an_site)])]
            };
        };
    } elsif ($id eq 'instream_block_open_player') {
        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf =>
                  ['open_player', _get_product_filter($short_product, $product_ids, products => [qw(video_an_site)])]
            };
        };
    } elsif ($id =~ /^(instream_block)_((?:preroll)|(?:postroll)|(?:midroll)|(?:pauseroll))_shows\z/) {
        #instream_block_preroll_shows
        $short_product = $1;
        $type          = $2;

        $depends = ['instream_block_shows'];

        my $type_id = (grep {$VIDEO_BLOCK_TYPES->{$_}{'bk'} eq $type} keys(%$VIDEO_BLOCK_TYPES))[0];

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => [
                    'shows',
                    [
                        'AND',
                        [
                            ['dsp_id', 'NOT IN', \[5, 10]],
                            _get_product_filter($short_product, $product_ids),
                            [
                                {
                                    dictGetUInt8 => [
                                        \'blocks_dict',
                                        \'video_block_type',
                                        {
                                            tuple => [
                                                {toString => ['product_id']},
                                                {toUInt32 => ['page_id']},
                                                {toUInt32 => ['block_id']}
                                            ]
                                        }
                                    ]
                                },
                                '=',
                                \$type_id
                            ]
                        ]
                    ]
                ]
            };
        };
    } elsif ($id =~ /^(instream_block)_((?:preroll)|(?:postroll)|(?:midroll)|(?:pauseroll))_hits\z/) {
        #instream_block_preroll_hits
        $short_product = $1;
        $type          = $2;

        $depends = ['instream_block_hits'];

        my $type_id = (grep {$VIDEO_BLOCK_TYPES->{$_}{'bk'} eq $type} keys(%$VIDEO_BLOCK_TYPES))[0];

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => [
                    'hits',
                    [
                        'AND',
                        [
                            ['dsp_id', 'NOT IN', \[5, 10]],
                            _get_product_filter($short_product, $product_ids),
                            [
                                {
                                    dictGetUInt8 => [
                                        \'blocks_dict',
                                        \'video_block_type',
                                        {
                                            tuple => [
                                                {toString => ['product_id']},
                                                {toUInt32 => ['page_id']},
                                                {toUInt32 => ['block_id']}
                                            ]
                                        }
                                    ]
                                },
                                '=',
                                \$type_id
                            ]
                        ]
                    ]
                ]
            };
        };
    } elsif ($id =~ /^(instream_block)_((?:preroll)|(?:postroll)|(?:midroll)|(?:pauseroll))_all_hits\z/) {
        #instream_block_preroll_all_hits
        $short_product = $1;
        $type          = $2;

        $depends = ['instream_block_hits', 'instream_block_hits_own_adv', 'instream_block_hits_unsold'];

        my $type_id = (grep {$VIDEO_BLOCK_TYPES->{$_}{'bk'} eq $type} keys(%$VIDEO_BLOCK_TYPES))[0];

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {
                sumIf => [
                    'hits',
                    [
                        'AND',
                        [
                            _get_product_filter($short_product, $product_ids),
                            [
                                {
                                    dictGetUInt8 => [
                                        \'blocks_dict',
                                        \'video_block_type',
                                        {
                                            tuple => [
                                                {toString => ['product_id']},
                                                {toUInt32 => ['page_id']},
                                                {toUInt32 => ['block_id']}
                                            ]
                                        }
                                    ]
                                },
                                '=',
                                \$type_id
                            ]
                        ]
                    ]
                ]
            };
        };
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_instream_block_metrics');
    }

    my @conflict_fields = qw(public_id adfox_block);

    $conflicts = {dimension_fields => \@conflict_fields};

    return get_field(
        $id,
        get      => 'db',
        type     => 'number',
        category => $CATEGORY_COUNTER,
        ($depends               ? (depends_on            => $depends)               : ()),
        ($clickhouse_expression ? (clickhouse_expression => $clickhouse_expression) : ()),
        ($conflicts             ? (conflicts             => $conflicts)             : ()),
    );
}

sub get_field_direct_ads {
    my ($id) = @_;

    my ($short_product, $depends, $conflicts, $ch_shows, $ch_page_ad_shows);
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_ads\z/) {
        #premium_ads

        $short_product = $1;

        $depends = ["${short_product}_shows", "${short_product}_page_ad_shows"];

        $ch_shows         = get_field_shows($depends->[0]);
        $ch_page_ad_shows = get_field_page_ad_shows($depends->[1]);
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_direct_ads');
    }

    my @conflict_fields = qw(public_id adfox_block);

    $conflicts = {dimension_fields => \@conflict_fields};

    return get_field(
        $id,
        get        => 'db',
        depends_on => $depends,
        type       => 'float',
        get        => sub {
            my ($stat) = @_;

            safe_div($stat->{$depends->[0]}, $stat->{$depends->[1]});
        },
        category              => $CATEGORY_CALCULATED,
        clickhouse_expression => sub {
            my ($field, $product_ids) = @_;

            return [
                '/' => [
                    $ch_shows->{'clickhouse_expression'}->($ch_shows, $product_ids),
                    $ch_page_ad_shows->{'clickhouse_expression'}->($ch_page_ad_shows, $product_ids),
                ]
            ];
        },
        ($conflicts ? (conflicts => $conflicts) : ()),
    );
}

sub get_field_direct_ad_metrics {
    my ($id) = @_;

    my ($depends, $conflicts, $has_forced, %ch_fields);
    if ($id eq 'direct_ad_visibility') {
        $depends = [qw(an_cover_direct_hits an_cover_senthits)];

        $has_forced = TRUE;
    } elsif ($id eq 'direct_ad_reach') {
        $depends = [qw(an_cover_senthits an_cover_hits)];
    } elsif ($id eq 'direct_ad_coverage') {
        $depends = [qw(an_cover_direct_hits an_cover_hits)];
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_direct_ad_metrics');
    }

    $ch_fields{$_} = get_field_an_cover_hits($_) foreach @$depends;

    my @conflict_fields = qw(public_id adfox_block);

    $conflicts = {dimension_fields => \@conflict_fields};

    return get_field(
        $id,
        get        => 'db',
        depends_on => $depends,
        type       => 'percent',
        get        => sub {
            my ($stat) = @_;

            safe_div($stat->{$depends->[0]} * 100, $stat->{$depends->[1]});
        },
        category              => $CATEGORY_CALCULATED,
        clickhouse_expression => sub {
            my ($field, $product_ids) = @_;

            return [
                '/' => [
                    [
                        '*' => [
                            $ch_fields{$depends->[0]}->{'clickhouse_expression'}
                              ->($ch_fields{$depends->[0]}, $product_ids),
                            \100
                        ]
                    ],
                    $ch_fields{$depends->[1]}->{'clickhouse_expression'}->($ch_fields{$depends->[1]}, $product_ids)
                ]
            ];
        },
        ($has_forced ? (forced    => TRUE)       : ()),
        ($conflicts  ? (conflicts => $conflicts) : ()),
    );
}

sub get_field_clicks {
    my ($id) = @_;

    my ($short_product, $clickhouse_expression, $has_default);
    if ($id =~ /^${SHORT_PRODUCTS_REGEXP}_clicks\z/) {
        #premium_clicks

        $short_product = $1;

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf => ['clicks', _get_product_filter($short_product, $product_ids)]};
        };
    } elsif ($id =~ /^${SHORT_PRODUCTS_REGEXP}_direct_clicks\z/) {
        #instream_block_direct_clicks

        $short_product = $1;

        $has_default = $short_product eq 'rtb_block';

        $clickhouse_expression = sub {
            my ($field, $product_ids) = @_;

            return {sumIf =>
                  ['direct_clicks', ['AND', [['dsp_id', '=', \1], _get_product_filter($short_product, $product_ids)]]]
            };
        };
    } else {
        throw sprintf('Incorrect id "%s" for sub "%s"', $id, 'get_field_clicks');
    }

    return get_field(
        $id,
        get      => 'db',
        type     => 'number',
        category => $CATEGORY_COUNTER,
        section  => 'default',
        ($has_default ? (default => TRUE) : ()),
        ($clickhouse_expression ? (clickhouse_expression => $clickhouse_expression) : ())
    );
}

TRUE;
