package Application::Model::Statistics::Product;

use Coro;
use Coro::Specific;

use qbit;
use Utils::Logger;

use Math::Round;

use base qw(Application::Model::Statistics::Update);

use QBit::Application::Model::DBManager::_Utils::Fields;
use Application::Model::Statistics::_Utils::LevelOpts;

use Application::Model::Statistics::Fields::Types qw(@FIELD_TYPES %FIELD_TYPES);
use Application::Model::Statistics::Fields::DimensionTypes qw(@DIMENSION_TYPES %DIMENSION_TYPES);

use Application::Model::Statistics::Fields::ClickhouseExpressions;

use Utils::DB qw(fields_to_filter);

use PiConstants qw(
  $STAT_MONEY_SCALE
  $YNDX_PARTNER_INTAPI_USER_ID
  );

__PACKAGE__->abstract_methods(qw(title fields dimension_fields get_event_fields query));

__PACKAGE__->model_accessors(
    clickhouse_db   => 'Application::Model::ClickhouseDB',
    memcached       => 'QBit::Application::Model::Memcached',
    partner_db      => 'Application::Model::PartnerDB',
    product_manager => 'Application::Model::ProductManager',
    statistics      => 'Application::Model::Statistics',
    users           => 'Application::Model::Users',
);

sub parent_accessor {undef}

sub clickhouse_table_name {'statistics'}

sub get_clickhouse_expressions {return $CLIKHOUSE_EXPRESSIONS}

sub add_authorization_filter {
    my ($self, $filter, $group_fields, %opts) = @_;

    my $product_ids = $self->get_product_ids(%opts);

    if (defined($product_ids)) {
        if (grep {$group_fields->{$_}} $self->statistics->get_fields_for_block_levels_only()) {
            # если есть группировка по этим поля - выкидываем уровни пейджей
            my %page_accessors = map {$_ => TRUE} @{$self->product_manager->get_page_model_accessors};

            $product_ids = [grep {!$page_accessors{$_}} @$product_ids];

            #TODO: кидать эксепшен если запросили поля с уровня пейджа
        }
    }

    my (%product_ids_by_rights, %product_ids_by_page_ids, %page_ids);
    foreach my $product_id (@$product_ids) {
        my $level = $self->statistics->get_level_by_product_accessor($product_id);

        if ($level->is_available_by_right()) {
            $product_ids_by_rights{$product_id} = TRUE;
        } else {
            my $tmp_rights = $self->app->add_tmp_rights($product_id . '_view_zero_block');

            my $model = $self->app->$product_id;
            my @product_page_ids =
              map {$_->{'page_id'}} @{
                $model->get_all(
                    fields => [qw(page_id)],
                    (
                        $model->isa('Application::Model::Page')
                          && $model->get_multistate_by_name('balance_registered')
                        ? (filter => {multistate => $model->get_multistates_by_filter('balance_registered')})
                        : ()
                    ),
                    distinct => TRUE
                )
              };

            if (@product_page_ids) {
                $product_ids_by_page_ids{$product_id} = TRUE;

                $page_ids{$_} = TRUE foreach @product_page_ids;
            }
        }
    }

    unless (%product_ids_by_rights || %product_ids_by_page_ids) {
        $filter->and(['AND' => [\undef]]);

        return [];
    }

    my $auth_filter = $self->clickhouse_db->filter();
    if (%product_ids_by_rights) {
        $auth_filter->or({product_id => [grep {$product_ids_by_rights{$_}} @$product_ids]});
    }

    if (%product_ids_by_page_ids) {
        $auth_filter->or(
            [
                'AND',
                [
                    ['product_id' => 'IN' => \[grep {$product_ids_by_page_ids{$_}} @$product_ids]],
                    ['page_id'    => 'IN' => \[sort keys(%page_ids)]]
                ]
            ]
        );
    }

    $filter->and($auth_filter);

    return [grep {$product_ids_by_rights{$_} || $product_ids_by_page_ids{$_}} @$product_ids];
}

sub is_available_by_right {$_[0]->check_short_rights('always_view')}

sub data_query {
    my ($self, %opts) = @_;

    my $entity_fields = $opts{'entity_fields'} // [];
    my $entity_filter = $opts{'entity_filter'};

    my %stat_fields =
      $self->_get_query_stat_fields(hash_transform(\%opts, [qw(stat_db_fields stat_db_group_fields entity_fields)]));

    return () unless %stat_fields;

    my $accessor = $opts{'entity_filter_accessor'} // $self->product->{'accessor'};

    my $level_opts = Application::Model::Statistics::_Utils::LevelOpts->new(
        fields       => \%stat_fields,
        filter       => $opts{'filter'},
        fields_rules => $self->get_fields_rules(),
        filter_rules => $self->get_filter_rules(),
    );

    return ()
      if $self->can('check_that_level_supported_request') && !$self->check_that_level_supported_request($level_opts);

    my $query = $self->query(
        fields => $level_opts->stat_fields,
        filter => $level_opts->stat_filter,
    );

    return () unless $query;

    if ($level_opts->need_join_model()) {
        $self->_join_required_tables(
            query        => $query,
            accessor     => $self->product->accessor,
            fields       => $level_opts->model_fields,
            filter       => $level_opts->model_filter,
            is_low_level => TRUE,
        );
    }

    if ($opts{'limit'}) {
        $query = $query->limit($opts{'offset'}, $opts{'limit'});
    }

    if ($opts{'order_by'}) {
        $query = $query->order_by($opts{'order_by'});
    }

    $query->distinct() if $opts{'distinct'};

    my @group_by = $self->_get_group_by_fields($opts{'group_by'});

    if (@group_by) {
        my %group_by = map {$_ => TRUE} @group_by;
        my %query_fields = map {%{$_->{'fields'}}} @{$query->{'__TABLES__'}};
        my @not_grouping_fields = grep {!$query_fields{$_} && !$group_by{$_}} keys(%query_fields);

        $query->group_by(sort @group_by, @not_grouping_fields);
    }

    if (
           defined($query)
        && $self->can('join_condition_data')
        && (@$entity_fields
            || defined($entity_filter))
       )
    {
        $self->_join_required_tables(
            query            => $query,
            accessor         => $accessor,
            is_entity_filter => TRUE,
            fields           => $entity_fields,
            filter           => $entity_filter,
        );
    }

    return ($query);
}

sub get_fields_rules {{}}

sub get_filter_rules {{}}

sub entity_filter_fields {
    my ($self) = @_;

    my $db_filter_fields;
    if ($self->can('product')) {
        $db_filter_fields = $self->product->get_db_filter_fields();
    } else {
        $db_filter_fields = $self->users->get_db_filter_fields();
    }

    return $db_filter_fields;
}

sub entity_filter_simple_fields {
    my ($self) = @_;

    my $db_filter_simple_fields;
    my %ignore;
    if ($self->can('product')) {
        $db_filter_simple_fields = $self->product->get_db_filter_simple_fields();
        %ignore = map {$_ => TRUE} qw(moderation_reason);
    } else {
        $db_filter_simple_fields = $self->users->get_db_filter_simple_fields();
        %ignore = map {$_ => TRUE} qw(moderation_reason multistate role_id);
    }
    if (ref($db_filter_simple_fields->[0]) eq 'ARRAY') {
        foreach my $list (@$db_filter_simple_fields) {
            $list = [grep {!$ignore{$_->{name}}} @$list];
        }
    } else {
        $db_filter_simple_fields = [grep {!$ignore{$_->{name}}} @$db_filter_simple_fields];
    }

    return $db_filter_simple_fields;
}

sub get_conflict_fields {
    my ($self) = @_;

    return $self->can('conflict_fields') ? $self->conflict_fields() : [];
}

sub get_dimension_fields {
    my ($self, %opts) = @_;

    my $fields_hs = $self->dimension_fields();

    my %avail_fields;

    foreach my $field (keys %$fields_hs) {
        my $fopts = $fields_hs->{$field};
        next if exists($fopts->{'check_rights'}) && !$self->app->check_rights($fopts->{'check_rights'});
        $avail_fields{$field} = 1;
    }

    my @fields = grep {exists($avail_fields{$_->{'id'}})} @DIMENSION_TYPES;

    return \@fields if $opts{'raw'};

    return [
        map {
            my $ft = $_;
            my %field = map {$_ => ref($ft->{$_}) eq 'CODE' ? $ft->{$_}() : $ft->{$_}}
              grep {exists($ft->{$_})}
              qw(
              id
              title
              values
              filter_values
              verbatim
              wo_group_by_date
              type
              ajax
              sort_as
              only_group
              );

            if ($ft->{'get'} eq 'db_select') {
                $field{'type'} = 'select';
                $field{'values'} = [map {[$_->[0], $_->[1]()]} @{$ft->{'values'}}];
            }

            if ($ft->{'get'} eq 'dictionary' && !$ft->{'ajax'} && !$ft->{'only_group'}) {
                $field{'filter_values'} = $self->statistics->get_dimension_field_values($ft);
            }

            \%field;
          } @fields
    ];
}

sub get_entity_fields {
    my ($self, %opts) = @_;

    my $entity_fields = $self->can('entity_fields') ? $self->entity_fields() : [];

    foreach (@$entity_fields) {
        $_->{'only_group'} = TRUE;
    }

    return $entity_fields;
}

sub get_entity_fields_obj {
    my ($self, $fields) = @_;

    # We do not have a product (entity table) for top levels.
    # In this case we supply all field structs manually.
    #
    return $self->can('product')
      ? $self->product->_get_fields_obj($fields)
      : QBit::Application::Model::DBManager::_Utils::Fields->new({map {$_ => {db => TRUE}} @$fields},
        {map {$_ => 0} @$fields}, $fields);
}

sub get_fields {
    my ($self, %opts) = @_;

    local $Application::Model::Statistics::Fields::Titles::RENAME_PARTNER_SHARE =
      !$self->check_rights('statistics_view__partner_share_text');

    my $fields_hs = $self->fields();

    my %avail_fields = map {$_->{'id'} => TRUE} @{$opts{'child_fields'}};

    foreach my $field (keys %$fields_hs) {
        my $fopts = $fields_hs->{$field};
        next if exists($fopts->{'check_rights'}) && !$self->app->check_rights($fopts->{'check_rights'});
        $avail_fields{$field} = TRUE;
    }

    if ($opts{'forced'}) {
        # Добавляем НЕдоступные поля, если от них зависят доступные
        foreach my $field (keys(%avail_fields)) {
            if (   exists($FIELD_TYPES{$field})
                && $FIELD_TYPES{$field}->{'forced'}
                && exists($FIELD_TYPES{$field}->{'depends_on'}))
            {
                $avail_fields{$_} = TRUE foreach @{$FIELD_TYPES{$field}->{'depends_on'}};
            }
        }
    }

    # Удаляем поля, которые зависят от недоступных
    my $cnt = 0;
    while ($cnt != keys(%avail_fields)) {
        $cnt = keys(%avail_fields);
        foreach my $field (keys(%avail_fields)) {
            delete($avail_fields{$field})
              if exists($FIELD_TYPES{$field}->{'depends_on'})
                  && !$FIELD_TYPES{$field}->{'forced'}
                  && grep {!exists($avail_fields{$_})} @{$FIELD_TYPES{$field}->{'depends_on'}};
        }
    }

    my @fields = grep {exists($avail_fields{$_->{'id'}})} @FIELD_TYPES;
    # PI-9107 удаляем поля с _w_nds
    if ($opts{'no_nds'}) {
        @fields = grep {$_->{'id'} !~ /_w_nds$/} @fields;
    }
    @fields = $self->remove_shared_fields(@fields) unless $opts{'with_shared'};
    unless ($opts{'clickhouse_expression'}) {
        @fields = @{clone(\@fields)};
        foreach (@fields) {
            delete($_->{'clickhouse_expression'});
        }
    }

    return \@fields if $opts{'raw'};

    return [
        map {
            my $ft = $_;
            scalar {
                map {$_ => ref($ft->{$_}) eq 'CODE' ? $ft->{$_}() : $ft->{$_}}
                  grep {exists($ft->{$_})} (
                    exists($opts{'fields'})
                    ? @{$opts{'fields'}}
                    : (qw(id title short_title type hint shared category conflicts section))
                  )
              }
          } @fields
    ];
}

sub remove_shared_fields {
    my ($self, @fields) = @_;

    return grep {!$_->{'shared'}} @fields;

    return @fields;
}

sub _join_required_tables {
    my ($self, %opts) = @_;

    my $query            = $opts{'query'};
    my $accessor         = $opts{'accessor'};
    my $is_entity_filter = $opts{'is_entity_filter'};
    my $fields           = $opts{'fields'};
    my $filter           = $opts{'filter'};
    my $is_low_level     = $opts{'is_low_level'};

    my $model = $self->$accessor;

    my $table =
        $model->can('partner_db_table')
      ? $model->partner_db_table
      : $model->_db_table;

    my $cond = $self->join_condition($accessor);

    if (ref($cond) eq 'HASH') {
        my $depends_on = $cond->{'depends_on'};

        if ($depends_on) {
            foreach my $accessor (@$depends_on) {
                $self->_join_required_tables(query => $query, accessor => $accessor);
            }
        }
        $cond = $cond->{'condition'};    # becomes legacy array ref
    }

    $query->join(
        fields => $is_low_level
        ? $fields // {}
        : (
              $fields && @$fields
            ? $model->_get_fields_obj($fields)->get_db_fields()
            : []
          ),
        table => $table,
        (
            $is_entity_filter
            ? (alias => 'entity_filter_table')
            : ()
        ),
        (
            defined($filter)
            ? (
                filter => $is_low_level
                ? $filter
                : $model->get_db_filter($filter)
              )
            : ()
        ),
        join_on => $cond
    );
}

my $coro_specific = Coro::Specific->new();

sub check_statistics_by_blocks { }

sub fix_statistics {
    my ($self, $stat, $cuted_stat, %opts) = @_;

    throw gettext('Excepted "fields"') unless $opts{'fields'} && @{$opts{'fields'}};

    my $SUM = {};

    foreach my $row (@$stat) {
        foreach (@{$opts{'fields'}}) {
            $SUM->{'stat'}{$_} += $row->{$_};
        }
    }

    foreach my $row (@$cuted_stat) {
        foreach (@{$opts{'fields'}}) {
            $SUM->{'cuted_stat'}{$_} += $row->{$_};
        }
    }

    my $params = {};

    foreach (@{$opts{'fields'}}) {
        $params->{'sum_' . $_}        = 0;
        $params->{'correction_' . $_} = 0;
        $params->{'coeff_' . $_} = $SUM->{'cuted_stat'}{$_} == 0 ? 0 : $SUM->{'stat'}{$_} / $SUM->{'cuted_stat'}{$_};
    }

    foreach my $row (@$cuted_stat) {
        foreach (@{$opts{'fields'}}) {
            if ($row->{$_}) {
                my $old_sum   = $params->{"sum_$_"};
                my $corrected = $row->{$_} * $params->{"coeff_$_"} - $params->{'correction_' . $_};
                $params->{"sum_$_"} += $corrected;
                $params->{'correction_' . $_} = ($params->{'sum_' . $_} - $old_sum) - $corrected;
                $row->{$_} = round($params->{"sum_$_"}) - round($old_sum);
            }
        }
    }

    if ($opts{'check_sub'}) {
        my $count = 1;
        my $delta = $opts{'check_sub'}->($cuted_stat->[0]);

        while (grep {$delta->{$_}} @{$opts{'fields'}} || $count == 1) {
            my $i = 0;
            while ($i < @$cuted_stat) {
                my $j = $i + 1;
                $j = 0 if $j == @$cuted_stat;

                foreach (@{$opts{'fields'}}) {
                    $cuted_stat->[$i]{$_} = $cuted_stat->[$i]{$_} - $delta->{$_};
                    $cuted_stat->[$j]{$_} = $cuted_stat->[$j]{$_} + $delta->{$_};
                }

                $delta = $opts{'check_sub'}->($cuted_stat->[$j]);

                last unless grep {$delta->{$_}} @{$opts{'fields'}} || $count == 1;

                $i++;
            }

            throw gettext('Bad data') if ++$count == 10;
        }
    }

    return $cuted_stat;
}

sub get_additional_text {
    my ($money_type) = @_;

    my $text;

    if (defined($money_type->{'additional_text'})) {
        return $money_type->{'additional_text'}();
    } else {
        return lc($money_type->{'title'}());
    }
}

sub get_data {
    my ($self, %opts) = @_;

    my $use_coro = $self->get_option('use_coro');
    my @threads  = ();
    my $results  = [];
    my $count    = 0;

    local $QBit::Application::Model::DB::DBH_KEY = $coro_specific if $use_coro;
    local $Coro::State::DIEHOOK = \&QBit::Exceptions::die_handler if $use_coro;

    my @queries = $self->data_query(%opts);

    my %db_config = ();
    if ($self->check_rights('statistics_from_replication') && !$ENV{'RUNNING_IN_CRON'}) {
        my $master_host = $self->partner_db->get_option('host');

        my @slave = grep {$master_host ne $_->{'host'}} @{$self->partner_db->get_option('slave', [])};

        %db_config = %{$slave[rand(@slave)] // {}};
    }

    foreach my $query (@queries) {
        if ($use_coro) {
            push(
                @threads,
                async {
                    my ($count) = @_;

                    $$coro_specific = "_STAT_$count";
                    $self->app->partner_db->_connect(use_coro_mysql => TRUE, %db_config);

                    return $query->get_all();
                }
                $count++
            );
        } else {
            push(@$results, [$query->get_all()]);
        }
    }

    if ($use_coro) {
        # @$results : array of arrays of results from coro routines
        $results = $self->join_all_coros(\@threads);
    }

    return [map {@{$_->[0]}} @$results];    # join all stat records from result arrays together
}

sub get_product {
    my ($self) = @_;

    $self->can('product') ? $self->product : $self->users;
}

sub has_statistics {TRUE}

sub has_raw_stat {FALSE}

sub is_available {
    my ($self) = @_;

    my $cur_user = $self->app->get_option('cur_user') or throw Exception gettext('Current user is undefined');
    my $cache_key = $cur_user->{'id'} . '_' . ref($self);

    my $res =
        $cur_user->{'id'} != $YNDX_PARTNER_INTAPI_USER_ID
      ? $self->memcached->get(available_stat_levels => $cache_key)
      : undef;

    return $res if defined($res);

    # TODO: Выгдядит так, что этот блок может быть удалён, но нужно разобраться почему начинаются 500-ки
    my $product    = $self->product;
    my $pk_fields  = $product->get_pk_fields();
    my $tmp_rights = $self->app->add_tmp_rights(
        map {ref($_) eq '' ? $_ : @$_}
        grep {defined($_)}
        map {$_->{'check_rights'}} values %{{hash_transform($product->get_model_fields(), $pk_fields)}}
    );
    # TODO: END

    $res = $self->is_available_by_right();

    unless ($res) {
        if ($self->support_clickhouse()) {
            my $filter = $self->clickhouse_db->filter();
            $self->add_authorization_filter($filter);

            $res = @{$self->clickhouse_db->statistics->get_all(fields => [qw(page_id)], filter => $filter, limit => 1)};
        } else {
            my $id_field = $self->get_fields(fields => ['id'])->[0]{'id'};

            $res = defined($id_field) && @{$self->query(fields => [$id_field])->limit(1)->get_all()};
        }
    }

    $self->memcached->set(available_stat_levels => $cache_key, $res, 30 * 60);

    return $res;
}

sub get_product_ids {
    my ($self, %opts) = @_;

    return undef if $self->id eq 'additional_income';

    my $product_ids = [];
    if ($self->has_statistics) {
        my @request_fields = keys(%{$opts{'fields'} // {}});

        if (@request_fields) {
            my $stat_fields = $self->fields();

            foreach my $field_name (@request_fields) {
                if (_level_has_field_or_depends($field_name, $stat_fields)) {
                    push(@$product_ids, $self->get_product->accessor);

                    last;
                }
            }
        } else {
            push(@$product_ids, $self->get_product->accessor);
        }
    }

    if ($self->can('children')) {
        foreach ($self->children) {
            push(@$product_ids, @{$_->get_product_ids(%opts)});
        }
    }

    return $product_ids;
}

sub _level_has_field_or_depends {
    my ($field_name, $stat_fields) = @_;

    if ($stat_fields->{$field_name}) {
        return TRUE;
    } elsif (exists($FIELD_TYPES{$field_name}->{'depends_on'})) {
        foreach (@{$FIELD_TYPES{$field_name}->{'depends_on'}}) {
            if (_level_has_field_or_depends($_, $stat_fields)) {
                return TRUE;
            }
        }
    }

    return FALSE;
}

sub join_condition {
    my ($self, $entity_filter_accessor) = @_;

    my %join_condition_data = $self->join_condition_data();

    return $join_condition_data{$entity_filter_accessor}
      || throw gettext("Unknown entity filter accessor '%s' for '%s'", $entity_filter_accessor, $self->id);
}

sub _get_group_by_fields {
    my ($self, $fields) = @_;

    return () unless $fields && @$fields;

    my %product_stat_fields = map {$_->name => TRUE} @{$self->_get_stat_table()->fields()};

    return grep {$product_stat_fields{$_} || $_ ne 'currency_id'} @$fields;
}

sub _get_query_stat_fields {
    my ($self, %opts) = @_;

    my %product_stat_fields = map {$_->name => TRUE} @{$self->_get_stat_table()->fields()};
    my %entity_fields = map {$_ => TRUE} @{$opts{'entity_fields'} // []};

    return (
        map {$_ => $opts{'stat_db_group_fields'}{$_}}
          grep {!$entity_fields{$_} && ($product_stat_fields{$_} || $_ ne 'currency_id')}
          keys(%{$opts{'stat_db_group_fields'}})
      ),
      (map {$_ => $opts{'stat_db_fields'}{$_}} grep {$product_stat_fields{$_}} keys(%{$opts{'stat_db_fields'}}));
}

sub _remove_currency_id_from_filter {
    my ($self, $filter) = @_;

    # interate over members of top AND
    my $filters = $filter->{__FILTER__}[1];
    foreach (my $i = 0; $i < @$filters; $i++) {
        # find currency_id filter in 2nd level ANDs
        # and cut it off
        if (ref($filters->[$i][1]) eq 'ARRAY') {
            if ($filters->[$i][1][0][0] eq 'currency_id') {
                splice(@$filters, $i, 1);
            }
        }
    }
}

sub print_diff {
    my ($self, $dt, $page_id, $sums, $page_stat, $diff_fields_format) = @_;

    # TODO: Sentry formatting
    WARN(
        sprintf(
            qq{
FIX STATISTICS!!!
Level: %s
Date: %s
Page ID: %s

 -                   pages          tags         diff, %%
--------------------------------------------------------
%s
},
            $self->accessor,
            $dt, $page_id,
            join(
                "\n",
                map {
                    sprintf('%-14s  %14s %14s %9.4f', @$_)
                  }
                  grep {
                    $_->[1] != $_->[2]
                  }
                  map {
                    my ($field_name, $format) = @$_;
                    my $sum_val  = $sums->{$field_name}      // 0;
                    my $page_val = $page_stat->{$field_name} // 0;
                    if ($field_name =~ /_nds$/) {
                        $sum_val  /= $STAT_MONEY_SCALE;
                        $page_val /= $STAT_MONEY_SCALE;
                    }
                    [
                        $field_name,
                        sprintf($format, $page_val),
                        sprintf($format, $sum_val),
                        $page_val
                        ? ($sum_val * 100 / $page_val - 100)
                        : 0
                    ]
                  } @$diff_fields_format
                )
               )
        );
}

sub support_clickhouse {TRUE}

sub sort_priority {0}

sub _can {
    $_[0]->can($_[1]) ne \&{"Application::Model::Statistics::Product::$_[1]"};
}

sub _get_pages_stat_places {undef}

TRUE;
