
=encoding UTF-8
=head1

Модель Statistics::ReportParamsDigest предназначена для хранения пар ключ-значение,
где значение - это любой текст, например JSON с параметрами отчета,
а ключ - хеш от этого значения по алгоритму SHA1.
В случае коллизий к ключу добавляется суффикс вида -1, -2, и так далее.

Добавление:

    my $hash = $app->stat_report_params_digest->add(
        params => 'asdf',
    );

    # '3da541559918a808c2402bba5012f6c60b27661c'

Получение:

    my $data = $app->stat_report_params_digest->get(
        '3da541559918a808c2402bba5012f6c60b27661c',
    );

    #{
    #    id     => "3da541559918a808c2402bba5012f6c60b27661c",
    #    params => "asdf",
    #}

=cut

package Application::Model::Statistics::ReportParamsDigest;

use qbit;

use base qw(Application::Model::DBManager::Base);

use QBit::Validator;
use Digest::SHA qw(sha1_hex);

use Exception::DB::DuplicateEntry;

sub accessor      {'stat_report_params_digest'}
sub db_table_name {'stat_report_params_digest'}

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB::Statistics');

__PACKAGE__->model_fields(
    id          => {db => TRUE, default => TRUE, pk => TRUE},
    params      => {db => TRUE, default => TRUE},
    update_date => {db => TRUE},
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {id => {type => 'text', label => d_gettext('ID')}, update_date => {type => 'date'}},
);

sub add {
    my ($self, %opts) = @_;

    my $params = $opts{'params'};

    QBit::Validator->new(data => $params, template => {len_min => 1}, throw => TRUE);

    my $hash   = sha1_hex($params);
    my $suffix = 0;

    my ($id, $data, $res);
    while (!defined($res)) {
        $id = $hash . ($suffix > 0 ? "-$suffix" : "");
        try {
            $res = $self->partner_db_table()->add(
                {
                    id          => $id,
                    params      => $params,
                    update_date => curdate(oformat => 'db_time'),
                }
            );
        }
        catch Exception::DB::DuplicateEntry with {
            $data = $self->get($id);
            if ($data && $data->{'params'} eq $params) {
                $res = 1;
            } else {
                $suffix++;
            }
        };
    }

    return $id;
}

sub update {
    my ($self, $id) = @_;

    $self->partner_db_table()->edit($id, {update_date => curdate(oformat => 'db_time')});
}

sub clean {
    my ($self) = @_;

    my $filter =
      $self->partner_db->filter(['update_date' => '<' => \date_sub(curdate(), month => 2, oformat => 'db_time')]);

    $self->partner_db_table()->delete($filter);
}

TRUE;
