package Application::Model::Statistics::Update;

=encoding UTF-8

=head1 Название

Application::Model::Statistics::Update - Все про перезабор статистики

=begin comment Описание работы

В этом классе собраны все методы, которые нужны для перезабора статистики из
баланса. Нужно отнаследоватся от этого класса и переопределить методы, которые
указаны в abstract_methods().

=end comment

=begin comment _get_last_update_day

Возвращает целое число $int с номером дня до которого включительно нужно
перезабирать статистику за прошлый месяц.

Например, если возвращается 1, то первый запуск перезабора статистики 1-ого
числа так же перезаберет статистику за весь прошлый месяц. Еще пример: если
возвращается 5, то c первого числа нового месяца, вплоть до 5-ого числа
включительно будет перезабиратся статистика за предыдущий месяц.

Если возвращается 0, то в новом месяце статистика за старый месяц уже не
перезабирается.

=end comment

=cut

use qbit;

use base qw(QBit::Application::Model Application::Model::Statistics::_Utils::Money);

use Utils::Logger qw(INFO INFOF ERROR WARN);

sub has_partner_money_fields {TRUE}

use Exception::Statistics::InvalidCurrency;
use Exception::BatchFail;
use Exception::Statistics::Update;
use Exception::Validation::BadArguments;
use Exception::DB::DuplicateEntry;

use PiConstants qw($DAYS_BEFORE_DELETE_TMP_TABLES $RUB_CURRENCY_ID $CLICKHOUSE_STATISTICS_PK);
use PiVariables;

__PACKAGE__->abstract_methods(
    qw(
      _filter_stat
      _get_last_update_day
      _get_stat_from_balance
      db_table_name
      clickhouse_table_name
      )
);

__PACKAGE__->mk_accessors(qw(table_fields_without_pk));

__PACKAGE__->model_accessors(
    kv_store      => 'QBit::Application::Model::KvStore',
    clickhouse_db => 'Application::Model::ClickhouseDB'
);

sub can_be_reloaded {FALSE}

sub check_statistics { }

sub get_table_fields_without_pk {
    my ($self) = @_;

    my $fields = $self->table_fields_without_pk() // [];

    unless (@$fields) {
        my %pk = map {$_ => TRUE} @{$self->_get_stat_table->primary_key};

        $fields = [grep {!$pk{$_}} $self->_get_stat_table->field_names()];

        $self->table_fields_without_pk($fields);
    }

    return $fields;
}

sub id {
    my ($self) = @_;

    my $id = $self->{'accessor'};
    $id =~ s/^statistics_//;

    return $id;
}

=head2 regular_update_statistics

B<Параметры:> 1) $self

B<Возвращаемое значение:> -

Метод, который вызывается по крону много раз в день и перезабирает статистику
за сегодня и вчера. При первом запуске в день метод перезабирает данные с
начала месяца (а в первых числах месяца еще и за прошлый мясяц).

    $app->statistics_dsp->regular_update_statistics(
        from => "2012-01-01",
        to => "2012-01-31",
    );

=cut

sub regular_update_statistics {
    my ($self, %opts) = @_;

    $opts{'storage'} //= 'mysql';
    my $is_update_for_month = $opts{'for_month'} //= FALSE;

    $self->update_statistics(
        from => $self->__get_regular_update_start_date($is_update_for_month),
        to   => $self->__get_regular_update_finish_date($is_update_for_month),
        %opts
    );

    $self->__save_regular_update_run_date($is_update_for_month, %opts);

    return TRUE;
}

sub save_statistics {
    my ($self, $date, $stat, %opts) = @_;

    my $table = $self->_get_stat_table();

    my %filter_opts = hash_transform(\%opts, [qw(logins page_ids)]);
    my $fixed_stat = $self->_filter_stat($stat, %filter_opts);

    my $data_fields = $self->get_table_fields_without_pk();

    foreach my $rec (@$fixed_stat) {
        foreach (@$data_fields) {
            $rec->{$_} = sprintf('%d', $rec->{$_} // 0);
        }
    }

    my $result = FALSE;
    try {
        if ($opts{'storage'} eq 'mysql') {
            $self->app->partner_db->transaction(
                sub {
                    my $filter = $self->app->partner_db->filter(
                        [AND => [[dt => '>=' => \$date], [dt => '<=' => \"$date 23:59:59"]]]);
                    $self->_fix_delete_filter($filter, %filter_opts);
                    $table->delete($filter);
                    $table->add_multi($fixed_stat) if @$fixed_stat;
                }
            );

            $result = TRUE;
        } elsif ($opts{'storage'} eq 'clickhouse') {
            $self->refresh_tmp_stat_table();

            my @filters = ();
            push(@filters, $opts{'filter'}) if defined($opts{'filter'});

            my @page_ids_for_delete = $self->_filter_opts2page_ids(%filter_opts);
            push(@filters, ['page_id', 'IN', \[@page_ids_for_delete]]) if @page_ids_for_delete;

            $self->move_statistics(
                'statistics', 'statistics_temporary',
                from => $date,
                to   => $date,
                (@filters ? (filter => ['AND', \@filters]) : ())
            );

            $self->clickhouse_db->statistics_temporary->add_multi($self->convert_mysql_to_clickhouse($fixed_stat));

            $self->move_statistics('statistics_temporary', 'statistics', from => $date, to => $date);

            $result = TRUE;

            $self->clickhouse_db->statistics_temporary->drop();
        } else {
            throw sprintf('Unknown storage: %s', $opts{'storage'});
        }
    }
    catch Exception::DB::DuplicateEntry with {
        throw sprintf('Exception "DuplicateEntry" does not exists for clickhouse') if $opts{'storage'} eq 'clickhouse';

        WARN(   "DuplicateEntry error while inserting in: "
              . $table->{name}
              . ". Will use INSERT IGNORE to bypass broken row!");
        $self->app->partner_db->transaction(
            sub {
                my $filter =
                  $self->app->partner_db->filter([AND => [[dt => '>=' => \$date], [dt => '<=' => \"$date 23:59:59"]]]);
                $self->_fix_delete_filter($filter, %filter_opts);
                $table->delete($filter);
                $table->add_multi($fixed_stat, ignore => TRUE);
            }
        );

        $result = TRUE;
    }
    catch {
        my ($exception) = @_;
        $self->app->exception_dumper->dump_as_html_file($exception);
    };

    return $result;
}

sub is_save_statistics {
    return TRUE;
}

## Public methods:

=head2 update_statistics

B<Параметры:> 1) $self 2) %opts

B<Возвращаемое значение:> -

Метод для обновления статистики продукта. Методу нужно передать
диапазон дат по которым нужно обновить статистику:

    $app->statistics_dsp->update_statistics(
        from => "2012-01-01",
        to => "2012-01-31",
    );

=cut

sub update_statistics {
    my ($self, %opts) = @_;

    throw Exception::Validation::BadArguments gettext("Expecting 'from'") unless defined($opts{'from'});
    throw Exception::Validation::BadArguments gettext("Expecting 'to'")   unless defined($opts{'to'});

    $opts{'storage'} //= 'mysql';

    my @date_period = reverse $self->__get_dates_by_period(%opts);

    return FALSE unless @date_period;

    local $PiVariables::IGNORE_RTB_ON_DIRECT_SWITHER = 1;

    my $skiped_dates = [];
    foreach my $date (@date_period) {

        my $stat;
        my $tries_left       = 3;
        my $is_day_got_error = 0;
      RETRY: {

            try {
                $stat = $self->_get_stat_from_balance(
                    from     => $date,
                    to       => $date,
                    logins   => $opts{'logins'},
                    page_ids => $opts{'page_ids'},
                    storage  => $opts{'storage'},
                ) // throw Exception::Statistics::Update 'undef result from _get_stat_from_balance';
            }
            catch Exception::Statistics::InvalidCurrency with {
                throw $_[0];
            }
            catch {
                my ($exception) = @_;

                # NOTE! делаем 3 попытки только для HTTP ошибок, иначе просто переходим ко следующему дню
                if (--$tries_left && $exception->isa('Exception::API::HTTP')) {
                    sleep 10;
                    goto RETRY;
                } else {
                    push @$skiped_dates, $date;
                    $self->app->exception_dumper->dump_as_html_file($exception);
                    $is_day_got_error = 1;
                }
            };
        }

        unless ($is_day_got_error) {
            my $stat_sums = $self->{'STAT_SUMS'};
            if ($stat_sums) {
                my $accessor = $self->accessor;
                INFO("[Cron::Methods] $accessor ($date)");
                foreach my $key (sort keys %$stat_sums) {
                    INFOF "\t%s = %s", $key, $stat_sums->{$key};
                }
            }

            if ($self->is_save_statistics()) {
                my $res = $self->save_statistics($date, $stat, %opts);

                push(@$skiped_dates, $date) unless $res;
            }

            $self->check_statistics();
        }

        if ($opts{for_month} && $opts{day_update_observer}) {
            $opts{day_update_observer}->($self->app);
        }
    }

    if ($opts{'for_month'}) {
        my $key_for_dates_with_errors = $self->__get_kv_store_key_for_dates_with_errors(%opts);

        $self->kv_store->set($key_for_dates_with_errors,
            to_json(array_uniq($skiped_dates, name2date('yesterday', oformat => 'db'))));
    } elsif (@$skiped_dates) {
        throw Exception::BatchFail "Some days has been skipped: " . join(", ", @$skiped_dates);
    }

    return TRUE;
}

sub add_recs {
    my ($self, $recs, %opts) = @_;

    my $data_fields = $self->get_table_fields_without_pk();

    foreach my $rec (@$recs) {
        $self->{STAT_SUMS}->{'rec_count'}++;
        foreach (@$data_fields) {
            $rec->{$_} = sprintf('%d', $rec->{$_} // 0);
            $self->{STAT_SUMS}->{$_} += $rec->{$_};
        }
    }

    if ($opts{'storage'} eq 'mysql') {
        $self->_get_stat_table()->add_multi($recs, %opts);
    } elsif ($opts{'storage'} eq 'clickhouse') {
        $self->clickhouse_db->statistics_temporary->add_multi($self->convert_mysql_to_clickhouse($recs));
    } else {
        throw sprintf('Unknown storage: %s', $opts{'storage'});
    }
}

sub __get_dates_by_period {
    my ($self, %opts) = @_;

    my %dates           = ();
    my @dates_for_retry = ();

    if ($opts{'for_month'}) {
        my $key_for_dates_with_errors = $self->__get_kv_store_key_for_dates_with_errors(%opts);
        my $json = $self->kv_store->get($key_for_dates_with_errors) // '[]';

        @dates_for_retry = @{from_json($json)};

        my $key_for_month = $self->__get_kv_store_key_for_month(%opts);
        my $last_run_date = $self->kv_store->get($key_for_month) // '';

        return sort @dates_for_retry if $last_run_date eq curdate(oformat => 'db');

        %dates = map {$_ => TRUE} @dates_for_retry;
    }

    map {$dates{$_} = TRUE} dates2array(
        $opts{'from'},
        $opts{'to'},
        iformat => 'db',
        oformat => 'db',
    );

    return sort keys(%dates);
}

sub __get_kv_store_key {
    my ($self, %opts) = @_;

    my $postfix = $opts{'storage'} eq 'mysql' ? '' : '_clickhouse';

    return "last_update_stat_" . $self->id() . $postfix;
}

sub __get_kv_store_key_for_month {
    my ($self, %opts) = @_;

    my $postfix = $opts{'storage'} eq 'mysql' ? '' : '_clickhouse';

    return "last_update_stat_for_month_" . $self->id() . $postfix;
}

sub __get_kv_store_key_for_dates_with_errors {
    my ($self, %opts) = @_;

    my $postfix = $opts{'storage'} eq 'mysql' ? '' : '_clickhouse';

    return "dates_with_errors_" . $self->id() . $postfix;
}

=begin comment __get_month_update_date

B<Параметры:> 1) $self 2) $date в формаете YYYY-MM-DD 3) $day

B<Возвращаемое значение:> 1) $start_date в формаете YYYY-MM-DD

$start_date зависит только от $date и $day и не зависит от других состояний
системы.

Этот метом можно было бы объединить с __get_regular_update_start_date, но я
вынес его в отдельный метод, чтобы было проще тестировать.

=end comment

=cut

sub __get_month_update_date {
    my (undef, $date, $user_day) = @_;

    my $start_date;

    my $qbit_date = trdate('db', 'norm', $date);
    my $date_day = $qbit_date->[2];

    if ($date_day <= $user_day) {
        my $month_ago = date_sub($qbit_date, month => 1, iformat => 'norm', oformat => 'norm');
        $month_ago->[2] = 1;
        $start_date = trdate('norm', 'db', $month_ago);
    } else {
        $qbit_date->[2] = 1;
        $start_date = trdate('norm', 'db', $qbit_date);
    }

    return $start_date;
}

sub __get_regular_update_finish_date {
    my ($self, $is_update_for_month) = @_;

    return $ENV{STAT_UPDATE_END_DATE} // (
        $is_update_for_month
        ? name2date('yesterday', oformat => 'db')
        : curdate(oformat => 'db')
    );
}

=begin comment __get_regular_update_start_date

B<Параметры:> 1) $self 2) $is_update_for_month двоичный флаг

B<Возвращаемое значение:> 1) $start_date - дату в формате YYYY-MM-DD с который
нужно начать регулярный забор статистики.

Метод исходя из из значения флага $is_update_for_month формирует дату с которой
нужно перезабирать данные.

=end comment

=cut

sub __get_regular_update_start_date {
    my ($self, $is_update_for_month) = @_;

    return $ENV{STAT_UPDATE_START_DATE} // (
          $is_update_for_month
        ? $self->__get_month_update_date(curdate(oformat => 'db'), $self->_get_last_update_day())
        : name2date('today', oformat => 'db')
    );
}

## Private methods:

=begin comment __is_update_for_month

B<Параметры:> 1) $self

B<Возвращаемое значение:> 1) TRUE/FALSE - ответ на вопрос, нужно ли забирать
статистику за месяц или же только за вчера и сегодня.

Метод смотрит на дату последнего запуска регулярного перезабора статистики
(которая хранится в базе), на время из опции statistics_update_for_month_min_time
Application.json, начиная с которого можно перезабирать статистику за месяц,
и на настройки продукта. И исходя из этих данных метод дает ответ на вопрос.
нужно перезабирать данные.

=end comment

=cut

sub __is_update_for_month {
    my ($self) = @_;

    my $key_for_month = $self->__get_kv_store_key_for_month();
    my $last_run_date = $self->kv_store->get($key_for_month);

    my $start_date                = curdate();
    my $update_for_month_min_time = $self->get_option('statistics_update_for_month_min_time');
    $start_date->[3] = $update_for_month_min_time->{'hour'}   // 4;
    $start_date->[4] = $update_for_month_min_time->{'minute'} // 0;
    $start_date->[5] = $update_for_month_min_time->{'second'} // 0;

    return (!defined($last_run_date) || ($last_run_date ne curdate(oformat => 'db')))
      && (compare_dates(curdate(), $start_date) >= 0);
}

sub __save_regular_update_run_date {
    my ($self, $is_update_for_month, %opts) = @_;

    my $key;
    if ($is_update_for_month) {
        $key = $self->__get_kv_store_key_for_month(%opts);
    } else {
        $key = $self->__get_kv_store_key(%opts);
    }

    $self->kv_store->set($key, curdate(oformat => 'db'),);

    return FALSE;
}

# This function is redefined in all page update modules
# and in all block update modules where page_id name is not 'page_id'
# TODO: create a subclass of this class to be a base class for block update modules
#
sub _create_stat_entry {
    my ($self, $dt, $page_id, $block_id, $dsp_id) = @_;

    return {
        dt          => $dt,
        page_id     => $page_id,
        block_id    => $block_id,
        dsp_id      => $dsp_id,
        currency_id => 2,           #RUB
        map {$_ => 0} @{$self->get_table_fields_without_pk()}
    };
}

# NOTE! Копия ф-ции partner_db_table() в статистике (т.к. в иерархии нет подходящих общих предков)
sub _get_stat_table {
    my ($self) = @_;
    my $db_table_name = $self->db_table_name();
    return $self->partner_db->$db_table_name;
}

sub without_only_zero_values {
    my ($self, $data, $table_fields_without_pk) = @_;

    my @result;

    my @key_fields = ();
    if (@$data) {
        my @all_fields = sort keys %{$data->[0]};
        my $fields_without_pk = {map {$_ => 1} @$table_fields_without_pk};
        @key_fields = grep {!$fields_without_pk->{$_}} @all_fields;
    }

    foreach my $row (@$data) {
        foreach (@$table_fields_without_pk) {
            if (!defined $row->{$_}) {
                my $key_str = join ', ', map {sprintf '%s=%s', $_, $row->{$_} // 'undef'} @key_fields;
                throw sprintf('Uninitialized value found: %s=undef (%s)', $_, $key_str);
            } elsif ($row->{$_} != 0) {
                push(@result, $row);
                last;
            }
        }
    }

    return \@result;
}

sub get_pages_tags_stat {
    my ($self, %opts) = @_;

    my $callbacks = $opts{'callbacks'};
    my $init      = $callbacks->{'init'};
    my $process   = $callbacks->{'process'};
    my $finish    = $callbacks->{'finish'};

    my $fh = $self->api_balance->get_pages_tags_stat(
        from        => $opts{'from'},
        to          => $opts{'to'},
        _get_tsv_fh => TRUE,
        model       => $self->product,
    );

    {
        my $line = <$fh>;
        chomp($line);
        my @headers = split(/\t/, $line);
        unless (@headers) {
            throw Exception::Statistics::Update
              sprintf(qq[Balance2.GetPagesTagsStat returns no headers. \nfirst line: "%s"], $line // 'undef');
        }

        $init->(\@headers, $opts{'from'});
    }

    my $ch_data;
    if ($self->api_balance->need_balance_stat_from_clickhouse(from => $opts{'from'}, to => $opts{'to'})) {
        $ch_data = $self->api_balance->get_pages_tags_stat_from_clickhouse(
            from       => $opts{'from'},
            to         => $opts{'to'},
            model      => $self->product,
            only_place => $opts{only_place},
        );
    }

    my @elems;
    while (my $line = <$fh>) {
        chomp($line);
        @elems = split(/\t/, $line);
        $process->(\@elems, $ch_data);
    }

    close($fh);

    return $finish->();
}

sub get_pages_stat {
    my ($self, %opts) = @_;

    my $callbacks = $opts{'callbacks'};
    my $init      = $callbacks->{'init'};
    my $process   = $callbacks->{'process'};
    my $finish    = $callbacks->{'finish'};

    my $fh = $self->api_balance->get_pages_stat(
        from        => $opts{'from'},
        to          => $opts{'to'},
        _get_tsv_fh => TRUE,
        model       => $self->product,
    );

    {
        my $line = <$fh>;
        chomp($line);
        my @headers = split(/\t/, $line);
        unless (@headers) {
            throw Exception::Statistics::Update
              sprintf(qq[Balance.GetPagesStat returns no headers. \nfirst line: "%s"], $line // 'undef');
        }
        $init->(\@headers);
    }

    my $ch_data;
    if ($self->api_balance->need_balance_stat_from_clickhouse(from => $opts{'from'}, to => $opts{'to'})) {
        $ch_data = $self->api_balance->get_pages_stat_from_clickhouse(
            from           => $opts{'from'},
            to             => $opts{'to'},
            model          => $self->product,
            only_place     => $opts{only_place},
            convert_places => TRUE,
        );
    }

    my @elems;
    while (my $line = <$fh>) {
        chomp($line);
        @elems = split(/\t/, $line);
        $process->(\@elems, $ch_data);
    }

    close($fh);

    return $finish->();
}

sub get_dsp_stat_with_tags {
    my ($self, %opts) = @_;

    my $callbacks = $opts{'callbacks'};
    my $init      = $callbacks->{'init'};
    my $process   = $callbacks->{'process'};
    my $finish    = $callbacks->{'finish'};

    my $fh = $self->api_balance->get_dsp_stat(
        from                    => $opts{'from'},
        to                      => $opts{'to'},
        include_partner_stat_id => 1,
        _get_tsv_fh             => TRUE,
        model                   => $self->product,
    );

    {
        my $line = <$fh>;
        chomp($line);
        my @headers = split(/\t/, $line);
        unless (@headers) {
            throw Exception::Statistics::Update
              sprintf(qq[Balance2.GetDspStat returns no headers. \nfirst line: "%s"], $line // 'undef');
        }
        $init->(\@headers, $opts{'from'});
    }

    my $ch_data;
    if ($self->api_balance->need_balance_stat_from_clickhouse(from => $opts{'from'}, to => $opts{'to'})) {
        $ch_data = $self->api_balance->get_dsp_stat_from_clickhouse(
            from                    => $opts{'from'},
            to                      => $opts{'to'},
            model                   => $self->product,
            include_partner_stat_id => 1,
        );
    }

    my @elems;
    while (my $line = <$fh>) {
        chomp($line);
        @elems = split(/\t/, $line);
        $process->(\@elems, $ch_data);
    }

    close($fh);

    return $finish->();
}

sub get_dsp_page_stat {
    my ($self, %opts) = @_;

    my $callbacks = $opts{'callbacks'};
    my $init      = $callbacks->{'init'};
    my $process   = $callbacks->{'process'};
    my $finish    = $callbacks->{'finish'};

    my $fh = $self->api_http_bk->get_dsp_page_stat(
        starttime          => $opts{'from'},
        stoptime           => $opts{'to'},
        'include-bad-stat' => 1,
        ':cached'          => 1,
        ':return_fh'       => 1,
    );

    {
        # headers not in input
        my @headers = qw(
          Date
          PageID
          ImpID
          DSPID
          WinHits
          WinPrice
          WinPartnerPrice
          Shows
          AllHits
          WinMaxPositionsCount
          AllRealPrice
          BadWinHits
          BadWinPrice
          BadWinPartnerPrice
          BadShows
          BadAllHits
          BadWinMaxPositionsCount
          BadAllRealPrice
          );
        $init->(\@headers);
    }

    my @elems;
    while (my $line = <$fh>) {
        chomp($line);
        @elems = split(/\t/, $line);
        $process->(\@elems);
    }

    close($fh);

    return $finish->();
}

sub get_direct_rtb_stat {
    my ($self, %opts) = @_;

    my $callbacks = $opts{'callbacks'};
    my $init      = $callbacks->{'init'};
    my $process   = $callbacks->{'process'};
    my $finish    = $callbacks->{'finish'};

    my $end_mark_seen;

    my $fh = $self->api_http_bk->get_direct_rtb_stat(
        date_from => $opts{'from'},
        date_to   => $opts{'to'},
        (defined($opts{'ad_type'}) ? (ad_type => $opts{'ad_type'}) : ()),
        ':cached'    => 1,
        ':return_fh' => 1,
    );

    {
        my $line = <$fh>;
        chomp($line);
        my @headers = split(/\t/, $line);
        unless (@headers) {
            throw Exception::Statistics::Update
              sprintf(qq[export/export_direct_rtb_stat.cgi returns no headers. \nfirst line: "%s"], $line // 'undef');
        }
        $headers[0] =~ s/^#//;    # '#EventDate' to 'EventDate'
        $init->(\@headers);
    }

    my @elems;
    while (my $line = <$fh>) {
        chomp($line);

        if ($line eq '#END') {
            $end_mark_seen++;
            next;
        }
        @elems = split(/\t/, $line);
        $process->(\@elems);
    }

    close($fh);

    throw 'export/export_direct_rtb_stat.cgi - no "#END" marker received' unless $end_mark_seen;

    return $finish->();
}

sub _fix_delete_filter {
    my ($self, $filter, %opts) = @_;

    my @page_ids = $self->_filter_opts2page_ids(%opts);

    $filter->and({$self->get_page_id_field_name() => \@page_ids}) if @page_ids;
}

sub get_tmp_table_prefix {'tmp_statistics_' . $_[0]->product->accessor()}

sub get_tmp_table_name {
    my ($self, $prefix) = @_;

    throw Exception 'Expected prefix' unless defined($prefix);

    return join('_', $prefix, "pid_$$", format_date(curdate(), 'date_%Y%m%d_%H%M%S'));
}

sub drop_all_tmp_tables_by_prefix {
    my ($self, $prefix) = @_;

    throw Exception 'Expected prefix' unless defined($prefix);

    my $tables = $self->clickhouse_db->_get_all("SHOW TABLES LIKE '${prefix}_pid_%' FORMAT JSONCompact");

    foreach my $table (@$tables) {
        my ($date) = $table->{'name'} =~ /^$prefix\_pid_\d+_date_(\d{8})_\d{6}\z/;

        if (dates_delta_days($date, curdate(oformat => 'date_only_numbers'), iformat => 'date_only_numbers') >
            $DAYS_BEFORE_DELETE_TMP_TABLES)
        {
            INFOF "drop table if exists `%s`", $table->{'name'};
            $self->clickhouse_db->_do("DROP TABLE IF EXISTS `$table->{'name'}`");
        }
    }
}

sub refresh_tmp_stat_table {
    my ($self) = @_;

    my $prefix = $self->get_tmp_table_prefix();

    $self->drop_all_tmp_tables_by_prefix($prefix);

    $self->clickhouse_db->statistics_temporary->{'name'} = $self->get_tmp_table_name($prefix);

    $self->clickhouse_db->init_db(['statistics_temporary']);
}

sub move_statistics {
    my ($self, $table_from, $table_to, %opts) = @_;

    my %pk = map {$_ => ''} @$CLICKHOUSE_STATISTICS_PK;
    my @metrics = grep {!exists($pk{$_})} $self->clickhouse_db->$table_from->field_names;

    my $query = $self->clickhouse_db->query->select(
        table => $self->clickhouse_db->$table_from,
        fields =>
          {%pk, map {$_ => ($table_from eq 'statistics' ? ['*' => [\-1, {SUM => [$_]}]] : {SUM => [$_]})} @metrics},
        filter => [
            'AND',
            [
                ['product_id', '=',  \$self->product->accessor],
                ['dt',         '>=', \$opts{'from'}],
                ['dt',         '<=', \$opts{'to'}],
                (defined($opts{'filter'}) ? $opts{'filter'} : ()),
            ]
        ]
    );

    $query->group_by(@$CLICKHOUSE_STATISTICS_PK);

    $self->clickhouse_db->$table_to->add($query);
}

sub get_page_id_field_name {'campaign_id'}

sub convert_mysql_to_clickhouse {
    my ($self, $stats) = @_;

    my $is_product = !$self->isa('Application::Model::Statistics::Hierarchy');

    my $product_accessor   = $self->product->accessor();
    my $page_id_field_name = $self->get_page_id_field_name();

    my $has_partner_money_fields = $self->has_partner_money_fields();

    my $prefix         = $self->can('prefix') ? $self->prefix() : 'NOT_DEFINED_';
    my $rtb_short_name = 'NOT_DEFINED_';
    my $page_type      = 'NOT_DEFINED_';

    if (!$is_product) {
        if (   $product_accessor eq 'ssp_context_on_site_campaign'
            || $product_accessor eq 'ssp_video_an_site'
            || $self->product->is_context_page)
        {
            $page_type = 'context_';
        } elsif ($self->product->is_search_page) {
            $page_type = 'search_';
        }

        foreach my $child ($self->children) {
            my $child_product = $child->product();

            if ($child_product->can('does') && $child_product->does('Application::Model::Role::Has::RTB')) {
                $rtb_short_name = $child->prefix();
                $rtb_short_name =~ s/block_$//;
            }
        }
    }

    my $is_video_product = $self->product->isa('Application::Model::Block::Video')
      || $self->product->isa('Application::Model::Block::Dooh');

    return [
        map {
            {
                'dt'          => $_->{'dt'},
                'product_id'  => $product_accessor,
                'page_id'     => $_->{$page_id_field_name},
                'currency_id' => $_->{'currency_id'} // $RUB_CURRENCY_ID,
                (
                    $is_product
                    ? (
                        'block_id'     => $_->{'block_id'}     // 0,
                        'dsp_id'       => $_->{'dsp_id'}       // 0,
                        'tag_id'       => $_->{'tag_id'}       // 0,
                        'monetizer_id' => $_->{'monetizer_id'} // 0,
                        'geo_id'       => $_->{'geo_id'}       // 0,
                        'shows'        => $_->{$prefix . 'shows'}
                          || $_->{$prefix . 'shows_own_adv'} // 0,
                        'clicks' => $_->{$prefix . 'clicks'} // 0,
                        'hits' => $_->{$prefix . 'hits'}
                          || $_->{$prefix . 'hits_own_adv'}
                          || $_->{$prefix . 'hits_unsold'} // 0,
                        (
                            $product_accessor eq 'mobile_mediation_block'
                            ? (
                                'impressions'                 => $_->{$prefix . 'impressions'}                 // 0,
                                'calculated_revenue'          => $_->{$prefix . 'calculated_revenue'}          // 0,
                                'calculated_revenue_original' => $_->{$prefix . 'calculated_revenue_original'} // 0
                              )
                            : ()
                        ),
                        (
                            $is_video_product
                            ? ('win_max_positions_count' => $_->{$prefix . 'win_hits'}
                                  || $_->{$prefix . 'win_hits_own_adv'}
                                  || $_->{$prefix . 'win_hits_unsold'} // 0)
                            : ()
                        ),
                        'direct_clicks' => $_->{$prefix . 'direct_clicks'} // 0,
                        'direct_shows'  => $_->{$prefix . 'direct_shows'}  // 0,
                        'all_w_nds'     => $_->{'all_w_nds'}               // 0,
                        'all_wo_nds'    => $_->{'all_wo_nds'}              // 0,
                        'bad_shows'     => $_->{$prefix . 'bad_shows'}
                          || $_->{$prefix . 'bad_shows_own_adv'} // 0,
                        'bad_hits' => $_->{$prefix . 'bad_win_hits'}
                          || $_->{$prefix . 'bad_win_hits_own_adv'}
                          || $_->{$prefix . 'bad_win_hits_unsold'} // 0,
                        'bad_win_price_w_nds'  => $_->{'bad_win_price_w_nds'}  // 0,
                        'bad_win_price_wo_nds' => $_->{'bad_win_price_wo_nds'} // 0,

                        (
                            $has_partner_money_fields
                            ? (
                                'partner_w_nds'                => $_->{'partner_w_nds'}                // 0,
                                'partner_wo_nds'               => $_->{'partner_wo_nds'}               // 0,
                                'bad_win_partner_price_w_nds'  => $_->{'bad_win_partner_price_w_nds'}  // 0,
                                'bad_win_partner_price_wo_nds' => $_->{'bad_win_partner_price_wo_nds'} // 0,
                              )
                            : ()
                        ),
                      )
                    : (
                        #PAGES
                        'an_fraud_shows'           => $_->{'an_fraud_shows'}                              // 0,
                        'an_fraud_clicks'          => $_->{'an_fraud_clicks'}                             // 0,
                        'an_fraud_hits'            => $_->{'an_fraud_hits'}                               // 0,
                        'an_cover_hits'            => $_->{'an_cover_hits'}                               // 0,
                        'an_cover_direct_hits'     => $_->{'an_cover_direct_hits'}                        // 0,
                        'an_cover_market_hits'     => $_->{'an_cover_market_hits'}                        // 0,
                        'an_cover_mcb_hits'        => $_->{'an_cover_mcb_hits'}                           // 0,
                        'an_cover_senthits'        => $_->{'an_cover_senthits'}                           // 0,
                        'an_rtb_cover_hits'        => $_->{'an_' . $rtb_short_name . 'cover_hits'}        // 0,
                        'an_rtb_cover_senthits'    => $_->{'an_' . $rtb_short_name . 'cover_senthits'}    // 0,
                        'an_rtb_cover_direct_hits' => $_->{'an_' . $rtb_short_name . 'cover_direct_hits'} // 0,
                        'direct_page_ad_shows'     => $_->{'direct_' . $page_type . 'page_ad_shows'}      // 0,
                        'premium_page_ad_shows'    => $_->{'premium_page_ad_shows'}                       // 0,
                        'direct_hits'              => $_->{'direct_' . $page_type . 'hits'}               // 0,
                        'mcb_hits'                 => $_->{'mcb_' . $page_type . 'hits'}                  // 0,
                        'premium_hits'             => $_->{'premium_hits'}                                // 0,
                        'market_hits'              => $_->{'market_' . $page_type . 'hits'}               // 0,
                        'category_id'              => $_->{'category_id'}                                 // 0,
                        'view'                     => $_->{'instream_block_view'}                         // 0,
                        'open_player'              => $_->{'instream_block_open_player'}                  // 0,
                      )
                ),
            }
          } @$stats
    ];
}

TRUE;
