package Application::Model::Statistics::_Utils::Query;

use qbit;

use base qw(QBit::Class);

use Exception::Validation::Statistics;
use Exception::Validation::Statistics::InvalidPeriod;
use Exception::Statistics;
use Exception::Statistics::InvalidLevel;
use Exception::Statistics::BadQuery;
use Exception::Statistics::CantGrouped;
use Exception::Statistics::InvalidField;

__PACKAGE__->mk_ro_accessors(qw(statistics));

sub currency_id_if_needed {
    my ($self) = @_;

    return $self->is_currency_needed ? ('currency_id') : ();
}

sub db_group_by_fields {
    my ($self) = @_;

    unless (defined($self->{'db_group_by_fields'})) {
        $self->{'db_group_by_fields'} = array_uniq(
            map {ref($_) eq 'ARRAY' ? @$_ : $_} (
                (map {$_->{'db_fields'} || $_->{'id'}} values(%{$self->dimension_fields})),
                (map {$_} keys(%{$self->need_db_entity_fields}), ($self->is_currency_needed ? ('currency_id') : ()))
            )
        );
    }

    return clone($self->{'db_group_by_fields'});
}

sub dimension_field_names {
    my ($self) = @_;

    return keys(%{$self->dimension_fields});
}

sub dimension_fields {
    my ($self) = @_;

    unless (defined($self->{'dimension_fields'})) {
        my %requested_dimension_fields =
          map {my ($id, $opt) = split('\|'); $id => [$opt]} @{$self->{'opts'}{'dimension_fields'} // []};

        my %dimension_fields =
          map {$_->{'id'} => $_}
          grep {$requested_dimension_fields{$_->{'id'}}} @{$self->stat_level->get_dimension_fields(raw => TRUE)};

        $_->{'value'} = $requested_dimension_fields{$_->{'id'}}->[0]
          foreach grep {$_->{'get'} eq 'db_select'} values(%dimension_fields);

        $self->{'dimension_fields'} = \%dimension_fields;
    }

    return $self->{'dimension_fields'};
}

=head2 dimension_filter

    Input example:

    'dimension_filter' => [
      'AND',
      [
        [
          'tag_id',
          'IN',
          [
            '51',
            '52',
            '53',
            '54'
          ]
        ]
      ]
    ],

=cut

sub dimension_filter {
    my ($self) = @_;

    unless (exists($self->{'dimension_filter'})) {
        my $dimension_filter = $self->{'opts'}{'dimension_filter'};

        $dimension_filter = ['AND', [$dimension_filter]]
          if ref($dimension_filter) eq 'ARRAY' && @$dimension_filter == 3;

        my @dimension_filters;

        if ($dimension_filter) {
            my %permitted_dimension_fields =
              map {$_->{'id'} => $_} @{$self->stat_level->get_dimension_fields(raw => TRUE)};

            foreach my $filter (@{$dimension_filter->[1]}) {
                my $field_name = $filter->[0];
                my $field      = $permitted_dimension_fields{$field_name};
                my $filter_class;

                next unless $field;

                if ($field->{'type'}) {
                    $filter_class = "QBit::Application::Model::DBManager::Filter::$field->{'type'}";
                } else {
                    # !!! UGLY HACK: if the field is not in dimension_fields use some basic filter type
                    #
                    $filter_class = "QBit::Application::Model::DBManager::Filter::text";
                }

                my $filter_fn = "$filter_class.pm";
                $filter_fn =~ s/::/\//g;
                require $filter_fn or throw $!;

                $filter_class->new(%$field, field_name => $field->{'db_fields'}[0], db_manager => $self->statistics);
                push @dimension_filters,
                  $filter_class->as_filter([$field->{'db_fields'}[0] => $filter->[1] => $filter->[2]],);
            }
        }

        $self->{'dimension_filter'} = @dimension_filters ? [AND => \@dimension_filters] : undef;
    }

    return $self->{'dimension_filter'} // ();
}

sub entity_fields {
    my ($self) = @_;

    unless (defined($self->{'entity_fields'})) {
        my %requested_entity_fields = map {$_ => TRUE} @{$self->{'opts'}{'entity_fields'} // []};

        my %entity_fields =
          map {$_->{'id'} => $_}
          grep {$requested_entity_fields{$_->{'id'}}} @{$self->stat_level->get_entity_fields(raw => TRUE)};

        $self->{'entity_fields'} = \%entity_fields;
    }

    return $self->{'entity_fields'};
}

sub entity_field_names {
    my ($self) = @_;

    unless (defined($self->{'entity_field_names'})) {
        my $entity_fields = $self->entity_fields();

        $self->{'entity_field_names'} = [sort keys(%$entity_fields)];
    }

    return @{$self->{'entity_field_names'}};
}

sub fields {
    my ($self) = @_;

    unless (defined($self->{'fields'})) {
        my %requested_fields = map {$_ => TRUE} @{$self->{'opts'}{'fields'} // []};

        my %fields =
          map {$_->{'id'} => $_}
          grep {$requested_fields{$_->{'id'}}} @{
            $self->stat_level->get_fields(
                raw                   => TRUE,
                with_shared           => $self->{'opts'}{'with_shared'},
                clickhouse_expression => TRUE
            )
          };

        $self->{'fields'} = \%fields;
    }

    return $self->{'fields'};
}

sub get_fields_in_right_order {
    my ($self) = @_;

    my @fields = values(%{$self->fields});

    return sort {_func_sort_fields($a, $b)} @fields;
}

# TODO: подобная, но чуть расширеная (умеет принимать $period одиночной датой,
# например "2016-11-01") функция реализована в библиотеке фильтров кубита:
# QBit/Application/Model/DBManager/Filter/period.pm
# Нужно здесь заиспользовать её же
sub get_parsed_period_filter {
    my ($self, $period, $date_field_name, $with_minutes, $with_refs) = @_;

    try {
        if ($period && !(ref($period) eq 'ARRAY')) {
            $period = [name2dates($period, '', '', iformat => 'db', oformat => 'db')];
        }

        if ($period && (ref($period) eq 'ARRAY') && @$period == 1) {
            $period = [name2dates($period->[0], '', '', iformat => 'db', oformat => 'db')]
              unless ref($period->[0]) eq 'ARRAY';
        }
    }
    catch Exception with {
        my $e = shift;
        throw Exception::Validation::Statistics::InvalidPeriod $e;
    };

    throw Exception::Validation::Statistics::InvalidPeriod gettext('Invalid period')
      unless $period
          && ref($period) eq 'ARRAY'
          && check_date($period->[0], iformat => 'db')
          && check_date($period->[1], iformat => 'db')
          && $period->[0] le $period->[1];

    my $date_start = $period->[0];
    my $date_end   = $period->[1];

    throw Exception::Validation::Statistics::InvalidPeriod gettext('Days in period "%s - %s" more than %d', $date_start,
        $date_end, $self->max_days_in_period)
      if $self->max_days_in_period
          && dates_delta_days($date_start, $date_end, iformat => 'db') > $self->max_days_in_period;

    $date_end .= ' 23:59:59' if $with_minutes;

    return (
        $period,
        [
            AND => [
                [$date_field_name => '>=' => ($with_refs ? \$date_start : $date_start)],
                [$date_field_name => '<=' => ($with_refs ? \$date_end   : $date_end)]
            ]
        ]
    );
}

sub group_by {
    my ($self) = @_;

    unless (defined($self->{'group_by'})) {
        my $group_by = {};
        foreach (@{$self->{'opts'}{'group_by'} // []}) {
            $group_by->{$_} = {id => $_};
        }

        $self->{'group_by'} = $group_by;
    }

    return $self->{'group_by'};
}

sub group_by_fields {
    my ($self) = @_;

    return [$self->dimension_field_names, $self->entity_field_names, $self->currency_id_if_needed];
}

sub init {
    my ($self) = @_;

    $self->_parse_period();
    $self->level();
    $self->_check_group_by_fields_conflicts();

    return FALSE;
}

sub is_currency_needed {
    my ($self) = @_;

    unless (defined($self->{'is_currency_needed'})) {
        if (defined($self->{'opts'}{'is_currency_needed'})) {
            $self->{'is_currency_needed'} = $self->{'opts'}{'is_currency_needed'};
        } else {
            $self->{'is_currency_needed'} = exists($self->dimension_fields->{'currency_id'})
              || grep {$_->{'type'} eq 'money'} values(%{$self->need_fields});
        }
    }

    return $self->{'is_currency_needed'};
}

sub level {
    my ($self) = @_;

    unless (defined($self->{'level'})) {
        throw Exception::Statistics::InvalidLevel gettext('Invalid level')
          unless $self->{'opts'}{'levels'} && ref($self->{'opts'}{'levels'}) eq 'ARRAY';

        my $level = $self->{'opts'}{'levels'}[0];

        throw Exception::Statistics::InvalidLevel gettext('Invalid level')
          unless ref($level) eq 'HASH' && exists($level->{'id'});

        $self->{'level'} = $level;
    }

    return clone($self->{'level'});
}

sub level_filter {
    my ($self) = @_;

    return $self->level->{'filter'};
}

sub level_id {
    my ($self) = @_;

    return $self->level->{'id'};
}

sub max_days_in_period {$_[0]->{'opts'}{'max_days_in_period'}}

sub need_db_entity_fields {
    my ($self) = @_;

    unless (defined($self->{'need_db_entity_fields'})) {
        $self->{'need_db_entity_fields'} =
          $self->stat_level->get_entity_fields_obj([$self->entity_field_names])->get_db_fields();
    }

    return $self->{'need_db_entity_fields'};
}

sub need_db_field_names {
    my ($self) = @_;

    return grep {$self->need_fields->{$_}{'get'} eq 'db'} keys(%{$self->need_fields});
}

sub need_fields {
    my ($self) = @_;

    unless (defined($self->{'need_fields'})) {
        my %all_level_fields_with_shared =
          map {$_->{id} => $_} @{$self->stat_level->get_fields(raw => TRUE, with_shared => TRUE, forced => TRUE)};

        # {id => $_, get => sub {0}} is for the rare case when summary fields like video_an_site_sum_all_hits
        # depend on fields from several levels and some levels may not be available for the login at the moment
        #
        my %need_fields =
          map {
            $_ => $all_level_fields_with_shared{$_} // {
                id   => $_,
                get  => sub {0},
                type => $Application::Model::Statistics::Product::FIELD_TYPES{$_}{type},
                clickhouse_expression =>
                  $Application::Model::Statistics::Product::FIELD_TYPES{$_}{clickhouse_expression},
              }
          }
          map {
            _get_depends_fields($_)
          }
          keys(%{$self->fields});

        $self->{'need_fields'} = \%need_fields;
    }

    return $self->{'need_fields'};
}

sub need_summary {
    my ($self) = @_;

    return FALSE unless %{$self->fields};

    return $self->{'opts'}{'total'} if defined($self->{'opts'}{'total'});

    return $self->entity_field_names || grep {$_ ne 'currency_id'} $self->dimension_field_names;
}

sub new {
    my ($package, $statistics, %opts) = @_;

    %opts = hash_transform(
        \%opts,
        [
            qw(
              dimension_fields
              dimension_filter
              entity_fields
              fields
              group_by
              is_currency_needed
              levels
              limits
              max_days_in_period
              no_money_format
              order_by
              period
              total
              vat
              with_shared
              )
        ]
    );

    my $self = $package->SUPER::new(statistics => $statistics, opts => \%opts);

    return $self;
}

sub no_money_format {$_[0]->{'opts'}{'no_money_format'}}

sub period_description {
    my ($self) = @_;

    return $self->{'period_description'};
}

sub period_filter {
    my ($self) = @_;

    return $self->{'period_filter'};
}

sub stat_db_fields {
    my ($self) = @_;

    unless (defined($self->{'stat_db_fields'})) {
        my %stat_db_fields = map {$_ => {SUM => [$_]}} $self->need_db_field_names;

        throw Exception::Statistics::BadQuery gettext('Bad query') unless keys(%stat_db_fields);

        $self->{'stat_db_fields'} = \%stat_db_fields;
    }

    return clone($self->{'stat_db_fields'});
}

sub stat_db_group_fields {
    my ($self) = @_;

    unless (defined($self->{'stat_db_group_fields'})) {
        my %stat_db_group_fields = ();

        foreach my $field (values(%{$self->dimension_fields})) {
            if ($field->{'get'} eq 'db') {
                $stat_db_group_fields{$field->{'id'}} = $field->{'id'};
            } elsif ($field->{'get'} eq 'db_select') {
                my %values = map {$_->[0] => $_->[2]} @{$field->{'values'}};
                $stat_db_group_fields{$field->{'id'}} = $values{$field->{'value'} // ''} // $field->{'values'}[0][2];
            } elsif ($field->{'get'} eq 'dictionary') {
                $stat_db_group_fields{$_} = '' foreach @{$field->{'db_fields'}};
            } else {
                throw gettext('Unknown getter "%s"', $field->{'get'});
            }
        }

        $stat_db_group_fields{'currency_id'} = '' if $self->is_currency_needed;

        $self->{'stat_db_group_fields'} = \%stat_db_group_fields;
    }

    return clone($self->{'stat_db_group_fields'});
}

sub stat_level {
    my ($self) = @_;

    unless (defined($self->{'stat_level'})) {
        my $stat_level = $self->statistics->get_level($self->level_id);

        throw Exception::Statistics::InvalidLevel gettext('Invalid level name "%s"', $self->level_id)
          unless defined($stat_level) && $stat_level->is_available();

        $self->{'stat_level'} = $stat_level;
    }

    return $self->{'stat_level'};
}

sub _check_group_by_fields_conflicts {
    my ($self) = @_;

    my %group_fields = map {$_ => TRUE} $self->entity_field_names, keys(%{$self->dimension_fields});

    foreach (@{$self->stat_level->get_conflict_fields()}) {
        throw Exception::Statistics::CantGrouped gettext('Can\'t be grouped in the following fields: %s, %s', @$_)
          if (@$_) == (grep {$group_fields{$_}} @$_);
    }
}

sub _check_requested_fields {
    my ($self, $requested_fields, $permitted_fields, $error_field_message) = @_;
    $permitted_fields = {map {$_ => TRUE} @$permitted_fields} if ref($permitted_fields) eq 'ARRAY';

    my @errors = ();
    foreach my $field (grep {!exists($permitted_fields->{$_})} keys(%$requested_fields)) {
        push(@errors, sprintf($error_field_message, $field));
    }
    throw Exception::Statistics::InvalidField join(' ', @errors) if @errors;
}

sub _func_sort_fields {
    my ($first, $second) = @_;

    if ($first->{'get'} eq 'db') {
        return -1;
    } elsif ($second->{'get'} eq 'db') {
        return 1;
    } else {
        return -1 if in_array($first->{'id'},  $second->{'depends_on'});
        return 1  if in_array($second->{'id'}, $first->{'depends_on'});
        return 0;
    }
}

sub _get_depends_fields {
    my ($field_name) = @_;

    my $FIELD_TYPES = \%Application::Model::Statistics::Product::FIELD_TYPES;

    my @result = ($field_name);

    foreach (@{$FIELD_TYPES->{$field_name}{'depends_on'} // []}) {
        push(@result, _get_depends_fields($_));
    }

    return @result;
}

sub _parse_period {
    my ($self, $date_field_name, $with_minutes) = @_;

    $date_field_name //= 'dt';
    $with_minutes    //= 1;

    my $period = $self->{'opts'}{'period'};
    my $is_cmp = FALSE;
    if ($period && (ref($period) eq 'ARRAY') && (ref($period->[0]) eq 'ARRAY')) {
        throw Exception::Validation::Statistics::InvalidPeriod gettext('Invalid cmp period')
          unless @$period == 2 && (ref($period->[0]) eq 'ARRAY');
        $is_cmp = TRUE;
    }

    my $format_period = sub {
        join(' - ', map {format_date($_, gettext('%d.%m.%Y'), iformat => 'db')} @_);
    };
    my $period_description;
    my $period_filter;
    if ($is_cmp) {
        ($period->[0], my $period_filter_fst) =
          $self->get_parsed_period_filter($period->[0], $date_field_name, $with_minutes, 1);
        ($period->[1], my $period_filter_snd) =
          $self->get_parsed_period_filter($period->[1], $date_field_name, $with_minutes, 1);
        $period_filter = [OR => [$period_filter_fst, $period_filter_snd]];
        $period_description = gettext(
            "%s in comparison with period %s",
            $format_period->(@{$period->[0]}),
            $format_period->(@{$period->[1]})
        );
    } else {
        ($period, $period_filter) = $self->get_parsed_period_filter($period, $date_field_name, $with_minutes, 1);
        $period_description = $format_period->(@$period);
    }

    $self->{'period_filter'}      = $period_filter;
    $self->{'period_description'} = $period_description;

    return TRUE;
}

sub offset {$_[0]->{'opts'}{'limits'}{'offset'} // 0}

sub limit {$_[0]->{'opts'}{'limits'}{'limit'} // 0}

sub order_by {
    my ($self) = @_;

    unless (defined($self->{'order_by'})) {
        my $all_fields = {%{$self->fields()}, %{$self->entity_fields()}, %{$self->dimension_fields()},};

        $self->{'order_by'} = [grep {$all_fields->{$_->{'field'}}} @{$self->{'opts'}{'order_by'} // []}];
    }

    return $self->{'order_by'};
}

TRUE;
