package Application::Model::StatisticsReports;

use base qw(Application::Model::Common Application::Model::ValidatableMixin RestApi::MultistateModel);

use qbit;

use QBit::Validator;

use Exception;
use Exception::Denied;
use Exception::Validation::BadArguments;

my $REPORT_TYPE_PI   = 'pi';
my $REPORT_TYPE_MOL  = 'mol';
my @REPORT_TYPE_LIST = ($REPORT_TYPE_PI, $REPORT_TYPE_MOL);

use PiConstants qw( $MOL_REPORT_TYPE_MAIN $MOL_REPORT_TYPE_DSP $MOL_REPORT_TYPE_MM $CATEGORY_NAMES %MOL_STAT_TYPES);

sub accessor         {'statistics_reports'}
sub db_table_name    {'statistics_reports'}
sub get_product_name {gettext('statistics_reports')}

my $FIELDS_DEPENDS;

my $CATEGORY_FLAGS = {
    1 => {
        category => 'report_type',
        is_on    => sub {$_[0]->{report_type} ne $REPORT_TYPE_PI}
    },
    2 => {
        category => 'user',
        is_on    => sub {
            TRUE
        },
    },
    4 => {
        category => 'main',
        is_on    => sub {
            'main' eq ($_[0]->{stat_type} // '');
        },
    },
    8 => {
        category => 'mm',
        is_on    => sub {
            'mm' eq ($_[0]->{stat_type} // '');
        },
    },
    16 => {
        category => 'dsp',
        is_on    => sub {
            'dsp' eq ($_[0]->{stat_type} // '');
        },
    },
    2**5 => {category => 25,},
    2**6 => {category => 26,},
    2**7 => {category => 27,},
    2**8 => {category => 28,},
};

sub get_structure_model_accessors {
    return {
        partner_db        => 'Application::Model::PartnerDB',
        rbac              => 'Application::Model::RBAC',
        kv_store          => 'QBit::Application::Model::KvStore',
        statistics        => 'Application::Model::Statistics',
        widgets           => 'Application::Model::Widgets',
        widget_statistics => 'Application::Model::Widgets::Widget::Statistics',
        users             => 'Application::Model::Users',
        bk_statistics     => 'Application::Model::BKStatistics'
    };
}

sub get_structure_rights_to_register {
    return [
        {
            name        => 'report',
            description => d_gettext('Right to manage reports'),
            rights      => {
                statistics_reports_view_action_log => d_gettext('Right to view statistics reports logs'),
                statistics_reports_view            => d_gettext('Right to view statistics reports'),
                statistics_reports_view_all        => d_gettext('Right to view all statistics reports'),
                statistics_reports_add             => d_gettext('Right to add statistics reports'),
                statistics_reports_edit            => d_gettext('Right to edit statistics reports'),
                statistics_reports_view_filters    => d_gettext('Right to view filters in statistics reports'),
                statistics_reports_delete          => d_gettext('Right to delete statistics reports'),
            },
        },
    ];
}

sub get_structure_model_fields {
    return {
        id => {db => 'statistics_reports', pk => TRUE, type => 'number', default => TRUE, api => 1, type => 'string',},
        public_id => {
            depends_on => ['id'],
            get        => sub {
                return $_[1]->{'id'};
            },
            type => 'string',
        },
        report_id => {
            db         => 'statistics_reports',
            default    => TRUE,
            need_check => {
                optional => TRUE,
                len_min  => 1,
                len_max  => 64,
            },
        },
        caption => {
            db         => 'statistics_reports',
            default    => TRUE,
            i18n       => TRUE,
            need_check => {
                len_min => 1,
                len_max => 255
            },
            api  => 1,
            type => 'string',
        },
        description => {
            db         => 'statistics_reports',
            default    => TRUE,
            need_check => {
                optional => TRUE,
                len_max  => 255,
            },
            api  => 1,
            type => 'string',
        },
        level => {
            db         => 'statistics_reports',
            default    => TRUE,
            need_check => {
                check => sub {
                    my ($qv, $data) = @_;
                    my $available_levels = $qv->app->bk_statistics->get_available_statistics_levels();
                    unless (in_array($data, $available_levels)) {
                        throw Exception::Validator::Fields gettext('Got value "%s" not in array: %s',
                            $data, join(', ', sort(@{array_uniq(@$available_levels)})));
                    }
                  }
            },
        },
        query => {
            db         => 'statistics_reports',
            default    => TRUE,
            need_check => {
                len_min => 1,
                len_max => 65_535,
            },
            api  => 1,
            type => 'string',
        },
        owner_id   => {db => 'statistics_reports',},
        not_show   => {db => 'statistics_reports',},
        multistate => {db => 'statistics_reports', label => d_gettext('Status'), api => 1, type => 'number',},
        use_as_widget => {
            depends_on => ['id'],
            label      => d_gettext('Use as widget'),
            get        => sub {
                $_[0]->{'__USE_AS_WIDGET__'}{$_[1]->{'id'}} // 0;
            },
            need_check => {
                type     => 'boolean',
                optional => TRUE
            },
            api  => 1,
            type => 'boolean',
        },
        widget_settings => {
            depends_on => ['id'],
            label      => d_gettext('Widget settings'),
            get        => sub {
                $_[0]->{'__WIDGET_SETTINGS__'}{$_[1]->{'id'}} // [];
            },
            api  => 1,
            type => 'complex',
        },
        multistate_name => {
            depends_on => ['multistate'],
            label      => d_gettext('Multistate name'),
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            api  => 1,
            type => 'string',
        },
        editable_fields => {
            depends_on => [qw(id multistate is_standart_report)],
            get        => sub {
                return $_[0]->model->get_editable_fields($_[1]);
            },
            api  => 1,
            type => 'complex',
        },
        available_fields => {
            depends_on => [qw(multistate)],
            get        => sub {
                return $_[0]->model->get_available_fields($_[1]);
            },
            api  => 1,
            type => 'complex',
        },
        is_standart_report => {
            depends_on => [qw(owner_id)],
            get        => sub {
                return $_[1]->{'owner_id'} ? 0 : 1;
            },
            api  => 1,
            type => 'boolean',
        },
        actions => {
            label      => d_gettext('Actions'),
            depends_on => [qw(is_standart_report multistate)],
            get        => sub {
                $_[0]->model->get_actions($_[1]);
            },
            api  => 1,
            type => 'complex',
        },
        order => {
            db   => 'statistics_reports_level_order',
            api  => 1,
            type => 'number',
        },
        report_order => {
            db      => 'statistics_reports',
            default => TRUE,
            api     => 1,
            type    => 'number',
        },
        report_type => {
            db         => 'statistics_reports',
            need_check => {
                optional => TRUE,
                in       => \@REPORT_TYPE_LIST,
            },
            api  => 1,
            type => 'string',
        },
        fields_depends => {
            get => sub {
                $FIELDS_DEPENDS //= $_[0]->model->get_fields_depends();

                return $FIELDS_DEPENDS;
            },
            type => 'complex',
        },
        category_bits => {
            db         => 'statistics_reports',
            need_check => {optional => TRUE,},
        },
        category => {
            depends_on => ['category_bits'],
            get        => sub {
                for my $bit (qw(6 7 8)) {
                    return $CATEGORY_FLAGS->{2**$bit}{category} if $_[1]->{'category_bits'} & 2**$bit;
                }
                return $CATEGORY_FLAGS->{2**5}{category};
            },
            api  => 1,
            type => 'string',
        },
        category_name => {
            depends_on => ['category'],
            get        => sub {
                return $CATEGORY_NAMES->{$_[1]->{'category'}}->();
            },
            api  => 1,
            type => 'string',
        },
    };
}

sub get_structure_model_filter {
    return {
        db_accessor => 'partner_db',
        fields      => {
            id                 => {type => 'text', label => d_gettext('Report ID')},
            caption            => {type => 'text', label => d_gettext('Caption')},
            description        => {type => 'text', label => d_gettext('Description')},
            not_show           => {type => 'number'},
            is_standart_report => {
                type      => 'boolean',
                label     => d_gettext('Is standart report'),
                db_filter => sub {
                    return ['owner_id', ($_[1]->[2] ? 'IS' : 'IS NOT'), \undef];
                  }
            },
            stat_type => {
                type   => 'dictionary',
                values => sub {
                    [
                        map {+{id => $_, label => $MOL_STAT_TYPES{$_}{label}->()}}
                        sort keys %MOL_STAT_TYPES
                    ];
                },
                db_filter => sub {
                    my $bits = {
                        'dsp'  => 16,
                        'mm'   => 8,
                        'main' => 4,
                    };
                    return $bits->{$_[1]->[2]}
                      ? ['category_bits' => '&' => \$bits->{$_[1]->[2]}]
                      : ['category_bits' => '&' => \0];
                  }
            },
            use_as_widget => {type => 'boolean', label => d_gettext('Use as widget')}
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        (
            $self->check_short_rights('view_filters')
            ? (
                [{name => 'id', label => gettext('Report ID')}, {name => 'caption', label => gettext('Caption')},],
                [{name => 'description', label => gettext('Description')},],
              )
            : ()
        )
    ];
}

sub get_structure_multistates_graph {
    return {
        empty_name  => d_gettext('New'),
        multistates => [
            [new     => d_pgettext('Report', 'New')],
            [edited  => d_pgettext('Report', 'Edited')],
            [deleted => d_pgettext('Report', 'Deleted')],
        ],
        actions => {
            add     => d_pgettext('Report',          'Add'),
            edit    => d_pgettext('Report',          'Edit'),
            delete  => d_pgettext('Report',          'Delete'),
            restore => d_pgettext('Campaign action', 'Restore'),
        },
        multistate_actions => [
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => ['new'],
            },
            {
                action    => 'edit',
                from      => 'not deleted',
                set_flags => ['edited'],
            },
            {
                action    => 'delete',
                from      => 'not deleted',
                set_flags => ['deleted'],
            },
            {
                action      => 'restore',
                from        => 'deleted',
                reset_flags => ['deleted']
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('use_as_widget')) {
        my $owner_ids = array_uniq(grep {defined($_)} map {$_->{'owner_id'}} @$result);

        throw gettext('Field "use_as_widget" only for one user') if @$owner_ids > 1;

        my $widgets = $self->widgets->get($self->get_option('cur_user', {})->{'id'},
            fields => [qw(widgets_settings default_deleted_ids)]) // {};

        my $cur_widgets_settings = $widgets->{'widgets_settings'}    // [];
        my $default_deleted_ids  = $widgets->{'default_deleted_ids'} // {};

        my @default_statistics_widgets = ();
        if ($self->widget_statistics->can_view_as_default()) {
            @default_statistics_widgets =
              $self->widgets->_get_widget_models_by_settings([$self->widget_statistics->get_default_settings()]);
        }

        my @cur_widgets = $self->widgets->_merge_widgets(
            [
                $self->widgets->_get_widget_models_by_settings(
                    [grep {$_->{'type'} eq 'statistics'} @$cur_widgets_settings]
                )
            ],
            \@default_statistics_widgets,
            $default_deleted_ids
        );

        for my $widget (@cur_widgets) {
            $fields->{'__USE_AS_WIDGET__'}{$widget->report_id} = TRUE;
            unless (exists $fields->{'__WIDGET_SETTINGS__'}{$widget->report_id}) {
                my @report_widgets = map {$_->{settings}{settings}}
                  grep {$widget->report_id eq $_->report_id} @cur_widgets;
                # for new fake showType = 'both', we store it as 2 widgets - chart + table
                # but Frontend expects showType = 'both', so we do this
                if (2 == @report_widgets) {
                    my %new_widget = (%{$report_widgets[0]}, 'showType' => 'both');
                    $fields->{'__WIDGET_SETTINGS__'}{$widget->report_id} = [\%new_widget];
                } else {
                    $fields->{'__WIDGET_SETTINGS__'}{$widget->report_id} = \@report_widgets;
                }
            }
        }
    }

    if (@$result == 1) {
        $self->{'__RESULT_OWNER_ID__'} = delete($result->[0]{'_OWNER_ID'});
        $self->{'__RESULT_LEVEL__'}    = delete($result->[0]{'_LEVEL'});
    }
}

sub get_available_fields {
    my ($self, $obj) = @_;

    return {} if $obj && $self->check_multistate_flag($obj->{'multistate'}, 'deleted');
    my %fields = map {$_ => TRUE} keys(%{$self->get_model_fields});

    foreach (
        qw(
        report_id
        category_bits
        )
      )
    {
        delete($fields{$_});
    }

    return \%fields;
}

sub get_add_fields {
    my ($self, %opts) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{'report_id'}   = TRUE;
    $fields->{'level'}       = TRUE;
    $fields->{'report_type'} = TRUE;
    $fields->{'stat_type'}   = TRUE;

    # такого поля нет, но фронт фильтрует через get_add_fields,
    # возможно нужно будет ещё добавить в get_editable_fields
    $fields->{'widget_settings'} = TRUE;

    return $fields;
}

sub get_editable_fields {
    my ($self, $obj) = @_;

    return {} unless $self->check_action($obj, 'edit');

    return {use_as_widget => 1} if $obj->{'is_standart_report'};

    my %res = %{$self->_get_common_add_edit_fields()};

    return \%res;
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              caption
              description
              query
              use_as_widget
              widget_settings
              )
        ]
    );

    return $fields;
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('You cannot add report') unless $self->check_short_rights('add');

    $opts{report_type} //= $REPORT_TYPE_MOL;
    my $stat_type = delete $opts{stat_type} // 'main';
    $opts{category_bits} = _calc_report_category({%opts, stat_type => $stat_type});

    QBit::Validator->new(
        data  => \%opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => [keys(%{$self->get_add_fields})]),
    );

    $self->partner_db->transaction(
        sub {
            $opts{'id'} = $self->get_next_report_id();

            my $use_as_widget = delete($opts{'use_as_widget'});
            my $widget_settings = delete($opts{'widget_settings'}) // {};

            $self->partner_db->statistics_reports->add({%opts, owner_id => $self->get_option('cur_user', {})->{'id'}});

            if ($use_as_widget) {
                $self->widgets->save(
                    {
                        type     => 'statistics',
                        accessor => 'widget_statistics',
                        settings => {%$widget_settings, report_id => $opts{'id'}, level => $opts{'level'}}
                    },
                    replace => 1,
                );
            }
        }
    );

    $self->do_action($opts{'id'}, "add");

    return $opts{'id'};
}

sub edit {
    my ($self, $report_id, %data) = @_;

    throw Exception::Denied gettext('You cannot edit report')
      unless $self->check_short_rights('edit');
    return $report_id unless %data;

    my $report_info = $self->get($report_id, fields => [qw(editable_fields owner_id level)]);

    my $editable_fields = $report_info->{'editable_fields'};
    my $owner_id        = $report_info->{'owner_id'};
    my $level           = $report_info->{'level'};

    $self->check_bad_fields($report_id, \%data, {%$editable_fields, 'widget_settings' => TRUE});
    my $old_report_settings = $self->get($report_id, fields => [sort keys(%$editable_fields)])
      // throw Exception gettext('Report with id "%s" not found', $report_id);
    my %new_report_settings = (%$old_report_settings, %data);

    QBit::Validator->new(
        data  => \%new_report_settings,
        app   => $self,
        throw => TRUE,
        $self->get_template(fields => [keys(%$editable_fields), 'widget_settings']),
    );
    $self->partner_db->transaction(
        sub {
            my $use_as_widget = delete($data{'use_as_widget'});
            my $widget_settings = delete($data{'widget_settings'}) // {};

            if (defined($use_as_widget) && $use_as_widget) {
                $self->widgets->save(
                    {
                        type     => 'statistics',
                        accessor => 'widget_statistics',
                        settings => {%$widget_settings, report_id => $report_id, level => $level}
                    },
                    replace => 1,
                );
            } elsif (defined($use_as_widget) && !$use_as_widget) {
                $self->widgets->do_action(
                    {user_id => $owner_id,},
                    'delete',
                    widgets => [
                        {
                            type     => 'statistics',
                            accessor => 'widget_statistics',
                            settings => {%$widget_settings, report_id => $report_id, level => $level}
                        },
                    ]
                );
            }

            $self->partner_db->statistics_reports->edit($report_id, \%data) if defined($owner_id) && %data;

        }
    );

    return $report_id;
}

sub on_action_edit {
    my ($self, $report, %data) = @_;

    $self->edit($report->{'id'}, %data);
}

sub delete {
    my ($self, $report_id, %opts) = @_;

    throw Exception::Denied gettext('You cannot delete report')
      unless $self->check_short_rights('delete');

    my $report = $self->get($report_id, fields => [qw(is_standart_report use_as_widget level)], with_deleted => TRUE,)
      // throw gettext('Report with ID %s not found', $report_id);

    $self->partner_db->transaction(
        sub {
            if ($report->{'use_as_widget'} && $self->widgets->get($self->get_option('cur_user', {})->{'id'})) {
                $self->widgets->do_action(
                    $self->get_option('cur_user', {})->{'id'},
                    'delete',
                    widgets => {
                        type     => 'statistics',
                        accessor => 'widget_statistics',
                        settings => {report_id => $report_id, level => $report->{'level'}}
                    },
                );
            }

            if ($report->{'is_standart_report'}) {
                $self->partner_db->deleted_standart_statistics_reports->add(
                    {user_id => $self->get_option('cur_user', {})->{'id'}, report_id => $report_id});
            } else {
                $self->do_action($report_id, 'delete') unless $opts{from_action};
            }
        }
    );

    return TRUE;
}

sub can_action_delete {
    my ($self, $obj) = @_;

    $obj = $self->_get_object_fields($obj, [qw(is_standart_report)]);

    return !$obj->{'is_standart_report'};
}

sub on_action_delete {
    my ($self, $obj) = @_;
    return $self->delete($obj->{id}, from_action => 1);
}

sub query {
    my ($self, %opts) = @_;

    throw Exception::Denied gettext('Access denied') unless $self->check_short_rights('view');

    my $filter = $self->partner_db->filter($opts{'filter'});

    my $dont_view_reports = $self->partner_db->deleted_standart_statistics_reports->get_all(
        fields => ['report_id'],
        filter => {user_id => $self->get_option('cur_user', {})->{'id'}}
    );

    $filter->and(
        [
            'AND',
            [
                (
                    $opts{options}{with_deleted} ? ()
                    : (['multistate' => '=' => \$self->get_multistates_by_filter('not deleted')])
                ),
                (
                    @$dont_view_reports
                    ? (['id' => 'NOT IN' => \array_uniq(map {$_->{'report_id'}} @$dont_view_reports)])
                    : ()
                )
            ]
        ]
    );

    my $statistics_reports_fields = $opts{'fields'}->get_db_fields('statistics_reports');

    if ($self->check_short_rights('view_all')) {
        $statistics_reports_fields->{'_OWNER_ID'} = 'owner_id';    # extra hidden field
        $statistics_reports_fields->{'_LEVEL'}    = 'level';       # extra hidden field
    } else {
        my $levels = $self->bk_statistics->get_available_statistics_levels();
        $filter->and(
            [
                'AND',
                [
                    ['level' => 'IN' => \$levels],
                    ['OR', [[owner_id => '=' => \$self->get_option('cur_user', {})->{'id'}], {isnull => ['owner_id']}]]
                ]
            ]
        );
    }

    my $user = $self->users->get($self->get_option('cur_user', {})->{id}, fields => [qw(features)]);

    unless ($opts{options}{force}) {
        $filter->and(['report_type' => '!=' => \$REPORT_TYPE_PI]);
    }

    return $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => $statistics_reports_fields,
        filter => $filter,
      )->join(
        table   => $self->partner_db->statistics_reports_level_order,
        alias   => 'a_statistics_reports_level_order',
        fields  => $opts{'fields'}->get_db_fields('statistics_reports_level_order'),
        join_on => [level_id => '=' => {level => $self->partner_db->statistics_reports}],
      )->order_by(['order', 0], ['report_order', 0]);
}

sub get {
    my ($self, $object, %opts) = @_;

    my $available_levels = $self->bk_statistics->get_available_statistics_levels();

    # Get ANY user's report with a requested key and then check against the current user
    # It is required to return 'denied' exception

    $self->{'__RESULT_OWNER_ID__'} = undef;
    $self->{'__RESULT_LEVEL__'}    = undef;
    my $res;
    {
        my $tmp_rights = $self->app->add_tmp_rights(qw(statistics_reports_view_all));
        $res = $self->SUPER::get($object, %opts);
    }

    if ($res && !$self->check_short_rights('view_all')) {
        if (
            (
                defined($self->{'__RESULT_OWNER_ID__'})
                && $self->{'__RESULT_OWNER_ID__'} != $self->get_option('cur_user', {})->{'id'}
            )
            || (defined($self->{'__RESULT_LEVEL__'})
                && !in_array($self->{'__RESULT_LEVEL__'}, $available_levels))
           )
        {
            throw Exception::Denied gettext('Access denied');
        }
    }
    return $res;
}

sub get_caller_by_context {
    my ($self, $context) = @_;

    my @caller = ();

    my $i = 0;
    while (@caller = caller($i)) {
        last if $caller[3] && $caller[3] eq $context;
        $i++;
    }

    return @caller;
}

sub get_next_report_id {
    my ($self) = @_;

    throw gettext('Must call in transaction') unless $self->partner_db->{'__SAVEPOINTS__'};

    my $id = $self->kv_store->get('report_next_id', for_update => TRUE);

    $id = 1 unless defined($id);

    $self->kv_store->set('report_next_id', $id + 1);

    return $id;
}

sub get_fields_depends {
    return {};
}

#
# API
#

sub api_available_actions {
    return qw(
      edit
      delete
      );
}

sub api_can_add         {TRUE}
sub api_can_edit        {TRUE}
sub api_can_delete      {TRUE}
sub api_check_public_id {$_[1] =~ /^\w{1,64}$/i}

sub _calc_report_category {
    my ($report) = @_;

    my $category_bits = 0;
    for my $bit (keys %$CATEGORY_FLAGS) {
        next unless exists $CATEGORY_FLAGS->{$bit}->{is_on};
        if ($CATEGORY_FLAGS->{$bit}->{is_on}->($report)) {
            $category_bits = $category_bits | $bit;
        }
    }
    return $category_bits;
}

TRUE;
