package Application::Model::TNSDict::Article;

use qbit;

use base qw(Application::Model::TNSDict RestApi::SimpleModel Utils::Dict);
use Tree::Simple 'use_weak_refs';

use Utils::Logger qw(ERRORF);

my $MAX_LOST_ARTICLES = 10;

my $cpm_multiplier = 1000;

my $REAL_ARTICLE_LEVELS = [0, 2, 3];

sub accessor      {'tns_dict_article'}
sub db_table_name {'tns_dict_article'}

__PACKAGE__->model_fields(
    sid  => {db => TRUE, pk      => TRUE, default => TRUE, type => 'number'},
    grid => {db => TRUE, default => TRUE},
    lev  => {db => TRUE, default => TRUE},
    name    => {db => TRUE, default => TRUE, i18n => TRUE, type => 'string', api => 1},
    note    => {db => TRUE},
    status  => {db => TRUE},
    file_id => {db => TRUE},
    hidden  => {db => TRUE},
    parent_id => {type => 'number', get => sub {'fake_parent_id'}, api => 1},
    public_id => {
        db      => TRUE,
        db_expr => 'sid',
        type    => 'string',
        api     => 1
    }
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        sid    => {type => 'number',  label => 'Article ID'},
        grid   => {type => 'number',  label => 'Parent ID'},
        lev    => {type => 'number',  label => 'Level'},
        name   => {type => 'text',    label => d_gettext('Article name')},
        status => {type => 'text',    label => d_gettext('Status')},
        hidden => {type => 'boolean', label => d_gettext('Is hidden')},
    }
);

sub get_product_name {gettext('tns_dict_article')}

sub get_complete_articles {
    my ($self, $articles) = @_;

    return $self->get_cmp_from_tree($articles, $self->get_real_article_tree);
}

sub api_available_actions {
    return qw();
}

sub api_check_public_id {$_[1] =~ /^-?[0-9]+$/}

sub api_get_all {
    my ($self, $controller, $resource, $one_object) = @_;

    $controller->check_params(($one_object ? 'public_id' : ()), "fields[$resource]");
    my @fields = $controller->get_fields($resource);

    my $url_simple_model_get = $controller->get_abs_for_sprintf('simple_model__resource_get', qw(resource public_id));

    my %fields = map {$_ => TRUE} @fields;

    my @result = ();
    my $data   = $self->get_real_article();
    for my $row (@$data) {
        my $res = {
            id   => $row->{'id'},
            type => $resource,
            ($one_object ? () : (links => {self => sprintf($url_simple_model_get, $resource, $row->{'id'}),})),
        };

        my $parent_id = $row->{'parent_id'} || undef;
        $res->{'attributes'} = {};
        $res->{'attributes'}{'parent_id'} = $parent_id     if $fields{'parent_id'};
        $res->{'attributes'}{'name'}      = $row->{'name'} if $fields{'name'};

        push(@result, $res);
    }

    return \@result;
}

sub get_data_from_tree {
    my ($self, $tree) = @_;

    my $data;

    $tree->traverse(
        sub {
            my ($element) = @_;

            push @{$data},
              {
                id        => $element->getNodeValue()->{sid},
                name      => $element->getNodeValue()->{name},
                parent_id => $element->getParent()->getNodeValue()->{sid},
              };

        }
    );

    return $data;
}

sub get_full_article_tree {
    my ($self) = @_;

    my $tree = $self->_get_article_tree('full');

    return $tree;
}

sub get_real_article {
    my ($self) = @_;

    my $tree = $self->get_real_article_tree;

    return $self->get_data_from_tree($tree);
}

sub get_real_article_tree {
    my ($self) = @_;

    my $tree = $self->_get_article_tree('real');

    return $tree;
}

sub _balance_query {
    ["t_article"],
      "(t_article.hidden IS NULL OR t_article.hidden = 0) AND (t_article.status = 'A' OR t_article.status = 'N')";
}

sub _dict_table {$_[0]->partner_db->tns_dict_article}

sub _get_article_tree {
    my ($self, $type) = @_;

    my $levels;    # Arrayref. Уровни тематик, которые достанем из базы

    if ($type eq 'full') {
        $levels = [0, 1, 2, 3, 4];
    } elsif ($type eq 'real') {
        $levels = $REAL_ARTICLE_LEVELS;
    } else {
        throw gettext("Incorect type '%s'", $type);
    }

    my $last_update = $self->app->kv_store->get('update_tns_dict_article') // '';
    if (defined $self->{has}{$type}{date} && $self->{has}{$type}{date} eq $last_update) {
        return $self->{has}{$type}{tree};
    }

    # Получаем данные из базы
    my $articles_list = $self->partner_db->tns_dict_article->get_all(
        fields => ['sid', 'grid', 'name', 'lev'],
        filter => {'lev' => $levels},
        order_by => ['lev', 'name'],
    );

    throw gettext('Table "tns_dict_article" is empty') if @$articles_list == 0;

    # Создаем дерево

    # Тут хранятся все объексты дерева (ветви и листья).
    # Ключ - sid (айдишник) из базы, значение - объект Tree::Simple с
    # элементом дерева
    my %sids;

    if ($articles_list->[0]{'lev'} != 0) {
        throw gettext("First element should be on level 0");
    }

    # Корневой элемент дерева, sid = 0, name = 'ОСНОВНАЯ ГРУППА'
    my $tree = Tree::Simple->new(
        {
            sid  => $articles_list->[0]{'sid'},
            name => $articles_list->[0]{'name'},
        }
    );

    $sids{$articles_list->[0]{'sid'}} = $tree;

    foreach my $article (@{$articles_list}) {

        next if $article->{'lev'} == 0;

        if ($type eq 'real') {
            # В дереве, которое мы показываем пользователю не должно быть
            # раздела "АЛКОГОЛЬНЫЕ НАПИТКИ" и его подразделов
            my $alcohol_sid = 37;
            if (   ($article->{'sid'} == $alcohol_sid)
                or ($article->{'grid'} == $alcohol_sid))
            {
                next;
            }
        }

        my $element = Tree::Simple->new(
            {
                sid  => $article->{'sid'},
                name => $article->{'name'},
            }
        );

        $sids{$article->{'sid'}} = $element;

        if ($type eq 'real' and $article->{'lev'} == 2) {
            # В том случае, если мы создаем дерево, которые показываем
            # пользователей, то мы пропускаем уровень 1. В этом случае для
            # элементов уровня 2 у нас нет родителей и мы просто делаем их
            # детьми корня дерева
            $tree->addChild($element);
        } else {
            # Стандартное поведение. Значение grid у элемента в базе указывает
            # на его родителя
            $sids{$article->{'grid'}}->addChild($element);
        }

    }

    $self->{has}{$type} = {
        tree => $tree,
        date => $last_update,
    };

    return $tree;
}

sub reset_settings_on_blocks {
    my ($self, $lost_sids) = @_;

    if (@$lost_sids > $MAX_LOST_ARTICLES) {
        ERRORF 'Lost articles: %s', join(', ', @$lost_sids);
    } else {
        $self->app->articles->remove_lost_articles($lost_sids);
    }
}

TRUE;
