package Application::Model::TNSDict::Brand;

use qbit;

use base qw(Application::Model::TNSDict RestApi::MultistateModel);

use Utils::Logger qw(INFOF ERRORF);

use Exception::Validation::BadArguments;

my $MAX_LOST_BRANDS = 100;

sub accessor      {'tns_dict_brand'}
sub db_table_name {'tns_dict_brand'}

__PACKAGE__->model_accessors(product_manager => 'Application::Model::ProductManager',);

__PACKAGE__->register_rights(
    [
        {
            name        => 'tns_dict_brand',
            description => d_gettext('Right to manage brands'),
            rights      => {tns_dict_brand_view => d_gettext('Right to view brands'),}
        }
    ]
);

my $FIELDS_DEPENDS;

__PACKAGE__->model_fields(
    bid  => {db => TRUE, pk      => TRUE, default => TRUE},
    name => {db => TRUE, default => TRUE, i18n    => TRUE, type => 'string', api => 1},
    note      => {db => TRUE},
    status    => {db => TRUE},
    file_id   => {db => TRUE},
    hidden    => {db => TRUE},
    public_id => {
        depends_on => ['bid'],
        get        => sub {
            return $_[1]->{'bid'};
        },
    },
    fields_depends => {
        get => sub {
            $FIELDS_DEPENDS //= $_[0]->model->get_fields_depends();

            return $FIELDS_DEPENDS;
        },
        type => 'complex',
    },
    available_fields => {
        get => sub {
            {name => \1, available_fields => \1};
        },
        type => 'complex',
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        bid    => {type => 'number',  label => d_gettext('Brand ID')},
        name   => {type => 'text',    label => d_gettext('Type')},
        status => {type => 'text',    label => d_gettext('Status')},
        hidden => {type => 'boolean', label => d_gettext('Is hidden')},
    },
);

sub get_product_name {gettext('tns_dict_brand')}

sub query {
    my ($self, %opts) = @_;

    my $filter = $self->partner_db->filter($opts{'filter'});

    $filter->and([bid => '<>' => \1]);

    return $self->partner_db->query->select(
        table  => $self->_dict_table,
        fields => $opts{'fields'}->get_db_fields(),
        filter => $filter,
    );
}

#API

sub api_available_actions {
    return qw();
}

sub api_check_public_id {$_[1] =~ /^-?[0-9]+$/}

sub get_brand_suggest {
    my ($self, %opts) = @_;

    my $block_id  = $opts{'block_id'};
    my $substring = $opts{'substring'};
    my $limit     = $opts{'limit'};

    return [] if $substring eq "";

    my $safe_substring = $substring;
    $safe_substring =~ s/[^\w\d"'()\\\/\.,!:&#\-\+_ ]//g;
    $safe_substring = quotemeta($safe_substring);
    $safe_substring =~ s/_/\\_/g;

    return [] if $safe_substring eq "";

    my @block_brand_ids;

    if ($block_id) {
        my $block;
        foreach (@{$self->product_manager->get_page_model_accessors()}) {
            foreach my $block_model ($self->app->$_->get_block_models()) {
                my $id = $block_model->_split_id($block_id);
                next if ref($id) ne 'HASH';
                $block = $block_model->get($id, fields => ['brands']);
                last if $block;
            }
        }

        throw Exception::Validation::BadArguments gettext('Incorrect public_id %s', $block_id) unless $block;

        @block_brand_ids =
          map {$_->{'bid'}} defined($block->{'brands'}) ? @{$block->{'brands'}} : ();
    }

    my $query = $self->partner_db->query->select(
        table  => $self->_dict_table,
        fields => {bid => '', name => '', name_len => {LENGTH => ['name']}},
        filter => [
            'AND' => [
                [name => 'LIKE' => \"%$safe_substring%"],
                [bid  => '<>'   => \1],
                (
                    @block_brand_ids
                    ? [bid => 'NOT IN' => \\@block_brand_ids]
                    : ()
                )
            ]
        ],
    )->order_by('name_len', 'name');

    $query->limit($limit) if $limit;

    my $found = $query->get_all();

    $safe_substring =~ s/[её]/\[её\]/ig;
    $safe_substring =~ s/\\_/_/g;

    foreach my $element (@{$found}) {
        $element->{'name'} =~ m/^(.*)($safe_substring)(.*)$/i;

        $element->{'pre'}   = $1;
        $element->{'found'} = $2;
        $element->{'post'}  = $3;
        delete($element->{'name_len'});
    }

    return $found;
}

sub _balance_query {
    ['t_brand'], "(t_brand.hidden IS NULL OR t_brand.hidden = 0) AND (t_brand.status = 'A' OR t_brand.status = 'N')";
}

sub _dict_table {$_[0]->partner_db->tns_dict_brand}

sub reset_settings_on_blocks {
    my ($self, $lost_bids) = @_;

    if (@$lost_bids > $MAX_LOST_BRANDS) {
        ERRORF 'Lost brands: %s', join(', ', @$lost_bids);
    } else {
        my $brands_to_removed = $self->partner_db->brands->get_all(filter => {bid => \@$lost_bids});

        INFOF 'Removed following brands: %s', to_json($brands_to_removed);

        $self->partner_db->brands->delete($self->partner_db->filter({bid => \@$lost_bids}));
    }
}

TRUE;
