package Application::Model::TextTemplate;

use qbit;

use Template;

use base qw(Application::Model::Common);

use Exception::TextTemplate;
use Exception::DB::DuplicateEntry;
use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validation::BadArguments::EmptyValue;

sub accessor      {'text_template'}
sub db_table_name {'text_template'}

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB',);

__PACKAGE__->register_rights(
    [
        {
            name        => 'text_template',
            description => d_gettext('Right to manage TT2 templates'),
            rights      => {
                do_text_template_add  => d_gettext('Right to add TT2 templates'),
                text_template_view    => d_gettext('Right to view TT2 templates'),
                do_text_template_edit => d_gettext('Right to edit TT2 templates'),
            }
        }
    ]
);

__PACKAGE__->model_fields(
    id      => {default => TRUE, db => TRUE, type => 'number', pk => TRUE},
    key     => {default => TRUE, db => TRUE},
    caption => {default => TRUE, db => TRUE},
    content         => {default => TRUE, db => TRUE, i18n => TRUE},
    multistate      => {db      => TRUE},
    multistate_name => {
        depends_on => ['multistate'],
        get        => sub {
            $_[0]->model->get_multistate_name($_[1]->{'multistate'});
        },
    },
    actions => {
        depends_on => [qw(id multistate)],
        get        => sub {
            $_[0]->model->get_actions($_[1]);
        },
    },
    editable_fields => {
        depends_on => [qw(id multistate)],
        get        => sub {
            my %res = ();

            return \%res unless $_[0]->model->check_action($_[1], 'edit');

            $res{$_} = TRUE foreach (qw(caption content));

            return \%res;
          }
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id         => {type => 'number',     label => d_gettext('ID')},
        key        => {type => 'text',       label => d_gettext('Key')},
        caption    => {type => 'text',       label => d_gettext('Caption')},
        multistate => {type => 'multistate', label => d_gettext('Multistate')},
    },
);

__PACKAGE__->multistates_graph(
    empty_name    => 'new',
    multistates   => [[added => d_gettext('Added')],],
    right_actions => {
        add  => d_gettext('Add'),
        edit => d_gettext('Edit'),
    },
    right_group        => [text_template => d_gettext('Right to manage TT2 templates')],
    right_name_prefix  => 'text_template_',
    multistate_actions => [
        {
            action    => 'add',
            from      => '__EMPTY__',
            set_flags => ['added'],
        },
        {
            action => 'edit',
            from   => 'added',
        },
    ],
);

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        {name => 'id',         label => gettext('Template ID')},
        {name => 'key',        label => gettext('Key')},
        {name => 'caption',    label => gettext('Caption')},
        {name => 'multistate', label => gettext('Status')}
    ];
}

sub add {
    my ($self, %opts) = @_;

    throw Exception::Denied unless $self->check_rights($self->get_rights_by_actions('add'));

    foreach my $field_name (qw(caption content)) {
        throw Exception::Validation::BadArguments gettext("Missed required field '%s'", $field_name)
          unless $opts{$field_name};
    }

    $self->check_key($opts{'key'}) if exists($opts{'key'});

    foreach my $field_name (qw(content)) {
        foreach my $locale (keys %{$self->get_option('locales')}) {
            throw Exception::Validation::BadArguments gettext("Missed field '%s' in locale '%s'", $field_name, $locale)
              unless $opts{$field_name}{$locale};
        }
    }

    foreach my $locale (keys %{$self->get_option('locales')}) {
        $self->process_template($opts{'content'}{$locale});
    }

    my $id;
    try {
        $self->partner_db->transaction(
            sub {
                $id = $self->partner_db_table()->add({hash_transform(\%opts, [qw(key caption content)])});
                $self->do_action($id, 'add', %opts);
            }
        );
    }
    catch Exception::DB::DuplicateEntry with {
        throw Exception::Validation::BadArguments gettext("Template with such key already exists"),;
    };

    return $id;
}

sub on_action_edit {
    my ($self, $obj, %data) = @_;

    throw Exception::Denied unless $self->check_rights('do_text_template_edit');

    my $editable_fields = $self->get($obj, fields => ['editable_fields'])->{'editable_fields'};
    %data = hash_transform(\%data, [keys %$editable_fields]);

    return unless %data;

    foreach my $field_name (grep {exists($data{$_})} qw(content)) {
        throw Exception::Validation::BadArguments::EmptyValue gettext("Missed required field '%s'", $field_name)
          unless $data{$field_name};
    }

    if (exists($data{'content'})) {
        my $field_name = 'content';

        foreach my $locale (keys %{$self->get_option('locales')}) {
            $data{$field_name}{$locale} =~ s/^\s+|\s+$//g if $data{$field_name}{$locale};

            throw Exception::Validation::BadArguments gettext("Missed field '%s' in locale '%s'", $field_name, $locale)
              unless $data{$field_name}{$locale};

            $self->process_template($data{$field_name}{$locale});
        }
    }

    try {
        $self->partner_db_table()->edit($obj, \%data);
    }
    catch Exception::DB::DuplicateEntry with {
        throw Exception::Validation::BadArguments gettext("Template with such key already exists"),;
    };

    return FALSE;
}

sub query {
    my ($self, %opts) = @_;

    return $self->partner_db->query->select(
        table  => $self->partner_db_table,
        fields => $opts{'fields'}->get_db_fields(undef, all_locales => $opts{'all_locales'}),
        filter => $opts{'filter'}
    )->order_by('id');
}

sub check_key {
    my ($self, $value) = @_;

    throw Exception::Validation::BadArguments gettext("Invalid value in field '%s'.", 'key')
      unless $value =~ /^[a-zA-Z0-9 '"._]+$/;
}

sub d_get_content {
    my ($self, $key) = @_;

    return sub {
        $self->get_content($key);
    };
}

sub d_process_content {
    my ($self, $key, $vars, %opts) = @_;

    return sub {
        $self->process_content($key, $vars, %opts);
    };
}

sub get_content {
    my ($self, $key) = @_;

    throw Exception::Validation::BadArguments gettext("Missed required field '%s'", 'key') unless $key;
    my $obj = $self->get_all(fields => [qw(content)], filter => {key => $key})->[0]
      // throw Exception::Validation::BadArguments gettext('Template with key "%s" does not exist.', $key);

    my $content = $obj->{'content'};

    return $content;
}

sub process_content {
    my ($self, $key, $vars, %opts) = @_;

    my $tmp_locale;
    $tmp_locale = $self->app->set_tmp_app_locale($opts{'locale'}) if $opts{'locale'};

    throw Exception::Validation::BadArguments gettext("Missed required field '%s'", 'key') unless $key;
    my $obj = $self->get_all(fields => [qw(content)], filter => {key => $key})->[0]
      // throw Exception::Validation::BadArguments gettext('Template with key "%s" does not exist.', $key);

    my $content = $obj->{'content'};

    return $self->process_template($content, $vars, $obj);
}

sub process_template {
    my ($self, $content, $vars, $obj) = @_;
    $vars ||= {};
    $obj  ||= {};

    my $template = Template->new(
        INCLUDE_PATH => [$self->get_option('ApplicationPath') . 'mail_templates',],
        COMPILE_EXT  => '.ttc',
        COMPILE_DIR  => $self->app->get_option('TemplateCachePath')
          || '/tmp/tt_cache-mail'
          . (
            ($self->app->can('request') && blessed($self->app->request))
            ? '_' . $self->app->request->server_name() . '-' . $self->app->request->server_port()
            : ''
          )
          . "_$<",
        EVAL_PERL => 1,
        MINIMIZE  => $self->get_option('MinimizeTemplate'),
        (
            defined($obj->{'content_type'}) && $obj->{'content_type'} ne 'text/plain'
            ? (
                PRE_CHOMP  => 2,
                POST_CHOMP => 2,
                TRIM       => 2,
              )
            : ()
        ),
        RECURSION  => 1,
        PRE_DEFINE => {

            check_rights => sub {$self->check_rights(@_)},

            to_json => sub {
                return to_json(shift);
            },
            format_date => sub {
                return format_date(shift, shift, %{shift || {}});
            },
            format_number => sub {
                return format_number(shift, %{shift || {}});
            },
        },
    );

    my $out = '';
    $template->process(\$content, $vars, \$out)
      || throw Exception::TextTemplate gettext('Template process error [%s]', $template->error());

    return $out;
}

TRUE;
