package Application::Model::ThematicFilters;

=encoding UTF-8

=cut

use qbit;

use base qw(Application::Model::Common RestApi::MultistateModel);

use Utils::JSON qw(fix_type_for_complex);

sub accessor      {'thematic_filters'}
sub db_table_name {'thematic_filters'}

__PACKAGE__->register_rights(
    [
        {
            name        => 'thematic_filters',
            description => d_gettext('Rights to use thematic filters'),
            rights      => {
                thematic_filters_edit_field__caption_en => d_gettext('Right to edit caption_en'),
                thematic_filters_edit_field__caption_ru => d_gettext('Right to edit caption_ru'),
            }
        }
    ]
);

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB',);

__PACKAGE__->model_fields(
    id        => {default => TRUE, db => TRUE, pk   => TRUE, type => 'number', api => 1},
    caption   => {default => TRUE, db => TRUE, i18n => TRUE, type => 'string', api => 1},
    bk_name   => {default => TRUE, db => TRUE,},
    public_id => {
        db      => TRUE,
        db_expr => 'id',
        type    => 'string',
        api     => 1,
    },
    fields_depends => {
        depends_on => ['id'],
        get        => sub {
            {};
        },
        type => 'complex',
        api  => 1,
    },
    multistate      => {default => TRUE, db => TRUE, type => 'number', api => 1,},
    multistate_name => {
        depends_on => ['multistate'],
        get        => sub {
            $_[0]->model->get_multistate_name($_[1]->{'multistate'});
        },
        type => 'string',
        api  => 1,
    },
    actions => {
        label      => d_gettext('Actions'),
        depends_on => [qw(id multistate)],
        get        => sub {
            $_[0]->model->get_actions($_[1]);
        },
        type     => 'complex',
        fix_type => \&fix_type_for_complex,
        api      => 1,
    },
    available_fields => {
        depends_on => [qw(id)],
        label      => d_gettext('Available fields'),
        get        => sub {
            return $_[0]->model->get_available_fields($_[1]);
        },
        type     => 'complex',
        fix_type => \&fix_type_for_complex,
        api      => 1,
    },
    editable_fields => {
        forced_depends_on => [qw(id multistate)],
        get               => sub {
            return $_[0]->model->get_editable_fields($_[1]);
        },
        type     => 'complex',
        fix_type => \&fix_type_for_complex,
        api      => 1,
    },
    status => {
        depends_on => ['id'],
        get        => sub {
            'sinc'
        },
        type => 'string',
        api  => 1,
    },
);

__PACKAGE__->model_filter(
    db_accessor => 'partner_db',
    fields      => {
        id         => {type => 'number',     label => d_gettext('ID')},
        caption    => {type => 'text',       label => d_gettext('Short caption')},
        multistate => {type => 'multistate', label => d_gettext('Status')},
    },
);

__PACKAGE__->multistates_graph(
    empty_name    => d_gettext('Working'),
    multistates   => [[deleted => d_gettext('Deleted')]],
    right_actions => {
        edit   => d_gettext('Edit'),
        delete => d_gettext('Delete'),
        add    => d_gettext('Add'),
    },
    right_group        => [thematic_filters_action => d_gettext('Right to manage thematic filters')],
    right_name_prefix  => 'thematic_filters_action_',
    multistate_actions => [
        {
            action    => 'delete',
            from      => 'not deleted',
            set_flags => ['deleted']
        },
        {
            action => 'add',
            from   => '__EMPTY__',
        },
        {
            action => 'edit',
            from   => 'not deleted'
        },

    ],
);

sub get_product_name {
    gettext('thematic_filters');
}

sub get_available_fields {
    my ($self, $obj) = @_;
    if ($obj && $self->check_multistate_flag($obj->{'multistate'}, 'deleted')) {
        return {};
    }

    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub get_add_fields {
    my ($self, %opts) = @_;
    my %fields;
    foreach my $field (qw(caption_en caption_ru)) {
        $fields{$field} = TRUE if $self->check_short_rights("edit_field__$field");
    }

    return \%fields;
}

sub get_editable_fields {
    my ($self, $obj) = @_;

    return {} unless $self->check_action($obj, 'edit');

    my %res;

    foreach (qw(caption)) {
        $res{$_} = TRUE if $self->check_short_rights('edit_field__' . $_);
    }

    return \%res;
}

sub add {
    my ($self, %opts) = @_;
    throw Exception::Denied gettext('Access denied')
      unless $self->check_rights($self->get_rights_by_actions('add'));

    my $add_fields = $self->get_add_fields();
    my @missed_req_fields = grep {!length($opts{$_})} keys(%{$add_fields});
    throw Exception::Validation::BadArguments gettext('Missed required fields: %s', join(', ', @missed_req_fields))
      if @missed_req_fields;

    my @extra_fields = grep {!$add_fields->{$_}} keys(%opts);
    throw Exception::Validation::BadArguments gettext('Not right to add fields: %s', join(', ', @extra_fields))
      if @extra_fields;

    my $id;
    my $locales = $self->get_option('locales', {});
    $opts{'caption'} = {map {$_ => $opts{"caption_$_"}} keys(%$locales)};
    delete($opts{"caption_$_"}) for (keys(%$locales));
    $self->partner_db->transaction(
        sub {
            $id = $self->partner_db_table()->add(\%opts);
            $self->do_action($id, 'add');
        }
    );
    return $id;
}

sub on_action_edit {
    my ($self, $obj, %opts) = @_;
    my $editable_fields = $self->get_editable_fields($obj);
    my @missed_req_fields = grep {!length($opts{$_})} keys(%{$editable_fields});
    throw Exception::Validation::BadArguments gettext('Missed required fields: %s', join(', ', @missed_req_fields))
      if @missed_req_fields;

    my @extra_fields = grep {!$editable_fields->{$_}} keys(%opts);
    throw Exception::Validation::BadArguments gettext('Not right to add fields: %s', join(', ', @extra_fields))
      if @extra_fields;

    if (%opts) {
        $self->partner_db_table->edit($obj, \%opts);
    }

    return TRUE;
}

sub api_available_actions {
    return qw(delete edit);
}

TRUE;
