package Application::Model::UserNotifications;

use qbit;

use base qw(
  Application::Model::Common
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  );

consume qw(
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::CreateDate
  );

use Utils::Logger qw(INFOF);
use PiConstants qw($NOTIFICATION_FNS);

sub accessor             {'user_notifications'}
sub db_table_name        {'user_notifications'}
sub get_product_name     {gettext('user_notifications')}
sub get_opts_schema_name {'user_notifications_opts'}

sub get_structure_model_accessors {
    return {
        notification => 'Application::Model::Notification',
        partner_db   => 'Application::Model::PartnerDB',
        queue        => 'Application::Model::Queue',
        users        => 'Application::Model::Users',
    };
}

sub get_structure_rights_to_register {
    return [
        {
            name        => 'user_notifications',
            description => d_gettext('Right to manage user_notifications'),
            rights      => {
                user_notifications_add => d_gettext('Right to add records to user_notifications'),
                user_notifications_add_other =>
                  d_gettext('Right to add records to user_notifications for another user'),
                user_notifications_view_all => d_gettext('Right to view all records in user_notifications'),
            }
        }
    ];
}

sub get_structure_model_fields {
    return {
        id => {
            default => TRUE,
            db      => TRUE,
            pk      => TRUE,
            label   => d_gettext('ID'),
            type    => 'number',
            api     => 1,
        },
        public_id => {
            db      => TRUE,
            db_expr => 'id',
            type    => 'string',
            api     => 1
        },
        notification_id => {
            db         => TRUE,
            label      => d_gettext('Notification ID'),
            type       => 'number',
            need_check => {type => 'int_un'},
            api        => 1,
        },
        user_id => {
            db         => TRUE,
            label      => d_gettext('User ID'),
            type       => 'number',
            need_check => {type => 'int_un'},
            api        => 1,
        },
        multistate =>
          {db => TRUE, label => d_gettext('Status'), type => 'number', need_check => {type => 'int_un'}, api => 1},
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1
        },
        # enum(custom;auto) custom- дефолт
        type => {
            default    => TRUE,
            label      => d_gettext('Notification type'),
            type       => 'string',
            api        => 1,
            depends_on => ['notification_id', 'notifications.id', 'notifications.type'],
            get        => sub {
                return $_[0]->{'notifications'}{$_[1]->{'notification_id'}}->{type};
            },
        },
        # default;common_offer;custom_link;survey;top
        view_type => {
            default    => TRUE,
            label      => d_gettext('Notification layout'),
            type       => 'string',
            api        => 1,
            depends_on => ['notification_id', 'notifications.id', 'notifications.view_type'],
            get        => sub {
                return $_[0]->{'notifications'}{$_[1]->{'notification_id'}}->{view_type};
            },
        },
        icon_id => {
            default    => TRUE,
            label      => d_gettext('Icon ID'),
            type       => 'string',
            api        => 1,
            depends_on => ['notification_id', 'notifications.id', 'notifications.icon_id'],
            get        => sub {
                return $_[0]->{'notifications'}{$_[1]->{'notification_id'}}->{icon_id};
            },
        },
        url => {
            default    => TRUE,
            label      => d_gettext('URL'),
            type       => 'string',
            api        => 1,
            depends_on => ['notification_id', 'notifications.id', 'notifications.url'],
            get        => sub {
                return $_[0]->{'notifications'}{$_[1]->{'notification_id'}}->{url};
            },
        },
        caption => {
            default    => TRUE,
            label      => d_gettext('Caption'),
            type       => 'string',
            api        => 1,
            depends_on => ['notification_id', 'notifications.id', 'notifications.caption', 'custom_data'],
            get        => sub {
                my $msg_template = $_[0]->{'notifications'}{$_[1]->{'notification_id'}}->{caption};
                return sprinttmpl($msg_template, custom_data => $_[1]->{custom_data},);
            },
        },
        message => {
            default => TRUE,
            label   => d_gettext('Notification text'),
            type    => 'string',
            api     => 1,
            depends_on =>
              ['notification_id', 'notifications.id', 'notifications.message', 'custom_data', 'items', 'accessor'],
            get => sub {
                my $msg_template = $_[0]->{'notifications'}{$_[1]->{'notification_id'}}->{message};
                # [% custom_url %]
                # [% item_url %]
                return sprinttmpl(
                    $msg_template,
                    custom_data => $_[1]->{custom_data},
                    accessor    => $_[1]->{accessor},
                    items       => $_[1]->{items}
                );
            },
        },
        button_caption => {
            default    => TRUE,
            label      => d_gettext('Button caption'),
            type       => 'string',
            api        => 1,
            depends_on => ['notification_id', 'notifications.id', 'notifications.button_caption'],
            get        => sub {
                return $_[0]->{'notifications'}{$_[1]->{'notification_id'}}->{button_caption};
            },
        },
        expectation_caption => {
            default    => TRUE,
            label      => d_gettext('Expectation caption'),
            type       => 'string',
            api        => 1,
            depends_on => ['notification_id', 'notifications.id', 'notifications.expectation_caption'],
            get        => sub {
                return $_[0]->{'notifications'}{$_[1]->{'notification_id'}}->{expectation_caption};
            },
        },
        estimated_value => {
            default    => TRUE,
            label      => d_gettext('Estimated value'),
            type       => 'number',
            api        => 1,
            depends_on => ['notification_id', 'notifications.id', 'notifications.expected_growth'],
            get        => sub {
                return $_[0]->{'notifications'}{$_[1]->{'notification_id'}}->{expected_growth};
            },
        },
        fields_depends => {
            depends_on => ['id'],
            get        => sub {
                {};
            },
            type => 'complex',
            api  => 1,
        },
        accessor => {
            default    => TRUE,
            db         => TRUE,
            label      => d_gettext('Model root for links'),
            type       => 'string',
            api        => 1,
            need_check => {optional => TRUE,}
        },
        items => {
            label      => d_gettext('Model IDs for links'),
            from_opts  => 'from_hash',
            need_trim  => TRUE,
            type       => 'array',
            sub_type   => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'array',
            }
        },
        custom_data => {
            label      => d_gettext('Custom user link'),
            from_opts  => 'from_hash',
            type       => 'complex',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'hash',
                extra    => TRUE,
            }
        },
        short_name => {
            default    => TRUE,
            label      => d_gettext('Short name'),
            type       => 'string',
            api        => 1,
            depends_on => ['notification_id', 'notifications.id', 'notifications.short_name'],
            get        => sub {
                return $_[0]->{'notifications'}{$_[1]->{'notification_id'}}->{short_name};
            },
        },
    };
}

sub related_models {
    my ($self) = @_;

    return {
        notifications => {
            accessor     => 'notification',
            grant_rights => ['notification_view'],
            filter       => sub {
                +{id => array_uniq(map {$_->{'notification_id'} // ()} @{$_[1]})};
            },
            key_fields => ['id'],
        }
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            id => {db => TRUE, type => 'number',},
            (
                $self->check_short_rights('add_other')
                ? (
                    user_id         => {db => TRUE, type => 'number',},
                    notification_id => {db => TRUE, type => 'number',},
                  )
                : ()
            ),
            login      => {type => 'text',},
            multistate => {db   => TRUE, type => 'number',},
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    my @f;
    if ($self->check_short_rights('add_other')) {
        push @f, {name => 'user_id', label => gettext('User ID')};
    }

    return \@f;
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name    => 'new',
        multistates   => [[deleted_by_user => d_gettext('Deleted by user')], [viewed => d_gettext('Viewed')],],
        right_actions => {
            add        => d_gettext('Add'),
            delete     => d_gettext('Delete'),
            set_viewed => d_gettext('Set Viewed'),
        },
        right_group        => [user_notifications => d_gettext('Rights for user notifications')],
        right_name_prefix  => 'user_notifications_',
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__',
            },
            {
                action    => 'set_viewed',
                from      => 'not (deleted_by_user or viewed)',
                set_flags => ['viewed'],
            },
            {
                action    => 'delete',
                from      => 'not deleted_by_user',
                set_flags => ['deleted_by_user'],
            }
        ],
    };
}

sub get_actions_depends {
    [
        qw(
          id
          multistate
          )
    ];
}

sub get_available_fields {
    my ($self, $obj) = @_;

    my %fields = map {$_ => TRUE} keys(%{$self->get_model_fields});

    return \%fields;
}

sub get_available_fields_depends {[qw(id)]}

sub get_add_fields {
    my ($self) = @_;

    my $res = {};
    foreach (
        qw(
        notification_id
        user_id
        accessor
        items
        custom_data
        )
      )
    {
        $res->{$_} = TRUE;
    }

    return $res;
}

sub query_filter {
    my ($self, $filter) = @_;

    return $self->check_short_rights('view_all') ? $filter : $self->limit_filter_by_user($filter);
}

sub hook_writing_to_database {
    my ($self, $opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights('user_notifications_view_all');
    $self->SUPER::hook_writing_to_database($opts);
}

sub on_action_delete {
    my ($self, $obj) = @_;

    INFOF('user login[%s] has dismissed user_notifications id[%s]', $self->get_option('cur_user')->{login}, $obj->{id});
}

sub on_action_set_viewed {
    my ($self, $obj) = @_;

    my $notification = $self->get($obj, fields => [qw(user_id short_name custom_data)]);
    if ($notification->{'short_name'} eq $NOTIFICATION_FNS) {
        my $user = $self->users->get({id => $notification->{user_id}}, fields => [qw(inn)]);
        my $tmp_rights = $self->app->add_tmp_rights('queue_add_mark_read_fns_notification');
        $self->queue->add(
            method_name => 'mark_read_fns_notification',
            params      => {inn => $user->{'inn'}, notification_id => [$notification->{'custom_data'}->{'id'}]},
        );
    }
}

sub api_available_actions {
    return qw(delete set_viewed);
}

sub get_unread_count {
    my ($self, $user_id_list) = @_;

    my $no_viewed = $self->get_multistate_by_action('set_viewed');

    my $data = $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => {
            user_id => '',
            unread  => {count => [\'*']},
        },
        filter => ['AND', [[user_id => 'IN' => \$user_id_list], ['multistate', 'IN', \$no_viewed]],],
    )->group_by('user_id')->get_all();

    return {map {$_->{'user_id'} => $_} @$data};
}

sub throw_error_by_action {
    my ($self, $object, $action) = @_;

    if ($action eq 'set_viewed' && $self->check_multistate_flag($object->{'multistate'}, 'deleted_by_user')) {
        throw Exception::Multistate::BadAction
          gettext('You cannot set_viewed this notification because are archived'),
          dont_send_to_sentry => 1,
          ;
    }

    return FALSE;
}

TRUE;
