package Application::Model::Users;

use qbit;
use QBit::Validator;

use base qw(
  Application::Model::Common
  Application::Model::Product
  RestApi::MultistateModel
  Application::Model::ValidatableMixin
  );

consume qw(
  Application::Model::Role::Has::EditRequisites
  Application::Model::Role::Has::ModerationReason::Users
  );

use Digest::MD5 qw(md5_hex);
use File::Basename;

use List::Util qw(any);
use Utils::JSON qw(fix_type_for_complex get_boolean_representation);
use Utils::Logger qw(ERROR WARNF);
use Utils::PublicID;

use Application::Model::AgreementChecker::_Agreement;

use Exception::API::AdFoxGraphQL;
use Exception::API::Payoneer;
use Exception::API::Payoneer::PayeeNotFound;
use Exception::Balance::IncorrectAnswer;
use Exception::Balance::NotFound;
use Exception::CommonOffer;
use Exception::Conflict;
use Exception::DB::DuplicateEntry;
use Exception::Denied;
use Exception::Validation::BadArguments;
use Exception::Validator::Fields;
use PiConstants qw(
  @ADFOX_PRESET_PAIDPRODUCTS
  $ADINSIDE_USER_ID
  @AVAILABLE_COOPERATION_FORMS
  $COMMON_OFFER_DOCS
  $CONTRACT_OFFER_ERROR_MAIL
  %CONTRACT_TYPE
  $DESIGN_TYPES
  $DOCS_MAIL
  $DOCS_PROJECT_MAIL
  $DSP_MEDIA_TYPE_ID
  $FIRM_ID_YANDEX_EUROPE_AG
  $FIRM_ID_YANDEX_LTD
  $PARTNER2_COMMON_OFFER_ERRORS
  $PARTNER2_CRON_MAIL
  $SELFEMPLOYED_COOPERATION_FORM
  @SELFEMPLOYED_STATUSES
  $SEPARATE_CPM_STRATEGY_ID
  $TUTBY_COP_MAIL
  $DEFAULT_CPM_CURRENCY
  $CPM_AVAILABLE_CURRENCIES
  :STAFF
  :ROLES
  $GAME_OFFER_SERVICE
  $GAME_OFFER_MANAGER
  $PAYMENT_TYPES
  $SELFEMPLOYED_STATUS_READY
  );

my $TYPES = {
    1 => {
        db          => 'is_tutby',
        label       => d_gettext('Partner TUT.BY'),
        short_right => 'view_field__is_tutby',
    },
    2 => {
        db          => 'is_mobile_mediation',
        label       => d_gettext('Mobile mediator'),
        short_right => 'view_field__is_mobile_mediation',
    },
    3 => {
        db    => 'is_adfox_partner',
        label => d_gettext('ADFOX user'),
    },
    4 => {
        db          => 'is_video_blogger',
        label       => d_gettext('Video blogger'),
        short_right => 'view_field__is_video_blogger',
    },
    5 => {
        db          => 'is_games',
        label       => d_gettext('Games'),
        short_right => 'view_field__is_games',
    },
    6 => {
        db          => 'is_efir_blogger',
        label       => d_gettext('Efir blogger'),
        short_right => 'view_field__is_efir_blogger',
    },
};

my $CONTRACTOR_TYPES = {
    1 => 'individual_entrepreneur',
    2 => 'individual',
    3 => 'selfemployed',
    4 => 'entity',
};

sub accessor             {'users'}
sub db_table_name        {'users'}
sub get_opts_schema_name {'users_opts'}
sub get_product_name     {gettext('users')}

sub get_structure_model_accessors {
    my ($self) = @_;

    return {
        all_pages                         => 'Application::Model::AllPages',
        api_adfox                         => 'Application::Model::API::Yandex::AdFox',
        api_adfox_graphql                 => 'Application::Model::API::Yandex::AdFoxGraphQL',
        api_balalayka                     => 'Application::Model::API::Yandex::Balalayka',
        api_balance                       => 'QBit::Application::Model::API::Yandex::Balance',
        api_blackbox                      => 'QBit::Application::Model::API::Yandex::BlackBox',
        api_http_banner_storage           => 'Application::Model::API::Yandex::BannerStorage::HTTP',
        api_media_storage_s3              => 'Application::Model::API::Yandex::MediaStorage::S3',
        api_yamoney                       => 'Application::Model::API::Yandex::YaMoney',
        business_rules                    => 'Application::Model::BusinessRules',
        currency                          => 'Application::Model::Currency',
        design_templates                  => 'Application::Model::DesignTemplates',
        documents                         => 'Application::Model::Documents',
        dsp                               => 'Application::Model::DSP',
        geo_base                          => 'Application::Model::GeoBase',
        internal_context_on_site_campaign => 'Application::Model::Product::InternalAN::InternalContextOnSite::Campaign',
        internal_search_on_site_campaign  => 'Application::Model::Product::InternalAN::InternalSearchOnSite::Campaign',
        inviter                           => 'Application::Model::Inviter',
        mail_notification                 => 'Application::Model::MailNotification',
        mailer                            => 'Application::Model::SendMail',
        managers                          => 'Application::Model::Managers',
        partner_db                        => 'Application::Model::PartnerDB',
        queue                             => 'Application::Model::Queue',
        rbac                              => 'Application::Model::RBAC',
        resources                         => 'Application::Model::Resources',
        user_features                     => 'Application::Model::Users::Features',
        user_global_excluded_domains      => 'Application::Model::Users::GlobalExcludedDomains',
        user_global_excluded_phones       => 'Application::Model::Users::GlobalExcludedPhones',
        user_notifications                => 'Application::Model::UserNotifications',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    return [
        {
            name        => 'users',
            description => sub {gettext('Right to manage users')},
            rights      => {
                users_edit_field__features          => d_gettext('Right to edit user\'s features'),
                users_view                          => d_gettext('Right to view users'),
                users_view_all                      => d_gettext('Right to view all users'),
                users_view_field__login             => d_gettext('Right to view field "Login"'),
                users_view_field__email             => d_gettext('Right to view field "E-mail"'),
                users_view_field__client_id         => d_gettext('Right to view field "Client id"'),
                users_view_field__contract_id       => d_gettext('Right to view field "Contract id"'),
                users_view_field__multistate        => d_gettext('Right to view field "Status"'),
                users_view_field__roles             => d_gettext('Right to view field "Roles"'),
                users_view_field__user_id           => d_gettext('Right to view field "User id"'),
                users_view_all_fields               => d_gettext('Right to view all fields'),
                users_view_all_developers           => d_gettext('Right to view all developers'),
                users_view_all_an_internal_managers => d_gettext('Right to view all internal managers'),
                users_view_all_dsp_partners         => d_gettext('Right to view all DSP partners'),
                users_view_all_dsp_managers         => d_gettext('Right to view all DSP managers'),
                users_view_all_an_partners          => d_gettext('Right to view all AN partners'),
                users_view_all_an_managers          => d_gettext('Right to view all AN managers'),
                users_view_all_video_an_partners    => d_gettext('Right to view all Video AN partners'),
                users_view_all_indoor_an_partners   => d_gettext('Right to view all Indoor AN partners'),
                users_view_all_outdoor_an_partners  => d_gettext('Right to view all Outdoor AN partners'),
                users_view_all_mobile_an_partners   => d_gettext('Right to view all Mobile AN partners'),
                users_view_all_adblock_an_partners  => d_gettext('Right to view all adblock AN partners'),
                view_search_filters__user_type      => d_gettext('Right to view search filter "user_type"'),
                map {$_ => $_}
                  qw(
                  users_edit_all
                  users_edit_field__allowed_design_auction_native_only
                  users_edit_field__client_id
                  users_edit_field__content_block_edit_template_allowed
                  users_edit_field__currency_rate
                  users_edit_field__current_currency
                  users_edit_field__domain_login
                  users_edit_field__has_approved
                  users_edit_field__has_approved_app
                  users_edit_field__has_approved_site
                  users_edit_field__has_common_offer
                  users_edit_field__has_tutby_agreement
                  users_edit_field__is_dm_lite
                  users_edit_field__is_efir_blogger
                  users_edit_field__is_games
                  users_edit_field__is_mobile_mediation
                  users_edit_field__is_tutby
                  users_edit_field__is_video_blogger
                  users_edit_field__need_to_email_processing
                  users_edit_field__next_currency
                  users_view_all_tutby_partners
                  users_view_field__current_currency
                  users_view_field__domain_login
                  users_view_field__has_common_offer
                  users_view_field__has_mobile_mediation
                  users_view_field__has_rsya
                  users_view_field__has_tutby_agreement
                  users_view_field__is_dm_lite
                  users_view_field__is_efir_blogger
                  users_view_field__is_games
                  users_view_field__is_mobile_mediation
                  users_view_field__is_tutby
                  users_view_field__is_video_blogger
                  users_view_field__need_to_email_processing
                  )
            },
        }
    ];
}

sub get_structure_model_fields {
    my ($self) = @_;

    my $FIELDS_DEPENDS;

    return {
        id    => {db => TRUE, type => 'number', api => 1, adjust_type => 'str', pk => 1,},
        uid   => {db => TRUE, type => 'number', api => 1, adjust_type => 'str',},
        login => {db => TRUE, type => 'string', api => 1, default     => 1,},
        name  => {
            default    => 1,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'scalar',
                len_max  => 255,
            }
        },
        lastname => {
            default    => 1,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'scalar',
                len_max  => 255,
            }
        },
        midname => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'scalar',
                len_max  => 255
            }
        },
        email => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                check => sub {
                    my ($qv, $tag) = @_;

                    my @emails = split(/\s*,\s*/, $qv->data->{'email'});

                    foreach my $email (@emails) {
                        throw Exception::Validator::Fields gettext('Incorrect E-Mail') if !check_email($email);
                    }

                    return FALSE;
                  }
            },
        },
        accountant_email => {db => TRUE, need_check => {optional => TRUE}},
        newsletter       => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean', optional => TRUE},
            adjust_type => 'str',
        },
        phone => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                type     => 'scalar',
                len_max  => 255,
                optional => TRUE,
            }
        },
        has_approved => {
            from_opts  => 'db_generated',
            type       => 'boolean',
            api        => 0,
            need_check => {type => 'boolean', optional => TRUE}
        },
        has_approved_site => {
            from_opts  => 'from_hash',
            type       => 'boolean',
            api        => 0,
            need_check => {type => 'boolean', optional => TRUE}
        },
        has_approved_app => {
            from_opts  => 'from_hash',
            type       => 'boolean',
            api        => 0,
            need_check => {type => 'boolean', optional => TRUE}
        },
        need_to_email_processing => {
            db           => TRUE,
            check_rights => 'users_view_field__need_to_email_processing',
            type         => 'boolean',
            api          => 0,
            need_check   => {type => 'boolean', optional => TRUE}
        },
        client_id => {
            db           => TRUE,
            check_rights => 'users_view_field__client_id',
            type         => 'number',
            api          => 1,
            need_check   => {type => 'int_un'},
            adjust_type  => 'str',
        },
        no_stat_monitoring_emails => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean', optional => TRUE},
            adjust_type => 'str',
        },
        multistate => {db => TRUE, check_rights => 'users_view_field__multistate', type => 'number', api => 1},
        country_id => {db => TRUE,},
        business_unit =>
          {db => TRUE, check_rights => 'users_view_field__client_id', type => 'boolean', adjust_type => 'str',},
        is_tutby => {
            db           => TRUE,
            check_rights => 'users_view_field__is_tutby',
            type         => 'boolean',
            api          => 1,
            need_check   => {type => 'boolean', optional => TRUE},
            adjust_type  => 'str',
        },
        has_tutby_agreement => {
            from_opts    => 'db_generated',
            check_rights => 'users_view_field__has_tutby_agreement',
            type         => 'boolean',
            api          => 1,
            need_check   => {type => 'boolean', optional => TRUE}
        },
        block_light_form_enabled => {
            db          => 'u',
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean', optional => TRUE},
            adjust_type => 'str',
        },
        has_common_offer => {
            from_opts  => 'db_generated',
            type       => 'boolean',
            need_check => {type => 'boolean', optional => TRUE}
        },
        has_mobile_mediation => {
            from_opts  => 'db_generated',
            type       => 'boolean',
            api        => 1,
            need_check => {
                type  => 'boolean',
                check => sub {
                    my ($qv, $tag) = @_;
                    throw Exception::Validator::Fields gettext('Incorrect has_mobile_mediation')
                      if $qv->data->{'is_mobile_mediation'} && !$qv->data->{'has_mobile_mediation'};
                },
                optional => TRUE,
            }
        },
        has_rsya => {
            from_opts  => 'db_generated',
            type       => 'boolean',
            api        => 1,
            need_check => {
                type  => 'boolean',
                check => sub {
                    my ($qv, $tag) = @_;
                    throw Exception::Validator::Fields gettext('Incorrect has_rsya')
                      if !$qv->data->{'is_mobile_mediation'} && !$qv->data->{'has_rsya'};
                },
                optional => TRUE
            }
        },
        is_games => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean', optional => TRUE},
            adjust_type => 'str',
        },
        is_mobile_mediation => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean', optional => TRUE},
            adjust_type => 'str',
        },
        is_video_blogger => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean', optional => TRUE},
            adjust_type => 'str',
        },
        is_efir_blogger => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean', optional => TRUE},
            adjust_type => 'str',
        },
        is_adfox_partner => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean', optional => TRUE},
            adjust_type => 'str',
        },
        adfox_offer => {
            from_opts  => 'db_expr',
            type       => 'boolean',
            api        => 1,
            need_check => {
                type     => 'boolean',
                optional => TRUE,
                check    => sub {
                    my ($qv, $value, $template) = @_;

                    # Проверка только для редактирования
                    my $hs = $qv->app->hook_stash;
                    if ($hs->inited && $hs->mode('edit')) {
                        my $current          = $hs->get('current');
                        my $adfox_offer      = $current->{adfox_offer} // 0;
                        my $paid_offer       = $current->{paid_offer};
                        my $has_common_offer = $current->{has_common_offer};
                        my $adfox_info       = $current->{adfox_info};

                        # Ошибка при изменении на TRUE и отсутствии флагов, разрешающих это
                        # Ошибка при изменении на FALSE, если уже было TRUE
                        throw Exception::Validator::Fields gettext('Incorrect offerta field')
                          if (
                            $value != $adfox_offer
                            && ($value && !($paid_offer && $has_common_offer && @$adfox_info)
                                || !$value && $adfox_offer)
                             );
                    }
                },
            }
        },
        paid_offer => {
            from_opts  => 'db_expr',
            type       => 'boolean',
            api        => 1,
            need_check => {type => 'boolean', optional => TRUE}
        },
        domain_login => {db => TRUE},
        public_id    => {
            depends_on => ['id'],
            get        => sub {
                return $_[1]->{'id'};
            },
            type => 'string',
        },
        multistate_name => {
            depends_on   => ['multistate'],
            check_rights => 'users_view_field__multistate',
            label        => d_gettext('Multistate name'),
            get          => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1,
        },
        roles => {
            depends_on   => [qw(id)],
            check_rights => 'users_view_field__roles',
            get          => sub {
                $_[0]->{'__USER_ROLES__'}{$_[1]->{'id'}} // [];
            },
            type     => 'complex',
            fix_type => sub {
                my ($model, $value) = @_;

                foreach my $row (@$value) {
                    $row->{'id'} += 0;
                    $row->{'is_internal'} = get_boolean_representation($row->{'is_internal'});
                    $row->{'group'} += 0;
                    $row->{'viewer'} = get_boolean_representation($row->{'viewer'});
                }

                return $value;
            },
            api        => 1,
            need_check => {type => 'array', optional => TRUE},
        },
        full_name => {
            depends_on => [qw(name midname lastname)],
            get        => sub {
                join(' ', map {$_[1]->{$_}} grep {$_[1]->{$_} =~ /\S/} qw(lastname name midname)) || gettext('No name');
            },
            type => 'string',
            api  => 1,
        },
        active_contract => {
            forced_depends_on => [qw(login client_id)],
            get               => sub {
                return $_[0]->{__ACTIVE_CONTRACTS__}{$_[1]->{client_id}};
            },
            type => 'complex',
            api  => 1,
        },
        actions => {
            forced_depends_on => [qw(id multistate roles)],
            get               => sub {
                $_[0]->model->get_actions($_[1]);
            },
            type => 'complex',
            api  => 1,
        },
        adfox_info => {
            depends_on => ['id'],
            get        => sub {
                $_[0]->{'__USER_ADFOX__'}{$_[1]->{'id'}} // [];
            },
            type     => 'complex',
            fix_type => sub {
                my ($model, $value) = @_;

                for (@$value) {
                    $_->{'id'} += 0;
                    $_->{'login'}       .= '';
                    $_->{'create_date'} .= '';
                }

                return $value;
            },
            api        => 1,
            need_check => {skip => TRUE,},
        },
        excluded_domains => {
            depends_on => ['id', 'user_global_excluded_domains.domain'],
            get        => sub {
                $_[0]->{'user_global_excluded_domains'}{$_[1]->{'id'}} // [];
            },
            type       => 'array',
            sub_type   => 'string',
            api        => 1,
            need_check => {
                type     => 'array',
                optional => TRUE,
                all      => {type => 'domain',},
            },
        },
        excluded_phones => {
            depends_on => ['id', 'user_global_excluded_phones.phone'],
            get        => sub {
                $_[0]->{'user_global_excluded_phones'}{$_[1]->{'id'}} // [];
            },
            type       => 'array',
            sub_type   => 'string',
            api        => 1,
            need_check => {
                type     => 'array',
                optional => TRUE,
                all      => {type => 'phone',},
            },
        },
        features => {
            depends_on => ['id'],
            get        => sub {
                $_[0]->{'__USER_FEATURES__'}{$_[1]->{'id'}} // [];
            },
            type       => 'array',
            sub_type   => 'string',
            api        => 1,
            need_check => {
                type     => 'array',
                optional => TRUE,
                check    => sub {
                    my ($qv, $value, $template) = @_;

                    my $hs = $qv->app->hook_stash;
                    if ($hs->inited && $hs->mode('edit')) {
                        my $user_id = $hs->get('id')->{id};

                        my $current = $hs->get('current');

                        my %current_features;
                        my %new_features = (map {$_ => TRUE} @$value);

                        foreach my $feature (@{$current->{features}}) {
                            $current_features{$feature} = TRUE;

                            unless ($new_features{$feature}) {
                                my $method = 'can_delete_feature_' . $feature;
                                if ($qv->app->can($method)) {
                                    my ($can_del, $msg) = $qv->app->$method($user_id, $qv->data);

                                    unless ($can_del) {
                                        $msg //= gettext("Can't delete feature: %s", $feature);
                                        throw Exception::Validator::Fields $msg;
                                    }
                                }
                            }
                        }

                        foreach my $feature (@$value) {
                            unless ($current_features{$feature}) {
                                my $method = 'can_add_feature_' . $feature;
                                if ($qv->app->can($method)) {
                                    my ($can_add, $msg) = $qv->app->$method($user_id, $qv->data);

                                    unless ($can_add) {
                                        $msg //= gettext("Can't add feature: %s", $feature);
                                        throw Exception::Validator::Fields $msg;
                                    }
                                }
                            }
                        }
                    }
                },
            },
        },
        business_rules_count => {
            depends_on => ['id'],
            label      => d_gettext('Amount of business rules'),
            get        => sub {
                $_[0]->{'__BUSINESS_RULES_COUNT__'}{$_[1]->{'id'}}->{'all'} // 0;
            },
            type => 'number',
            api  => 1,
        },
        business_rules_active_count => {
            depends_on => ['id'],
            label      => d_gettext('Amount of active business rules'),
            get        => sub {
                $_[0]->{'__BUSINESS_RULES_COUNT__'}{$_[1]->{'id'}}->{'active'} // 0;
            },
            type => 'number',
            api  => 1,
        },
        available_fields => {
            forced_depends_on => [qw(multistate roles is_mobile_mediation)],
            label             => d_gettext('Available fields'),
            get               => sub {
                return $_[0]->model->get_available_fields($_[0], $_[1]);
            },
            type        => 'complex',
            fix_type    => \&fix_type_for_complex,
            api         => 1,
            adjust_type => 'hash_int',
        },
        editable_fields => {
            forced_depends_on => [
                qw(id multistate roles is_tutby paid_offer has_common_offer adfox_info features current_currency next_currency)
            ],
            get => sub {
                return $_[0]->model->get_editable_fields($_[1]);
            },
            type        => 'complex',
            fix_type    => \&fix_type_for_complex,
            api         => 1,
            adjust_type => 'hash_int',
        },
        lang => {
            forced_depends_on => [qw(id)],
            get               => sub {
                return $_[0]->{'__LANG__'}{$_[1]->{'id'}} // 'ru';
            },
            type => 'string',
        },
        avatar => {
            forced_depends_on => [qw(id)],
            get               => sub {
                return $_[0]->{'__AVATAR__'}{$_[1]->{'id'}} // '0/0-0';
            },
            type => 'string',
            api  => 1,
        },
        fields_depends => {
            get => sub {
                $FIELDS_DEPENDS //= $_[0]->model->get_fields_depends();

                return $FIELDS_DEPENDS;
            },
            type => 'complex',
        },
        status => {
            depends_on => ['id'],
            get        => sub {
                'sinc'
            },
            type => 'string',
            api  => 1,
        },
        last_payout => {
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {
                optional => TRUE,
                type     => 'date',
            },
            depends_on => ['create_date'],
            get        => sub {
                return $_[1]->{'last_payout'} // $_[1]->{'create_date'};
            },
        },
        # можно принять оферту common_offer is defined
        # обязан принять оферту common_offer < 0
        common_offer => {
            depends_on => ['id'],
            get        => sub {
                $_[0]{'__COMMON_OFFER__'}{$_[1]{id}};
            },
            type => 'number',
            api  => 1,
        },
        create_date => {
            db   => 'u',
            type => 'string',
            api  => 1,
        },
        notifications_count => {
            depends_on => ['id'],
            label      => d_gettext('Notifications count'),
            get        => sub {
                return $_[0]->{'__NOTIFICATIONS_COUNT__'}{$_[1]->{'id'}}->{'unread'} // 0;
            },
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
        is_assessor => {
            depends_on => ['id'],
            type       => 'boolean',
            get        => sub {
                return $_[0]->{'__IS_ASSESSOR__'}{$_[1]->{'id'}} // 0;
            },
            api => 1,
        },
        is_dm_lite => {
            db          => TRUE,
            type        => 'boolean',
            api         => 1,
            need_check  => {type => 'boolean', optional => TRUE},
            adjust_type => 'str',
        },
        allowed_design_auction_native_only => {
            from_opts  => 'from_hash',
            type       => 'boolean',
            api        => 1,
            need_check => {
                type     => 'boolean',
                optional => TRUE,
                check    => sub {
                    my ($qv, $value, $template) = @_;

                    my $hs = $qv->app->hook_stash;
                    if ($hs->inited && $hs->mode('edit')) {
                        # Проверка только для редактирования

                        my $current = $hs->get('current');

                        # При отключении проверить наличие tga
                        if ($current->{allowed_design_auction_native_only} && !$value) {
                            my $user_id = $hs->get('id')->{id};

                            my ($check, $msg) = $qv->app->_can_delete_feature_design_auction_native($user_id);
                            throw Exception::Validator::Fields $msg unless $check;
                        }
                    }
                  }
            },
        },
        content_block_edit_template_allowed => {
            from_opts  => 'from_hash',
            type       => 'boolean',
            api        => 1,
            need_check => {
                type     => 'boolean',
                optional => TRUE,
            },
        },
        inn => {
            from_opts  => 'from_hash',
            need_check => {
                type     => 'scalar',
                optional => TRUE,
            },
            api          => 1,
            type         => 'string',
            api_can_edit => FALSE,
        },
        cooperation_form => {
            from_opts  => 'from_hash',
            type       => 'string',
            need_check => {
                type     => 'scalar',
                optional => TRUE,
                in       => \@AVAILABLE_COOPERATION_FORMS,
            },
            api          => 1,
            type         => 'string',
            api_can_edit => FALSE,
        },
        self_employed => {
            from_opts  => 'from_hash',
            type       => 'number',
            need_check => {
                type     => 'int_un',
                optional => TRUE,
                in       => \@SELFEMPLOYED_STATUSES,
            },
            api          => 1,
            type         => 'number',
            api_can_edit => FALSE,
        },
        self_employed_request_id => {
            from_opts  => 'from_hash',
            type       => 'string',
            need_check => {
                type     => 'scalar',
                optional => TRUE,
            },
        },
        payoneer => {
            api          => 1,
            api_can_edit => FALSE,
            from_opts    => 'from_hash',
            need_check   => {
                optional => TRUE,
                type     => 'boolean',
            },
            type => 'boolean',
        },
        payoneer_currency => {
            api          => 1,
            api_can_edit => FALSE,
            from_opts    => 'from_hash',
            need_check   => {
                in       => [qw(EUR USD)],
                optional => TRUE,
                type     => 'scalar',
            },
            type => 'string',
        },
        payoneer_payee_id => {
            api          => 1,
            api_can_edit => FALSE,
            from_opts    => 'from_hash',
            need_check   => {
                len_max  => 30,
                optional => TRUE,
                type     => 'scalar',
            },
            type => 'string',
        },
        payoneer_step => {
            api          => 1,
            api_can_edit => FALSE,
            from_opts    => 'from_hash',
            need_check   => {
                max      => 8,
                min      => 0,
                optional => TRUE,
                type     => 'int_un',
            },
            type => 'number',
        },
        payoneer_url => {
            api          => 1,
            api_can_edit => FALSE,
            from_opts    => 'from_hash',
            need_check   => {
                optional => TRUE,
                type     => 'scalar',
            },
            type => 'string',
        },
        current_currency => {
            api       => 1,
            from_opts => 'from_hash',
            get       => sub {
                $_[1]->{'current_currency'} // $DEFAULT_CPM_CURRENCY;
            },
            need_check => {
                in       => $CPM_AVAILABLE_CURRENCIES,
                optional => TRUE,
                type     => 'scalar',
            },
            type => 'string',
        },
        next_currency => {
            api        => 1,
            from_opts  => 'from_hash',
            depends_on => [qw(current_currency)],
            need_check => {
                in       => $CPM_AVAILABLE_CURRENCIES,
                optional => TRUE,
                type     => 'scalar',
            },
            get => sub {
                $_[1]->{'next_currency'} // $_[1]->{'current_currency'};
            },
            type => 'string',
        },
        currency_rate => {
            api        => 1,
            from_opts  => 'from_hash',
            need_check => {
                optional => TRUE,
                check    => sub {
                    my ($qv, $value, $template) = @_;
                    throw Exception::Validator::Fields gettext("Field \"%s\" should be a positive number",
                        'currency_rate')
                      unless $value > 0;
                },
                type => 'scalar',
            },
            type => 'number',
        },
        has_game_offer => {
            api          => 1,
            api_can_edit => FALSE,
            from_opts    => 'from_hash',
            need_check   => {
                optional => TRUE,
                type     => 'boolean',
            },
            type => 'boolean',
        },
        contractor_type => {
            api          => 1,
            api_can_edit => FALSE,
            type         => 'string',
            label        => d_gettext('Contractor type'),
            depends_on   => [qw(cooperation_form self_employed inn)],
            get          => sub {
                my $type;

                if (defined($_[1]->{'self_employed'}) && $_[1]->{'self_employed'} == $SELFEMPLOYED_STATUS_READY) {
                    $type = 3;
                } elsif (defined $_[1]->{'cooperation_form'}) {
                    if (in_array($_[1]->{'cooperation_form'}, ['byu', 'sw_yt', 'sw_ur', 'ua', 'yt', 'ur'])) {
                        $type =
                          (      $_[1]->{'cooperation_form'} eq 'ur'
                              && defined($_[1]->{'inn'})
                              && length($_[1]->{'inn'}) == 12) ? 1 : 4;
                    } elsif (in_array($_[1]->{'cooperation_form'}, ['ph', 'sw_ytph', 'ytph'])) {
                        $type = 2;
                    }
                }
                return $type ? gettext($CONTRACTOR_TYPES->{$type}) : undef;
              }
        },
        is_deleted    => {db => TRUE, api => 1, type => 'boolean'},
        deletion_date => {db => TRUE, api => 1, type => 'string'},
        is_form_done  => {
            api          => 1,
            api_can_edit => FALSE,
            from_opts    => 'from_hash',
            need_check   => {
                optional => TRUE,
                type     => 'boolean',
            },
            type => 'boolean',
        },
    };
}

sub get_fields_depends {
    return {};
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            id                  => {type => 'number',     label => d_gettext('UID')},
            client_id           => {type => 'number',     label => d_gettext('Client ID')},
            login               => {type => 'login',      label => d_gettext('Login')},
            name                => {type => 'text',       label => d_gettext('Name')},
            lastname            => {type => 'text',       label => d_gettext('Lastname')},
            email               => {type => 'text',       label => d_gettext('E-Mail')},
            phone               => {type => 'text',       label => d_gettext('Phone')},
            accountant_email    => {type => 'text',       label => d_gettext('Accountant E-Mail')},
            newsletter          => {type => 'boolean'},
            is_tutby            => {type => 'boolean'},
            has_tutby_agreement => {type => 'boolean'},
            has_common_offer    => {type => 'boolean'},
            business_unit       => {type => 'boolean'},
            multistate          => {type => 'multistate', label => d_gettext('Status')},
            role_id             => {
                db_filter => sub {
                    my $filter = $_[0]->{'__DB_FILTER__'}{$_[1]->[0]}->as_filter($_[1], $_[2]);
                    return [
                        id => '= ANY' => $_[0]->partner_db->query->select(
                            table  => $_[0]->partner_db->user_role,
                            fields => ['user_id'],
                            filter => $filter
                        )
                    ];
                },
                type   => 'dictionary',
                label  => d_gettext('Roles'),
                values => sub {
                    [
                        map {
                            {hash_transform($_, ['id'], {name => 'label'})}
                          } @{$_[0]->rbac->get_roles()}
                    ];
                  }
            },
            contract_id => {
                type  => 'contractnumber',
                label => d_gettext('Contract ID'),
            },
            adfox_id => {
                db_filter => sub {
                    my $filter = $_[0]->{'__DB_FILTER__'}{$_[1]->[0]}->as_filter($_[1], $_[2]);
                    return [
                        id => '= ANY' => $_[0]->partner_db->query->select(
                            table  => $_[0]->partner_db->user_adfox,
                            fields => ['user_id'],
                            filter => $filter
                        )
                    ];
                },
                type  => 'number',
                label => d_gettext('AdFox ID'),
            },
            no_stat_monitoring_emails => {type => 'boolean'},
            need_to_email_processing  => {type => 'boolean'},
            has_business_rule         => {
                type      => 'boolean',
                label     => d_gettext('Is user has business rules'),
                db_filter => sub {
                    my ($self, $data, $field, %opts) = @_;

                    my $query = $self->partner_db->query->select(
                        table  => $self->partner_db->business_rules,
                        fields => ['owner_id'],
                    )->distinct(TRUE);

                    return [id => ($data->[2] ? '=' : '<>') . 'ANY' => $query];
                },
            },
            domain_login => {
                type  => 'text',
                label => d_gettext('Domain login'),
            },
            is_yandex => {
                type      => 'boolean',
                db_filter => sub {
                    my ($self, $data, $field, %opts) = @_;

                    return [id => ($data->[2] ? '=' : '<>') => \$ADINSIDE_USER_ID];
                },
            },
            is_adfox_partner => {type => 'boolean',},
            user_type        => {
                db_filter => sub {
                    my ($self, $data, $field, %opts) = @_;

                    my $positive = ($data->[1] eq '=' ? 1 : 0);
                    my %types;
                    @types{@{$data->[2]}} = undef;
                    my $filter = [
                        ($positive ? 'OR' : 'AND'),
                        [
                            map {
                                exists($types{$_})
                                  && (!$TYPES->{$_}{'short_right'}
                                    || $self->check_short_rights($TYPES->{$_}{'short_right'}))
                                  ? [$TYPES->{$_}{'db'} => '=' => \$positive]
                                  : ()
                              } keys(%$TYPES)
                        ]
                    ];

                    return (@{$filter->[1]} ? $filter : \TRUE);
                },
                label  => d_gettext('User type'),
                type   => 'dictionary',
                values => sub {
                    my ($self) = @_;

                    return [
                        map {{id => $_, label => $TYPES->{$_}{'label'}->()}}
                          sort {$a <=> $b}
                          grep {!$TYPES->{$_}{'short_right'} || $self->check_short_rights($TYPES->{$_}{'short_right'})}
                          keys(%$TYPES)
                    ];
                },
            },
            self_employed    => {type => 'json', value_type => 'text'},
            current_currency => {type => 'json', value_type => 'text'},
            uid              => {type => 'number'},
            is_deleted       => {type => 'boolean'},
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        $self->check_short_rights('view')
        ? (
            {name => 'multistate', label => gettext('Status')},
            ($self->check_short_rights('view_field__roles') ? {name => 'role_id', label => gettext('Role')} : ()),
            {name => 'login', label => gettext('Login')},
            (
                $self->check_rights('view_search_filters__user_type')
                ? {
                    name  => 'user_type',
                    label => gettext('User type'),
                  }
                : ()
            ),
            {name => 'email', label => gettext('E-Mail')},
            {name => 'phone', label => gettext('Phone')},
            (
                $self->check_short_rights('view_field__client_id')
                ? {
                    name  => 'client_id',
                    label => gettext('Client ID'),
                    hint  => gettext('The unique identifier of a partner in Balance')
                  }
                : ()
            ),
            (
                $self->check_short_rights('view_field__contract_id')
                ? {
                    name  => 'contract_id',
                    label => gettext('Contract ID'),
                  }
                : ()
            ),
            {name => 'lastname', label => gettext('Lastname')},
            {name => 'name',     label => gettext('Name')},
            (
                $self->check_short_rights('view_field__user_id')
                ? {name => 'id', label => gettext('User ID')}
                : ()
            ),
          )
        : ()
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {
        empty_name  => gettext('New'),
        multistates => [
            [contacts_provided           => d_gettext('Contacts provided')],
            [need_create_in_banner_store => d_gettext('Need create partner in BannerStore')],
            [free_multistate             => d_gettext('Free multistate'), private => TRUE],
            [need_yan_contract           => d_gettext('Need YAN contract')],
            [blocked                     => d_gettext('User blocked')],
        ],
        actions => {
            edit                                    => d_gettext('Edit'),
            created_partner_in_banner_store         => d_gettext('Created partner in BannerStore'),
            change_contract                         => d_gettext('Edit requisites'),
            provide_contacts                        => d_gettext('Provide contacts'),
            unsubscribe_from_stat_monitoring_emails => d_gettext('Unsubscribe from statistics monitoring emails'),
            set_excluded_domains                    => d_gettext('Set excluded domains'),
            set_excluded_phones                     => d_gettext('Set excluded phones'),
        },
        right_actions => {
            add                            => d_gettext('Add'),
            link_adfox_user                => d_gettext('Link AdFox user'),
            request_create_in_banner_store => d_gettext('Request create partner in BannerStore'),
            request_yan_contract           => d_gettext('Request YAN contract'),
            reset_blocked                  => d_gettext('Reset blocked'),
            revoke_roles                   => d_gettext('Revoke of the user roles'),
            set_blocked                    => d_gettext('Set blocked'),
            set_user_role                  => d_gettext('Adding roles to a user'),
            unlink_adfox_user              => d_gettext('Unlink AdFox user'),
            yan_contract_ready             => d_gettext('YAN contract ready'),
        },
        right_group        => [user_actions => d_gettext('Rights for users actions')],
        right_name_prefix  => 'user_action_',
        multistate_actions => [
            {
                action => 'add',
                from   => '__EMPTY__'
            },
            {
                action    => 'provide_contacts',
                from      => 'not contacts_provided',
                set_flags => ['contacts_provided']
            },
            {
                action => 'change_contract',
                from   => 'not blocked',
            },
            {
                action    => 'request_create_in_banner_store',
                from      => 'not need_create_in_banner_store',
                set_flags => ['need_create_in_banner_store']
            },
            {
                action      => 'created_partner_in_banner_store',
                from        => 'need_create_in_banner_store',
                reset_flags => ['need_create_in_banner_store']
            },
            {
                action => 'set_user_role',
                from   => 'not blocked',
            },
            {
                action => 'revoke_roles',
                from   => '__EMPTY__ or not __EMPTY__',
            },
            {
                action    => 'request_yan_contract',
                from      => 'not need_yan_contract',
                set_flags => ['need_yan_contract']
            },
            {
                action      => 'yan_contract_ready',
                from        => 'need_yan_contract',
                reset_flags => ['need_yan_contract']
            },
            {
                action => 'link_adfox_user',
                from   => '__EMPTY__ or not __EMPTY__',
            },
            {
                action => 'unlink_adfox_user',
                from   => '__EMPTY__ or not __EMPTY__',
            },
            {
                action => 'unsubscribe_from_stat_monitoring_emails',
                from   => 'not blocked',
            },
            {
                action => 'edit',
                from   => 'not blocked',
            },
            {
                action => 'set_excluded_domains',
                from   => 'not blocked',
            },
            {
                action => 'set_excluded_phones',
                from   => 'not blocked',
            },
            {
                action    => 'set_blocked',
                from      => 'not blocked',
                set_flags => ['blocked'],
            },
            {
                action      => 'reset_blocked',
                from        => 'blocked',
                reset_flags => ['blocked'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    if ($fields->need('active_contract')) {
        my %banks = ();

        for (@$result) {
            $fields->{'__ACTIVE_CONTRACTS__'}{$_->{'client_id'}} = $self->documents->get_active_contract($_);

            if (defined $fields->{'__ACTIVE_CONTRACTS__'}{$_->{'client_id'}}) {
                my ($bank_id, $bank_id_type) =
                  $fields->{'__ACTIVE_CONTRACTS__'}{$_->{'client_id'}}{Person}{bik}
                  ? ($fields->{'__ACTIVE_CONTRACTS__'}{$_->{'client_id'}}{Person}{bik}, 'bik')
                  : ($fields->{'__ACTIVE_CONTRACTS__'}{$_->{'client_id'}}{Person}{swift}, 'swift');

                next unless $bank_id;

                if (not exists $banks{$bank_id}) {
                    try {
                        # get_bank always returns only 1 bank
                        my $bank_raw = $self->api_balance->get_bank(ucfirst($bank_id_type) => $bank_id)->[0];

                        $banks{$bank_id} = {
                            bank_id      => $bank_id,
                            bank_id_type => $bank_id_type,
                            name         => $bank_raw->{name},
                            active       => ($bank_raw->{hidden} || $bank_raw->{hidden_dt})
                            ? JSON::XS::false
                            : JSON::XS::true,
                        };
                    };
                }
                $fields->{'__ACTIVE_CONTRACTS__'}{$_->{'client_id'}}{Person}{bank} =
                  $banks{$fields->{'__ACTIVE_CONTRACTS__'}{$_->{'client_id'}}{Person}{$bank_id_type}};
            }
        }
    }

    my $user_ids;
    if (   $fields->need('excluded_domains')
        || $fields->need('excluded_phones')
        || $fields->need('features')
        || $fields->need('adfox_info')
        || $fields->need('avatar')
        || $fields->need('common_offer')
        || $fields->need('roles')
        || $fields->need('is_assessor')
        || $fields->need('business_rules_count')
        || $fields->need('notifications_count'))
    {
        $user_ids = [map {$_->{'id'}} @$result];
    }

    if ($fields->need('roles')) {

        my %role_names = map {$_->{'id'} => $_} @{$self->rbac->get_roles};

        my $user_roles = $self->partner_db->user_role->get_all(
            fields => [qw(user_id role_id)],
            filter => {user_id => $user_ids}
        );

        foreach (@$user_roles) {
            push(
                @{$fields->{'__USER_ROLES__'}{$_->{'user_id'}}},
                $role_names{$_->{'role_id'}} // {id => $_->{'role_id'}}
            );
            $fields->{'__IS_ASSESSOR__'}{$_->{'user_id'}} = TRUE if $_->{'role_id'} == $ASSESSOR_ROLE_ID;
        }
    }

    if ($fields->need('is_assessor') and !$fields->need('roles')) {
        my $user_roles = $self->partner_db->user_role->get_all(
            fields => [qw(user_id)],
            filter => [AND => [[user_id => IN => \$user_ids], [role_id => "=" => \$ASSESSOR_ROLE_ID]]],
        );

        foreach (@$user_roles) {
            $fields->{'__IS_ASSESSOR__'}{$_->{'user_id'}} = TRUE;
        }
    }

    if ($fields->need('adfox_info')) {
        my $user_adfox = $self->partner_db->user_adfox->get_all(
            fields => [qw(user_id create_date adfox_id adfox_login)],
            filter => {user_id => $user_ids}
        );

        foreach (@$user_adfox) {
            push(
                @{$fields->{'__USER_ADFOX__'}{$_->{'user_id'}}},
                {id => $_->{'adfox_id'}, login => $_->{'adfox_login'}, create_date => $_->{'create_date'}}
            );
        }
    }

    if ($fields->need('features')) {
        my $features = $self->user_features->get_all(
            fields => [qw(user_id feature)],
            filter => {user_id => $user_ids}
        );

        foreach (@$features) {
            push(@{$fields->{'__USER_FEATURES__'}{$_->{'user_id'}}}, $_->{'feature'});
        }
    }

    if ($fields->need('lang') || $fields->need('avatar')) {
        my @tmp = @$user_ids;
        # ББ рекомендует ограничивать по 200
        while (my @part = splice @tmp, 0, 200) {
            my $data = $self->api_blackbox->get_users_avatar_and_lang(@part);
            for my $uid (keys %$data) {
                $fields->{'__LANG__'}{$uid}   = $data->{$uid}{lang};
                $fields->{'__AVATAR__'}{$uid} = $data->{$uid}{avatar};
            }
        }
    }

    if ($fields->need('business_rules_count') || $fields->need('business_rules_active_count')) {
        $fields->{'__BUSINESS_RULES_COUNT__'} = $self->business_rules->get_cnt($user_ids) // {};
    }
    if ($fields->need('notifications_count')) {
        $fields->{'__NOTIFICATIONS_COUNT__'} = $self->user_notifications->get_unread_count($user_ids) // {};
    }

    if ($fields->need('common_offer')) {
        my $list = $self->partner_db->common_offer_allowed_users->get_all(
            filter => [user_id => 'IN' => \$user_ids],
            fields => [qw(user_id deadline accept_date)]
        );
        my $curdate = curdate();
        for my $row (@$list) {
            unless ($row->{accept_date}) {
                $fields->{'__COMMON_OFFER__'}{$row->{user_id}} =
                  dates_delta_days($curdate, trdate(db => norm => $row->{deadline}));
            }
        }
    }
}

sub get_available_fields {
    my ($self, $fields, $obj) = @_;

    my $self_view =
         $obj
      && $obj->{'id'}
      && $obj->{'id'} eq $self->app->get_option('cur_user', {})->{'id'};

    my %fields;
    if ($self->check_rights('users_view_all_fields') || $self_view) {
        my $model_fields = $self->get_model_fields;
        %fields = map {$_ => TRUE} keys(%$model_fields);
    } else {
        %fields = map {$_ => TRUE} qw(
          allowed_design_auction_native_only
          content_block_edit_template_allowed
          features
          id
          login
          avatar
          available_fields
          editable_fields
          );
    }

    delete($fields{'email'}) if (!$self_view && !$self->check_rights('users_view_field__email'));

    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(
        \%fields,
        {
            $accessor . '_view_field__%s' => [
                qw(
                  client_id
                  domain_login
                  has_common_offer
                  has_mobile_mediation
                  has_rsya
                  has_tutby_agreement
                  is_dm_lite
                  is_efir_blogger
                  is_games
                  is_mobile_mediation
                  is_tutby
                  is_video_blogger
                  multistate
                  need_to_email_processing
                  roles
                  )
            ],
            $accessor . '_view_field__user_id'     => 'id',
            $accessor . '_view_field__contract_id' => 'contracts',
            $accessor . '_view_field__multistate'  => 'multistate_name',
        }
    );

    delete(@fields{'excluded_domains', 'excluded_phones'})
      unless grep {$_->{'id'} eq $SITE_PARTNER_ROLE_ID || $_->{'id'} eq $MOBILE_PARTNER_ROLE_ID} @{$obj->{'roles'}};

    $fields{'has_rsya'} = TRUE if $obj->{'is_mobile_mediation'};
    $fields{'is_form_done'} = TRUE;

    return \%fields;
}

sub get_add_fields {
    my ($self) = @_;

    my %fields = map {$_ => TRUE} qw(
      id
      uid
      login
      accountant_email
      adfox_offer
      name
      client_id
      name
      midname
      lastname
      email
      newsletter
      phone
      is_form_done
      is_video_blogger
      country_id
      is_tutby
      is_games
      is_mobile_mediation
      has_approved
      has_approved_app
      has_approved_site
      has_common_offer
      has_rsya
      has_mobile_mediation
      has_tutby_agreement
      need_to_email_processing
      is_dm_lite
      is_efir_blogger
      inn
      cooperation_form
      self_employed
      current_currency
      features
      );

    return \%fields;
}

sub get_editable_fields {
    my ($self, $object) = @_;

    return {}
      unless $self->check_action($object, 'edit');

    return $self->collect_editable_fields($object);
}

sub get_actions_depends {
    [];
}

sub collect_editable_fields {
    my ($self, $object) = @_;

    my %fields = ();

    #Костыль. Делается потому что при запросе get или get_all приходит массив хэшей.
    #А при редактировании приходит массив чисел. Сделано при переводе на хуки
    my @roles = map {ref($_) ? $_->{'id'} : $_} @{$object->{roles}};

    if ($object->{'id'} == $self->get_option('cur_user', {})->{'id'} || $self->check_short_rights('edit_all')) {
        $fields{$_} = TRUE foreach (
            qw(
            lastname
            name
            midname
            email
            phone
            newsletter
            no_stat_monitoring_emails
            block_light_form_enabled
            has_mobile_mediation
            has_rsya
            last_payout
            inn
            cooperation_form
            )
        );

        # Если включен флаг показа оферты АФ для платных услуг
        # то можно включать флаг оферты
        $fields{adfox_offer} = TRUE if $object->{paid_offer} && $object->{has_common_offer} && @{$object->{adfox_info}};

        $fields{$_} = TRUE foreach (
            qw(
            payoneer
            payoneer_currency
            payoneer_payee_id
            payoneer_step
            payoneer_url
            self_employed
            self_employed_request_id
            is_form_done
            )
        );
    }

    if (
        (
               $self->check_rights('do_context_on_site_add')
            || $self->check_rights('do_search_on_site_add')
            || $self->check_rights('do_mobile_app_add')
        )
        && (grep {$_ == $SITE_PARTNER_ROLE_ID || $_ == $MOBILE_PARTNER_ROLE_ID} @roles)
       )
    {
        $fields{'excluded_domains'} = TRUE;
        $fields{'excluded_phones'}  = TRUE;
    }

    $fields{'features'} = TRUE if $self->check_rights('users_edit_field__features');

    foreach my $field (
        qw(
        adfox_offer
        allowed_design_auction_native_only
        client_id is_tutby
        content_block_edit_template_allowed
        has_approved
        has_approved_app
        has_approved_site
        has_common_offer
        is_dm_lite
        is_efir_blogger
        is_games
        is_mobile_mediation
        is_video_blogger
        need_to_email_processing
        currency_rate
        current_currency
        next_currency
        has_game_offer
        )
      )
    {
        $fields{$field} = TRUE if $self->check_short_rights("edit_field__$field");
    }

    my $current_currency = $object->{current_currency};
    my $next_currency    = $object->{next_currency};

    my $hs = $self->hook_stash;
    if ($hs->inited && $hs->mode('edit')) {
        my $current = $hs->get('current');
        $current_currency = $current->{current_currency};
        $next_currency    = $current->{next_currency};
    }

    if ($current_currency ne $next_currency
        || !in_array('cpm_currency', $object->{features}))
    {
        delete $fields{$_} foreach (qw(next_currency currency_rate));
    } else {
        delete $fields{current_currency};
    }

    $fields{'has_tutby_agreement'} = TRUE
      if $self->check_short_rights('edit_field__has_tutby_agreement') && $object->{'is_tutby'};

    $fields{'roles'} = TRUE
      if $self->check_rights('do_user_action_set_user_role') && $self->check_rights('do_user_action_revoke_roles');

    return \%fields;
}

=head add

Добавляет пользователя в базу данных.

При добавлении обращается к балансу и выясняет Client ID, который записывается
в базу. При наличии ключей client_id и force_client_id сверка с балансом не производится.

    my $user = $app->users->add(
        id => '35309619',
        login => 'ivanbessarabov',
    );

=cut

sub hook_fields_processing_before_validation {
    my ($self, $opts) = @_;

    if ($self->hook_stash->mode('add')) {
        $opts->{has_mobile_mediation}               //= 0;
        $opts->{has_approved}                       //= 0;
        $opts->{has_approved_app}                   //= 0;
        $opts->{has_approved_site}                  //= 0;
        $opts->{has_tutby_agreement}                //= 0;
        $opts->{has_rsya}                           //= 1;
        $opts->{has_common_offer}                   //= 0;
        $opts->{allowed_design_auction_native_only} //= 0;
        $opts->{next_currency}                      //= $opts->{current_currency} if defined $opts->{current_currency};

        $opts->{'login'} = fix_login($opts->{'login'});

        # костыль, чтобы дальше по хукам был id
        # убрать эту строку в рамках https://st.yandex-team.ru/PI-26912
        $opts->{'id'} //= $opts->{'uid'};

        unless ($opts->{'client_id'} && delete($opts->{'force_client_id'})) {
            my $balance_client_id = $self->get_client_id($opts->{'id'});

            if (exists($opts->{'client_id'})) {
                my $client_id_from_user = $opts->{'client_id'} // '';

                if ($client_id_from_user ne $balance_client_id) {
                    throw gettext("Incorrect data. User ID: '%s'; Client ID: '%s'; Balance Client ID: '%s'",
                        $opts->{'id'}, $client_id_from_user, $balance_client_id);
                }
            } else {
                $opts->{'client_id'} = $balance_client_id;
            }
        }
    }

    $opts->{'excluded_domains'} = [map {get_domain($_) // $_} @{$opts->{'excluded_domains'}}]
      if exists($opts->{'excluded_domains'});
    $opts->{'excluded_phones'} = [map {get_normalized_phone($_) // $_} @{$opts->{'excluded_phones'}}]
      if exists($opts->{'excluded_phones'});

    $self->SUPER::hook_fields_processing_before_validation($opts);
}

sub hook_owner_processing { }

sub hook_preparing_fields_to_save {
    my ($self, $opts) = @_;

    # После установки флага оферты АФ для платных услуг
    # можно сбросить флаг разрешения показа этой оферты
    $opts->{paid_offer} = 0 if $opts->{adfox_offer};

    $self->SUPER::hook_preparing_fields_to_save($opts);
}

sub hook_save_fields_from_related_models {
    my ($self, $opts) = @_;

    my $need_update_resources = FALSE;
    my $related               = $self->hook_stash->get('fields_from_related_models');
    my $current               = $self->hook_stash->get('current');
    my $id                    = $self->hook_stash->get('id');

    if (my $excluded_domains = $related->{excluded_domains}) {
        $self->do_action($id, 'set_excluded_domains', domains => $excluded_domains);
    }

    if (my $excluded_phones = $related->{excluded_phones}) {
        $self->do_action($id, 'set_excluded_phones', phones => $excluded_phones);
    }

    if (my $features = $related->{features}) {
        $self->user_features->replace_multi($id->{id}, $features);
    }

    if (my $roles = $related->{'roles'}) {
        my %old_roles = map {$_->{id} => TRUE} @{$current->{'roles'}};
        my @roles_added;
        for my $role (@{array_uniq($roles)}) {
            unless (delete $old_roles{$role}) {
                push @roles_added, $role;
            }
        }
        my @roles_revoked = keys %old_roles;

        $self->do_action(
            $id, 'revoke_roles',
            roles_id                     => \@roles_revoked,
            'suppress_mail_notification' => (@$roles ? FALSE : TRUE),
        ) if @roles_revoked;

        $self->do_action($id, 'set_user_role', role_id => \@roles_added)
          if @roles_added;

        $need_update_resources ||= @roles_added || @roles_revoked;

        $self->mail_notification->add_when_blocking_login({id => $id->{id}, login => $current->{login}})
          unless @$roles;
    }

    foreach (qw(has_mobile_mediation has_rsya)) {
        $need_update_resources ||= defined $opts->{$_} && $current->{$_} != $opts->{$_};
    }

    if ($need_update_resources) {
        my $cur_user = $self->get_option('cur_user', {});
        if ($cur_user->{'id'} == $id->{id}) {
            foreach (qw(has_mobile_mediation has_rsya)) {
                $cur_user->{$_} = $opts->{$_} // $current->{$_};
            }
            $self->set_option('cur_user', $cur_user);
        }
        $self->resources->invalidate_cache($id->{id});
    }
}

sub hook_processing_after_insert {
    my ($self, $opts) = @_;

    $self->user_features->auto_assign($self->hook_stash->get('settings')->{'id'});

    $self->do_action($self->hook_stash->get('id'), 'provide_contacts')
      if $opts->{'provide_contacts'} && $self->check_action($self->hook_stash->get('id'), 'provide_contacts');
}

sub hook_processing_after_update {
    my ($self, $opts) = @_;

    my $current     = $self->hook_stash->get('current');
    my $adfox_offer = $current->{adfox_offer};
    if ($opts->{adfox_offer} && !$adfox_offer) {
        my $adfox_contract_created;
        try {
            # creating new person + contract in Balance
            $adfox_contract_created = $self->create_contract_adfox_paid_products(
                user_id   => $self->hook_stash->get('id')->{id},
                mail_info => {
                    user_id    => $self->hook_stash->get('id')->{id},
                    adfox_info => $current->{adfox_info}[0],
                },
            );

            if ($adfox_contract_created && !$adfox_contract_created->{is_billing_data_already_saved_in_adfox}) {
                # passing client_id + contract_id to AdFox
                $self->api_adfox_graphql->set_billing_data(
                    user_id      => $self->hook_stash->get('id')->{id},
                    billing_data => {
                        client_id   => $adfox_contract_created->{client_id},
                        contract_id => $adfox_contract_created->{id},
                    },
                    presets => \@ADFOX_PRESET_PAIDPRODUCTS,
                );
            }
        }
        catch {
            my ($exception) = @_;

            $self->_on_offer_error_send_email(
                'case_add_adfox_paidprooducts',
                $exception,
                {
                    adfox_contract_created => $adfox_contract_created,
                    user                   => {%$current, id => $self->hook_stash->get('id')->{id}},
                }
            );

            throw $exception;
        };
    }

    if (defined $opts->{'next_currency'}
        && $opts->{'next_currency'} ne ($current->{'next_currency'} // $DEFAULT_CPM_CURRENCY))
    {
        my $tmp_rights = $self->app->add_tmp_rights('queue_add_convert_cpm_in_user_blocks');
        $self->queue->add(
            method_name => 'convert_cpm_in_user_blocks',
            params      => {
                currency      => $opts->{next_currency},
                currency_rate => $opts->{currency_rate},
                user          => $self->hook_stash->get('id')->{id},
            },
        );
    }

    $self->SUPER::hook_processing_after_update($opts);
}

# obsolete
#
sub edit {
    my ($self, $user_id, %data) = @_;

    $self->do_action($user_id, 'edit', %data);
}

sub can_action_edit {
    my ($self, $user) = @_;

    my $cur_user = $self->get_option('cur_user');

    return TRUE if ($user->{'id'} == $cur_user->{'id'} || $self->check_short_rights('edit_all'));
}

sub on_delete_feature_design_auction_native {
    my ($self, $user_id) = @_;

    $self->_rollback_native_auction($user_id, [qw(desktop mobile)]);
}

sub on_delete_feature_design_auction_native_turbo {
    my ($self, $user_id) = @_;

    $self->_rollback_native_auction($user_id, [qw(turbo turbo_desktop)]);
}

sub on_delete_feature_mobile_fullscreen_available {
    my ($self, $user_id) = @_;

    $self->_delete_featured_site_versions_blocks($user_id, [qw(context_on_site_rtb internal_context_on_site_rtb)],
        [qw(mobile_fullscreen)]);
}

sub on_delete_feature_mobile_rewarded_available {
    my ($self, $user_id) = @_;

    $self->_delete_featured_site_versions_blocks($user_id, [qw(context_on_site_rtb internal_context_on_site_rtb)],
        [qw(mobile_rewarded)]);
}

sub on_delete_feature_mobile_floorad_available {
    my ($self, $user_id) = @_;

    $self->_delete_featured_site_versions_blocks($user_id, [qw(context_on_site_rtb internal_context_on_site_rtb)],
        [qw(mobile_floorad)]);
}

sub _delete_featured_site_versions_blocks {
    my ($self, $user_id, $models, $versions) = @_;
    foreach my $model (@$models) {
        my $page_accessor = $self->app->$model->page->accessor;

        my @page_ids = map {$_->{'page_id'}} @{
            $self->all_pages->get_all(
                fields    => ['page_id'],
                filter    => {owner_id => $user_id},
                from_view => TRUE,
            )
          };

        next unless @page_ids;

        my $blocks;
        if (   $self->app->$model->DOES('Application::Model::Role::JavaJsonApiProxy')
            && $self->app->$model->can_use_java_for_do_action())
        {
            my $page_id_field_name = $self->app->$model->get_page_id_field_name();

            $blocks = $self->app->$model->partner_db_table()->get_all(
                fields => [$page_id_field_name, 'id'],
                filter => [
                    'AND',
                    [
                        ($self->app->$model->is_block_table_with_multiple_models() ? ['model', '=', \$model] : ()),
                        ['site_version',      'IN', \$versions],
                        ['multistate',        'IN', \$self->app->$model->get_multistates_by_filter('not deleted')],
                        [$page_id_field_name, 'IN', \\@page_ids],
                    ]
                ]
            );

            foreach (@$blocks) {
                $_->{'page_id'}   = $_->{$page_id_field_name};
                $_->{'public_id'} = $self->app->$model->public_id($_);
            }
        } else {
            $blocks = $self->app->$model->get_all(
                fields => ['page_id', 'public_id', @{$self->app->$model->get_depends_for_field('actions')}],
                filter => {site_version => $versions, multistate => 'not deleted', page_id => \@page_ids}
            );
        }

        my @pages         = ();
        my @failed_blocks = ();
        foreach my $block (@$blocks) {
            try {
                $self->app->$model->do_action($block, 'delete', do_not_update_in_bk => TRUE);

                push(@pages, {model => $page_accessor, id => $block->{'page_id'}});
            }
            catch {
                push(@failed_blocks, $block->{'public_id'});
            };
        }

        ERROR 'Can not delete blocks: ' . join(', ', @failed_blocks) if @failed_blocks;

        $self->app->all_pages->mark_pages_for_async_update(pages => \@pages);
    }

}

sub _get_design_auction_native_filter {
    my ($self, $user_id, $site_version) = @_;

    my $db_filter = $self->partner_db->filter();

    $db_filter->and(
        [
            {'' => [qw(page_id block_id)]},
            'IN',
            $self->partner_db->query->select(
                table  => $self->partner_db->all_pages,
                fields => [],
                filter => ['owner_id', '=', \$user_id],
              )->join(
                table   => $self->partner_db->context_on_site_rtb,
                fields  => [qw(campaign_id id)],
                join_on => ['campaign_id' => '=' => {page_id => $self->partner_db->all_pages}],
                ($site_version ? (filter => ['site_version', 'IN', \$site_version]) : ())
              )
        ]
    );

    $db_filter->and(['multistate', 'IN', \$self->app->design_templates->get_multistates_by_filter('not deleted')]);

    return $db_filter;
}

sub can_delete_feature_design_auction_native_turbo {
    my ($self, $user_id) = @_;

    my ($res, $msg) = $self->_can_delete_feature_design_auction_native($user_id, [qw(turbo turbo_desktop)]);

    return ($res, $msg);
}

sub can_delete_feature_design_auction_native {
    my ($self, $user_id) = @_;

    my ($res, $msg) = $self->_can_delete_feature_design_auction_native($user_id, [qw(desktop mobile)]);

    return ($res, $msg);
}

sub _can_delete_feature_design_auction_native {
    my ($self, $user_id, $site_version) = @_;

    my $db_filter = $self->_get_design_auction_native_filter($user_id, $site_version);

    my @fields = qw(page_id block_id);

    my $sub_query = $self->partner_db->query->select(
        table  => $self->partner_db->design_templates,
        fields => \@fields,
        filter => $self->partner_db->filter(['type', '=', \$DESIGN_TYPES->{'TGA'}])->and($db_filter)
    )->distinct();

    $sub_query->fields_order(@fields);

    my $query = $self->partner_db->query->select(
        table  => $self->partner_db->design_templates,
        fields => \@fields,
        filter => $self->partner_db->filter(['type', '=', \$DESIGN_TYPES->{'NATIVE'}])->and($db_filter)
          ->and([{'' => \@fields}, 'NOT IN', $sub_query])
    );
    $query->limit(1);

    if (scalar @{$query->get_all()}) {
        return (wantarray ? (FALSE, gettext("Can't turn off, since there are blocks with native design only")) : FALSE);
    } else {
        return TRUE;
    }
}

sub _rollback_native_auction {
    my ($self, $user_id, $site_version) = @_;

    my $db_filter = $self->_get_design_auction_native_filter($user_id, $site_version);

    my %need_update    = ();
    my $native_designs = $self->partner_db->query->select(
        table  => $self->partner_db->design_templates,
        fields => [qw(page_id)],
        filter => $self->partner_db->filter(['type', '=', \$DESIGN_TYPES->{'NATIVE'}])->and($db_filter)
    )->get_all();

    map {$need_update{$_->{'page_id'}} = TRUE} @$native_designs;

    $self->partner_db->design_templates->edit(
        $self->partner_db->filter(['type', '=', \$DESIGN_TYPES->{'NATIVE'}])->and($db_filter),
        {multistate => $self->app->design_templates->get_multistate_by_name('deleted')}
    ) if @$native_designs;

    $self->app->all_pages->mark_pages_for_async_update(page_ids => [sort {$a <=> $b} keys(%need_update)])
      if %need_update;
}

sub on_action_edit {
    my ($self, $user, %opts) = @_;

    $self->do_action($user, 'provide_contacts')
      if delete($opts{'provide_contacts'}) && $self->check_action($user, 'provide_contacts');

    $self->SUPER::on_action_edit($user, %opts);
}

sub can_action_created_partner_in_banner_store {
    my ($self, $user) = @_;

    my $cur_user = $self->get_option('cur_user');

    return $user->{'id'} == $cur_user->{'id'};
}

sub on_action_created_partner_in_banner_store {
    my ($self, $user, %opts) = @_;

    my $cur_user = $self->get_option('cur_user');

    $self->api_http_banner_storage->partner_customers(
        login    => $cur_user->{'login'},
        password => $opts{'password'},
        name     => $opts{'company_name'},
        user_id  => $cur_user->{'id'},
    );

    return TRUE;
}

sub on_action_link_adfox_user {
    my ($self, $user, %opts) = @_;

    my $adfox_id = $opts{'adfox_id'};
    throw Exception::Validation::BadArguments(gettext('Not exists next arguments : adfox_id')) if (!$adfox_id);

    my $login;
    if ($opts{'mobile_mediation'}) {
        $login = $opts{'adfox_login'} || '';
    } else {
        my $user_info = $self->api_adfox->get_adfox_user_info(adfox_id => $adfox_id);
        $login = $user_info->{'login'};
    }

    try {
        $self->partner_db->user_adfox->add(
            {
                user_id     => $user->{'id'},
                create_date => curdate(oformat => 'db_time'),
                adfox_id    => $adfox_id,
                adfox_login => $login
            }
        );

        my $tmp_rights = $self->app->add_tmp_rights('users_view_field__roles');

        unless (grep {$_->{'is_internal'}} @{$self->get($user, fields => [qw(roles)])->{'roles'}}) {
            $self->partner_db_table()->edit($user, {is_adfox_partner => 1});
        }
    }
    catch Exception::DB::DuplicateEntry with {
        #        throw gettext('AdFox user is already linked');
    };

    return TRUE;
}

# Заглушка для переопределения ролью
sub on_action_reset_blocked { }

sub on_action_revoke_roles {
    my ($self, $user, %opts) = @_;

    throw Exception::Validation::BadArguments(gettext('Not exists next arguments : roles_id'))
      if (!$opts{'roles_id'});
    $opts{'roles_id'} = [$opts{'roles_id'}] if ref($opts{'roles_id'}) ne 'ARRAY';

    $user = $self->_get_object_fields($user, [qw(id login client_id)]);
    my ($user_id, $login, $client_id) = @$user{qw(id login client_id)};

    my %roles = map {$_->{'id'} => TRUE} @{$self->rbac->get_roles};
    throw Exception::Validation::BadArguments(
        gettext('Such a roles with ID %s does not exist', join(', ', grep {!$roles{$_}} @{$opts{'roles_id'}})))
      if grep {!$roles{$_}} @{$opts{'roles_id'}};

    $self->rbac->revoke_roles($user_id, $opts{'roles_id'}, %opts);

    $self->queue->add(
        method_name => 'on_revoke_user_roles',
        params      => {%opts, (user => $user)},
    );

    return TRUE;
}

# Заглушка для переопределения ролью
sub can_action_set_blocked {TRUE}

sub on_action_set_blocked {
    my ($self, $user, %opts) = @_;

    $user = $self->_get_object_fields($user, [qw(id login client_id login roles)]);
    my $roles_id = [map {$_->{id}} @{delete $user->{roles}}];

    if (@$roles_id) {
        my $tmp_rights = $self->app->add_tmp_rights('rbac_revoke_roles');
        $self->rbac->revoke_roles($user->{id}, $roles_id, %opts);
        undef $tmp_rights;

        $self->queue->add(
            method_name => 'on_revoke_user_roles',
            params      => {
                roles_id => $roles_id,
                user     => $user,
            },
        );

        $self->mail_notification->add_when_blocking_login({id => $user->{id}, login => $user->{login}});
    }

    return TRUE;
}

sub _get_owner_pages_based_on_roles {
    my ($self, $user, $roles_id, %opts) = @_;

    # Соответствие: какие роли какие модели закрывают.
    my $partner_owner_roles = $self->rbac->get_partner_owner_roles();
    my %role_page_accessor  = map {
        $_->{'id'} => (
             !$opts{related_only} && $_->{all_external}
            ? $self->app->product_manager->get_external_product_accessors
            : $_->{'related_page_accessor'}
          )
    } @$partner_owner_roles;

    my %in_roles_id = map {$_ => undef} @$roles_id;
    my %affected_pages;

    my %have_accessors;
    while (my ($role, $page_accessors) = each(%role_page_accessor)) {
        if (exists($in_roles_id{$role})) {

            foreach my $page_accessor (@$page_accessors) {
                next if $have_accessors{$page_accessor};
                $have_accessors{$page_accessor} = TRUE;
                my $pages = $self->app->$page_accessor->get_all(
                    fields => [qw(id page_id public_id caption domain)],
                    filter => {multistate => $opts{multistate}, owner_id => $user->{id}}
                );

                # pages should be uniq across all roles
                push @{$affected_pages{$page_accessor}}, @$pages if @$pages;
            }
        }
    }

    return \%affected_pages;
}

sub on_action_set_excluded_domains {
    my ($self, $user, %opts) = @_;

    $self->user_global_excluded_domains->replace_multi($user->{'id'}, @{$opts{'domains'}}) if $opts{'domains'};

    return 1;
}

sub on_action_set_excluded_phones {
    my ($self, $user, %opts) = @_;

    $self->user_global_excluded_phones->replace_multi($user->{'id'}, @{$opts{'phones'}}) if $opts{'phones'};

    return 1;
}

# TODO: удалить после того как Фронт перейдет напрямую в модель user_features
sub get_all_features {
    my ($self) = @_;

    return $self->user_features->get_all_features();
}

sub on_action_set_user_role {
    my ($self, $user, %opts) = @_;

    throw Exception::Validation::BadArguments(gettext('Not exists next arguments : role_id'))
      if (!$opts{'role_id'});
    $opts{'role_id'} = [$opts{'role_id'}] if ref($opts{'role_id'}) ne 'ARRAY';

    $self->rbac->set_user_role($user->{'id'}, $opts{'role_id'}, %opts);

    my $pages_to_restore = $self->_get_owner_pages_based_on_roles(
        $user, $opts{'role_id'},
        multistate   => 'deleted',
        related_only => TRUE
    );

    foreach my $page_accessor (keys %$pages_to_restore) {
        my $pa = $self->app->$page_accessor;
        foreach (@{$pages_to_restore->{$page_accessor}}) {
            $pa->maybe_do_action($_, 'restore');
        }
    }

    return TRUE;
}

sub on_action_unlink_adfox_user {
    my ($self, $user, %opts) = @_;

    throw Exception::Validation::BadArguments(gettext('Not exists next arguments : adfox_id'))
      if (!$opts{'adfox_id'});

    $self->partner_db->user_adfox->delete(
        {
            user_id  => $user->{'id'},
            adfox_id => $opts{'adfox_id'},
        }
    );

    return TRUE;
}

sub on_action_unsubscribe_from_stat_monitoring_emails {
    my ($self, $user, %opts) = @_;

    $self->partner_db->users->edit($user->{'id'}, {no_stat_monitoring_emails => 1});
}

sub query {
    my ($self, %opts) = @_;

    my $filter    = $self->partner_db->filter($opts{'filter'});
    my $filter_or = $self->partner_db->filter();

    my @role_ids;
    unless ($self->check_short_rights('view_all')) {
        my %users_rights = (
            users_view_all_developers            => $DEVELOPER_ROLE_ID,
            users_view_all_an_internal_managers  => $INTERNAL_YAN_MANAGER_ROLE_ID,
            users_view_all_dsp_partners          => $DSP_PARTNER_ROLE_ID,
            users_view_all_dsp_managers          => $DSP_MANAGER_ROLE_ID,
            users_view_all_an_partners           => $SITE_PARTNER_ROLE_ID,
            users_view_all_video_an_partners     => $VIDEO_PARTNER_ROLE_ID,
            users_view_all_mobile_an_partners    => $MOBILE_PARTNER_ROLE_ID,
            users_view_all_adblock_an_partners   => $ADBLOCK_PARTNER_ROLE_ID,
            users_view_all_an_partner_assistants => $YAN_PARTNER_ASSISTANT_ROLE_ID,
            users_view_all_tutby_aggregators     => $TUTBY_ROLE_ID,
            users_view_all_indoor_an_partners    => $INDOOR_PARTNER_ROLE_ID,
            users_view_all_outdoor_an_partners   => $OUTDOOR_PARTNER_ROLE_ID,
        );

        foreach (keys(%users_rights)) {
            push(@role_ids, $users_rights{$_}) if $self->check_rights($_);
        }

        if ($self->check_rights('yndx_login')) {
            my $roles_company_users = $self->rbac->get_roles_by_rights('yndx_login');
            @role_ids = @{array_uniq(@role_ids, $roles_company_users)};
        }

        $filter_or->or(
            [
                id => '= ANY' => $self->partner_db->query->select(
                    table  => $self->partner_db->user_role,
                    fields => ['user_id'],
                    filter => {role_id => [sort {$a <=> $b} @role_ids]}
                )
            ]
        ) if @role_ids;

        if ($self->rbac->has_role_from_group_yndx_services([keys(%{$self->rbac->get_cur_user_roles() || {}})])) {
            $filter_or->or({id => $ADINSIDE_USER_ID});
        }

        if ($self->check_short_rights('view_all_tutby_partners')) {
            $filter_or->or({is_tutby => TRUE});
        }

        my $cur_user_id = $self->get_option('cur_user', {})->{'id'};

        if ($self->check_rights('is_assistant')) {
            my @owner_ids = map {$_->{'owner_id'}} @{
                $self->partner_db->query->select(
                    table  => $self->partner_db->all_pages,
                    fields => [qw(owner_id)],
                    filter => [
                        'page_id' => 'IN' => $self->partner_db->query->select(
                            table  => $self->partner_db->assistants,
                            fields => [qw(page_id)],
                            filter => {user_id => $cur_user_id}
                        )
                    ]
                  )->get_all()
              };

            $filter_or->or({id => \@owner_ids}) if @owner_ids;
        }

        $filter_or->or({id => $cur_user_id});
    }

    $filter_or->and($filter);

    my $query = $self->partner_db->query->select(
        table  => $self->partner_db_table(),
        fields => $opts{'fields'}->get_db_fields(),
        filter => $filter_or
    );

    return $query;
}

#
# API
#

sub api_available_actions {
    return qw(
      change_contract
      edit
      link_adfox_user
      reset_blocked
      revoke_roles
      set_blocked
      set_excluded_domains
      set_excluded_phones
      set_user_role
      unsubscribe_from_stat_monitoring_emails
      );
}

sub api_check_public_id {check_public_id($_[1], 'simple_id') || $_[1] eq 'current'}

sub api_get {
    my ($self, $pk, %opts) = @_;

    my $tmp_rights;
    if ($pk eq 'current') {
        $tmp_rights = $self->app->add_tmp_rights($self->get_right('view_field__login'));

        $pk = $self->get_option('cur_user', {})->{'id'};
    }

    return $self->SUPER::api_get($pk, %opts);
}

sub related_models {
    return {
        user_global_excluded_domains => {
            accessor => 'user_global_excluded_domains',
            filter   => sub {
                return {user_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
            },
            key_fields => ['user_id'],
            value_type => 'array_domain',
        },
        user_global_excluded_phones => {
            accessor => 'user_global_excluded_phones',
            filter   => sub {
                return {user_id => array_uniq(map {$_->{'id'} // ()} @{$_[1]})};
            },
            key_fields => ['user_id'],
            value_type => 'array_phone',
        },
    };
}

sub check_adfox_user {
    my ($self, %opts) = @_;

    my $error;

    $error = 'denied'    unless $self->check_rights('do_user_action_link_adfox_user');
    $error = 'signature' unless $self->api_adfox->check_signature($opts{'url'});

    my $adfox_id = $opts{'adfox_id'};

    $error = 'linked'
      if $self->partner_db->user_adfox->get({user_id => $opts{'id'}, adfox_id => $adfox_id},
        fields => [qw(user_id adfox_id)]);

    return {
        ($error ? (error => $error) : ()),
        url_ok     => $self->api_adfox->get_signed_adfox_uri("/RtbBlocks.php?status=ok&adfox_id=$adfox_id"),
        url_failed => $self->api_adfox->get_signed_adfox_uri("/RtbBlocks.php?status=failed&adfox_id=$adfox_id"),
        url_cancel => $self->api_adfox->get_signed_adfox_uri("/RtbBlocks.php?status=cancel&adfox_id=$adfox_id"),
    };
}

sub get_active_contract {
    my ($self, $opts) = @_;

    my $active_contract = {};
    foreach my $contract (@{$opts->{'contracts'}}) {
        if ($contract->{'Contract'}{'pi_info'}{'is_live'}) {
            $active_contract = $contract;

            last;
        }
    }

    my $no_delete = {
        Person => {
            map {$_ => TRUE} qw(longname name phone email fax legaladdress representative),
            ($self->check_short_rights('view_field__client_id') ? qw(id client_id) : ())
        },
        Contract => {map {$_ => TRUE} qw(external_id dt is_signed is_faxed partner_pct reward_type pi_info)},
    };

    foreach my $type (keys(%$active_contract)) {
        if (exists($no_delete->{$type})) {
            foreach my $field (keys(%{$active_contract->{$type}})) {
                delete($active_contract->{$type}{$field}) unless $no_delete->{$type}{$field};
            }
        } else {
            delete($active_contract->{$type});
        }
    }

    $active_contract->{'Person'}{'login'} = $opts->{'login'}
      if $self->check_rights('partner_acts_view_filter__login');

    return $active_contract;
}

sub get_all_assistants {
    my ($self, %opts) = @_;

    my $tmp = $self->app->add_tmp_rights('users_view_all_an_partner_assistants');

    return $self->get_by_roles(%opts, roles => [$YAN_PARTNER_ASSISTANT_ROLE_ID]);
}

sub get_all_managers {
    my ($self, %opts) = @_;

    return $self->get_by_roles(
        %opts,
        roles => [
            $DEVELOPER_ROLE_ID,   $INTERNAL_YAN_MANAGER_ROLE_ID, $INTERNAL_YAN_ADMINISTRATOR_ROLE_ID,
            $DSP_MANAGER_ROLE_ID, $YAN_MANAGER_ROLE_ID
        ]
    );
}

sub get_by_login {
    my ($self, $login, %opts) = @_;

    throw Exception::Validation::BadArguments gettext('Login is not specified') unless defined($login);

    return $self->get_all(%opts, filter => {login => fix_login($login)})->[0];
}

sub get_by_roles {
    my ($self, %opts) = @_;

    my @ids =
      map {$_->{'user_id'}}
      @{$self->partner_db->user_role->get_all(filter => {role_id => $opts{roles}}, fields => ['user_id'])};

    return [] unless @ids;

    return $self->get_all(%opts, filter => {id => \@ids});
}

sub get_by_uid {
    my ($self, $uid, @opts) = @_;

    return $self->get_all(filter => {uid => $uid, is_deleted => FALSE}, @opts)->[0];
}

=head2

Получает из баланса client_id для указанного uid или создает новый.

=cut

sub get_client_id {
    my ($self, $uid) = @_;

    my $client_id = $self->api_balance->get_client_id_by_uid($uid);
    unless ($client_id) {
        $client_id = $self->api_balance->create_client(
            operator_uid => $uid,
            NAME         => 'Client',
        );
        $self->api_balance->create_user_client_association(
            operator_uid => $uid,
            client_id    => $client_id,
            user_id      => $uid
        );
    }

    return $client_id;
}

sub get_dsp_managers {
    my ($self, %opts) = @_;

    $opts{order_by} = [['name'], ['lastname']] unless exists($opts{order_by});
    return $self->get_by_roles(%opts, roles => [$DSP_MANAGER_ROLE_ID]);
}

sub get_login_by_client_id {
    my ($self, $client_id) = @_;

    throw Exception::Validation::BadArguments gettext('Client ID is not specified') unless defined($client_id);
    throw Exception::Validation::BadArguments gettext('Incorrect Client ID') unless $client_id =~ /^\d+$/;

    my $maybe_dirty_login = $self->get_all(filter => {client_id => $client_id})->[0]{'login'};
    throw Exception::Validation::BadArguments gettext('Client ID %s is unknown', $client_id)
      unless defined($maybe_dirty_login);

    my $login = fix_login($maybe_dirty_login);
    return $login;
}

sub suggest_login {
    my ($self, %opts) = @_;

    my $login_filter = ($opts{'login_part'} ? [login => 'LIKE' => $opts{'login_part'}] : undef);

    my @filtered_roles;
    push(
        @filtered_roles,
        (
            $DSP_PARTNER_ROLE_ID,    $SITE_PARTNER_ROLE_ID,   $VIDEO_PARTNER_ROLE_ID,
            $MOBILE_PARTNER_ROLE_ID, $INDOOR_PARTNER_ROLE_ID, $OUTDOOR_PARTNER_ROLE_ID
        )
    ) if $opts{'partner'};
    push(@filtered_roles, $INTERNAL_YAN_MANAGER_ROLE_ID)  if $opts{'adv_net_internal_manager'};
    push(@filtered_roles, $SITE_PARTNER_ROLE_ID)          if $opts{'adv_net_partner'};
    push(@filtered_roles, $DSP_PARTNER_ROLE_ID)           if $opts{'dsp_partner'};
    push(@filtered_roles, $DSP_MANAGER_ROLE_ID)           if $opts{'dsp_manager'};
    push(@filtered_roles, $VIDEO_PARTNER_ROLE_ID)         if $opts{'video_an_partner'};
    push(@filtered_roles, $MOBILE_PARTNER_ROLE_ID)        if $opts{'mobile_an_partner'};
    push(@filtered_roles, $INDOOR_PARTNER_ROLE_ID)        if $opts{'indoor_an_partner'};
    push(@filtered_roles, $OUTDOOR_PARTNER_ROLE_ID)       if $opts{'outdoor_an_partner'};
    push(@filtered_roles, $YAN_PARTNER_ASSISTANT_ROLE_ID) if $opts{'assistant'};

    my $users = $self->get_all(
        fields => [qw(id login full_name avatar)],
        (
            $login_filter || @filtered_roles
            ? (
                filter => [
                    AND => [
                        ($login_filter ? $login_filter : ()),
                        (@filtered_roles ? [role_id => '=' => [@filtered_roles]] : ()),
                    ]
                ],
              )
            : ()
        ),
        limit => 5,
    );

    return $users;
}

=head2 send_email_to_processing

    $app->users->send_email_to_processing( user_id => $user_id );

=cut

sub send_email_to_processing {
    my ($self, %opts) = @_;

    my $tmp_rights = $self->app->add_all_tmp_rights();

    my $user_id = delete($opts{user_id});
    throw Exception::Validation::BadArguments "Must specify 'user_id'" unless $user_id;

    my $user = $self->get($user_id, fields => [qw(is_tutby login)]);
    throw Exception::Validation::BadArguments "User with user_id = $user_id not found" unless $user;

    # Забираем последнюю запись
    my $form_data = $self->partner_db->form_data->get_all(
        filter => {user_id => $user_id,},
        order_by => [['dt', 1]],
        limit    => 1,
    )->[0];

    # Эта саба выполняется после первой успешной модерации площадки/приложения партнера.
    # У всех партнеров, которые есть в системе уже есть успешно промодерированные сущности (иначе их не было бы в ПИ2)
    throw "No data in table form_data for user_id = $user_id" unless defined($form_data);

    $form_data->{data} = from_json($form_data->{data});

    my @attachment_fields = @{$form_data->{data}->{attachment_fields} // []};
    my @attachments;

    foreach my $field_id (@attachment_fields) {
        foreach my $s3_key (@{$form_data->{data}->{fields}->{$field_id}}) {

            my $s3_data = $self->api_media_storage_s3->get($s3_key);

            $s3_key =~ s/\//_/g;

            push @attachments,
              {
                filename => $s3_key,
                data     => $s3_data->{content},
              };
        }
    }

    if ($user->{is_tutby}) {
        my $body = 'login: ' . $user->{login} . "\n";

        foreach my $id (sort keys %{$form_data->{data}->{fields}}) {
            next if ref($form_data->{data}->{fields}->{$id}) ne '';
            $body .= sprintf "%s: %s\n", $id, $form_data->{data}->{fields}->{$id} // '';
        }

        $body .= "\nhttps://partner2.yandex.ru/v2/users/$user_id/edit/settings/";

        $self->mailer->send(
            from    => 'default',
            to      => $TUTBY_COP_MAIL,
            subject => 'Новый партнер TUT.BY в РСЯ',
            body    => $body,
        );
    } else {
        my $body = sprintf(
"Необходимо сформировать договор РСЯ.\n\nЛогин: %s\nClient ID: %s\nPerson ID: %s\n",
            $user->{login},
            $form_data->{client_id},
            $form_data->{person_id} // '',
        );

        if ($form_data->{data}->{contract}) {
            $body .= sprintf "Договор оферты: %s\n", $form_data->{data}->{contract}->{EXTERNAL_ID};
        }

        $body .= "\n\n\nВсе данные, которые партнер указал в анкете:\n\n";

        my $country_id = $form_data->{country_id};
        my $country_name;

        if ($form_data->{'data'}{'country_name_ru'}) {
            $country_name = $form_data->{'data'}{'country_name_ru'};
        } else {
            my $geo_data = $self->app->geo_base->get($country_id, fields => ['name']) // {name => '???'};
            $country_name = $geo_data->{name};
        }

        $body .= sprintf "country: %s (id: %s)\n", $country_name, $country_id;

        foreach my $id (sort keys %{$form_data->{data}->{fields}}) {
            if (ref($form_data->{data}->{fields}->{$id}) eq '') {
                $body .= sprintf "%s: %s\n", $id, $form_data->{data}->{fields}->{$id} // '';
            } else {
                $body .= sprintf "%s: %s\n", $id, to_json($form_data->{data}->{fields}->{$id});
            }
        }

        $self->mailer->send(
            from     => 'default',
            reply_to => {user_id => $user_id},
            to       => $DOCS_MAIL,
            subject  => 'РСЯ: заявка на создание договора',
            body     => $body,
            (@attachments ? (attachments => \@attachments) : ()),
        );
    }

    return 1;
}

sub check_pi_adfox_contracts {
    my ($self, %opts) = @_;

    @_ = ($self, '[FILTERED]');    # Защита от сохранения пароля в стектрейсе sentry

    my ($user_id, $adfox_login, $adfox_psw, $from_offer, $allow_paid_services, $allow_existing_test_mode_contract) =
      @opts{qw(user_id adfox_login adfox_password from_offer allow_paid_services allow_existing_test_mode_contract)};
    my %extra = (
        adfox_login         => $adfox_login,
        allow_paid_services => $allow_paid_services,
        from_offer          => $from_offer,
        user_id             => $user_id,
    );
    my ($pi_contract, $adfox_current_contract, $adfox_id, $adfox_pi_accounts);

    try {
        throw Exception::Validation::BadArguments gettext('User id is missed') unless ($user_id);

        my $tmp_rights = $self->app->add_tmp_rights('users_view_field__client_id');
        my $user = $self->get($user_id, fields => [qw(login client_id)]) // {};
        undef $tmp_rights;

        my $pi_login = $extra{pi_login} = $user->{login} // '';
        # Для существующего пользователя берём сохранённый client_id, если есть
        my $pi_client_id = $extra{pi_client_id} = $user->{client_id}
          // $self->api_balance->get_client_id_by_uid($user_id);
        my $pi_raw_contracts;
        $pi_raw_contracts = $self->documents->get_contracts(client_id => $pi_client_id) if ($pi_client_id);
        $pi_contract = $extra{pi_contract} = (
            grep {
                Application::Model::AgreementChecker::_Agreement->new(
                    contract    => $_->{Contract},
                    collaterals => $_->{Collaterals},
                  )->is_live_today()
              } @$pi_raw_contracts
          )[0]
          if ($pi_raw_contracts);

        if ($adfox_login) {
            my $adfox_info;
            try {
                $adfox_info = $extra{adfox_info} = $self->api_adfox_graphql->get_billing_client_info(
                    adfox_login => $adfox_login,
                    adfox_psw   => $adfox_psw,
                    uid         => $user_id,
                );
            }
            catch Exception::API::AdFoxGraphQL with {
                my ($exception) = @_;

                if ($exception->{adfox_error_category}) {
                    my $token = 'adfox_api_' . $exception->{adfox_error_category};
                    throw Exception::CommonOffer($token, parent => $exception, sentry => $exception->{sentry},)
                      if (exists $Exception::CommonOffer::TOKENS{$token});
                }
                throw $exception;
            };
            # Флаг отмены проверок на белый список, client_id и договор, если установлен в false
            my $need_close_contract = $extra{need_close_contract} = $adfox_info->{needCloseContract};

            # Пользователю adfox не разрешено заполнять единую оферту
            throw Exception::CommonOffer 'adfox_not_allowed'
              if (!$adfox_info->{inWhitelist});

            # Пользователю adfox по договору предоставляются платные услуги
            throw Exception::CommonOffer 'adfox_has_paid'
              if ($adfox_info->{hasPaidServices} && !$allow_paid_services);

            # У пользователю adfox на договоре висят другие пользователи
            throw Exception::CommonOffer 'adfox_shared_contract'
              if ($adfox_info->{hasOtherAccounts});

            # этот yandex_uid связан с другим полльзователем AdFox
            throw Exception::CommonOffer 'adfox_other_account_same_uid'
              if ($adfox_info->{hasSameUidInOtherAccount});

            # Пользователь adfox имеет сложные связи
            # (такого не должно быть, проверка на случай рассинхрона)
            throw Exception::CommonOffer 'adfox_complex'
              if ($adfox_info->{complexPiLinks});

            # Пользователь adfox связан с несколькими пользователями ПИ
            # (такого не должно быть, проверка на случай рассинхрона)
            $adfox_pi_accounts = $extra{adfox_pi_accounts} = $adfox_info->{piAccounts};
            throw Exception::CommonOffer 'adfox_multilink'
              if (@$adfox_pi_accounts > ($from_offer ? 1 : 0));

            $adfox_id = $extra{adfox_id} = $adfox_info->{userId};
            my $adfox_link = $extra{adfox_link} = $self->partner_db->query->select(
                table  => $self->partner_db->user_adfox,
                fields => [],
                filter => ($adfox_id ? [adfox_id => '=' => \$adfox_id] : [adfox_login => '=' => \$adfox_login]),
              )->join(
                table   => $self->partner_db->users,
                fields  => ['login'],
                join_on => [id => '=' => {user_id => $self->partner_db->user_adfox}],
              )->get_all();
            # Пользователь указал аккаунт adfox,
            # который уже имеет связь с каким-то другим пользователем ПИ
            throw Exception::CommonOffer 'adfox_has_link'
              if (@$adfox_link == 1 && $adfox_link->[0]{login} ne $pi_login || @$adfox_link > 1);

            if ($need_close_contract) {
                my $adfox_client_id = $extra{adfox_client_id} = $adfox_info->{billingClientId};
                # У пользователя adfox нет client_id
                throw Exception::CommonOffer 'adfox_no_client_id' unless ($adfox_client_id);

                my $adfox_contract = $extra{adfox_contract} = $adfox_info->{billingContractId};
                # Договора у пользователя adfox нет
                throw Exception::CommonOffer 'adfox_no_contract' unless ($adfox_contract);

                my $adfox_raw_contracts = $self->api_balance->get_partner_contracts(ClientID => $adfox_client_id);
                ($adfox_current_contract) =
                  grep {$_->{Contract}{contract2_id} eq $adfox_contract} @$adfox_raw_contracts;

                # Проверка на нужные параметры договора
                my $contract_type = $extra{adfox_contract_type} = $adfox_current_contract->{Contract}{type} // '';
                throw Exception::CommonOffer 'unsupported_adfox_contract_type'
                  if ($contract_type ne 'GENERAL');
            }
        }

        if ($from_offer) {    # существующий пользователь (со страницы оферты)

            # У действующего пользователя ПИ почему-то нет client_id
            throw Exception::CommonOffer 'user_no_client_id' unless ($pi_client_id);

            # У действующего пользователя ПИ почему-то нет активного договора
            unless ($pi_contract) {
                if ($pi_raw_contracts && @$pi_raw_contracts) {
                    # У него может быть не активный договор
                    $pi_contract = $extra{pi_contract_old} = $pi_raw_contracts->[-1];
                } else {
                    # Или совсем не быть никакого
                    throw Exception::CommonOffer 'user_no_contract' unless ($pi_contract);
                }
            }

            my $pi_contract_type = $extra{pi_contract_type} = $pi_contract->{Contract}{contract_type} // '';
            my $person_type      = $extra{pi_person_type}   = $pi_contract->{Person}{type}            // '';
            my $currency         = $extra{pi_currency}      = $pi_contract->{Contract}{currency}      // '';
            my $firm_id          = $extra{pi_firm}          = $pi_contract->{Contract}{firm}          // '';
            # У действующего пользователя неподдерживаемый тип договора
            throw Exception::CommonOffer 'unsupported_contract'
              unless ($self->documents->is_resident($pi_contract)
                || $self->documents->is_nonresident($pi_contract));

            my $allow = $extra{allow} = $self->partner_db->common_offer_allowed_users->get_all(
                fields => ['accept_date'],
                filter => [user_id => '=' => \$user_id],
            )->[0];
            # Менеджеры не добавили пользователя к списку готовых подписывать оферту
            throw Exception::CommonOffer 'user_not_allowed' unless ($allow);

            my $deny = $extra{deny} = $self->partner_db->common_offer_black_list->get_all(
                fields => ['opts'],
                filter => [user_id => '=' => \$user_id],
            )->[0];
            # Менеджеры добавили пользователя в чёрный список
            throw Exception::CommonOffer 'user_not_allowed' if ($deny);

            # Пользователь adfox связан с другим пользователям ПИ
            # (такого не должно быть, проверка на случай рассинхрона)
            throw Exception::CommonOffer 'adfox_other_link'
              if ( $adfox_pi_accounts
                && @$adfox_pi_accounts == 1
                && fix_login($adfox_pi_accounts->[0]) ne $pi_login);

            my $pi_link = $extra{pi_link} = $self->partner_db->user_adfox->get_all(
                fields => [qw(adfox_id adfox_login)],
                filter => [user_id => '=' => \$user_id],
            );
            # Пользователь указал аккаунт adfox (или не указал ничего) в то время,
            # как уже имеет связь с каким-то другим пользователем adfox
            throw Exception::CommonOffer 'user_has_link'
              if (
                   @$pi_link > 1
                || @$pi_link == 1
                && (
                    !$adfox_login
                    || $adfox_login && ($adfox_id && $adfox_id != $pi_link->[0]{adfox_id}
                        || !$adfox_id && lc($pi_link->[0]{adfox_login}) ne lc($adfox_login))
                   )
                 );
        } else {    # новый пользователь (из анкеты)
            if ($pi_contract) {
                if ($allow_existing_test_mode_contract) {
                    # Эта ветка используется для асессоров
                    # При старте тестирования каждого релиза асессор регистрируется в анкете под своим логином.
                    # Когда на препрод выезжает новый билд, используется чистая база, в которой этого юзера уже нет.
                    # Но в балансе договор остается и мешает создать новый.
                    # Поэтому нужно подтягивать существующий, если он имеется.
                    # Дополнительно проверяем, что договор похож на асессорский.
                    if (exists($pi_contract->{Person}) && $pi_contract->{Person}{id}
                        || !$pi_contract->{Contract}{test_mode})
                    {
                        throw Exception::CommonOffer 'assessor_suspicious_contract';
                    }
                } else {
                    # У нового пользователя ПИ почему-то уже есть client_id и активный договор
                    # Такого не должно быть и должно отсеиваться на этапе can_fill_form в анкете
                    throw Exception::CommonOffer 'user_with_contract';
                }
            }
        }
    }
    catch {
        # Поймать, добавить инфу, отправить сообщение и бросить дальше
        my ($exception) = @_;

        $exception->{sentry}{extra}{common_offer} = \%extra;
        ERROR $exception;

        local $Data::Dumper::Sortkeys = 1;
        my $message = join(
            "\n",
            'Пользователь не смог пройти проверки для '
              . (
                $extra{from_offer}
                ? 'перехода на единую оферту'
                : 'регистрации по единой оферте'
              ),
            '-' x 40,
            'Error: ' . $exception->message(),
            sprintf(
                'PI: login=%s, id=%s, client_id=%s',
                ($extra{pi_login}     // ''),
                ($extra{user_id}      // ''),
                ($extra{pi_client_id} // '')
            ),
            sprintf('AdFox: login=%s, client_id=%s', ($extra{adfox_login} // ''), ($extra{adfox_client_id} // '')),
            '',
            'Справочные данные:',
            '-' x 40,
            (
                $extra{adfox_login}
                ? (
                    'AdFox:',
                    '    Login: ' . $extra{adfox_login},
                    (
                        $extra{adfox_info}
                        ? (
                            '    флаг необходимости закрытия договора: '
                              . ($extra{adfox_info}{needCloseContract} ? 'да' : 'нет'),
                            '    балансовый номер клиента: '
                              . ($extra{adfox_info}{billingClientId} // ''),
                            '    номер договора: ' . ($extra{adfox_info}{billingContractId} // ''),
                            '    платные услуги: '
                              . ($extra{adfox_info}{hasPaidServices} ? 'есть' : 'нет'),
                            '    общий договор на несколько аккаунтов: '
                              . ($extra{adfox_info}{hasOtherAccounts} ? 'да' : 'нет'),
                            '    сложные связи: '
                              . ($extra{adfox_info}{complexPiLinks} ? 'есть' : 'нет'),
                            '    белый список: ' . ($extra{adfox_info}{inWhitelist} ? 'есть' : 'нет'),
                            sprintf(
                                '    связи с аккаунтами ПИ в AdFox: [ %s ]',
                                join(
                                    ', ',
                                    sort @{
                                        ref $extra{adfox_info}{piAccounts} eq 'ARRAY'
                                        ? $extra{adfox_info}{piAccounts}
                                        : []
                                      }
                                    )
                                   ),
                          )
                        : ()
                    ),
                    sprintf(
                        '    связи с аккаунтами ПИ в ПИ: [ %s ]',
                        join(', ',
                            map {$_->{login}} sort @{ref $extra{adfox_link} eq 'ARRAY' ? $extra{adfox_link} : []})
                           ),
                  )
                : ()
            ),
            'Партнёрский Интерфейс:',
            '    Login: ' . ($extra{pi_login} // ''),
            (
                $extra{pi_client_id}
                ? '    балансовый номер клиента: ' . $extra{pi_client_id}
                : ()
            ),
            (
                $extra{pi_contract} && $extra{pi_contract}{Contract} && $extra{pi_contract}{Contract}{contract2_id}
                ? '    номер договора: ' . $extra{pi_contract}{Contract}{contract2_id}
                : ()
            ),
            sprintf('    связи с аккаунтами AdFox: [ %s ]',
                join(', ', map {$_->{adfox_login}} sort @{ref $extra{pi_link} eq 'ARRAY' ? $extra{pi_link} : []})),
            'Согласие на подключение платных услуг: '
              . ($allow_paid_services ? 'да' : 'нет'),
            '',
            'Технические детали:',
            '-' x 40,
            $exception,
            Dumper \%extra
        );
        $self->mail_notification->add_common_offer_error(
            subject => sprintf(
                'Common offer failed - Оферта, login=%s, adfox_login=%s',
                ($extra{pi_login} || $extra{user_id} // ''),
                ($extra{adfox_login} // '')
            ),
            to      => [$PARTNER2_COMMON_OFFER_ERRORS, $CONTRACT_OFFER_ERROR_MAIL],
            message => $message,
        );
        $exception->{'__mailed'} = TRUE;

        throw $exception;
    };

    return {
        adfox_contract => $adfox_current_contract,
        pi_contract    => $pi_contract,
    };
}

sub accept_common_offer {
    my ($self, %opts) = @_;

    my ($client_id, $contracts) = @opts{qw(client_id contracts)};
    my %extra = %opts;
    my $user_id = $extra{user_id} = $self->get_option('cur_user', {})->{'id'};
    my $contract;

    try {
        throw Exception::Denied gettext('Unknown user') unless ($user_id);
        my $pi_login = $extra{pi_login} = ($self->get($user_id, fields => [qw(login)]) // {})->{login} // '';

        my $curdate = curdate(oformat => 'db');
        my $delta   = 7;
        my $end_dt  = date_add(curdate(), day => $delta, oformat => 'iso8601');
        my $next_dt;
        foreach my $part (keys %$contracts) {
            my $cntrct = $contracts->{$part};

            next unless $cntrct;

            $extra{closing} = $cntrct;
            if ($cntrct->{Contract}{end_dt} || $cntrct->{Contract}{is_cancelled}) {
                # У договора есть дата окончания или он аннулирован

                # Если это договор ПИ, новый должен начаться сразу
                $next_dt = $curdate if $part eq 'pi_contract';
            } elsif ($part eq 'pi_contract'
                && $cntrct->{Contract}{contract_type}
                && $cntrct->{Contract}{contract_type} == $CONTRACT_TYPE{OFFER})
            {
                # Текущую оферту не переоформляем, используем как есть
                $contract = $extra{new_contract} = {
                    EXTERNAL_ID => $cntrct->{Contract}{external_id},
                    ID          => $cntrct->{Contract}{contract2_id},
                };
                next;
            } elsif ($part eq 'pi_contract'
                && $cntrct->{Contract}{contract_type} == $CONTRACT_TYPE{YAN_2014}
                && !$cntrct->{Contract}{is_faxed}
                && !$cntrct->{Contract}{is_signed})
            {
                # У пользователя есть старый неактивный договор
                # Допник к нему создать нельзя, но ему можно обновить дату окончания
                # Теоретически сюда не должно попасть, так как Коля аннулировал все такие договора PAYSUP-464353
                my $c_id = $cntrct->{Contract}{contract2_id};
                $next_dt = $curdate;
                WARNF('User "%s" (%s) had old contract %s / %s',
                    $pi_login, $user_id, $c_id, $cntrct->{Contract}{external_id});
            } else {
                # Для остальных случаев создаём допник на закрытие
                $self->api_balance->close_contract(
                    END_DT        => $end_dt,
                    contract_id   => $cntrct->{Contract}{contract2_id},
                    contract_type => $cntrct->{Contract}{type},
                    operator_uid  => $user_id,
                );
            }
            $extra{close}{$part} = $cntrct;
            delete $extra{closing};
            if ($part eq 'adfox_contract') {
                $self->mail_notification->add(
                    type    => 0,
                    user_id => 0,
                    opts    => {
                        check_alive => FALSE,
                        subject     => sprintf(
                            'Закрытие договора Adfox N %s / %s',
                            ($cntrct->{Contract}{contract2_id} // ''),
                            ($cntrct->{Contract}{external_id}  // ''),
                        ),
                        to     => [$PARTNER2_COMMON_OFFER_ERRORS, $COMMON_OFFER_DOCS],
                        values => {
                            message_body => join(
                                "\n",
                                'В рамках перехода на Единую Оферту',
                                sprintf('для пользователя "%s"', $extra{adfox_login}),
                                sprintf(
'создано доп.соглашение о расторжении договора N %s / %s',
                                    ($cntrct->{Contract}{contract2_id} // ''),
                                    ($cntrct->{Contract}{external_id}  // ''),
                                ),
                            ),
                        },
                    },
                );
            }
        }

        # Закрыли в ПИ договор, создаём новый
        unless ($contract) {
            my %params;
            if ($opts{from_offer}) {
                my $start_dt = $next_dt // date_add(curdate(), day => $delta + 1, oformat => 'db');
                my $currency      = $contracts->{pi_contract}{Contract}{currency}      // '';
                my $contract_type = $contracts->{pi_contract}{Contract}{contract_type} // '';
                my $test_mode = ($contracts->{pi_contract}{Contract}{test_mode} ? TRUE : FALSE);
                my $signed    = !$test_mode;
                my $firm_id   = (
                      $self->documents->is_nonresident($contracts->{pi_contract})
                    ? $FIRM_ID_YANDEX_EUROPE_AG
                    : $FIRM_ID_YANDEX_LTD
                );
                %params =
                  ( # Параметры аналогичны тем, что в форме или предыдущем договоре
                    operator_uid => $user_id,
                    client_id    => $client_id,
                    (
                        $contracts->{pi_contract}{Person}{id}
                        ? (person_id => $contracts->{pi_contract}{Person}{id},)
                        : ()
                    ),

                    currency     => $self->documents->get_currency_for_offer($currency),
                    ctype        => 'PARTNERS',
                    firm_id      => $firm_id,
                    payment_type => 1,
                    partner_pct  => 43,

                    start_dt         => $start_dt,
                    service_start_dt => $start_dt,

                    test_mode => ($test_mode ? 1 : 0),
                    signed    => ($signed    ? 1 : 0),
                    nds       => $contracts->{pi_contract}{Contract}{nds},
                  );
            } else {
                throw Exception::Validation::BadArguments gettext('Missed required fields: %s', 'params')
                  unless ($opts{params});

                %params = %{$opts{params}};
            }
            $contract = $extra{new_contract} = $self->api_balance->create_offer(%params);
        }
    }
    catch {
        my ($exception) = @_;

        $exception->{sentry}{extra}{common_offer} = \%extra;
        ERROR $exception;

        local $Data::Dumper::Sortkeys = 1;
        my $message = join(
            "\n",
            'Пользователь не смог перейти на оферту',
            '-' x 40,
            'Error: ' . $exception->message(),
            sprintf(
                'PI: login=%s, id=%s, client_id=%s',
                ($extra{pi_login}  // ''),
                ($extra{user_id}   // ''),
                ($extra{client_id} // '')
            ),
            'AdFox: login=' . ($extra{adfox_login} // ''),
            '',
            'Справочные данные:',
            '-' x 40,
            (
                ref($extra{close}) eq 'ARRAY' && @{$extra{close}}
                ? (
                    'Созданы доп.соглашения на закрытие договоров:',
                    map {
                        sprintf(
'    (%s) основной тип=%s, номер=%s, внешний номер=%s, подтип=%s',
                            $_,
                            ($extra{close}{$_}{Contract}{type}          // ''),
                            ($extra{close}{$_}{Contract}{contract2_id}  // ''),
                            ($extra{close}{$_}{Contract}{external_id}   // ''),
                            ($extra{close}{$_}{Contract}{contract_type} // ''),
                          )
                      } keys %{$extra{close}}
                  )
                : ()
            ),
            (
                ref($extra{closing}) eq 'HASH'
                ? (
                    'Попытка закрыть договор:',
                    sprintf(
                        '    основной тип=%s, номер=%s, внешний номер=%s, подтип=%s',
                        ($extra{closing}{Contract}{type}          // ''),
                        ($extra{closing}{Contract}{contract2_id}  // ''),
                        ($extra{closing}{Contract}{external_id}   // ''),
                        ($extra{closing}{Contract}{contract_type} // ''),
                    )
                  )
                : ()
            ),
            (
                ref($extra{new_contract}) eq 'HASH'
                ? (
                    (
                        $contracts->{pi_contract}
                          && $contracts->{pi_contract}{Contract}{contract_type}
                          && $contracts->{pi_contract}{Contract}{contract_type} == 9
                        ? 'Договор ПИ оставшийся в силе в качестве договора Единой Оферты:'
                        : 'Новый договор Единой Оферты:'
                    ),
                    sprintf(
                        '    основной тип=%s, номер=%s, внешний номер=%s',
                        ($opts{params} && $opts{params}{ctype} // 'PARTNERS'),
                        ($extra{new_contract}{ID}          // ''),
                        ($extra{new_contract}{EXTERNAL_ID} // ''),
                    )
                  )
                : ()
            ),
            '',
            'Технические детали:',
            '-' x 40,
            $exception,
            Dumper \%extra
        );
        $self->mail_notification->add_common_offer_error(
            subject => sprintf(
                'Common offer failed - Оферта, login=%s, adfox_login=%s',
                ($extra{pi_login} || $extra{user_id} // ''),
                ($extra{adfox_login} // '')
            ),
            to      => [$PARTNER2_COMMON_OFFER_ERRORS, $CONTRACT_OFFER_ERROR_MAIL],
            message => $message,
        );
        $exception->{'__mailed'} = TRUE;

        throw $exception;
    };

    return $contract;
}

sub adfox_offer_email_notification {
    my ($self, %opts) = @_;

    my $user_id = $opts{user_id};
    throw Exception::Validation::BadArguments gettext("Must specify 'user_id'") unless $user_id;

    if (grep {!exists $opts{$_}} qw(adfox_info client_id contract login)) {
        my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_field__client_id));
        my $user = $self->get($user_id, fields => [qw(active_contract adfox_info client_id login)]);
        throw Exception::Validation::BadArguments gettext('User with id "%s" not found', $user_id) unless $user;
        $opts{$_} //= $user->{$_} foreach (qw(client_id login));
        $opts{adfox_info} //= $user->{adfox_info}[0];
        $opts{contract}   //= {};
        $opts{contract}{$_} //= $user->{active_contract}{Contract}{$_} foreach (qw(contract2_id external_id));
    }

    $self->mail_notification->add(
        opts => {
            values => {
                (
                    $opts{adfox_contract}
                    ? (
                        af_client_id       => $opts{adfox_contract}{Client}{id},
                        af_contract_ext_id => $opts{adfox_contract}{Contract}{external_id},
                        af_contract_id     => $opts{adfox_contract}{Contract}{contract2_id},
                      )
                    : ()
                ),
                af_login           => $opts{adfox_info}{login},
                af_user_id         => $opts{adfox_info}{id},
                pi_client_id       => $opts{client_id},
                pi_contract_ext_id => $opts{contract}{external_id},
                pi_contract_id     => $opts{contract}{contract2_id},
                pi_login           => $opts{login},
                pi_user_id         => $user_id,
            },
        },
        type    => 7,
        user_id => 0,
    );
}

sub _on_offer_error_send_email {
    my ($self, $case, $exception, $extra) = @_;
    if ('case_add_adfox_paidprooducts' eq $case) {
        $exception->{sentry}{extra}{common_offer} = $extra;
        ERROR $exception;

        local $Data::Dumper::Sortkeys = 1;
        my $message = join(
            "\n",
            'Пользователь не смог принять оферту на платные услуги AdFox',
            '-' x 40,
            'Error: ' . $exception->message(),
            sprintf('PI: id=%s, client_id=%s', ($extra->{user}{id} // ''), ($extra->{user}{client_id} // '')),
            '',
            'Справочные данные:',
            '-' x 40,
            (
                exists $extra->{adfox_conract_created}
                ? (
                    'Создана оферта для AdFox PAIDPRODUCTS:',
                    sprintf(
                        'клиент в Балансе=%s, номер договора=%s',
                        ($extra->{adfox_conract_created}{client_id} // ''),
                        ($extra->{adfox_conract_created}{id}        // ''),
                    )
                  )
                : ()
            ),
            '',
            'Технические детали:',
            '-' x 40,
            $exception,
            Dumper $extra
        );
        $self->mail_notification->add_common_offer_error(
            subject => sprintf(
                'Common offer failed - Оферта, PI users.id=%s, adfox_login=%s',
                ($extra->{user}->{id} // ''),
                ($extra->{user}{adfox_info}[0]{login} // '')
            ),
            to      => [$PARTNER2_COMMON_OFFER_ERRORS, $CONTRACT_OFFER_ERROR_MAIL],
            message => $message,
        );
        $exception->{'__mailed'} = TRUE;
    }
}

=head3
    my %person_to_clone_from = $self->get_balance_person_to_clone($balance_client_id, $balance_person_id);

    returns a hash which may be passed to Balance.CreatePerson to create a new person for given client
    TODO: it will become obsolete soon; one person will fit all client's contracts
=cut

sub _get_balance_person_to_clone {
    my ($self, $balance_client_id, $balance_person_id) = @_;

    my ($IS_PARTNER, $RETURN_FULL_DATA) = (1, 1);
    my $balance_person_list_for_client =
      $self->api_balance->get_client_persons($balance_client_id, $IS_PARTNER, $RETURN_FULL_DATA);

    my %person2clone = %{[grep {$_->{id} eq $balance_person_id} @$balance_person_list_for_client]->[0]};
    unless (%person2clone) {
        throw Exception::Validation::BadArguments gettext(
            'Balance.GetClientPersons does not return person with id=[%s] for client_id=[%s]',
            $balance_person_id, $balance_client_id);
    }
    delete $person2clone{id};

    if (exists $person2clone{birthday} && check_date($person2clone{birthday}, iformat => 'db_time')) {
        $person2clone{birthday} = format_date($person2clone{birthday}, '%Y-%m-%d', iformat => 'db_time');
    }

    return %person2clone;
}

sub create_contract_adfox_paid_products {
    my ($self, %opts) = @_;

    my $user_id = $opts{user_id};
    throw Exception::Validation::BadArguments gettext("Must specify 'user_id'") unless $user_id;

    my $tmp_rights = $self->app->add_tmp_rights(qw(users_view_field__client_id users_view_field__id));
    my $user = $self->get($user_id, fields => [qw(adfox_info client_id)]);
    throw Exception::Validation::BadArguments gettext('User with id "%s" not found', $user_id) unless $user;

    if (my $common_offer_data = $self->partner_db->common_offer_allowed_users->get($user_id, fields => [qw(opts)])) {
        my $opts = from_json($common_offer_data->{opts});
        if (my $contract_paid_products = $opts->{contracts}{adfox_contract_paid_products}) {
            WARNF(
                'User[%s] already has offer for AdFox with PAIDPRODUCTS: client_id[%s] contract_id[%s]',
                $user_id,
                $contract_paid_products->{client_id},
                $contract_paid_products->{contract_id}
            );
            return {
                id                                     => $contract_paid_products->{contract_id},
                client_id                              => $contract_paid_products->{client_id},
                is_billing_data_already_saved_in_adfox => TRUE,
            };
        }
    }

    my $contract = $opts{contract};
    unless ($contract) {
        my $raw_contracts = $self->documents->get_contracts(client_id => $user->{client_id});
        $contract = (
            grep {
                Application::Model::AgreementChecker::_Agreement->new(
                    contract    => $_->{Contract},
                    collaterals => $_->{Collaterals},
                  )->is_live_today()
              } @$raw_contracts
          )[0]
          if ($raw_contracts);
        unless ($contract) {
            if ($raw_contracts && @$raw_contracts) {
                # У него может быть не активный договор
                $contract = $raw_contracts->[-1];
            } else {
                # Или совсем не быть никакого
                throw Exception::CommonOffer 'user_no_contract' unless ($contract);
            }
        }
    }
    if ($self->documents->is_nonresident($contract)) {
        $self->adfox_offer_email_notification(
            exists $opts{mail_info}
            ? %{$opts{mail_info}}
            : (user_id => $user_id,)
        );
        return FALSE;
    }

    # 1. create new Person from existing Person from PI contract:
    # can not use this Person for cloning bc it does not have all the required fields
    throw Exception::Validation::BadArguments(gettext('Can not create AdFox offer: no ID in contract.Person'))
      unless $contract->{Person}{id};
    my %person2clone = $self->_get_balance_person_to_clone($user->{client_id}, $contract->{Person}{id});

    my $person_id_new = $self->api_balance->create_person(
        %person2clone,
        operator_uid => $user_id,
        client_id    => $user->{'client_id'},
        person_id  => -1,    # -1 означает что нужно создать нового плательщика
        is_partner => 0
    );

    # 2. create new contract for newly added person
    my $args4create_offer_in_balance = $self->api_balance->make_create_offer_params_adfox_paid_services(
        operator_uid      => $user_id,
        client_id         => $user->{'client_id'},
        person_id         => $person_id_new,
        contract_currency => $self->documents->get_currency_for_offer($contract->{Contract}{currency} // ''),
    );

    # { 'EXTERNAL_ID' => '702491/19', 'ID' => '1236739' }
    my $contract_created = $self->api_balance->create_offer(%$args4create_offer_in_balance);
    throw Exception::Validation::BadArguments(gettext('Can not create AdFox offer: no ID from Balance2.CreateOffer'))
      unless ($contract_created && $contract_created->{ID});

    return {
        id        => $contract_created->{ID},
        client_id => $user->{'client_id'},
    };
}

sub api_can_edit {TRUE}

sub api_can_add {TRUE}

sub get_roles_and_rights {
    my ($self, $user_id) = @_;

    my $roles    = $self->rbac->get_roles_by_user_id($user_id);
    my @role_ids = keys %$roles;

    my @rights = $self->user_features->get_user_rights($user_id, \@role_ids);
    push @rights, map {$_->{'right'}} @{$self->rbac->get_roles_rights(fields => [qw(right)], role_id => \@role_ids)};

    my $only_feature_rights = 1;
    push @rights, keys %{$self->app->assistants->get_rights_for_assistant($user_id, $roles, $only_feature_rights)};

    return ($roles, [sort @rights]);
}

sub has_feature {
    my ($self, $user_id, $feature) = @_;

    return 0 < @{
        $self->user_features->get_all(
            fields => [qw(id)],
            filter => {user_id => $user_id, feature => $feature}
        )
      };
}

sub _stop_and_delete_user_pages {
    my ($self, %opts) = @_;

    my $user = $opts{user};

    # Send to archive pages
    my $pages_to_delete =
      $self->_get_owner_pages_based_on_roles($user, $opts{'roles_id'}, multistate => 'not (deleted or protected)');

    foreach my $page_accessor (keys %$pages_to_delete) {
        my $pa = $self->app->$page_accessor;
        foreach (@{$pages_to_delete->{$page_accessor}}) {
            $pa->maybe_do_action($_, 'stop_testing');
            $pa->maybe_do_action(
                $_, 'stop',
                'suppress_mail_notification' => $opts{'suppress_mail_notification'},
                'on_revoke_user_roles'       => 1
            );
            $pa->maybe_do_action(
                $_, 'delete',
                'suppress_mail_notification' => $opts{'suppress_mail_notification'},
                'on_revoke_user_roles'       => 1
            );
        }
    }

    # Create StarTrek task for protected pages
    my $protected_pages = $self->_get_owner_pages_based_on_roles($user, $opts{'roles_id'}, multistate => 'protected');
    return TRUE unless %$protected_pages;

    my $message = $self->_get_message($protected_pages, %opts);
    my $issue_id;
    eval {
        $issue_id = $self->app->startrek->create_issue(
            {
                queue       => 'PISUP',
                type        => 'task',
                summary     => $message->{title},
                description => $message->{message},
                followers   => [$DEPARTMENT_HEAD_PI, $DEV_GROUP_HEAD_PI, $RESPONSIBLE_TEST_ENGINEER_PI],
                tags        => [qw(partner2-cron failed_role_revoke)]
            }
        );
    };

    throw Exception::Queue::StartrekIssues('Failed to create issue for roles revoke and protected pages')
      unless $issue_id;

    return TRUE;
}

sub _get_message {
    my ($self, $protected_pages, %opts) = @_;

    my $user = $opts{user};
    my ($user_id, $login, $client_id) = @$user{qw(id login client_id)};

    my $roles = $self->rbac->get_roles(ids => $opts{'roles_id'});

    my $base_url = 'https://partner2.yandex.ru/v2/';
    my %route    = (
        users                    => 'users/',
        context_on_site_campaign => 'context/campaigns/',
        search_on_site_campaign  => 'search/campaigns/',
        video_an_site            => 'video/sites/',
        mobile_app_settings      => 'mobile/applications/'
    );

    my $page_link = sub {
        $base_url . $route{$_[0]} . '?search=%7B%22page_id%22%3A%5B%22%3D%22%2C%22' . $_[1] . '%22%5D%7D';
    };
    my $user_link =
      $base_url . $route{'users'} . '?search=%7B%22client_id%22%3A%5B%22%3D%22%2C%22' . $client_id . '%22%5D%7D';

    my $title =
"Площадки владельца $login не могут быть перенесены в архив автоматически";
    my $message = sprintf(
q[При снятии ролей %s с пользователя %s, его площадки не могут быть автоматически заархивированы, так как сейчас имеют статус **protected**.
Требуется отправить их в архив вручную, отправить информацию команде БК.
Список площадок:],
        join(', ', map {"'" . $_->{'name'} . '(' . $_->{'id'} . ')' . "'"} @$roles),
        "(($user_link $login)) (id: $user_id client_id: $client_id)"
    );

    my @pages_list;
    foreach my $page_accessor (keys %$protected_pages) {
        foreach (@{$protected_pages->{$page_accessor}}) {
            push @pages_list,
              sprintf(
                "caption: %s domain: %s id: %d page_id: %d ((%s link))",
                $_->{'caption'} || '-//-',
                $_->{'domain'}  || '-//-',
                $_->{'id'}, $_->{'page_id'}, $page_link->($self->app->$page_accessor->accessor(), $_->{'page_id'})
              );
        }
    }
    $message .= "\n" . join("\n", @pages_list);

    return {
        title   => $title,
        message => $message,
    };
}

sub payoneer_get_payee_id {
    my ($self, $user) = @_;

    my $payee_id = $user->{payoneer_payee_id} || substr(md5_hex(reverse $user->{id}), 0, 30);

    return $payee_id;
}

sub payoneer_save_user {
    my ($self, $user_id, %opts) = @_;

    try {
        $self->do_action($user_id, 'edit', %opts);
    }
    catch {
        my ($exception) = @_;
        $exception->{sentry}{extra}{fingerprint} //= ['Users', 'payoneer_save_user'];
        ERROR $@;
        throw Exception::Validation::BadArguments gettext('Error while save user');
    };

    return TRUE;
}

sub payoneer_error_message {
    my ($self, %opts) = @_;

    local $Data::Dumper::Sortkeys = 1;
    my $message = join(
        "\n",
'Не удалось создать договор для перехода на платёжную систему Payoneer',
'Надо проанализировать ситуацию и довести создание договора до конца или аннулировать допник на закрытие предыдущего.',
        'Error: ' . $opts{exception}->message(),
        sprintf(
            'login=%s, id=%s, client_id=%s',
            ($opts{user}{login}     // ''),
            ($opts{user}{id}        // ''),
            ($opts{user}{client_id} // '')
        ),
        '',
        'Технические детали:',
        '-' x 40,
        Dumper \%opts
    );

    $self->mail_notification->add(
        type    => 0,
        user_id => 0,
        opts    => {
            check_alive => FALSE,
            subject =>
              sprintf('Payoneer offer failed - login=%s, id=%s', ($opts{user}{login} // ''), ($opts{user}{id} // '')),
            to     => $PARTNER2_COMMON_OFFER_ERRORS,
            values => {
                message_body       => $message,
                plain_text_wrapper => TRUE,
            },
        },
    );

    return TRUE;
}

sub payoneer_check_currency {
    my ($self, %opts) = @_;

    my $user = $self->get($self->get_option('cur_user'), fields => [qw(active_contract id payoneer payoneer_step)]);

    # Ошибка, если не достутпно
    throw Exception::Denied gettext('Access denied') unless $user->{payoneer};

    # Ошибка, если этот шаг уже прошли
    # Разрешаем вернуться, если ещё не прошли 3 шаг
    $user->{payoneer_step} //= 0;
    throw Exception::Denied gettext('Access denied')
      unless !$user->{payoneer_step} || $opts{force} && $user->{payoneer_step} < 3;

    # Ошибка, если нет договора
    throw Exception::Validation::BadArguments gettext('User has not contract')
      unless $user->{active_contract} && $user->{active_contract}{Contract};

    # Проверка валюты текущего договора
    my $currency = $user->{active_contract}{Contract}{currency};
    if ($self->documents->is_currency_eur($currency) || $self->documents->is_currency_usd($currency)) {
        # Получение буквенного обозначения валюты
        $currency = $self->documents->get_currency_for_offer($currency);

        # Валюта подходящая, пропускаем шаг, используя то, что есть
        $self->payoneer_save_user(
            $user->{id},
            payoneer_currency => $currency,
            payoneer_step     => 1,
            ($opts{force} ? (payoneer_url => '') : ()),
        );
    } else {
        # Валюта не подходящая, надо выбрать что-то нужное и перезаключить договор
        $currency = '';
    }

    return $currency;
}

sub payoneer_set_currency {
    my ($self, %opts) = @_;

    # Проверим, а можно ли было сюда попадать
    throw Exception::Denied gettext('Access denied') if $self->payoneer_check_currency(force => $opts{force},);

    # Проверка переданной валюты
    my $currency = $opts{currency};
    if ($self->documents->is_currency_eur($currency) || $self->documents->is_currency_usd($currency)) {
        # Получение буквенного обозначения валюты
        $currency = $self->documents->get_currency_for_offer($currency);

        # Валюта подходящая, записываем и переходим на следующий шаг
        $self->payoneer_save_user(
            $self->get_option('cur_user')->{id},
            payoneer_currency => $currency,
            payoneer_step     => 1,
            ($opts{force} ? (payoneer_url => '') : ()),
        );
    } else {
        throw Exception::Validation::BadArguments gettext('Invalid currency');
    }

    return TRUE;
}

sub payoneer_get_login_link {
    my ($self, %opts) = @_;

    my $user = $self->get($self->get_option('cur_user'),
        fields => [qw(id payoneer payoneer_currency payoneer_payee_id payoneer_step)]);

    # Ошибка, если не достутпно
    throw Exception::Denied gettext('Access denied') unless $user->{payoneer};

    # Ошибка, если этот шаг уже прошли или не дошли
    # Разрешаем вернуться, если ещё не прошли 3 шаг
    $user->{payoneer_step} //= 0;
    throw Exception::Denied gettext('Access denied')
      unless 1 == $user->{payoneer_step}
          || $opts{force} && $user->{payoneer_step} < 3;

    my $currency = $user->{payoneer_currency};
    throw Exception::Validation::BadArguments gettext('Invalid currency') unless $currency;

    # Получим идентификатор плательщика в Пионере
    # или то, что в БД, или из метода
    my $payee_id = $self->payoneer_get_payee_id($user);

    # Получим линк из Пионера
    my $link;
    try {
        $link = $self->api_balalayka->get_payoneer_login_link($currency, $payee_id,);
    }
    catch Exception::API::Payoneer with {
        my ($exception) = @_;
        $exception->{sentry}{extra}{fingerprint} //= ['Users', 'payoneer_get_login_link'];
        ERROR $@;
        throw Exception::Validation::BadArguments gettext('Cannot get Payoneer login-link');
    };

    # Сохраним позицию
    $self->payoneer_save_user(
        $user->{id},
        payoneer_payee_id => $payee_id,
        payoneer_step     => 2,
        payoneer_url      => $link,
    );

    return $link;
}

sub payoneer_check_payee_status {
    my ($self, %opts) = @_;

    my $tmp_rights = $self->app->add_tmp_rights('users_view_field__client_id');
    my $user       = $self->get($self->get_option('cur_user'),
        fields => [qw(client_id id payoneer payoneer_currency payoneer_payee_id payoneer_step)]);
    undef $tmp_rights;

    # Ошибка, если не достутпно
    throw Exception::Denied gettext('Access denied') unless $user->{payoneer};

    # Ошибка, если до этого шага ещё не дошли
    $user->{payoneer_step} //= 0;
    throw Exception::Denied gettext('Access denied')
      if $user->{payoneer_step} < 2;

    throw Exception::Validation::BadArguments gettext('Invalid currency') unless $user->{payoneer_currency};

    # на шаге проверки активации линка
    if (2 == $user->{payoneer_step} || 3 == $user->{payoneer_step}) {
        try {
            if (
                $self->api_balalayka->get_payoneer_payee_status(
                    $user->{payoneer_currency}, $user->{payoneer_payee_id},
                )
               )
            {
                # Сохраним позицию
                $self->payoneer_save_user($user->{id}, payoneer_step => 4,);
                $user->{payoneer_step} = 4;
            } else {
                # Сохраним позицию
                $self->payoneer_save_user($user->{id}, payoneer_step => 3,);
                throw Exception::Validation::BadArguments gettext('Payoneer account is not activated yet');
            }
        }
        catch Exception::API::Payoneer::PayeeNotFound with {
            throw Exception::Validation::BadArguments gettext(
                'Payoneer account is not linked yet. Please follow by link above.');
        }
        catch Exception::API::Payoneer with {
            my ($exception) = @_;
            $exception->{sentry}{extra}{fingerprint} //= ['Users', 'payoneer_check_payee_status'];
            ERROR $@;
            throw Exception::Validation::BadArguments gettext('Error while check Payoneer payee status');
        };
    }

    # Ошибка, если нет договора
    my $contract;
    try {
        $contract = $self->documents->get_live_contracts(client_id => $user->{client_id})->[0];
    }
    catch Exception::Balance::IncorrectAnswer with {
        my ($exception) = @_;
        $exception->{sentry}{extra}{fingerprint} //= ['Users', 'payoneer_check_payee_status'];
        ERROR $@;
        throw Exception::Validation::BadArguments gettext('Error while get contract');
    };
    throw Exception::Validation::BadArguments gettext('User has not contract')
      unless $contract;

    # на шаге обновления плательщика
    if (4 == $user->{payoneer_step}) {
        try {
            $self->api_balance->update_person_for_payoneer(
                client_id    => $user->{client_id},
                operator_uid => $user->{id},
                payee_id     => $user->{payoneer_payee_id},
                person_id    => $contract->{Person}{id},
                type         => $contract->{Person}{type},
            );
        }
        catch Exception::Balance::IncorrectAnswer with {
            my ($exception) = @_;
            $exception->{sentry}{extra}{fingerprint} //= ['Users', 'payoneer_check_payee_status'];
            ERROR $@;
            throw Exception::Validation::BadArguments gettext('Error while update payee in contract');
        };
        if ($contract->{Contract}{is_signed}) {
            my $currency = $contract->{Contract}{currency};
            if (   $self->documents->is_currency_eur($currency)
                || $self->documents->is_currency_usd($currency))
            {
                $user->{payoneer_step} = 7;
            } else {
                $user->{payoneer_step} = 5;
            }
        } else {
            try {
                $self->api_balance->update_contract(
                    contract_id  => $contract->{Contract}{contract2_id},
                    currency     => $user->{payoneer_currency},
                    operator_uid => $user->{id},
                );
            }
            catch Exception::Balance::IncorrectAnswer with {
                my ($exception) = @_;
                $exception->{sentry}{extra}{fingerprint} //= ['Users', 'payoneer_check_payee_status'];
                ERROR $@;
                throw Exception::Validation::BadArguments gettext('Error while update contract');
            };
            $user->{payoneer_step} = 8;
        }
        # Сохраним позицию
        $self->payoneer_save_user($user->{id}, payoneer_step => $user->{payoneer_step},);
    }

    # на шаге закрытия договора
    if (5 == $user->{payoneer_step}) {
        try {
            $self->api_balance->close_contract(
                END_DT        => date_add(curdate(), day => 5, oformat => 'iso8601'),
                contract_id   => $contract->{Contract}{contract2_id},
                contract_type => 'PARTNERS',
                operator_uid  => $user->{id},
            );
        }
        catch Exception::Balance::IncorrectAnswer with {
            my ($exception) = @_;
            $exception->{sentry}{extra}{fingerprint} //= ['Users', 'payoneer_check_payee_status'];
            ERROR $@;
            throw Exception::Validation::BadArguments gettext('Error while close contract');
        };
        # Сохраним позицию
        $self->payoneer_save_user($user->{id}, payoneer_step => 6,);
        $user->{payoneer_step} = 6;
    }

    # на шаге копирования договора
    if (6 == $user->{payoneer_step}) {
        try {
            $self->api_balance->create_new_contract_for_payoneer(
                contract     => $contract->{Contract},
                currency     => $user->{payoneer_currency},
                dt           => date_add(curdate(), day => 6, oformat => 'db'),
                operator_uid => $user->{id},
            );
        }
        catch Exception::Balance::IncorrectAnswer with {
            my ($exception) = @_;

            $self->payoneer_error_message(
                contract  => $contract,
                exception => $exception,
                user      => $user,
            );
            $exception->{sentry}{extra}{fingerprint} //= ['Users', 'payoneer_check_payee_status'];
            ERROR $@;
            throw Exception::Validation::BadArguments gettext('Error while create contract');
        }
        catch {
            my ($exception) = @_;

            $self->payoneer_error_message(
                contract  => $contract,
                exception => $exception,
                user      => $user,
            );

            throw $exception;
        };
        # Сохраним позицию
        $self->payoneer_save_user($user->{id}, payoneer_step => 8,);
        $user->{payoneer_step} = 8;
    }

    # на шаге обновления договора
    if (7 == $user->{payoneer_step}) {
        try {
            $self->api_balance->update_contract_payment_type(
                contract_id  => $contract->{Contract}{contract2_id},
                operator_uid => $user->{id},
                pay_to       => $PAYMENT_TYPES->{PAYONEER},
            );
        }
        catch Exception::Balance::IncorrectAnswer with {
            my ($exception) = @_;
            $exception->{sentry}{extra}{fingerprint} //= ['Users', 'payoneer_check_payee_status'];
            ERROR $@;
            throw Exception::Validation::BadArguments gettext('Error while create collateral for contract');
        };
        # Сохраним позицию
        $self->payoneer_save_user($user->{id}, payoneer_step => 8,);
        $user->{payoneer_step} = 8;
    }

    return 8 == $user->{payoneer_step} ? TRUE : FALSE;
}

sub get_currency_rates {
    my ($self, $user_id) = @_;
    $user_id //= $self->get_option('cur_user', {})->{'id'};
    my $current_currency =
      $self->get_all(fields => [qw(current_currency)], filter => ['id', '=', $user_id])->[0]->{'current_currency'};
    my $base_currency_rate = $self->api_balance->get_currency_rate(currency => $current_currency);

    my $rates = $self->currency->get_all(
        fields => [qw(symbol code country)],
        filter => ['code', 'IN', $CPM_AVAILABLE_CURRENCIES]
    );
    foreach (@$rates) {
        $_->{rate} =
          sprintf("%.4f", $self->api_balance->get_currency_rate(currency => $_->{code}) / $base_currency_rate);
    }
    return $rates;
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{'next_currency'}) {
        $result{'currencies'} = $self->get_currency_rates($opts->{attributes}->{id});
    }
    if ($need_fields->{'roles'}) {
        my $roles = $self->rbac->get_roles();
        my @role_list = sort {$a->{'name'} cmp $b->{'name'}} map {{id => $_->{'id'}, name => $_->{'name'}}} @$roles;
        $result{'roles'} = \@role_list;
    }
    if ($need_fields->{'features'}) {
        my $features = $self->get_all_features();
        my @feature_list =
          sort {$a->{'feature'} cmp $b->{'feature'}}
          map {{feature => $_, name => $features->{$_}{'name'},}} keys %$features;
        $result{'features'} = \@feature_list;
    }

    return \%result;
}

sub check_game_offer {
    my ($self, $user) = @_;

    $user = $self->_get_object_fields($user, [qw(id is_games has_game_offer)]);

    if ($user->{has_game_offer}) {
        return $user;
    } elsif ($user->{is_games}) {
        my $tmp_rights = $self->app->add_tmp_rights('users_view_field__client_id');
        $user = $self->_get_object_fields($user, [qw(id client_id active_contract)]);
        $user->{has_game_offer} = scalar $self->documents->get_games_contracts($user->{client_id}, live_only => TRUE);
        if (   $user->{active_contract}
            && $user->{active_contract}{Contract}{person_id}
            && $user->{active_contract}{Contract}{is_signed}
            && !$user->{active_contract}{Contract}{test_mode})
        {
            return $user;
        }
    }

    return FALSE;
}

sub accept_game_offer {
    my ($self, $user) = @_;

    $user = $self->check_game_offer($user);

    throw Exception::Denied 'Cannot accept game_offer' unless $user;

    my $id = TRUE;
    unless ($user->{has_game_offer}) {
        $id = $self->app->api_balance->create_contract(
            'operator_uid' => $user->{id},
            'client_id'    => $user->{client_id},
            'manager_code' => $GAME_OFFER_MANAGER,
            'services'     => [$GAME_OFFER_SERVICE],
            'start_dt'     => curdate(oformat => 'db'),
            'is_offer'     => 1,
            'person_id'    => $user->{active_contract}{Contract}{person_id},
            'currency'     => $self->documents->get_currency_for_offer($user->{active_contract}{Contract}{currency}),
            'signed'       => 1,
            'nds'          => $user->{active_contract}{Contract}{nds},
            'firm_id'      => $user->{active_contract}{Contract}{firm},
            'selfemployed' => $user->{active_contract}{Contract}{selfemployed} ? 1 : 0,
        );

        try {
            $self->app->api_balance->create_collateral_game_offer(
                'operator_uid' => $user->{id},
                'contract_id'  => $user->{active_contract}{Contract}{contract2_id},
                'memo' =>
'Создано автоматически (PI-25718). Дополнительное соглашение Я.Игры оформлено в '
                  . to_json($id),
            );
        }
        catch {
            my ($exception) = @_;
            ERROR {
                exception => $exception,
                message   => 'cannot create_collateral_game_offer',
                extra     => {
                    'operator_uid' => $user->{id},
                    'contract_id'  => $user->{active_contract}{Contract}{contract2_id},
                    'memo'         => to_json($id),
                },
                fingerprint => ['Users', 'accept_game_offer'],
            };
        };

        my $tmp_rights = $self->app->add_tmp_rights('users_edit_field__has_game_offer');
        $self->do_action($user, 'edit', has_game_offer => TRUE);
    }

    return $id;
}

TRUE;
