package Application::Model::Users::Features;

use qbit;

use base qw(Application::Model::DBManager::Base RestApi::DBModel Application::Model::DAC);

use Utils::JSON qw(fix_type_for_complex);
use Utils::PublicID qw(check_public_id);

use Exception::DB::DuplicateEntry;
use Exception::Validation::BadArguments;

use PiConstants qw(
  $ASSESSOR_ROLE_ID
  $AUTOASSIGN_FEATURE_KEY
  $ADINSIDE_USER_ID
  );

my %PRODUCT_FEATURES = (
    additional_income => {
        name   => d_gettext('Feature additional_income'),
        rights => [qw(statistics_additional_income_view)],
    },
    business_rules => {
        name   => d_gettext('Feature business_rules'),
        rights => [qw(do_business_rules_add business_rules_edit business_rules_view)]
    },
    context_on_site_natural => {
        name   => d_gettext('Feature context_on_site_natural'),
        rights => [
            qw(do_context_on_site_natural_add context_on_site_natural_view context_on_site_natural_edit context_on_site_natural_edit_field__articles context_on_site_natural_edit_field__picategories context_on_site_natural_edit_field__strategy context_on_site_natural_view_field__readonly)
        ],
        auto_assign_for_assessors => TRUE,
    },
    cpm_currency => {
        auto_assign => TRUE,
        name        => d_gettext('Allows change display currency'),
        rights      => [
            qw(users_edit_field__currency_rate users_edit_field__next_currency users_view_field__current_currency users_view_field__next_currency)
        ],
    },
    design_auction_native => {
        auto_assign => TRUE,
        name        => d_gettext('Allows native designs in Design Auction'),
        on_delete   => 'on_delete_feature_design_auction_native',
        rights      => [],
    },
    design_auction_native_turbo => {
        auto_assign_for_assessors => TRUE,
        name                      => d_gettext('Allows native designs in Design Auction for turbo site version'),
        on_delete                 => 'on_delete_feature_design_auction_native_turbo',
        rights                    => [],
    },
    instream_vmap => {
        name   => d_gettext('VMap in instream blocks'),
        rights => [qw(video_scenaries_edit video_scenaries_view do_video_scenaries_add)],
    },
    support_chat => {
        name   => d_gettext('Enable support chat'),
        rights => [],
    },
    old_stat_api => {
        name   => d_gettext('Old stat API'),
        rights => ['api_statistics'],
    },
    picategories => {
        name        => d_gettext('PiCategoriesIAB'),
        rights      => [],
        auto_assign => TRUE,
        undeletable => TRUE,
    },
    statistics_renewed => {
        auto_assign => TRUE,
        name        => d_gettext('Renewed statistics'),
        rights      => [
            qw(statistics_reports_mol_view statistics_mol_view statistics_billing_view bk_statistics_view_field__block_level bk_statistics_view_field__page_level)
        ],
    },
    widget_balance => {
        name   => d_gettext('Actual Balance Widget'),
        rights => [],
    },
    turbo_desktop_available => {
        name   => d_gettext('Enable turbo desktop'),
        rights => [],
    },
    editable_requisites => {
        # При выпиливании фичи надо не забыть, что она проставляется из анкеты
        name   => d_gettext('Enable edit requsites'),
        rights => ['settings_in_menu_view'],
    },
    simple_inapp => {
        auto_assign => TRUE,
        name        => d_gettext('Simple inapp'),
        rights      => [qw(mobile_app_view do_mobile_app_edit mobile_app_rtb_edit_field__show_video)],
    },
    internal_simple_inapp => {
        name   => d_gettext('Internal simple inapp'),
        rights => [],
    },
    simple_menu => {
        auto_assign => TRUE,
        name        => d_gettext('Simple menu'),
        rights      => [],
    },
    new_mobmed => {
        name   => d_gettext('New Mobile mediation'),
        rights => [
            map "bk_statistics_view_field__$_",
            qw/clicks_mm requests_mm served_requests_mm show_rate_v2_mm fill_rate_v2_mm ctr_mm ecpm_mm/
        ],
    },
    inapp_upd_metrics => {
        name   => d_gettext('MM statistics updated metrics'),
        rights => [
            map "bk_statistics_view_field__$_",
            qw/requests_external_mm served_requests_external_mm impressions_external_mm clicks_external_mm show_rate_external_mm fill_rate_external_mm revenue_external_mm ctr_external_mm ecpm_external_mm/
        ],
    },
    inapp_ab_exp => {
        name   => d_gettext('A/B tests'),
        rights => [],
    },
    mobile_fullscreen_available => {
        auto_assign => TRUE,
        name        => d_gettext('Mobile fullscreen site version'),
        rights      => [],
        on_delete   => 'on_delete_feature_mobile_fullscreen_available',
    },
    mobile_rewarded_available => {
        name      => d_gettext('Mobile rewarded site version'),
        rights    => [],
        on_delete => 'on_delete_feature_mobile_rewarded_available',
    },
    mobile_floorad_available => {
        auto_assign => TRUE,
        name        => d_gettext('Mobile floor_add site version'),
        rights      => [],
        on_delete   => 'on_delete_feature_mobile_floorad_available',
    },
    mobile_statistics => {
        name   => d_gettext('Statistics mobile version'),
        rights => [],
    },
    new_dashboard => {
        name   => d_gettext('New Dashboard'),
        rights => [],
    },
    new_users_page => {
        name   => d_gettext('New interface for the users tab'),
        rights => [],
    },
    ua_payment_banner => {
        name   => d_gettext('UA payment banner'),
        rights => []
    },
);

my %EXPERIMENT_FEATURES = ();

my %HIDDEN_FEATURES = ();

my %VISIBLE_FEATURES = (%PRODUCT_FEATURES, %EXPERIMENT_FEATURES);

our %ALL_FEATURES = (%VISIBLE_FEATURES, %HIDDEN_FEATURES);

sub accessor         {'user_features'}
sub db_table_name    {'user_features'}
sub get_product_name {gettext('user_features')}

sub get_structure_model_accessors {
    return {
        partner_db => 'Application::Model::PartnerDB::Users',
        users      => 'Application::Model::Users',
        resources  => 'Application::Model::Resources',
        assistants => 'Application::Model::Assistants',
        cur_user   => 'Application::Model::CurUser',
        all_pages  => 'Application::Model::AllPages',
        kv_store   => 'QBit::Application::Model::KvStore',
        rbac       => 'Application::Model::RBAC'
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = [
        {
            name        => $self->accessor(),
            description => d_gettext('Right to manage ' . $self->accessor()),
            rights      => {user_features_view_all => d_gettext('Right to view all users features'),},
        },
    ];

    return $rights;
}

sub get_structure_model_fields {
    return {
        id => {db => TRUE, pk => TRUE, type => 'number'},
        public_id => {db      => TRUE, db_expr => 'id', type => 'string', api => 1},
        user_id   => {default => TRUE, db      => TRUE, type => 'number', api => 1},
        login     => {
            depends_on => ['user_id', 'users.login'],
            get        => sub {
                return $_[0]->{'users'}{$_[1]->{'user_id'}}{'login'} // '';
            },
            type => 'string',
            api  => 1
        },
        feature          => {default => TRUE, db => TRUE, type => 'string', api => 1},
        available_fields => {
            get => sub {
                return $_[0]->model->get_available_fields();
            },
            type     => 'complex',
            fix_type => \&fix_type_for_complex,
            api      => 1
        }
    };
}

sub get_structure_model_filter {
    return {
        db_accessor => 'partner_db',
        fields      => {
            id      => {type => 'number', label => d_gettext('ID')},
            user_id => {type => 'number', label => d_gettext('User ID')},
            feature => {type => 'text',   label => d_gettext('Right')},
            users   => {
                type           => 'subfilter',
                model_accessor => 'users',
                field          => 'user_id',
                fk_field       => 'id',
                label          => d_gettext('Login'),
            },
        }
    };
}

sub get_available_fields {
    my ($self) = @_;
    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);

    return \%fields;
}

sub add {
    my ($self, $user_id, $feature) = @_;

    throw Exception::Validation::BadArguments gettext('Users feature "%s" does not exists', $feature)
      unless $ALL_FEATURES{$feature};

    my $primary_key;
    try {
        $primary_key = $self->partner_db_table()->add(
            {
                user_id => $user_id,
                feature => $feature,
            }
        );
        $self->resources->invalidate_cache($user_id);
    }
    catch Exception::DB::DuplicateEntry with {
        throw Exception::Validation::BadArguments gettext('Users feature "%s" already added', $feature);
    };

    if (my $on_add = $ALL_FEATURES{$feature}{on_add}) {
        $self->users->$on_add($user_id);
    }

    return $primary_key;
}

sub delete {
    my ($self, $user_id, @features) = @_;

    for my $feature (@features) {
        throw Exception::Validation::BadArguments gettext('Users feature "%s" cannot delete', $feature)
          if $ALL_FEATURES{$feature} && $ALL_FEATURES{$feature}{undeletable};
    }

    for my $feature (@features) {
        if ($ALL_FEATURES{$feature} and my $on_delete = $ALL_FEATURES{$feature}{on_delete}) {
            $self->users->$on_delete($user_id);
        }
    }
    my $ret =
      $self->partner_db_table()->delete($self->partner_db->filter({user_id => $user_id, feature => \@features}));

    $self->resources->invalidate_cache($user_id);

    return $ret;
}

sub api_available_actions { }

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_assistant($filter, tutby => TRUE);

    return $filter;
}

sub get_all_features {
    my ($self) = @_;

    my %feats =
      map {$_ => {name => $VISIBLE_FEATURES{$_}->{name}->(), rights => $VISIBLE_FEATURES{$_}->{rights}}}
      keys %VISIBLE_FEATURES;

    return \%feats;
}

sub get_user_rights {
    my ($self, $user_id, $user_roles) = @_;

    my %rights;
    my @feature_filter =
      $user_roles
      ? (
        feature => [
            grep {@$user_roles || $ALL_FEATURES{$_}{apply_without_roles}}
              keys(%ALL_FEATURES)
        ]
      )
      : ();

    # to get page owners rigths for assistants
    my $tmp_rights = $self->app->add_tmp_rights('user_features_view_all');
    foreach my $row (
        grep {$ALL_FEATURES{$_->{'feature'}}} @{
            $self->get_all(
                fields => [qw(feature)],
                filter => {
                    user_id => $user_id,
                    @feature_filter
                }
            )
        }
      )
    {
        foreach my $right (@{$ALL_FEATURES{$row->{'feature'}}->{rights}}) {
            $rights{$right} = 1;
        }
    }

    return keys %rights;
}

sub init {
    my ($self) = @_;

    $self->SUPER::init();
    $self->register_rights($self->get_structure_rights_to_register());
    $self->model_fields($self->get_structure_model_fields());
    $self->model_filter($self->get_structure_model_filter());
}

sub related_models {
    my ($self) = @_;

    return {
        users => {
            accessor => 'users',
            filter   => sub {
                +{id => array_uniq(map {$_->{'user_id'}} @{$_[1]})};
            },
            key_fields => ['id'],
        },
    };
}

sub replace_multi {
    my ($self, $user_id, $features, %opts) = @_;

    foreach my $feature (@$features) {
        throw Exception::Validation::BadArguments gettext('Users feature "%s" does not exists', $feature)
          unless $ALL_FEATURES{$feature};
    }

    $self->partner_db->transaction(
        sub {
            my $table = $self->partner_db_table();

            my %current =
              map {$_->{'feature'} => TRUE} @{$table->get_all(fields => ['feature'], filter => {user_id => $user_id})};

            unless ($opts{'only_add'}) {
                my %new = map {$_ => TRUE} @$features;

                my @drop;
                for my $feature (grep {!$new{$_}} keys(%current)) {
                    throw Exception::Validation::BadArguments gettext('Users feature "%s" cannot delete', $feature)
                      if $ALL_FEATURES{$feature}{undeletable};
                    push @drop, $feature;
                }

                $self->delete($user_id, @drop) if @drop;
            }

            my @add = grep {!$current{$_}} @$features;
            foreach (@add) {
                if (my $on_add = $ALL_FEATURES{$_}{on_add}) {
                    $self->users->$on_add($user_id);
                }
            }

            $table->add_multi([map {{user_id => $user_id, feature => $_}} @add], duplicate_update => TRUE)
              if @add;

            $self->resources->invalidate_cache($user_id);
        }
    );

    return 1;
}

sub auto_assign {
    my ($self, $user_id, $roles) = @_;

    my %roles = map {$_ => $_} @$roles;

    my @features     = qw();
    my $cd           = curdate(oformat => 'db_time');
    my $autofeatures = $self->_get_autofeatures_from_db();
    foreach my $feature (keys(%ALL_FEATURES)) {
        my $assign_interval = $ALL_FEATURES{$feature}{auto_assign_interval};
        if (   ($autofeatures->{$feature} // $ALL_FEATURES{$feature}{auto_assign})
            || ($ALL_FEATURES{$feature}{auto_assign_for_assessors} && $roles{$ASSESSOR_ROLE_ID})
            || ($assign_interval && (($cd ge $assign_interval->[0]) && ($cd lt $assign_interval->[1]))))
        {
            push(@features, $feature);
        }
    }

    $self->replace_multi($user_id, \@features, only_add => TRUE) if @features;
}

=hint
  enable feature auto-assign:
    UPDATE `kv_store`
      SET value = JSON_SET(`value`,'$.<feature_id>',TRUE)
    WHERE `key` = 'autoassign_feature'

  disable feature auto-assign:
    UPDATE `kv_store`
      SET value = JSON_SET(`value`,'$.<feature_id>',FALSE)
    WHERE `key` = 'autoassign_feature'

  use feature auto-assign settings from code:
    UPDATE `kv_store`
      SET value = JSON_REMOVE(`value`,'$.<feature_id>')
    WHERE `key` = 'autoassign_feature'
=cut

sub _get_autofeatures_from_db {
    my ($self) = @_;

    return eval {from_json($self->kv_store->get($AUTOASSIGN_FEATURE_KEY))} // {};
}

# добавляю сюда, чтобы потом можно было быстро найти все места, где вызывается, и выпилить
sub has_feature_simple_inapp {
    my ($self) = @_;
    return $self->app->users->has_feature($self->get_option('cur_user', {})->{id}, 'simple_inapp');
}

sub has_feature_internal_simple_inapp {
    my ($self) = @_;
    return $self->app->users->has_feature($self->get_option('cur_user', {})->{id}, 'internal_simple_inapp');
}

sub page_owner_has_feature_simple_inapp {
    my ($self, $obj) = @_;

    my $page_id = $obj->{context_page_id} // $obj->{page_id};
    my $user_id =
      $self->partner_db->mobile_app_settings->get_all(fields => ['owner_id'], filter => {context_page_id => $page_id})
      ->[0]->{owner_id};

    return $self->app->users->has_feature($user_id, 'simple_inapp');
}

sub page_owner_has_feature_internal_simple_inapp {
    my ($self, $obj) = @_;

    return $self->app->users->has_feature($ADINSIDE_USER_ID, 'internal_simple_inapp');
}

TRUE;
