package Application::Model::Users::GlobalExcludedPhones;

use qbit;

use Utils::JSON qw(fix_type_for_complex);
use Utils::PublicID qw(check_public_id);
use base qw(
  Application::Model::DBManager::Base
  RestApi::DBModel Application::Model::ValidatableMixin
  Application::Model::DAC
  );

use Exception::Validator::Fields;
use Exception::DB::DuplicateEntry;
use Exception::Validation::BadArguments::InvalidPhone;

sub accessor         {'user_global_excluded_phones'}
sub db_table_name    {'user_global_excluded_phones'}
sub get_product_name {gettext('user_global_excluded_phones')}

sub get_structure_model_accessors {
    return {
        partner_db => 'Application::Model::PartnerDB::Users',
        users      => 'Application::Model::Users',
        assistants => 'Application::Model::Assistants',
        cur_user   => 'Application::Model::CurUser',
        all_pages  => 'Application::Model::AllPages'
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = [
        {
            name        => $self->accessor(),
            description => d_gettext('Right to manage ' . $self->accessor()),
            rights      => {
                (
                    user_global_excluded_phones_view     => d_gettext('Right to view global excluded phones'),
                    user_global_excluded_phones_view_all => d_gettext('Right to view all global excluded phones'),
                    user_global_excluded_phones_view_field__login =>
                      d_gettext('Right to view login in global excluded phones'),
                ),
            }
        },
    ];

    return $rights;
}

sub get_structure_model_fields {
    return {
        id => {db => TRUE, pk => TRUE, type => 'number'},
        public_id => {db      => TRUE, db_expr => 'id', type => 'string', api => 1},
        user_id   => {default => TRUE, db      => TRUE, type => 'number', api => 1},
        login     => {
            depends_on => ['user_id', 'users.login'],
            get        => sub {
                return $_[0]->{'users'}{$_[1]->{'user_id'}}{'login'} // '';
            },
            type => 'string',
            api  => 1
        },
        phone => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            api        => 1,
            need_check => {type => 'phone',}
        },
        available_fields => {
            get => sub {
                return $_[0]->model->get_available_fields();
            },
            type     => 'complex',
            fix_type => \&fix_type_for_complex,
            api      => 1
        }
    };
}

sub get_structure_model_filter {
    return {
        db_accessor => 'partner_db',
        fields      => {
            id      => {type => 'number', label => d_gettext('ID')},
            user_id => {type => 'number', label => d_gettext('User ID')},
            phone   => {type => 'text',   label => d_gettext('Phone')},
            users   => {
                type           => 'subfilter',
                model_accessor => 'users',
                field          => 'user_id',
                fk_field       => 'id',
                label          => d_gettext('Login'),
            },
        }
    };
}

sub get_available_fields {
    my ($self) = @_;
    my $model_fields = $self->get_model_fields;
    my %fields = map {$_ => TRUE} keys(%$model_fields);
    my $accessor = $self->accessor();

    $self->app->delete_field_by_rights(\%fields, {$accessor . '_view_field__login' => [qw(user_id login)]});

    return \%fields;
}

sub add {
    my ($self, $user_id, $raw_phone) = @_;

    my $phone = get_normalized_phone_or_throw($raw_phone);

    my $opts = {
        user_id => $user_id,
        phone   => $phone,
    };

    $self->app->validator->check(
        data  => $opts,
        app   => $self,
        throw => TRUE,
        $self->get_template(),
    );

    my $primary_key;
    try {
        $primary_key = $self->partner_db_table()->add($opts);
    }
    catch Exception::DB::DuplicateEntry with {
        throw Exception::Validation::BadArguments::InvalidPhone gettext('Excluded phone "%s" is exists', $phone);
    };

    my $pages = $self->cur_user->get_all_page_ids_owned_by_user_for_global_concurents($user_id);

    $self->all_pages->mark_pages_for_async_update(pages => $pages);

    return $primary_key;
}

sub delete {
    my ($self, $user_id, $phone) = @_;

    my $res = $self->partner_db_table()->delete($self->partner_db->filter({user_id => $user_id, phone => $phone}));

    my $pages = $self->cur_user->get_all_page_ids_owned_by_user_for_global_concurents($user_id);

    $self->all_pages->mark_pages_for_async_update(pages => $pages);

    return $res;
}

sub api_available_fields {
    my ($self) = @_;

    my @fields = $self->SUPER::api_available_fields();

    #need to remove user_id
    unless ($self->check_short_rights('view_field__login')) {
        @fields = grep {!/user_id/} @fields;
    }

    return @fields;
}

sub api_available_actions { }

sub api_check_public_id {
    return check_public_id($_[1], 'simple_id');
}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_assistant($filter);

    return $filter;
}

sub get_all_phones {
    my ($self, $user_id) = @_;

    my @phones = map {$_->{'phone'}} @{
        $self->get_all(
            fields => [qw(phone)],
            filter => {user_id => $user_id,}
        )
      };

    return \@phones;
}

sub get_all_phones_for_cur_user {
    my ($self) = @_;

    return $self->get_all_phones($self->cur_user->get_cur_user_id());
}

sub init {
    my ($self) = @_;

    $self->SUPER::init();
    $self->register_rights($self->get_structure_rights_to_register());
    $self->model_fields($self->get_structure_model_fields());
    $self->model_filter($self->get_structure_model_filter());
}

sub related_models {
    my ($self) = @_;

    return {
        users => {
            accessor => 'users',
            filter   => sub {
                +{id => array_uniq(map {$_->{'user_id'}} @{$_[1]})};
            },
            key_fields => ['id'],
        },
    };
}

sub replace_multi {
    my ($self, $user_id, @phones) = @_;

    $self->app->validator->check(
        data     => \@phones,
        app      => $self,
        throw    => TRUE,
        template => {
            type => 'array',
            all  => {type => 'phone'},
        },
    );

    $self->partner_db->transaction(
        sub {
            $self->partner_db_table()->delete($self->partner_db->filter({user_id => $user_id}));

            $self->partner_db_table()->add_multi([map {{user_id => $user_id, phone => $_}} @{array_uniq(@phones)}]);
        }
    );

    my $pages = $self->cur_user->get_all_page_ids_owned_by_user_for_global_concurents($user_id);

    $self->all_pages->mark_pages_for_async_update(pages => $pages);

    return 1;
}

TRUE;
