package Application::Model::VideoScenaries;

use qbit;

use Exception::Validator::Fields;

use base qw(
  Application::Model::Product
  Application::Model::ValidatableMixin
  RestApi::MultistateModel
  );

consume qw(
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::EditableFields
  Application::Model::Role::Has::CreateDate
  );

sub accessor {'video_scenaries'}

sub db_table_name {'video_scenaries'}

sub get_product_name {gettext('video_scenaries')}

sub get_opts_schema_name {'video_scenaries_opts'}

sub public_id_prefix {'VS-'}

sub public_id {
    my ($self, $obj) = @_;

    return $self->public_id_prefix . "$obj->{page_id}-$obj->{id}";
}

sub get_structure_model_accessors {
    my ($self) = @_;

    return {
        partner_db             => 'Application::Model::PartnerDB',
        video_an_site          => 'Application::Model::Page::Video',
        video_an_site_instream => 'Application::Model::Product::VideoAN::Site::InStream',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;

    my $rights = $self->SUPER::get_structure_rights_to_register();

    $rights->[0]{'rights'} = {
        map {$self->get_description_right($_)}
          qw(
          view
          edit
          view_action_log
          ),
    };

    return $rights;
}

sub get_structure_model_fields {
    my ($self) = @_;

    return {
        id => {
            db          => TRUE,
            type        => 'number',
            pk          => TRUE,
            api         => 1,
            adjust_type => 'str',
        },
        page_id => {
            db         => TRUE,
            type       => 'number',
            pk         => TRUE,
            need_check => {
                type  => 'int_un',
                min   => 1,
                check => sub {
                    my ($qv, $page_id_value) = @_;
                    throw Exception::Validator::Fields gettext('Can not find page: %s', $page_id_value // 'UNDEF')
                      unless $qv->app->video_an_site_instream->is_page_reachable($page_id_value);
                  }
            },
            api         => 1,
            adjust_type => 'str',
        },
        public_id => {
            default    => TRUE,
            depends_on => [qw(page_id id)],
            label      => d_gettext('Scenario ID'),
            get        => sub {
                $_[0]->model->public_id_prefix . "$_[1]->{'page_id'}-$_[1]->{'id'}";
            },
            type        => 'string',
            api         => 1,
            adjust_type => 'str',
        },
        caption => {
            default    => TRUE,
            db         => TRUE,
            type       => 'string',
            need_check => {
                len_min => 1,
                len_max => 255,
            },
            api => 1,
        },
        single_ad_session => {
            default     => TRUE,
            from_opts   => 'from_hash',
            type        => 'boolean',
            need_check  => {type => 'boolean',},
            api         => 1,
            adjust_type => 'int',
        },
        use_better_places => {
            default     => TRUE,
            from_opts   => 'from_hash',
            type        => 'boolean',
            need_check  => {type => 'boolean',},
            api         => 1,
            adjust_type => 'int',
        },
        ad_session_interval => {
            default    => TRUE,
            from_opts  => 'from_hash',
            type       => 'number',
            need_check => {
                type     => 'int_un',
                optional => TRUE,
                check    => sub {
                    my ($qv, $value) = @_;
                    throw Exception::Validator::Fields gettext('Must be a whole number from %d to %d', 1, 30)
                      if $qv->data->{'single_ad_session'} && ($value < 1 || $value > 30);
                  }
            },
            api => 1,
        },
        resource => {
            depends_on => ['page_id', 'pages.domain'],
            label      => d_gettext('Domain'),
            get        => sub {
                $_[0]->{'pages'}{$_[1]->{'page_id'}}{'domain'};
            },
            type => 'string',
            api  => 1
        },
        blocks_amount => {
            depends_on => ['page_id', 'id', 'blocks.id'],
            label      => d_gettext('Blocks amount'),
            get        => sub {
                scalar @{$_[0]->{'blocks'}{$_[1]->{'page_id'}, $_[1]->{'id'}} // []};
            },
            type => 'number',
            api  => 1
        },
        multistate => {
            db          => TRUE,
            type        => 'number',
            api         => 1,
            adjust_type => 'str',
        },
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            type => 'string',
            api  => 1,
        },
        #API
        fields_depends => {
            depends_on => ['id'],
            get        => sub {{}},
            type       => 'complex',
            api        => 1,
        },
        status => {
            depends_on => ['id'],
            get        => sub {'sync'},
            type       => 'string',
            api        => 1,
        },
        for_adfox => {
            from_opts  => 'from_hash',
            type       => 'boolean',
            need_check => {type => 'boolean', optional => TRUE},
            api        => 1,
        },
    };
}

sub get_structure_model_filter {
    my ($self) = @_;

    return {
        db_accessor => 'partner_db',
        fields      => {
            id        => {type => 'number', label => d_gettext('ID')},
            public_id => {
                type   => 'publicid',
                label  => d_gettext('Scenario ID'),
                regexp => {'^' . $self->public_id_prefix . '([0-9]+)-([0-9]+)\z' => ['page_id', 'id'],}
            },
            caption    => {type => 'text',       label => d_gettext('Caption')},
            multistate => {type => 'multistate', label => d_gettext('Multistate')},
            page_id    => {type => 'number',     label => d_gettext('Page ID')},
            for_adfox  => {
                type       => 'json',
                value_type => 'boolean',
                label      => d_gettext('Adfox scenario'),
            },
        },
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;

    return [
        {name => 'public_id', label => gettext('Scenario ID')},
        {name => 'caption',   label => gettext('Caption')},
        {name => 'page_id',   label => gettext('Page ID')},
    ];
}

sub get_structure_multistates_graph {
    my ($self) = @_;

    return {

        empty_name  => 'new',
        multistates => [
            [deleted => d_gettext('Deleted')],
            [working => d_gettext('Working')],
            [
                deleted_with_page => d_gettext('Archived with page'),
                private           => TRUE,
            ]

        ],
        right_name_prefix => $self->accessor . '_',
        actions           => {
            add               => d_gettext('Add'),
            edit              => d_gettext('Edit'),
            delete            => d_gettext('Delete'),
            duplicate         => d_gettext('Duplicate'),
            restore           => d_gettext('Restore'),
            delete_with_page  => d_gettext('Archive with page'),
            restore_with_page => d_gettext('Restore with page'),
        },
        right_actions      => {add             => d_gettext('Add'),},
        right_group        => [video_scenaries => d_gettext('Rights for video scenaries')],
        right_name_prefix  => 'video_scenaries_',
        multistate_actions => [
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => ['working'],
            },
            {
                action      => 'delete',
                from        => 'not deleted',
                set_flags   => ['deleted'],
                reset_flags => ['working'],
            },
            {
                action      => 'delete_with_page',
                from        => 'not deleted',
                set_flags   => ['deleted_with_page', 'deleted'],
                reset_flags => ['working'],
            },
            {
                action      => 'restore',
                from        => 'deleted and not deleted_with_page',
                set_flags   => ['working'],
                reset_flags => ['deleted'],
            },
            {
                action      => 'restore_with_page',
                from        => 'deleted and deleted_with_page',
                set_flags   => ['working'],
                reset_flags => ['deleted_with_page', 'deleted'],
            },
            {
                action => 'edit',
                from   => 'not deleted',
            },
            {
                action => 'duplicate',
                from   => 'not (deleted or deleted_with_page)',
            },
        ],
    };
}

sub get_actions_depends {
    [
        qw(
          id
          page_id
          multistate
          for_adfox
          )
    ];
}

sub get_available_fields {
    my ($self) = @_;

    my %fields = map {$_ => TRUE} keys(%{$self->get_model_fields});

    return \%fields;
}

sub get_available_fields_depends {
    return [];
}

sub get_add_fields {
    my ($self) = @_;

    my $fields = $self->_get_common_add_edit_fields();

    $fields->{$_} = TRUE foreach (qw(page_id for_adfox));

    return $fields;
}

sub get_editable_fields {
    my ($self, $data) = @_;

    return {} if $self->check_multistate_flag($data->{multistate}, 'deleted');
    return {} if $data->{for_adfox};

    my $fields = $self->_get_common_add_edit_fields();

    delete $fields->{ad_session_interval} unless $data->{single_ad_session};

    return $fields;
}

sub get_editable_fields_depends {
    return [qw(multistate single_ad_session for_adfox)];
}

sub _get_common_add_edit_fields {
    my ($self) = @_;

    my $fields = $self->get_fields_by_right(
        no_right_fields => [
            qw(
              ad_session_interval
              caption
              single_ad_session
              use_better_places
              )
        ]
    );

    return $fields;
}

sub can_action_delete {
    my ($self, $obj) = @_;

    $obj = $self->_get_object_fields($obj, ['for_adfox']);
    return !$obj->{for_adfox};
}

sub on_action_delete {
    my ($self, $obj) = @_;

    my $tmp_rights = $self->video_an_site->get_view_edit_rights_for_all_block('stop');
    $self->video_an_site->do_all_block_action(
        $obj, 'delete_with_vmap',
        models => [qw(video_an_site_instream)],
        filter => {vmap_id => $obj->{id}}
    );
}

sub on_action_duplicate {
    my ($self, $obj, %opts) = @_;

    my $public_id      = $self->SUPER::on_action_duplicate($obj, %opts);
    my $id             = $self->_split_id($public_id->{public_id})->{id};
    my $tmp_rights     = $self->app->add_all_tmp_rights();
    my $instream_block = $self->video_an_site_instream;
    my $blocks         = $instream_block->get_all(
        fields => ['*'],
        filter => {
            vmap_id => $obj->{id},
            page_id => $obj->{page_id},
            #TODO Поправить фильтр в соответствии с результатами проработки PI-20527
            multistate => $instream_block->get_multistates_by_filter('working or not working or deleted')
        },
    );
    foreach my $block (@$blocks) {
        my $block_id = $instream_block->on_action_duplicate(
            $block,
            modify_fields => {vmap_id => $id},
            unset_fields  => [qw(adfox_block)]
        )->{public_id};

        $instream_block->do_action($instream_block->_split_id($block_id), 'delete') if ($block->{is_deleted});
    }

    return $public_id;
}

sub on_action_restore {
    my ($self, $obj) = @_;

    $self->video_an_site->do_all_block_action(
        $obj, 'restore_with_vmap',
        models => [qw(video_an_site_instream)],
        filter => {vmap_id => $obj->{id}}
    );
}

sub hook_owner_processing { }

sub hook_set_initialize_settings {
    my ($self, $opts) = @_;

    $self->SUPER::hook_set_initialize_settings($opts);

    $self->partner_db->transaction(
        sub {
            $opts->{'id'} = $self->video_an_site->get_next_scenario_id($opts->{'page_id'});
        }
    );
}

sub related_models {
    my ($self) = @_;

    return {
        pages => {
            accessor => 'video_an_site',
            filter   => sub {
                +{page_id => array_uniq(map {$_->{'page_id'} // ()} @{$_[1]})};
            },
            key_fields => ['page_id'],
        },
        blocks => {
            accessor => 'video_an_site_instream',
            filter   => sub {
                +{
                    page_id    => array_uniq(map {$_->{'page_id'}} @{$_[1]}),
                    multistate => 'not deleted'
                 };
            },
            key_fields => ['page_id', 'vmap_id'],
            value_type => 'array',
        },
    };
}

sub get_or_create_default_scenario {
    my ($self, $page_id, $is_adfox) = @_;

    my @filter;
    if ($is_adfox) {
        @filter = ({'for_adfox' => 1});
    } else {
        @filter = ([OR => [['for_adfox', 'IS', undef], ['for_adfox', '=', 0]]]);
    }
    my $page_scenaries = $self->get_all(
        fields   => [qw(id)],
        filter   => ['AND', [{page_id => $page_id}, {multistate => 'not deleted'}, @filter]],
        order_by => [qw(id)],
        limit    => 1
    );
    return $page_scenaries->[0]->{id} if defined $page_scenaries->[0]->{id};

    my $tmp_rights       = $self->app->add_tmp_rights('do_video_scenaries_add');
    my $default_scenario = $self->add(
        page_id           => $page_id,
        caption           => $is_adfox ? "Adfox scenario" : "Default scenario 1",
        single_ad_session => 0,
        use_better_places => 0,
        for_adfox         => $is_adfox,
    );
    return $self->_split_id($default_scenario)->{'id'};
}

# API

sub api_available_actions {
    my ($self) = @_;

    return qw(add edit delete restore duplicate);
}

sub api_can_edit {TRUE}

sub api_can_add {TRUE}

sub query_filter {
    my ($self, $filter) = @_;

    $filter = $self->limit_filter_by_current_user_pages($filter);

    return $filter;
}

sub api_check_public_id {
    my ($self, $id) = @_;

    my $prefix = $self->public_id_prefix();

    return $id =~ /^$prefix(?:\d+)-(?:\d+)$/;
}

sub _split_id {
    my ($self, $id) = @_;

    return defined($id) && !ref($id) && $id =~ /(\d+)-(\d+)\z/
      ? {page_id => $1, id => $2}
      : $id;
}

sub hook_fields_validation {
    my ($self, $opts) = @_;

    $self->SUPER::hook_fields_validation($opts);

    if ($opts->{for_adfox}) {
        my $vmap = $self->get_all(
            fields => ['id'],
            filter => {
                page_id   => $opts->{page_id},
                for_adfox => 1,
            }
        )->[0];

        throw Exception::Validation::BadArguments gettext('Only one adfox scenary is allowed') if $vmap;
    }
}

TRUE;
