package Application::Model::Widgets;

=encoding UTF-8

=cut

use qbit;

use base qw(
  Application::Model::Common
  RestApi::MultistateModel
  Application::Model::ValidatableMixin
  );

consume qw(
  Application::Model::Role::Has::Actions
  Application::Model::Role::Has::AvailableFields
  Application::Model::Role::Has::EditableFields
  );

use Utils::Logger qw(ERROR);

use Exception::Validation::BadArguments;

sub accessor      {'widgets'}
sub db_table_name {'widgets'}

sub get_product_name {gettext('widgets')}

sub public_id {
    my ($self, $obj) = @_;
    return $obj->{'user_id'};
}

sub get_structure_model_accessors {
    return {
        partner_db      => 'Application::Model::PartnerDB::Widgets',
        product_manager => 'Application::Model::ProductManager',
        text_template   => 'Application::Model::TextTemplate',
    };
}

sub get_structure_rights_to_register {
    my ($self) = @_;
    return [
        {
            name        => 'widgets',
            description => d_gettext('Rights to use widgets'),
            rights      => {
                widgets_view_all => d_gettext('Right to view widgets'),
                map {$self->get_description_right($_)}
                  qw(
                  view_field__widgets_settings
                  view_field__default_deleted_ids
                  ),
            }
        }
    ];
}

sub get_structure_model_fields {
    return {
        user_widgets => {
            depends_on => ['public_id', 'json_widgets_settings', 'json_default_deleted_ids'],
            default    => TRUE,
            api        => TRUE,
            type       => 'complex',
            need_check => {
                type     => 'array',
                optional => TRUE
                , # TODO: убрать когда фронт перестанет дергать метод "save" через rosetta
                all => {type => 'widget'}
            },
            get => sub {
                return $_[0]->{__USER_WIDGETS__}->{$_[1]->{'public_id'}} // [];
            },
        },
        'is_exists' => {
            db_expr => {'if' => [['is', ['user_id', \undef]], \0, \1]},    # if(user_id is null, 1, 0)
            db      => TRUE,
            api     => 1,
            type    => 'boolean',
        },
        widgets_for_restore => {
            depends_on => ['public_id', 'default_deleted_ids'],
            api        => TRUE,
            type       => 'complex',
            get        => sub {
                return $_[0]->{__WIDGETS_FOR_RESTORE__}->{$_[1]->{'public_id'}} // [];
            },
        },
        public_id => {    # нужно для API
            db      => TRUE,
            db_expr => 'user_id',
            type    => 'string',
            api     => TRUE,
        },
        fields_depends => {
            depends_on => [qw(user_id)],
            get        => sub {
                return {};
            },
            type => 'complex',
            api  => TRUE,
        },
        user_id                  => {default => TRUE, db => TRUE, pk   => TRUE, type => 'number'},
        json_widgets_settings    => {default => TRUE, db => TRUE, type => 'string',},
        json_default_deleted_ids => {default => TRUE, db => TRUE, type => 'string',},
        multistate               => {
            db   => TRUE,
            api  => TRUE,
            type => 'number',
            get  => sub {
                $_[1]->{'multistate'} // 0;
            },
        },
        multistate_name => {
            depends_on => ['multistate'],
            get        => sub {
                $_[0]->model->get_multistate_name($_[1]->{'multistate'});
            },
            api  => TRUE,
            type => 'string',
        },
        widgets_settings => {
            depends_on => ['json_widgets_settings'],
            api        => TRUE,
            type       => 'complex',
            get        => sub {
                return defined($_[1]->{'json_widgets_settings'}) ? from_json($_[1]->{'json_widgets_settings'}) : undef;
            },
        },
        default_deleted_ids => {
            depends_on => ['json_default_deleted_ids'],
            api        => TRUE,
            type       => 'complex',
            get        => sub {
                return
                  defined($_[1]->{'json_default_deleted_ids'}) ? from_json($_[1]->{'json_default_deleted_ids'}) : {};
            },
        },
    };
}

sub get_structure_model_filter {
    return {
        db_accessor => 'partner_db',
        fields      => {user_id => {type => 'number', label => d_gettext('User ID')},},
    };
}

sub get_db_filter_simple_fields {
    my ($self) = @_;
    return [];
}

sub get_structure_multistates_graph {
    return {
        empty_name  => d_gettext('New'),
        multistates => [[new => d_gettext('New widgets')], [edited => d_gettext('Edited Widgets')],],
        actions     => {
            add    => d_pgettext('Widgets', 'Add'),
            edit   => d_pgettext('Widgets', 'Edit'),
            delete => d_pgettext('Widgets', 'Delete'),
        },
        right_group        => [widgets => d_gettext('Right to manage Widgets')],
        right_name_prefix  => 'widgets_',
        multistate_actions => [
            {
                action    => 'add',
                from      => '__EMPTY__',
                set_flags => ['new'],
            },
            {
                action      => 'edit',
                from        => 'new or edited',
                set_flags   => ['edited'],
                reset_flags => ['new'],
            },
            {
                action      => 'delete',
                from        => 'new or edited',
                set_flags   => ['edited'],
                reset_flags => ['new'],
            },
        ],
    };
}

sub pre_process_fields {
    my ($self, $fields, $result) = @_;

    my $cur_user_id = $self->get_option(cur_user => {})->{'id'};
    my $public_id = $result->[0]->{public_id} // $cur_user_id;

    throw Exception::Validation::BadArguments gettext('You cannot get "user_widgets" for other users')
      if @$result > 1 || $public_id != $cur_user_id;

    if ($fields->need('user_widgets')) {
        my $user_widgets = {
            widgets_settings    => from_json($result->[0]->{json_widgets_settings}    // '[]'),
            default_deleted_ids => from_json($result->[0]->{json_default_deleted_ids} // '{}')
        };

        $fields->{'__USER_WIDGETS__'} = {$public_id => $self->get_widgets($user_widgets)};
    }

    if ($fields->need('widgets_for_restore')) {
        $fields->{'__WIDGETS_FOR_RESTORE__'} =
          {$public_id => $self->get_widgets_for_restore($fields->{default_deleted_ids})};
    }
}

sub get_actions_depends {[qw(user_id multistate)]}

sub get_available_fields_depends {[]}

sub get_editable_fields_depends {[]}

sub get_available_fields {
    my ($self, $obj) = @_;

    my $fields = {map {$_ => TRUE} keys %{$self->get_model_fields()}};

    my $accessor = $self->accessor();
    $self->app->delete_field_by_rights($fields,
        {$accessor . '_view_field__%s' => [qw(default_deleted_ids  widgets_settings)],});

    return $fields;
}

sub get_add_fields {
    my ($self) = @_;
    my $fields = $self->_get_common_add_edit_fields();
    return $fields;
}

sub get_editable_fields {
    my ($self, $object) = @_;
    my $fields = $self->_get_common_add_edit_fields();
    return $fields;
}

sub _get_common_add_edit_fields {
    return {
        user_widgets             => TRUE,
        json_default_deleted_ids => TRUE,
        json_widgets_settings    => TRUE,
    };
}

sub hook_fields_processing_before_validation {
    my ($self, $obj) = @_;

    $self->SUPER::hook_fields_processing_before_validation($obj);

    if (exists $obj->{'user_widgets'}) {
        QBit::Validator->new(
            data     => $obj,
            template => {
                type   => 'hash',
                fields => {user_widgets => {type => 'array',},}
            },
            app   => $self,
            throw => TRUE,
        );

        foreach my $widget (@{$obj->{'user_widgets'}}) {
            # не сохраняем автосгенеренные данные
            delete $widget->{'server_settings'};
        }
    }

    return 1;
}

sub hook_owner_processing {
    # заглушка нужна чтобы не простявлять поле owner_id
    return 1;
}

sub hook_set_initialize_settings {
    my ($self, $obj, %opts) = @_;
    $obj->{'multistate'} = 0;
    $obj->{user_id} = $self->get_option(cur_user => {})->{'id'};
}

sub hook_preparing_fields_to_save {
    my ($self, $obj) = @_;

    if (exists $obj->{'user_widgets'}) {
        my $new_obj = $self->_get_updated_widgets(delete $obj->{'user_widgets'});
        $obj = {%$obj, %$new_obj};
    }

    $self->SUPER::hook_preparing_fields_to_save($obj);
}

sub on_action_delete {
    my ($self, $obj, %opts) = @_;

    $opts{'widgets'} = [$opts{'widgets'}] unless ref($opts{'widgets'}) eq 'ARRAY';

    QBit::Validator->new(
        data     => \%opts,
        template => {
            type   => 'hash',
            fields => {
                widgets => {
                    type => 'array',
                    all  => {type => 'widget'},
                },
            }
        },
        app   => $self,
        throw => TRUE,
    );

    my @deleted_widgets = $self->_get_widget_models_by_settings($opts{'widgets'});

    my %widgets_ids = ();
    foreach my $widget (@deleted_widgets) {
        throw Exception::Validation::BadArguments gettext('Can not delete widget "%s"', $widget->title)
          unless $widget->can_delete;

        $widgets_ids{$widget->id} = TRUE;
    }

    my $user_widgets = $self->get($obj, fields => [qw(widgets_settings default_deleted_ids)]) // {};

    my $cur_widgets_settings = $user_widgets->{'widgets_settings'}    // [];
    my $default_deleted_ids  = $user_widgets->{'default_deleted_ids'} // {};

    my @defualt_widgets = $self->_get_default_widgets();

    my %default_widgets_ids = map {$_->id() => TRUE} @defualt_widgets;

    my @cur_widgets = $self->_merge_widgets([$self->_get_widget_models_by_settings($cur_widgets_settings)],
        \@defualt_widgets, $default_deleted_ids);

    foreach (@deleted_widgets) {
        $default_deleted_ids->{$_->id()} = TRUE if $default_widgets_ids{$_->id()};
    }

    my @need_save = map {$_->settings} grep {$_->is_available && !$widgets_ids{$_->id}} @cur_widgets;

    $self->partner_db->transaction(
        sub {
            $self->partner_db->widgets->edit(
                $obj,
                {
                    json_default_deleted_ids => to_json($default_deleted_ids, canonical => TRUE),
                    json_widgets_settings    => to_json(\@need_save,          canonical => TRUE),
                }
            );
        }
    );

    return TRUE;
}

sub query {
    my ($self, %opts) = @_;

    my $user_id = $self->get_option(cur_user => {})->{'id'};

    my $query = $self->partner_db->query->select(
        fields => {'public_id' => 'id',},
        table  => $self->partner_db->users(),
        filter => {id          => $user_id},
      )->left_join(
        alias   => 'widgets',
        table   => $self->partner_db_table(),
        fields  => $opts{'fields'}->get_db_fields(),
        join_on => [user_id => '=' => {'id' => $self->partner_db->users()}]
      );

    return $query;
}

sub get_fields_depends {{}}

sub api_can_delete {TRUE}

sub api_available_actions {qw(edit delete)}
sub api_can_add           {TRUE}
sub api_can_edit          {TRUE}

sub get_widget_models {
    my ($self) = @_;

    return map {$self->app->$_} @{$self->product_manager->get_accessors_by_tag('widgets')};
}

sub get_widgets {
    my ($self, $user_widgets) = @_;

    my $user_id = $self->get_option(cur_user => {})->{'id'}
      // throw Exception::Validation::BadArguments gettext('No user ID');

    unless ($user_widgets) {
        $user_widgets = $self->get($user_id, fields => [qw(widgets_settings default_deleted_ids)]) // {};
    }

    my $cur_widgets_settings = $user_widgets->{'widgets_settings'}    // [];
    my $default_deleted_ids  = $user_widgets->{'default_deleted_ids'} // {};

    my @defualt_widgets = $self->_get_default_widgets();

    my @cur_widgets = $self->_merge_widgets([$self->_get_widget_models_by_settings($cur_widgets_settings)],
        \@defualt_widgets, $default_deleted_ids);

    my @report_id_list = map {$_->report_id} grep {'statistics' eq $_->type} @cur_widgets;
    my $reports = $self->partner_db->statistics_reports->get_all(
        fields => [qw(id report_id report_type)],
        filter => ['OR', [['id' => 'IN' => \\@report_id_list], ['report_id' => 'IN' => \\@report_id_list],]],
    );

    my %report_hash =
      map {$_->{id} => {report_type => $_->{report_type}, converted_from => $_->{report_id},}} @$reports;

    my @widget_list;
    my $standard_report_map = {
        advnet_report                        => 'context_income_total_30days',
        all_payments                         => 'all_income',
        all_payments_adfox                   => 'all_income_adfox',
        context_report                       => 'context_income_30days',
        context_rtb_detailed_report          => 'context_rtb_detailed_report_30days',
        context_rtb_report                   => 'context_rtb_report_30days',
        context_rtb_tags_report              => 'context_rtb_tags_report_30days',
        mobile_report                        => 'mobile_income_30days',
        mobile_rtb_detailed_report           => 'mobile_rtb_detailed_report_30days',
        mobile_rtb_report                    => 'mobile_rtb_report_30days',
        video_instream_block_detailed_report => 'video_instream_block_detailed_report_30days',
        video_instream_block_report          => 'video_instream_block_report_30days',
        video_site_report                    => 'video_blocks_report',
    };

    foreach my $widget (@cur_widgets) {
        if ('statistics' ne $widget->type) {
            push @widget_list, $widget;
        } else {
            if ('mol' eq $report_hash{$widget->report_id}{report_type}) {
                push @widget_list, $widget;
            } else {
                if (exists $standard_report_map->{$widget->report_id}) {
                    $widget->{settings}{settings}{report_id} = $standard_report_map->{$widget->report_id};
                    $widget->{settings}{settings}{showType}  = delete $widget->{settings}{settings}{show_type}
                      // 'chart';
                    push @widget_list, $widget;
                } else {
                    my @converted_to = grep {
                        defined $report_hash{$_}{converted_from}
                          && $report_hash{$_}{converted_from} eq $widget->report_id
                    } keys %report_hash;
                    if ($converted_to[0]) {
                        $widget->{settings}{settings}{report_id} = $converted_to[0];
                        $widget->{settings}{settings}{showType}  = delete $widget->{settings}{settings}{show_type}
                          // 'chart';
                        push @widget_list, $widget;
                    }
                }
            }
        }
    }

    $self->_add_widgets_templates(\@widget_list);

    return [map {$_->set_server_settings(); $_->settings()} @widget_list];
}

sub get_widgets_for_restore {
    my ($self, $default_deleted_ids) = @_;

    my $user_id = $self->get_option(cur_user => {})->{'id'}
      // throw Exception::Validation::BadArguments gettext('No user ID');

    unless (defined $default_deleted_ids) {
        my $user_widgets = $self->get($user_id, fields => [qw(default_deleted_ids)]) // {};
        $default_deleted_ids = $user_widgets->{'default_deleted_ids'};
    }

    my %widget_model_by_type = map {$_->type() => $_} $self->get_widget_models();

    my @widget_settings = ();
    foreach my $id (keys(%$default_deleted_ids)) {
        my $widget;

        foreach my $type (keys(%widget_model_by_type)) {
            if ($id =~ m/^$type/) {
                $widget = $widget_model_by_type{$type};
                last;
            }
        }

        unless ($widget) {
            my $exception = Exception->new(gettext('Accessor not found in app for widget with id "%s"', $id));

            $self->app->exception_dumper->dump_as_html_file($exception);

            next;
        }

        next unless $widget->can_restore() && $widget->can_view_as_default();

        push(@widget_settings, $widget->get_default_settings());
    }

    my @widgets = $self->_get_widget_models_by_settings(\@widget_settings);

    $self->_add_widgets_templates(\@widgets);

    return [
        map {$_->set_server_settings(); $_->settings()}
        sort {$a->priority() <=> $b->priority()} grep {$_->is_available()} @widgets
    ];
}

sub _get_updated_widgets {
    my ($self, $new_widgets_settings, %opts) = @_;

    my $user_id = $self->get_option(cur_user => {})->{'id'};

    $new_widgets_settings = [$new_widgets_settings] unless ref($new_widgets_settings) eq 'ARRAY';

    my @new_widgets = $self->_get_widget_models_by_settings($new_widgets_settings);

    my $user_widgets = $self->get($user_id, fields => [qw(widgets_settings default_deleted_ids)]) // {};

    my $cur_widgets_settings = $user_widgets->{'widgets_settings'}    // [];
    my $default_deleted_ids  = $user_widgets->{'default_deleted_ids'} // {};
    my @defualt_widgets      = $self->_get_default_widgets();

    my %default_widgets_ids = map {$_->id() => TRUE} @defualt_widgets;

    my @cur_widgets = $self->_merge_widgets([$self->_get_widget_models_by_settings($cur_widgets_settings)],
        \@defualt_widgets, $default_deleted_ids);

    my %new_widget_ids = ();
    foreach my $widget (@new_widgets) {
        throw Exception::Validation::BadArguments gettext('Can not save widget "%s"', $widget->title())
          unless $widget->is_available();

        $new_widget_ids{$widget->id()} = TRUE;
    }

    my @need_delete = ();
    if ($opts{'append'}) {
        if ($opts{'append'} eq 'last') {
            $new_widgets_settings = [@$cur_widgets_settings, @$new_widgets_settings];
        } elsif ($opts{'append'} eq 'first') {
            $new_widgets_settings = [@$new_widgets_settings, @$cur_widgets_settings];
        } else {
            throw Exception::Validation::BadArguments gettext('Unknow value "%s" in "append"', $opts{'append'});
        }
    } elsif ($opts{'replace'}) {
        my @cur_widget_settings_filtered;
        my $is_replaced = FALSE;
        $cur_widgets_settings = $new_widgets_settings unless @$cur_widgets_settings;
        my @new_widget_settings_both_show_type_aware;
        if ('both' eq $new_widgets_settings->[0]{settings}{showType}) {
            for my $showType (qw(chart table)) {
                my %settings_clone = %{$new_widgets_settings->[0]{settings}};
                $settings_clone{showType} = $showType;
                push @new_widget_settings_both_show_type_aware,
                  {%{$new_widgets_settings->[0]}, settings => \%settings_clone};
            }
        } else {
            push @new_widget_settings_both_show_type_aware, $new_widgets_settings->[0];
        }
        for (my $i = 0; $i < @$cur_widgets_settings; $i++) {
            if ('statistics' eq $cur_widgets_settings->[$i]{type}
                && $cur_widgets_settings->[$i]{settings}{report_id} eq $new_widgets_settings->[0]{settings}{report_id})
            {
                next if $is_replaced || !exists $new_widgets_settings->[0]{settings}{showType};
                push @cur_widget_settings_filtered, @new_widget_settings_both_show_type_aware;
                $is_replaced = TRUE;
            } else {
                push @cur_widget_settings_filtered, $cur_widgets_settings->[$i];
            }
        }
        $new_widgets_settings =
          $is_replaced
          ? \@cur_widget_settings_filtered
          : [@new_widget_settings_both_show_type_aware, @cur_widget_settings_filtered];
    } else {
        foreach my $widget (@cur_widgets) {
            my $id = $widget->id();

            unless ($new_widget_ids{$id}) {
                throw Exception::Validation::BadArguments gettext('Can not delete widget "%s"', $widget->title)
                  unless $widget->can_delete;

                push(@need_delete, $id) if $default_widgets_ids{$id};
            }
        }
    }

    $default_deleted_ids->{$_} = TRUE foreach @need_delete;
    foreach my $id (keys(%$default_deleted_ids)) {
        #убираем из удалённых восстановленные виджеты
        delete($default_deleted_ids->{$id}) if $new_widget_ids{$id};
    }

    my $json_default_deleted_ids = to_json($default_deleted_ids,  canonical => TRUE);
    my $json_widgets_settings    = to_json($new_widgets_settings, canonical => TRUE);

    my $new_widgets = {
        json_default_deleted_ids => $json_default_deleted_ids,
        json_widgets_settings    => $json_widgets_settings,
    };

    return $new_widgets;
}

sub save {
    my ($self, $new_widgets_settings, %opts) = @_;

    my $user_id = $self->get_option(cur_user => {})->{'id'};

    my $is_exists = $self->get($user_id, fields => [qw(is_exists)])->{is_exists};

    my $new_widgets = $self->_get_updated_widgets($new_widgets_settings, %opts);

    if (!$is_exists) {
        $self->partner_db->widgets->add({user_id => $user_id, %$new_widgets});
        $self->do_action($user_id, 'add', %$new_widgets);
    } else {
        $self->do_action($user_id, 'edit', %$new_widgets);
    }

    return TRUE;
}

sub make_fields_defaults {
    my ($self, $opts, $need_fields) = @_;

    my %result = ();

    if ($need_fields->{user_widgets}) {

        my @defualt_widgets = $self->_get_default_widgets();
        my $widget_settings = [map {$_->settings} @defualt_widgets];
        my $widgets         = $self->get_widgets({'widgets_settings' => $widget_settings,});
        $result{user_widgets} = $widgets;
    }

    return \%result;
}

sub _get_default_widgets {
    my ($self) = @_;

    my @widget_model_list = grep {$_->can_view_as_default()} $self->get_widget_models();
    my @default_settings =
      map  {$_->get_default_settings()}
      sort {$a->priority() <=> $b->priority()} @widget_model_list;

    my @default_widgets = $self->_get_widget_models_by_settings(\@default_settings);

    return grep {$_->is_available()} @default_widgets;
}

sub _get_widget_models_by_settings {
    my ($self, $widgets_settings) = @_;

    my @widgets = ();
    foreach my $settings (@$widgets_settings) {
        my $accessor = $settings->{'accessor'};

        throw Exception::Validation::BadArguments gettext('Expected settings "accessor"') unless defined($accessor);

        throw Exception::Validation::BadArguments gettext('Unknown accessor "%s"', $accessor)
          unless $self->app->can($accessor);

        push(@widgets, $self->app->$accessor->widget($settings));
    }

    return @widgets;
}

sub _merge_widgets {
    my ($self, $cur_widgets, $default_widgts, $default_deleted_ids) = @_;

    my %default_deleted_ids = %$default_deleted_ids;

    my @cur_widgets = ();
    foreach (@$cur_widgets) {
        push(@cur_widgets, $_) if $_->is_available();

        $default_deleted_ids{$_->id()} = TRUE;
    }

    foreach my $widget (@$default_widgts) {
        next if $default_deleted_ids{$widget->id()};

        push(@cur_widgets, $widget);
    }

    return @cur_widgets;
}

sub _add_widgets_templates {
    my ($self, $widgets) = @_;

    my %template_keys = ();

    for (my $i = 0; $i < @$widgets; $i++) {
        next unless $widgets->[$i]->can('get_template_key');

        $template_keys{$widgets->[$i]->get_template_key()} = $i;
    }

    if (%template_keys) {
        my $templates = {
            map {$_->{key} => $_->{content}} @{
                $self->text_template->get_all(
                    fields => [qw(key content)],
                    filter => [key => 'IN' => [keys %template_keys]]
                )
              }
        };

        for my $key (keys %template_keys) {
            throw gettext('Not found template for key "%s"', $key) unless exists $templates->{$key};

            my $widget = $widgets->[$template_keys{$key}];

            my $settings = $widget->settings();

            $settings->{'server_settings'}{'message'} = $templates->{$key};

            $widget->settings($settings);
        }
    }
}

TRUE;
