package Application::Model::Widgets::Widget::Statistics;

use qbit;

use base qw(Application::Model::Widgets::Widget);

use Exception::Validation::BadArguments;

my @DEFAULT_IDS = qw(
  all_payments
  dsp_report
  dsp_detailed_report
  context_income
  context_report
  search_income
  search_report
  mobile_income
  mobile_report
  video_income
  video_site_report
  internal_context_income
  internal_search_income
  internal_mobile_income
  internal_advnet_report
  internal_context_report
  internal_search_report
  internal_mobile_report
  );

my $REPORTS = [map {{report_id => $_, right_view_as_default => "widget_statistics_view_as_default__$_"}} @DEFAULT_IDS];

sub accessor {'widget_statistics'}

__PACKAGE__->model_accessors(
    statistics_reports => 'Application::Model::StatisticsReports',
    bk_statistics      => 'Application::Model::BKStatistics',
);

__PACKAGE__->register_rights(
    [
        {
            name        => 'widget_statistics',
            description => d_gettext('Rights to use widget statistics'),
            rights      => {
                widget_statistics_view_as_default => d_gettext('Right to view widget statistics as default'),
                map {
                    scalar "widget_statistics_view_as_default__$_" =>
                      d_gettext('Right to view widget statistics "%s" as default', $_)
                  } @DEFAULT_IDS
            }
        }
    ]
);

sub can_view_as_default {
    $_[0]->check_short_rights('view_as_default');
}

sub check_available {
    my ($self) = @_;

    my $level = $self->settings()->{'settings'}{'level'};

    my %available_levels = map {$_ => 1} @{$self->bk_statistics->get_available_statistics_levels()};

    $self->is_available(TRUE) if exists $available_levels{$level};
}

sub check_settings {
    my ($self) = @_;

    $self->SUPER::check_settings();

    throw Exception::Validation::BadArguments gettext('Expected "report_id" in settings')
      unless exists($self->settings->{'settings'}{'report_id'});

    throw Exception::Validation::BadArguments gettext('Expected "level" in settings')
      unless exists($self->settings->{'settings'}{'level'});
}

sub get_default_settings {
    my ($self) = @_;

    my $type     = $self->type();
    my $accessor = $self->accessor();

    my @ids = map {$_->{'report_id'}} grep {$self->check_rights($_->{'right_view_as_default'})} @$REPORTS;

    my $reports = $self->statistics_reports->get_all(
        fields   => [qw(id level)],
        filter   => {id => \@ids, is_standart_report => TRUE},
        order_by => ['order'],
        force    => TRUE,
    );

    return
      map {{type => $type, accessor => $accessor, settings => {report_id => $_->{'id'}, level => $_->{'level'},},}}
      @$reports;
}

sub id {
    my ($self) = @_;

    return $self->type . '_' . $self->settings->{'settings'}{'report_id'};
}

sub priority {500}

sub report_id {
    my ($self) = @_;

    return $self->settings()->{'settings'}{'report_id'};
}

sub resizable {TRUE}

sub title {gettext('Statistics')}

sub type {'statistics'}

TRUE;
