package Application::Model::_Utils::MethodOptions;

use qbit;

use base qw(QBit::Class);

use Exception::Validation::BadArguments::InvalidJSON;
use Exception::Validation::BadArguments;

__PACKAGE__->mk_ro_accessors(
    qw(options values missing_fields invalid_value_fields valid_value_fields incompatible_value_fields));

my %TYPES = (
    date                         => {check => sub {return scalar($_[0] =~ /^[1-9][0-9]{3}-[0-9]{2}-[0-9]{2}$/);},},
    number                       => {check => sub {return scalar($_[0] =~ /^[0-9]+$/);},},
    decimal_number               => {check => sub {return scalar($_[0] =~ /^[0-9]+(?:[.,][0-9]+)?$/);},},
    not_negative_integral_number => {check => sub {return scalar($_[0] =~ /^[0-9]|[1-9][0-9]+$/);},},
    text     => {check => \&_check_text,},
    textarea => {check => \&_check_text,},
    json     => {
        check => sub {
            my ($value) = @_;

            return TRUE unless defined($value);

            my $invalid_json = FALSE;
            try {
                from_json($value);
            }
            catch Exception::Validation::BadArguments::InvalidJSON with {
                $invalid_json = TRUE;
            };

            return !$invalid_json;
        },
    },
);

sub errors {
    my ($self) = @_;

    return join("\n", @{$self->{'errors'}});
}

sub fields {
    my ($self) = @_;

    return {%{$self->missing_fields}, %{$self->invalid_value_fields}, %{$self->valid_value_fields}};
}

sub init {
    my ($self) = @_;

    $self->{'missing_fields'}            //= {};
    $self->{'invalid_value_fields'}      //= {};
    $self->{'valid_value_fields'}        //= {};
    $self->{'incompatible_value_fields'} //= {};
    $self->{'generated_fields'}          //= {};
    $self->{'errors'}                    //= [];

    $self->_check_and_expand($self->options);
    $self->_generate_field_values();
    $self->_check_incompatible_values();

    $self->{'check'}->($self, $self->{'values'}, $self->{'errors'}) if defined($self->{'check'});

    $self->{'throw_exception'} = TRUE unless defined($self->{'throw_exception'});
    $self->throw_exception() if $self->{'throw_exception'};
}

sub problem_fields {
    my ($self) = @_;

    return {%{$self->missing_fields}, %{$self->invalid_value_fields}, %{$self->incompatible_value_fields}};
}

sub throw_exception {
    my ($self) = @_;

    my $errors = $self->{'errors'};

    push(@$errors, (map {gettext("'%s': has invalid value", $_)} keys(%{$self->{'invalid_value_fields'}})))
      if %{$self->{'invalid_value_fields'}};
    push(@$errors, (map {gettext("'%s': is missing", $_)} keys(%{$self->{'missing_fields'}})))
      if %{$self->{'missing_fields'}};

    throw Exception::Validation::BadArguments $self->errors() if @$errors;
}

sub _add_label {
    my ($self, $field, $field_description, $value) = @_;

    my $boolean = ($field_description->{'type'} // '') eq 'boolean';

    if (defined($value)) {
        if (   exists($field_description->{'values'})
            && exists($field_description->{'values'}{$value})
            && exists($field_description->{'values'}{$value}{'label'}))
        {
            my $label = $field_description->{'values'}{$value}{'label'};
            $self->{'values'}{$field . '_label'} = ref($label) eq 'CODE' ? $label->() : $label;
        } elsif ($boolean) {
            $self->{'values'}{$field . '_label'} = $value ? gettext('Yes') : gettext('No');
        } else {
            $self->{'values'}{$field . '_label'} = $value;
        }
    }
}

sub _check_and_expand {
    my ($self, $options) = @_;

    foreach my $field (keys(%{$options})) {
        $self->{'values'}{$field} =~ s/^\s+|\s+$//g
          if defined($self->{'values'}{$field}) && !ref($self->{'values'}{$field});

        my $field_description = $options->{$field};
        my $value             = $self->{'values'}{$field};
        my $boolean           = ($field_description->{'type'} // '') eq 'boolean';

        $field_description->{'values'} //= [0, 1] if $boolean;

        if (exists($field_description->{'values'})) {
            $field_description->{'values'} = $field_description->{'values'}->()
              if ref($field_description->{'values'}) eq 'CODE';
            $field_description->{'values'} = {map {$_ => undef} @{$field_description->{'values'}}}
              if ref($field_description->{'values'}) eq 'ARRAY';
        }

        my $can_not_be_generated = FALSE;
        my $generated            = exists($field_description->{'value'});
        if ($generated) {
            $self->{'generated_fields'}{$field} = $field_description;
            $can_not_be_generated = !$self->_generate_field_value($field, $field_description);
            $value = $self->{'values'}{$field} unless $can_not_be_generated;
        }

        unless ($generated && $can_not_be_generated) {
            if (defined($value) && ($boolean || length($value))) {
                if (ref($value)) {
                    $self->{'invalid_value_fields'}{$field} = $field_description;
                } elsif (exists($field_description->{'values'})) {
                    $value = $value ? 1 : 0 if $boolean;

                    if (exists($field_description->{'values'}{$value})) {
                        $self->{'valid_value_fields'}{$field} = $field_description unless $generated;

                        if (exists($field_description->{'values'}{$value}{'fields'})) {
                            $self->_check_and_expand($field_description->{'values'}{$value}{'fields'});
                        }
                        if (exists($field_description->{'fields'})) {
                            $self->_check_and_expand($field_description->{'fields'});
                        }

                        $self->_add_label($field, $field_description, $value);
                    } else {
                        $self->{'invalid_value_fields'}{$field} = $field_description;
                    }
                } elsif ($generated) {
                    # do nothing.
                } elsif (defined($field_description->{'type'})) {
                    my $check = $field_description->{'check'}
                      // ($TYPES{$field_description->{'type'}} // {})->{'check'};
                    $field_description->{'check'} = $check if defined($check);

                    if (defined($check) && $check->($value, $field_description)) {
                        $self->{'valid_value_fields'}{$field} = $field_description unless $generated;
                    } else {
                        $self->{'invalid_value_fields'}{$field} = $field_description;
                    }
                } else {
                    throw 'Broken field dependencies';
                }
            } elsif ($field_description->{'optional'}) {
                $self->{'valid_value_fields'}{$field} = $field_description unless $generated;
                if (exists($field_description->{'fields'})) {
                    $self->_check_and_expand($field_description->{'fields'});
                }
            } else {
                $self->{'missing_fields'}{$field} = $field_description;
            }
        }
    }
}

sub _check_incompatible_values {
    my ($self) = @_;

    foreach my $combination (@{$self->{'incompatible_values'}}) {
        my @sub_conditions;
        foreach my $field_set (@{$combination->{'condition'}}) {
            my @set_valid_value_fields = grep {exists($self->valid_value_fields->{$_})} keys(%$field_set);
            if (@set_valid_value_fields) {
                unless (@sub_conditions) {
                    @sub_conditions = map {
                        {$_ => $field_set->{$_}}
                    } @set_valid_value_fields;
                } else {
                    my @extended_sub_conditions;
                    foreach my $sub_condition (@sub_conditions) {
                        push(
                            @extended_sub_conditions,
                            map {
                                {%$sub_condition, $_ => $field_set->{$_}}
                              } @set_valid_value_fields
                            );
                    }
                    @sub_conditions = @extended_sub_conditions;
                }
            } else {
                @sub_conditions = ();
                last;
            }
        }
        foreach my $sub_condition (@sub_conditions) {
            my $match = TRUE;
            foreach my $field (keys(%$sub_condition)) {
                my $sub_condition_value = $sub_condition->{$field};
                my $value               = $self->values->{$field};

                if (($self->valid_value_fields->{$field}->{'type'} // '') eq 'boolean') {
                    $sub_condition_value = $sub_condition_value ? 1 : 0;
                    $value = $value ? 1 : 0;
                }

                if ($sub_condition_value ne $value) {
                    $match = FALSE;
                    last;
                }
            }

            if ($match) {
                if (exists($combination->{'message'})) {
                    push(@{$self->{'errors'}}, $combination->{'message'});
                } else {
                    my %labeled_values =
                      map {($self->valid_value_fields->{$_}->{'label'} // $_) => $self->values->{$_ . '_label'}}
                      keys(%$sub_condition);
                    push(
                        @{$self->{'errors'}},
                        gettext(
                            "Incompatible values combination: %s",
                            join(', ', map {"$_ = $labeled_values{$_}"} sort keys(%labeled_values))
                        )
                    );
                }
                $self->incompatible_value_fields->{$_} = $self->valid_value_fields->{$_} foreach keys(%$sub_condition);

                last;
            }
        }
    }
}

sub _check_text {
    my ($value, $field_description) = @_;

    my $length = ($_[1] // {})->{'length'};

    return FALSE unless defined($value);
    return length($value) <= $length if defined($length);
    return TRUE;
}

sub _generate_field_value {
    my ($self, $field, $field_description) = @_;

    my @depends_on =
        !$field_description->{'depends_on'}     ? ()
      : ref($field_description->{'depends_on'}) ? @{$field_description->{'depends_on'}}
      :                                           $field_description->{'depends_on'};

    return FALSE
      if grep {!($self->{'valid_value_fields'}{$_} || $self->{'generated_fields'}{$_}) || !exists($self->values->{$_})}
          @depends_on;

    $self->{'values'}{$field} = $field_description->{'value'};
    $self->{'values'}{$field} = $self->{'values'}{$field}->({map {$_ => $self->{'values'}{$_}} @depends_on})
      if ref($self->{'values'}{$field}) eq 'CODE';

    $self->_add_label($field, $field_description, $self->{'values'}{$field});

    return TRUE;
}

sub _generate_field_values {
    my ($self) = @_;

    my @generated_fields =
      sort {exists($self->{'generated_fields'}{$a}{'depends_on'}) ? 1 : 0} keys(%{$self->{'generated_fields'}});
    foreach my $field (@generated_fields) {
        my $field_description = $self->{'generated_fields'}{$field};

        $self->_generate_field_value($field, $field_description);
    }
}

TRUE;
