package Cron;

use qbit;
use Time::Local;

use base qw(QBit::Cron Application);

use Utils::Logger qw(ERROR);
use Utils::Cron;
use Utils::MonitoringUtils qw(send_to_graphite get_pjapi);
use Utils::Logger qw(INFO), {logger => 'Cron',};
use Schedule::Cron::Events;

use Cron::Methods::AdvNet path             => 'adv_net';
use Cron::Methods::AdvNetDooh path         => 'adv_net_dooh';
use Cron::Methods::AdvNetMobile path       => 'adv_net_mobile';
use Cron::Methods::AdvNetVideo path        => 'adv_net_video';
use Cron::Methods::Assistants path         => 'assistants';
use Cron::Methods::BadDomains path         => 'bad_domains';
use Cron::Methods::Balance path            => 'balance';
use Cron::Methods::BusinessRules path      => 'business_rule_methods';
use Cron::Methods::DSP path                => 'dsp';
use Cron::Methods::Dictionaries path       => 'dictionaries';
use Cron::Methods::DictionariesYT path     => 'dictionaries_yt';
use Cron::Methods::Domain path             => 'domain';
use Cron::Methods::FastBan path            => 'fast_ban';
use Cron::Methods::Fias path               => 'fias';
use Cron::Methods::Frontend path           => 'frontend';
use Cron::Methods::LastPayout path         => 'last_payout';
use Cron::Methods::MailNotification path   => 'mail_notification';
use Cron::Methods::MDSAvatarsCleanup path  => 'mds_avatars_cleanup';
use Cron::Methods::Metrika path            => 'metrika';
use Cron::Methods::Moderation path         => 'moderation';
use Cron::Methods::Monitoring path         => 'monitoring';
use Cron::Methods::PagesBS path            => 'get_partner_pages';
use Cron::Methods::ReportParamsDigest path => 'report_params_digest';
use Cron::Methods::SSP path                => 'ssp';
use Cron::Methods::ShortLink path          => 'short_link';
use Cron::Methods::System path             => 'system';
use Cron::Methods::TNSDict path            => 'tns_dict';
use Cron::Methods::UpdateBKAsync path      => 'update_bk_async';
use Cron::Methods::Webmaster path          => 'webmaster';
use Cron::Methods::UserNotifications path  => 'user_notifications';
use Cron::Methods::Assessors path          => 'assessors';
use Cron::Methods::CustomBkData path       => 'custom_bk_data';
use Cron::Methods::SelfEmployedCheck path  => 'self_employed_check';
use Cron::Methods::AutoMigrations path     => 'automigrations';

use Exception::BatchFail;

sub do {
    my ($self, $path, $method, $ignore_stage) = @_;

    $ENV{'RUNNING_IN_CRON'} = TRUE;
    $ENV{'SYSTEM'}          = 'cron';

    my $exception;
    try {
        $self->SUPER::do($path, $method, $ignore_stage);
    }
    catch {
        ($exception) = @_;

        unless ($exception->isa('Exception::BatchFail')) {
            $self->exception_dumper->dump_as_html_file($exception);

            if ($ENV{'TAP_VERSION'}) {
                throw $exception;
            }
        }
    };

    unless ($ENV{'TAP_VERSION'}) {
        $self->send_event(
            defined($exception)
            ? (
                status      => 'CRIT',
                description => substr($exception->message, 0, 1000),
              )
            : (status => 'OK',),
        );

        # NOTE! в тестах досрочный выход не делаем
        exit(defined($exception) ? 1 : 0);
    }
}

sub send_event {
    my ($self, %event) = @_;
    if (!$ENV{'TAP_VERSION'} && in_array($self->get_option('stage', 'unknown'), [qw(test preprod production)])) {
        get_pjapi($self)->send(
            events => [
                {
                    service => $self->_get_juggler_service(),
                    %event,
                }
            ]
        );
    }
}

sub send_heartbeat {
    my ($self) = @_;
    $self->send_event(status => 'OK');
}

sub _get_juggler_service {
    my ($self, %opts) = @_;

    return sprintf 'cron__%s__%s', map {$opts{$_} // $self->get_option($_)} qw(cron_path cron_method);
}

sub post_run {
    my ($self) = @_;
    $self->SUPER::post_run();
    my $path            = $self->get_option('cron_path');
    my $full_method     = $self->get_option('cron_method');
    my $method          = $self->get_option('cron_instance_method');
    my $instance_number = $self->get_option('cron_instance_number');
    my $elapsed         = $self->get_time();

    INFO("[Cron $path - $full_method] END ($elapsed sec)");

    unless ($ENV{TAP_VERSION}) {
        my $memory = proc_memory();
        my ($process_memory) = $memory =~ /(\d+)/;
        send_to_graphite(
            interval => "one_min",
            path     => "cron.${path}.${full_method}.memory",
            value    => $process_memory,
            solomon  => {
                type            => 'cron',
                cron_path       => $path,
                cron_method     => $full_method,
                instance_method => $method,
                instance_number => $instance_number,
                sensor          => 'memory',
            }
        );
        send_to_graphite(
            interval => "one_min",
            path     => "cron.${path}.${full_method}.timing",
            value    => $elapsed,
            solomon  => {
                type            => 'cron',
                cron_path       => $path,
                cron_method     => $full_method,
                instance_method => $method,
                instance_number => $instance_number,
                sensor          => 'timing',
            }
        );
    }
}

sub generate_juggler {
    my ($self, $pja, %opts) = @_;

    my $stage = $opts{'stage'} // $self->get_option('stage', 'unknown');

    return FALSE unless in_array($stage, [qw(production preprod)]);

    my $methods = $self->get_cron_methods();

    foreach my $path (sort keys(%$methods)) {
        foreach my $method (sort keys(%{$methods->{$path}})) {

            my $attrs = $methods->{$path}{$method}{'attrs'};
            my $time  = $methods->{$path}{$method}{'time'};

            my $method_stages = $attrs->{'stage'} // {production => TRUE,};
            next unless $method_stages->{$stage};

            foreach my $i (1 .. ($attrs->{'instances'} // 1)) {
                my $full_method  = $attrs->{'instances'} ? $method . '_' . $i : $method;
                my @tags         = ('show-on-dashboard');
                my $default_tags = $self->get_option('api_juggler')->{tags};
                push @tags, @$default_tags        if $default_tags;
                push @tags, $attrs->{juggler_tag} if $attrs->{juggler_tag};
                $pja->add_or_update(
                    service => $self->_get_juggler_service(cron_path => $path, cron_method => $full_method),
                    ttl => $attrs->{'ttl'} // get_ttl_from_cron_time_date_fields($time) . 's',
                    tags => \@tags,
                    meta => {
                        urls => [
                            {
                                title => 'Что делать, если горит проверка',
                                url   => 'https://wiki.yandex-team.ru/partner/w/juggler-checks/cron/',
                                type  => 'wiki'
                            }
                        ]
                    }
                );
            }
        }
    }
    for my $service (qw(soon_auto_stop.crit soon_auto_stop.warn)) {
        $pja->add_or_update(
            service => $service,
            ttl     => '1200s',
            tags    => ['show-on-dashboard'],
            meta    => {
                urls => [
                    {
                        title => 'Что делать, если горит проверка',
                        url   => 'https://wiki.yandex-team.ru/partner/w/juggler-checks/soon_auto_stop/',
                        type  => 'wiki'
                    }
                ]
            }
        );
    }
}

TRUE;
