package Cron::Methods;

use qbit;

use Utils::Logger qw( INFO  INFOF  WARN );

use Exception::Validation::BadArguments;
use Exception::IncorrectParams;
use Exception::BatchFail;

use Utils::MonitoringUtils qw/ send_to_graphite /;

use PiConstants qw($MYSQL_DEFAULT_DATETIME);

sub _regular_update_statistics {
    my ($self, %opts) = @_;

    my $products  = $opts{'products'};
    my $for_month = $opts{'for_month'};
    my $storage   = $opts{'storage'};

    my @failed_products = ();

    foreach my $product (@$products) {

        INFO("[Cron::Methods] $product start");

        try {
            $self->app->$product->regular_update_statistics(
                for_month           => $for_month,
                storage             => $storage,
                day_update_observer => sub {
                    my ($app) = @_;
                    $app->send_heartbeat();
                },
            );
        }
        catch {
            my ($exception) = @_;
            unless ($exception->isa('Exception::BatchFail')) {
                $self->app->exception_dumper->dump_as_html_file($exception);
            }
            push @failed_products, $product;
        };

        INFO("[Cron::Methods] $product stop");
    }

    throw Exception::BatchFail
      sprintf('regular update statistics for products "%s" failed', join ', ', @failed_products)
      if @failed_products;

    return TRUE;
}

sub check_model_items_update_time {
    my ($self, %opts) = @_;

    my ($accessor, $field_names, $graphite_metrics, $solomon_metrics, $filter_data, $update_time_limit, $warn,
        $separate_metrics)
      = @opts{
        qw( accessor fields  graphite_metrics solomon_metrics filter update_time_limit warn_if_hang_too_long separate_metrics)
      };
    $filter_data //= {};
    $filter_data = clone $filter_data;

    my ($field, $is_not_null, $model_filter) = @$filter_data{qw( field  not_null  model_filter )};

    my $model = $self->app->$accessor;

    my $now = curdate(oformat => 'db_time');
    my $min_update_time = date_sub($now, second => $update_time_limit, iformat => 'db_time', oformat => 'db_time');

    my $is_create_date_exists = $model->get_model_fields()->{create_date};
    push @$field_names, ('update_time', ($is_create_date_exists ? 'create_date' : ()));

    $filter_data->{model_filter} =
      $model_filter
      ? ['AND', [$model_filter, ['update_time' => '<' => \$min_update_time]]]
      : ['update_time' => '<' => \$min_update_time];

    my $found_rows = $self->get_model_items_to_update(
        accessor => $accessor,
        fields   => $field_names,
        filter   => $filter_data,
        dontlog  => 1,
    );

    if (defined($separate_metrics)) {
        send_to_graphite(
            path    => $separate_metrics->{'graphite_metrics'},
            value   => scalar(@{$found_rows // []}),
            solomon => $separate_metrics->{'solomon_metrics'},
        );
    }

    my $pages_to_send = $self->partner_db->need_update_pages->get_all(
        fields => [qw(page_id update_time)],
        filter =>
          ['AND', [['model', '=', \$accessor], ['processed', '=', \0], ['update_time', '<', \$min_update_time],]]
    );

    my %map_rows = map {$_->{'page_id'} => $_} @$pages_to_send;

    my $heavy_pages     = {};
    my $page_field_name = '';
    unless ($accessor eq 'business_blocks') {
        $page_field_name = $model->get_page_id_field_name();
        $heavy_pages = {map {$_->{'page_id'} => 1} @{$self->partner_db->heavy_pages->get_all()}};
    }

    my $count  = 0;
    my $report = '';
    foreach my $row (@$found_rows) {
        my $is_heavy = $page_field_name && $heavy_pages->{$row->{$page_field_name}} ? 1 : 0;

        $report .= $self->_get_report($row, $field_names, count => $count, now => $now, is_heavy => $is_heavy);

        delete($map_rows{$row->{$page_field_name}});
    }

    foreach my $row (values(%map_rows)) {
        push(@$found_rows, $row);

        my $is_heavy = $page_field_name && $heavy_pages->{$row->{'page_id'}} ? 1 : 0;

        $report .=
          $self->_get_report($row, [qw(page_id update_time)], count => $count, now => $now, is_heavy => $is_heavy);
    }

    if ($warn && @$found_rows) {
        WARN {
            message => sprintf(
                "Some items are waiting to be updated for a long time (%d %s):\n%s",
                scalar(@$found_rows), $accessor, $report
            ),
            extra => {
                accessor => $accessor,
                items    => $found_rows,
            },
            fingerprint => ['Cron', 'check_update_in_bk', $accessor],
        };
    }

    send_to_graphite(
        path    => $graphite_metrics,
        value   => scalar(@{$found_rows // []}),
        solomon => $solomon_metrics,
    );

    return $found_rows;
}

sub _get_report {
    my ($self, $row, $field_names, %opts) = @_;

    my $delta_days;
    if ($row->{update_time} eq $MYSQL_DEFAULT_DATETIME) {
        $delta_days = '???';
    } else {
        $delta_days = dates_delta_days($row->{update_time}, $opts{'now'}, iformat => 'db_time');
        $row->{days_ago} = $delta_days;
    }

    return sprintf(
        qq[\t%d. %s (%s days ago%s)\n],
        ++$opts{'count'}, join(', ', map {sprintf '%s="%s"', $_, $row->{$_} // 'NULL'} @$field_names),
        $delta_days, ($opts{'is_heavy'} ? ', HEAVY' : '')
    );
}

=head2 get_model_items_to_update

Функция возвращает список строк таблицы указанной модели со статусом "need_update or updating"
  [ { field1 => <val>, field2 => <val> }, ... ]

accessor    - аксессор модели
field_names - список полей (обычно PK)
filter:
 - field            на какое поле накладывается фильтр
 - not_null         добавляет условие field is not null
 - only_ids         добавляет условие на записи
 - except_ids       добавляет условие на исключение записей
 - instances_count  добавляет условие на остаток от деления:
 - instance_number  field % instances_count == nstance_number
 - limit            лимит

=cut

sub get_model_items_to_update {
    my ($self, %opts) = @_;

    my ($accessor, $field_names, $filter_data, $order_by, $dontlog) =
      @opts{qw( accessor fields  filter order_by dontlog)};
    $filter_data //= {};

    my (
        $field,           $is_not_null,  $only_ids, $except_ids, $instances_count,
        $instance_number, $model_filter, $limit,    $multistate
       )
      = @$filter_data{
        qw( field  not_null  only_ids  except_ids  instances_count  instance_number  model_filter  limit multistate)};

    $instance_number -= 1;
    $only_ids   = [split /,/, $only_ids]   if $only_ids   && ref($only_ids)   ne 'ARRAY';
    $except_ids = [split /,/, $except_ids] if $except_ids && ref($except_ids) ne 'ARRAY';

    my $model = $self->app->$accessor;

    unless ($multistate) {
        throw Exception::IncorrectParams sprintf('Model "%s" has no multistate "need_update"', $accessor)
          unless $model->get_multistate_by_name('need_update');
        $multistate = $model->get_multistate_by_action('start_update');
    } else {
        $multistate = $model->get_multistates_by_filter($multistate);
    }

    my $field_name = $field // $field_names->[0];
    unless ($order_by) {
        $order_by = [$field_name];
    }

    my $filter = [[multistate => 'IN' => \$multistate]];
    push @$filter, [multistate => 'IN' => \$model->get_multistates_by_filter('balance_registered')]
      if $is_not_null && $model->get_multistate_by_name('balance_registered');
    push @$filter, [$field => 'IN'     => \$only_ids]   if $only_ids   && @$only_ids;
    push @$filter, [$field => 'NOT IN' => \$except_ids] if $except_ids && @$except_ids;
    push @$filter, [{'MOD' => [$field, \$instances_count]}, '=', \$instance_number]
      if $instances_count && defined $instance_number && !$only_ids;
    push @$filter, $model_filter if $model_filter;

    my $found_rows = $model->partner_db_table()->get_all(
        fields   => $field_names,
        filter   => ['AND' => $filter],
        order_by => $order_by,
        ($limit ? (limit => $limit) : ())
    );

    if (!$dontlog && $found_rows && @$found_rows) {
        INFOF(
            '%s - %d rows found: %s',
            $accessor,
            scalar(@$found_rows),
            '(' . join(
                ")\n (",
                map {
                    my $row = $_;
                    join '; ', map {join '=', $_, $row->{$_}} @$field_names
                  }
                  grep {
                    $_
                  }
                  map {
                    $found_rows->[$_]
                  } 0 .. 4
              )
              . ')'
             );
    }

    return $found_rows;
}

TRUE;
