package Cron::Methods::AdvNetDooh;

use qbit;
use Utils::Logger qw(ERROR);

use base qw(QBit::Cron::Methods Cron::Methods);

__PACKAGE__->model_accessors(
    api_selfservice => 'Application::Model::API::Yandex::SelfService',
    auto_start      => 'Application::Model::AutoStart',
    auto_stop       => 'Application::Model::AutoStop',
    indoor          => 'Application::Model::Page::InDoor',
    outdoor         => 'Application::Model::Page::OutDoor',
    indoor_block    => 'Application::Model::Block::InDoor',
    outdoor_block   => 'Application::Model::Block::OutDoor',
    kv_store        => 'QBit::Application::Model::KvStore',
);

my $page_accessor_indoor  = 'indoor';
my $page_accessor_outdoor = 'outdoor';

sub model_path {'adv_net_dooh'}

sub cron_prepare_auto_stop_indoor : CRON('0 * * * *') : LOCK {
    my ($self) = @_;

    $self->auto_stop->prepare_auto_stop(
        model => $self->indoor,
        products_for_agreement_checker =>
          $self->app->product_manager->get_block_accessors_by_page_n_tag($page_accessor_indoor, 'can_auto_stop'),
    );

    return TRUE;
}

sub cron_auto_stop_indoor : CRON('*/5 * * * *') : LOCK {
    my ($self) = @_;

    $self->auto_stop->do_auto_stop(
        email_subject => gettext('Autostop indoor'),
        model         => $self->indoor,
        products_for_agreement_checker =>
          $self->app->product_manager->get_block_accessors_by_page_n_tag($page_accessor_indoor, 'can_auto_stop'),
    );

    return TRUE;
}

sub cron_prepare_auto_stop_outdoor : CRON('0 * * * *') : LOCK {
    my ($self) = @_;

    $self->auto_stop->prepare_auto_stop(
        model => $self->outdoor,
        products_for_agreement_checker =>
          $self->app->product_manager->get_block_accessors_by_page_n_tag($page_accessor_outdoor, 'can_auto_stop'),
    );

    return TRUE;
}

sub cron_auto_stop_outdoor : CRON('*/5 * * * *') : LOCK {
    my ($self) = @_;

    $self->auto_stop->do_auto_stop(
        email_subject => gettext('Autostop outdoor'),
        model         => $self->outdoor,
        products_for_agreement_checker =>
          $self->app->product_manager->get_block_accessors_by_page_n_tag($page_accessor_outdoor, 'can_auto_stop'),
    );

    return TRUE;
}

sub send_updated_items_to_logbroker : CRON('*/1 * * * *') : LOCK {
    my ($self) = @_;

    my $current_time = curdate(oformat => 'db_time');
    my $chunk_size = 1000;

    my $pages_topic  = $self->get_option('logbroker')->{pages_topic};
    my $blocks_topic = $self->get_option('logbroker')->{blocks_topic};

    for my $page_model (qw(indoor outdoor)) {
        my $key              = $self->get_moderation_update_time_key($page_model);
        my $last_update_time = $self->kv_store->get($key);

        my $pages = [];
        my $i     = 0;
        do {
            $pages = $self->$page_model->get_all(
                fields => [qw(page_id update_time bk_state_id)],
                filter => [update_time => '>=' => $last_update_time],
                limit  => $chunk_size,
                offset => $i * $chunk_size,
            );

            for my $page (@$pages) {
                $page->{model} = $page_model;
                $page->{state} = delete $page->{bk_state_id};
            }

            $self->api_selfservice->logbroker(
                data  => $pages,
                topic => $pages_topic,
            );

            my %pages_dict = map {$_->{page_id} => $_} @$pages;
            my $page_ids = [map {$_->{page_id}} @$pages];

            for my $block_model (@{$self->$page_model->get_block_model_names()}) {
                my $blocks = [];
                my $j      = 0;
                do {
                    $blocks = $self->$block_model->get_all(
                        fields => [qw(page_id id is_deleted)],
                        filter => [page_id => 'IN' => $page_ids],
                        limit  => $chunk_size,
                        offset => $j * $chunk_size,
                    );

                    for my $block (@$blocks) {
                        my $is_deleted = delete $block->{is_deleted};
                        $block->{model}       = $block_model;
                        $block->{state}       = $is_deleted ? -1 : $pages_dict{$block->{page_id}}{state};
                        $block->{update_time} = $pages_dict{$block->{page_id}}{update_time};
                    }

                    $self->api_selfservice->logbroker(
                        data  => $blocks,
                        topic => $blocks_topic,
                    );

                    $j += 1;
                } while (@$blocks);
            }

            $i += 1;
        } while (@$pages);

        $self->kv_store->set($key, $current_time);
    }
}

sub get_moderation_update_time_key {
    my ($self, $model) = @_;
    return "${model}_logbroker_last_update_time";
}

TRUE;
