package Cron::Methods::Assessors;

use qbit;

use base qw(QBit::Cron::Methods);
use PiConstants qw($ASSESSOR_ROLE_ID);
use Utils::MonitoringUtils;
use Utils::Logger qw(INFO INFOF ERROR);

my $DELETE_AFTER_DAYS = 7;

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB');

sub model_path {'assessors'}

sub delete_old_assessor_pages : CRON('10 * * * *') : LOCK : FREQUENCY_LIMIT('1d') : TTL('2h') {
    my ($self) = @_;

    my $query = $self->partner_db->query->select(
        table  => $self->partner_db->all_pages,
        fields => ['model', 'page_id'],
        filter => [
            'AND' => [
                [
                    owner_id => 'IN' => $self->partner_db->query->select(
                        table  => $self->partner_db->user_role,
                        fields => ['user_id'],
                        filter => ['role_id' => '=' => \$ASSESSOR_ROLE_ID],
                    ),
                ],
                [create_date => '<' => \date_sub(curdate(), day => $DELETE_AFTER_DAYS, oformat => "db")],
                [is_deleted  => '=' => \0],
            ]
        ],
    );

    my %by_accessor;
    for my $row (@{$query->get_all}) {
        push @{$by_accessor{$row->{model}}}, $row->{page_id};
    }

    my %result;
    for my $accessor (sort keys %by_accessor) {
        $self->app->send_heartbeat();

        my $model = $self->app->$accessor;
        my $pages = $model->get_all(
            filter => [AND => [{multistate => 'not deleted'}, {page_id => $by_accessor{$accessor}}]],
            fields => ['id', 'page_id', 'multistate']
        );
        for my $page (@$pages) {
            try {
                my $multistate = $page->{multistate};
                if ($model->check_multistate_action($multistate, 'stop_testing')) {
                    $multistate = $model->do_action($page->{id}, 'stop_testing');
                }
                if ($model->check_multistate_action($multistate, 'stop')) {
                    $multistate = $model->do_action($page->{id}, 'stop');
                }
                $model->do_action($page->{id}, 'delete');
                $result{$accessor}++;
            }
            catch {
                my ($e) = @_;
                ERROR {
                    exception => $e,
                    message   => 'Cannot delete assessor page',
                    extra     => {
                        model => $accessor,
                        page  => $page,
                    }
                };
            };
        }
    }

    if (%result) {
        for my $accessor (sort keys %result) {
            Utils::MonitoringUtils::send_to_graphite(
                interval => 'one_day',
                path     => "assessor_delete_page.$accessor",
                value    => $result{$accessor},
            );
            INFOF("Delete %d pages in %s", $result{$accessor}, $accessor);
        }
    } else {
        INFO("No pages for delete");
    }

    return;
}

1;
