package Cron::Methods::Balance;

use qbit;

use base qw(QBit::Cron::Methods Cron::Methods);

use Utils::Logger qw(ERROR INFO);

use Exception::API::HTTP;

my $CONTRACT_STREAM           = '//logs/balance-contract/stream/5min';
my $CONTRACTSTREAM_OFFSET_KEY = 'contract_stream_last_offset';
my $CONTRACT_STREAM_LIMIT     = 100;

__PACKAGE__->model_accessors(
    api_yql                           => 'QBit::Application::Model::API::Yandex::YQL',
    api_balance                       => 'Application::Model::API::Yandex::Balance',
    context_on_site_campaign          => 'Application::Model::Product::AN::ContextOnSite::Campaign',
    contracts                         => 'Application::Model::Contracts',
    documents                         => 'Application::Model::Documents',
    indoor                            => 'Application::Model::Page::InDoor',
    internal_context_on_site_campaign => 'Application::Model::Product::InternalAN::InternalContextOnSite::Campaign',
    internal_search_on_site_campaign  => 'Application::Model::Product::InternalAN::InternalSearchOnSite::Campaign',
    kv_store                          => 'QBit::Application::Model::KvStore',
    outdoor                           => 'Application::Model::Page::OutDoor',
    search_on_site_campaign           => 'Application::Model::Product::AN::SearchOnSite::Campaign',
    user_features                     => 'Application::Model::Users::Features',
);

sub model_path {'balance'}

sub register_in_balance : CRON('* * * * *') : LOCK : STAGE('PREPROD') : STAGE('PRODUCTION') {
    my ($self) = @_;

    my @accessors = qw(
      context_on_site_campaign
      search_on_site_campaign

      internal_context_on_site_campaign
      internal_search_on_site_campaign

      indoor
      outdoor
      );

    foreach (@accessors) {
        try {
            $self->$_->cron_init_campaigns();
        }
        catch {
            ERROR shift;
        };
    }

    return TRUE;
}

sub update_contracts : CRON('*/5 * * * *') : LOCK : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;

    my $last_offset = $opts{'--offset'} // $self->kv_store->get($CONTRACTSTREAM_OFFSET_KEY) // 0;
    my $table_name_limit = $opts{'--table_name_limit'}
      // ($opts{'--without_table_name_limit'} ? '20' : date_sub(curdate(), day => 1, oformat => 'db_time_t'));
    my $limit = $opts{'--limit'} // $CONTRACT_STREAM_LIMIT;
    my $query = sprintf(
        q(
SELECT
    _offset as offset,
    CAST(Yson::SerializeJson(Object) AS String) as contract,
    TableName() AS table_name,
    TableRecordIndex() AS row_number
FROM
    RANGE(`%s`, '%s')
WHERE
    Yson::ConvertToString(Yson::Parse(Object).type) = 'PARTNERS'
    AND
    _offset > %d
ORDER BY
    offset
LIMIT %d
;
        ), $CONTRACT_STREAM, $table_name_limit, $last_offset, $limit
    );
    INFO $query;
    my $yql_operation_result;
    try {
        $yql_operation_result = $self->api_yql->yql_start_operation_and_get_result(
            clusters => $opts{'--cluster'} // $self->app->get_option('yt')->{replicas},
            start_params => {params => {content => $query,}},
            get_params   => {format => 'json',},
        );
    }
    catch Exception::API::HTTP with {
        my ($exception) = @_;

        if (   $exception->{'response'}->code() == 400
            && $exception->{'response'}->decoded_content() =~ /Operation\s+result\s+contains\s+no\s+data/)
        {
            INFO 'Got code 400:' . $exception->message();
        } else {
            throw $exception;
        }
    };

    return unless defined($yql_operation_result);

    foreach my $contract (@{from_jsonl($yql_operation_result)}) {
        INFO $contract;
        $contract->{raw_contract} = from_json delete $contract->{contract};
        $self->contracts->update($contract);
        $last_offset = $contract->{offset} if $last_offset < $contract->{offset};
    }

    $self->kv_store->set($CONTRACTSTREAM_OFFSET_KEY, $last_offset) unless $opts{'--offset'};
}

sub close_ua_contracts : CRON('*/5 * * * *') : LOCK : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;
    INFO "Start cron";

    my $user_ids_hash = $self->user_features->get_all(
        fields => [qw(user_id)],
        filter => {feature => 'ua_payment_banner'},
    );

    unless (@$user_ids_hash) {
        INFO "End cron. Not found users with feature";
        # Нечего обрабатывать
        return;
    }

    my @user_ids = map {$self->app->partner_db->quote($_->{user_id})} @$user_ids_hash;
    my $user_ids_str = join(",", @user_ids);
    my $query =
      "SELECT uid FROM `//home/forms/answers/forms_ext/production/13455809/data` WHERE uid IN ($user_ids_str)";

    INFO "Execute yql: $query";
    my $yql_operation_result;
    try {
        $yql_operation_result = $self->api_yql->yql_start_operation_and_get_result(
            clusters     => ['hahn'],
            start_params => {params => {content => $query,}},
            get_params => {format => 'json',},
        );
    }
    catch Exception::API::HTTP with {
        my ($exception) = @_;

        if (   $exception->{'response'}->code() == 400
            && $exception->{'response'}->decoded_content() =~ /Operation\s+result\s+contains\s+no\s+data/)
        {
            INFO "Got code 400:" . $exception->message();
        } else {
            throw $exception;
        }
    };

    unless (defined($yql_operation_result)) {
        INFO "End cron. Yql result is not defined";
        return;
    }

    my @uids = map {$_->{uid}} @{from_jsonl($yql_operation_result)};

    unless (@uids) {
        INFO "End cron. Yql result is empty";
        return;
    }
    INFO "Execute uids: " . join(",", @uids);

    my $users = $self->app->users->get_all(
        fields => [qw(id client_id)],
        filter => [id => 'IN' => \@uids],
    );

    my $error_uids;
    for my $user (@$users) {
        INFO "(user_id=$user->{id}). Start";
        my $contracts;
        my $is_error = FALSE;
        try {
            # Закрываем все живые договоры РСЯ
            $contracts = $self->documents->get_live_contracts(client_id => $user->{client_id});
        }
        catch {
            my ($exception) = @_;
            $error_uids->{$user->{id}} = TRUE;
            ERROR "(user_id=$user->{id}). Error get contracts. Message: " . $exception->message();
            $is_error = TRUE;
        };

        next if $is_error;

        INFO "(user_id=$user->{id}). Start close contracts: "
          . join(",", map {$_->{Contract}{contract2_id}} @$contracts);

        my $end_dt = curdate(oformat => 'iso8601');
        for my $contract (@$contracts) {
            INFO "(user_id=$user->{id}). Close contract_id=$contract->{Contract}{contract2_id}";
            try {
                $self->api_balance->close_contract(
                    operator_uid  => $user->{id},
                    contract_id   => $contract->{Contract}{contract2_id},
                    contract_type => $contract->{Contract}{type},
                    END_DT        => $end_dt,
                    memo          => "перевод к селлеру NEWSERVICE-2990",
                );
            }
            catch {
                my ($exception) = @_;
                $error_uids->{$user->{id}} = TRUE;
                # Возможно это test_mode
                # https://a.yandex-team.ru/arcadia/partner/perl/partner2/lib/Application/Model/AgreementChecker/_Agreement.pm?rev=r9647613#L144-148
                # Мы такой не умеем расторгать нужно отдать на ручной разбор

                ERROR
"(user_id=$user->{id}) (contract_id=$contract->{Contract}{contract2_id}). Error close contract. Message: "
                  . $exception->message();
            };
        }
        INFO "(user_id=$user->{id}). End close contracts";

        unless (defined $error_uids && exists $error_uids->{$user->{id}}) {
            # Удаляем фичу, чтобы больше баннер не показывался, тк все успешно обработали у него
            $self->user_features->delete($user->{id}, 'ua_payment_banner');
            INFO "(user_id=$user->{id}). End";
        }
    }

    # Проверка что не было ошибок
    if (defined($error_uids) && $error_uids) {
        throw Exception::BatchFail sprintf('Failed user ids: %s', join ', ', keys %{$error_uids});
    }
}

TRUE;
