package Cron::Methods::BusinessRules;

use qbit;

use base qw(QBit::Cron::Methods Cron::Methods);

use PiConstants qw( $BK_MONEY_KOEF );
use Utils::Logger qw( INFO  INFOF );
use Exception::API::YT;
use QBit::StringUtils qw( unset_utf);

__PACKAGE__->model_accessors(partner_db => 'Application::Model::PartnerDB',);

sub model_path {'business_rule_methods'}

sub export_busines_rules_dict : CRON('*/20 * * * *') : LOCK {
    my ($self) = @_;

    my $rules =
      $self->app->business_rules->get_all(fields => [qw(rule_id  caption  update_time)], order_by => [qw(rule_id)]);

    my @data =
      map {{id => 0 + $_->{rule_id}, caption => unset_utf($_->{caption}), update_time => $_->{update_time}}} @$rules;

    # yt create  replicated_table ...
    # yt create  table_replica    //home/partner/business_rules_dict \
    #   --attributes $(cat <<EOF | tr "\n" " " | sed 's| ||g'
    #   {
    #       dynamic=%true;
    #       schema=[
    #         { name=id;          type=uint64; sort_order=ascending };
    #         { name=caption;     type=string                       };
    #         { name=update_time; type=string                       }
    #       ]
    #   }
    #   EOF
    # ) --proxy=pythia

    while (@data) {
        my @batch = splice @data, 0, 1000;
        $self->app->api_yt->insert_rows(
            path                 => '//home/partner/business_rules_dict',
            data                 => \@batch,
            require_sync_replica => FALSE,
            params               => {
                ':timeout'  => 20,
                ':attempts' => 3,
                ':delay'    => 0,
            }
        );
    }
}

# Метод проверяет разницу во времени постановки на отправку и текущим времененем, если оно больше порога, ругается в почту
#
# > LAZY_LOAD=1 perl -I./lib -MCron -e'Cron->new->do'  business_rule_methods  check_yt_update
#
sub check_yt_update : CRON('*/5 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self) = @_;

    $self->check_model_items_update_time(
        accessor => 'business_blocks',
        fields   => ['rule_id', 'page_id', 'block_id'],
        filter   => {
            field    => 'page_id',
            not_null => TRUE,
        },
        graphite_metrics => 'BusinessRules.rules_outdated',
    );

    return TRUE;
}

# Метод осуществяет "асинхронную" отправку в БК - всех правил на которых стоит флаг "set_need_update"
# NOTE! Ошибки не блочат всю отправку, но их нужно мониторть на борде отправки!
#
# > LAZY_LOAD=1 perl -I./lib -MCron -e'Cron->new->do'  business_rule_methods  async_update_yt
#
sub async_update : CRON('*/5 * * * *') : LOCK : STAGE('TEST') : STAGE('PRODUCTION') {
    my ($self, %opts) = @_;

    ####### business_blocks
    my $found_rows = $self->get_model_items_to_update(
        accessor => 'business_blocks',
        fields   => [qw( rule_id  page_id  block_id  is_deleted )],
        filter   => {
            field      => 'page_id',
            not_null   => TRUE,
            only_ids   => $opts{'only_page_ids'},
            except_ids => $opts{'except_page_ids'},
        },
    );

    unless ($found_rows && @$found_rows) {
        INFO 'Nothing to update';
        return 1;
    }

    my $blocks   = {};
    my $rule_ids = {};
    foreach my $row (@$found_rows) {
        my $block_pub_id = $row->{block_pub_id} = join '-', @$row{qw( page_id  block_id )};
        my $rule_id = $row->{rule_id};

        my $block_data = $blocks->{$block_pub_id} //= {
            page_id  => $row->{page_id},
            block_id => $row->{block_id},
            rules    => {}
        };
        $block_data->{rules}->{$rule_id} = {is_deleted => $row->{is_deleted}};
        $rule_ids->{$rule_id} = 1;

    }

    ####### business_rules
    my $model = $self->app->business_rules;
    my $rules = $model->get_all(
        fields => [qw( rule_id  cpm  conditions  multistate  position )],
        filter => {'rule_id' => [sort keys %$rule_ids]},
    );

    my $rules_data = {
        map {
            $_->{rule_id} => {
                is_working => $model->check_multistate_flag($_->{multistate}, 'working'),
                position   => $_->{position},
                bk_data    => {
                    rule_id => 0 + $_->{rule_id},
                    context => $model->get_bk_expression($_->{conditions}),
                    min_cpm => $_->{cpm} /
                      1000    # NOTE! БК считает CPM за 1 показ, а не 1000 (#PI-12171)
                }
              }
          } @$rules
    };

    ####### prepare bk_data
    my @buffer_yt_to_update = ();
    my @buffer_yt_to_delete = ();
    my @buffer_pi_to_keep   = ();
    my @buffer_pi_to_delete = ();
    foreach my $block_pub_id (sort keys %$blocks) {
        my ($page_id, $block_id, $block_rules) = @{$blocks->{$block_pub_id}}{qw( page_id  block_id  rules )};

        my @block_rules_sorted =
          sort {$rules_data->{$a}->{position} <=> $rules_data->{$b}->{position}} keys %$block_rules;

        my $block_rules_bk_data = [];
        foreach my $rule_id (@block_rules_sorted) {
            my ($rule_is_working, $bk_data) = @{$rules_data->{$rule_id}}{qw( is_working  bk_data )};

            my $is_rule_block_deleted = $block_rules->{$rule_id}->{is_deleted};

            my $block_pk = {
                rule_id  => $rule_id,
                page_id  => $page_id,
                block_id => $block_id
            };

            if ($is_rule_block_deleted) {
                push @buffer_pi_to_delete, $block_pk;
            } else {
                push @buffer_pi_to_keep, $block_pk;
            }

            if ($rule_is_working && !$is_rule_block_deleted) {
                push @$block_rules_bk_data, $bk_data;
            }
        }

        if (@$block_rules_bk_data) {
            push @buffer_yt_to_update,
              {
                page_id  => 0 + $page_id,
                block_id => 0 + $block_id,
                data     => to_json(
                    {
                        rules    => $block_rules_bk_data,
                        page_imp => [
                            {
                                page_id => 0 + $page_id,
                                imp_id  => 0 + $block_id
                            }
                        ]
                    }
                )
              };
        } else {
            push @buffer_yt_to_delete,
              {
                page_id  => 0 + $page_id,
                block_id => 0 + $block_id
              };
        }

        # Delete
        if (@buffer_yt_to_delete >= 1_000) {
            $self->_delete_from_yt(\@buffer_yt_to_delete, \@buffer_pi_to_delete);

            @buffer_yt_to_delete = ();
            @buffer_pi_to_delete = ();
        }

        # Update
        if (@buffer_yt_to_update >= 1_000) {
            $self->_send_to_yt(\@buffer_yt_to_update, \@buffer_pi_to_keep);

            @buffer_yt_to_update = ();
            @buffer_pi_to_keep   = ();
        }
    }

    # Delete
    $self->_delete_from_yt(\@buffer_yt_to_delete, \@buffer_pi_to_delete)
      if @buffer_yt_to_delete || @buffer_pi_to_delete;

    # Update
    $self->_send_to_yt(\@buffer_yt_to_update, \@buffer_pi_to_keep)
      if @buffer_yt_to_update || @buffer_pi_to_keep;

    INFO('end');

    return TRUE;
}

sub _delete_from_yt {
    my ($self, $buffer_yt_to_delete, $buffer_pi_to_delete) = @_;

    # delete from YT table
    if (@$buffer_yt_to_delete) {

        INFOF 'start deleting %s blocks from YT', scalar(@$buffer_yt_to_delete);

        $self->app->api_yt->delete_rows(
            path                 => '//home/partner/business_rules',
            data                 => $buffer_yt_to_delete,
            require_sync_replica => FALSE,
            params               => {
                ':timeout'  => 20,
                ':attempts' => 3,
                ':delay'    => 0,
            }
        );
    }

    # delete from DB
    if (@$buffer_pi_to_delete) {
        INFOF 'start deleting %s blocks', scalar(@$buffer_pi_to_delete);
        $self->app->business_blocks->partner_db_table()->delete($buffer_pi_to_delete);
    }

    return 1;
}

sub _send_to_yt {
    my ($self, $buffer_yt_to_update, $buffer_pi_to_keep) = @_;

    INFO 'start mark start_update';
    foreach my $pk (@$buffer_pi_to_keep) {
        $self->app->business_blocks->do_action($pk, 'start_update');
    }

    ####### update YT table
    if (@$buffer_yt_to_update) {
        # insert to YT table
        INFOF 'start updating %s blocks to YT', scalar(@$buffer_yt_to_update);

        # yt create  replicated_table ...
        # yt create  table_replica    //home/partner/business_rules \
        #   --attributes $(cat <<EOF | tr "\n" " " | sed 's| ||g'
        #   {
        #       dynamic=%true;
        #       schema=[
        #         { name=page_id;     type=uint64; sort_order=ascending };
        #         { name=block_id;    type=uint64; sort_order=ascending  };
        #         { name=data;        type=string                       }
        #       ]
        #   }
        #   EOF
        # ) --proxy=pythia

        $self->app->api_yt->insert_rows(
            path                 => '//home/partner/business_rules',
            data                 => $buffer_yt_to_update,
            require_sync_replica => FALSE,
            params               => {
                ':timeout'  => 20,
                ':attempts' => 3,
                ':delay'    => 0,
            }
        );
    }

    # unset "updating"
    INFO 'start mark stop_update';
    foreach my $pk (@$buffer_pi_to_keep) {
        $self->app->business_blocks->maybe_do_action($pk, 'stop_update');
    }
}

TRUE;
