package Cron::Methods::CustomBkData;

use qbit;

use base qw(QBit::Cron::Methods);

use Utils::Logger qw(INFO ERROR WARN);
use Utils::MonitoringUtils qw(send_to_graphite);
use PiConstants qw($STAT_MONEY_SCALE $DAYS_BEFORE_UNSET_GODMODE);

my $UNSET_GODMODE_LIMIT = 10;

sub model_path {'custom_bk_data'}

__PACKAGE__->model_accessors(
    partner_db      => 'Application::Model::PartnerDB',
    product_manager => 'Application::Model::ProductManager',
    clickhouse_db   => 'Application::Model::ClickhouseDB',
    all_pages       => 'Application::Model::AllPages',
    api_yt          => 'QBit::Application::Model::API::Yandex::YT',
);

my %HASH_LOGINS = (
    'rambler-p' => 'rambler_p',
    'adinside'  => 'adinside',
);

my %HASH_DOMAINS = ('game.yandex.ru' => 'game_yandex_ru');

my $MONTH_FOR_MONEY = 6;

sub unset_godmode : CRON('1 * * * *') : LOCK : FREQUENCY_LIMIT('1d') : TTL('2h') {
    my ($self, %opts) = @_;

    my $skip = $self->_fetch_exception_unset_gm(%opts);

    my $models = $self->model_with_godmode(need_create_date => TRUE);

    my $limit = $self->app->kv_store->get('unset_godmode_limit');

    $limit = defined($limit) ? int($limit) : $UNSET_GODMODE_LIMIT;

    INFO(sprintf("unset_godmode_limit: %s", $limit));

    return if (defined($limit) && $limit < 1);

    $limit--;

    for my $accessor (@$models) {
        my $model       = $self->app->$accessor;
        my $block_table = $model->partner_db_table;

        my $campaign_model         = $model->get_campaign_model;
        my $campaign_id_field_name = $campaign_model->get_page_id_field_name;
        my $page_id_field_name     = $model->get_page_id_field_name;
        my $block_action_log       = $model->partner_db_action_log_table;
        my $log_elem_page_id_field = "elem_$page_id_field_name";
        my $block_action_log2      = $model->action_log_accessor;

        my $blocks = $self->_get_unset_godmod_blocks(
            block_action_log       => $block_action_log,
            block_table            => $block_table,
            campaign_table         => $campaign_model->partner_db_table,
            page_id_field_name     => $page_id_field_name,
            campaign_id_field_name => $campaign_id_field_name,
            campaign_model         => $campaign_model,
            accessor               => $accessor,
            log_elem_page_id_field => $log_elem_page_id_field,
            limit                  => $limit,
            prefix                 => $model->public_id_prefix,
        );

        my $money = $self->get_money_mol([map {$_->{public_id}} @$blocks]);
        my %pages;
        my ($count, $error) = (0, 0);

        INFO(sprintf("PREPARE_UNSET_GM: found = %s\t(%s)", scalar @$blocks, $accessor));
        for my $block (@$blocks) {
            if ($skip->{$block->{public_id}}) {
                INFO("SKIP_UNSET_GM $block->{public_id}");
                next;
            }
            next if $money->{$block->{public_id}};
            try {
                if (!$opts{'--dry_run'}) {
                    $block_table->edit({id => $block->{id}, $page_id_field_name => $block->{$page_id_field_name}},
                        {is_custom_bk_data => 0});

                    $block_action_log->add(
                        {
                            action                  => 'edit',
                            comment                 => 'force unset gm',
                            dt                      => curdate(oformat => 'db_time'),
                            elem_id                 => $block->{id},
                            $log_elem_page_id_field => $block->{$page_id_field_name},
                            old_multistate          => $block->{multistate},
                            new_multistate          => $block->{multistate},
                            opts                    => '{"is_custom_bk_data":0}',
                        }
                    );
                }
                INFO("MAKE_UNSET_GM $block->{public_id} bk_data: " . to_json($block->{bk_data}));
                $pages{$block->{$page_id_field_name}} = 1;
                $count++;
            }
            catch {
                my ($e) = @_;
                WARN("CANNOT_UNSET_GM $block->{public_id}: " . $e->message);
                $error++;
            };
        }
        INFO(sprintf("RESULT_UNSET_GM: proccessed=%s error=%s", $count, $error))
          if scalar @$blocks;

        $self->all_pages->mark_pages_for_async_update(page_ids => [sort {$a <=> $b} keys %pages])
          unless $opts{'--dry_run'};
    }
}

sub quantity_blocks : CRON('1 * * * *') : LOCK : FREQUENCY_LIMIT('1d') : TTL('2h') {
    my ($self, %opts) = @_;

    my $models = $self->model_with_godmode;
    my $prefix = 'gm_quantity_blocks';

    my @result;
    for my $accessor (sort @$models) {
        my $model = $self->app->$accessor;
        my $list  = $model->get_all(
            fields => ['public_id', 'page_id', 'id', 'multistate', 'is_protected', 'domain', 'login', 'is_deleted'],
            filter => {is_custom_bk_data => 1},
        );
        push @result, @$list;
    }

    my @filter;
    for my $row (@result) {
        push @filter, $row->{public_id};
    }
    my $money = $self->get_money_mol(\@filter);

    my %data;
    for my $row (@result) {
        my @type;
        if ($row->{is_protected}) {
            push @type, 'protected';
        }
        if (my $type = $HASH_DOMAINS{$row->{domain}}) {
            push @type, $type;
        }
        if (my $type = $HASH_LOGINS{$row->{login}}) {
            push @type, $type;
        }
        my $has_money = $money->{$row->{public_id}} ? 'with_money' : 'no_money';
        my $active    = $row->{is_deleted}          ? 'deleted'    : 'working';

        $data{all}{$has_money}{$active}++;
        for my $type (@type) {
            $data{$type}{$has_money}{$active}++;
        }
    }

    for my $type (keys %data) {
        for my $money (keys %{$data{$type}}) {
            for my $act (keys %{$data{$type}{$money}}) {
                my %params = (
                    interval => 'one_day',
                    path     => "$prefix.$type.$money.$act",
                    value    => $data{$type}{$money}{$act},
                    solomon  => {
                        owner  => $type,
                        money  => $money,
                        active => $act,
                        sensor => $prefix,
                    }
                );
                send_to_graphite(%params, log_only => $opts{'--log_only'});
            }
        }
    }

    return 1;
}

sub model_with_godmode {
    my ($self, %opts) = @_;

    my $block_models = $self->product_manager->get_block_model_names;

    my @model_with_godmode;
    for my $accessor (grep {!/^ssp/} @$block_models) {
        my $model_fields = $self->app->$accessor->get_model_fields;
        next if $opts{need_create_date} and !$model_fields->{create_date};
        push @model_with_godmode, $accessor if $model_fields->{is_custom_bk_data};
    }

    return \@model_with_godmode;
}

sub get_money_mol {
    my ($self, $blocks) = @_;

    my $to = curdate(oformat => 'db');
    my $from = date_sub(curdate(), month => $MONTH_FOR_MONEY, oformat => 'db');

    my %result;
    while (my @part = splice @$blocks, 0, 100) {
        my $data = $self->app->bk_statistics->get_statistics2(
            fields        => ['all_wo_nds'],
            period        => [$from, $to],
            stat_type     => 'main',
            entity_fields => ['complex_block_id'],
            levels        => [{filter => ["AND", [["complex_block_id", "IN", \@part]]]}]
        );
        for my $row (@{$data->{points}}) {
            $result{$row->{dimensions}{complex_block_id}} = $row->{measures}[0]{all_wo_nds};
        }
    }

    return \%result;
}

sub _fetch_exception_unset_gm {
    my ($self, %opts) = @_;

    return {} if $opts{'--exception_skip'};

    my $data = $self->api_yt->select_rows(
        host    => $self->get_option('yt')->{'meta_cluster'},
        yql     => 'public_id from [//home/partner/dict/exception_unset_gm]',
        path    => '//home/partner/dict/exception_unset_gm',
        headers => {'X-YT-Output-Format' => 'json'},
        params  => {
            ':timeout'  => 60,
            ':attempts' => 3,
            ':delay'    => 0,
        }
    );

    $data = $data ? from_jsonl($data) : [];
    my %result;
    for my $row (@$data) {
        $result{$row->{public_id}} = TRUE;
    }
    return \%result;
}

sub _get_unset_godmod_blocks {
    my ($self, %opts) = @_;

    my $date = date_sub(curdate(), day => $DAYS_BEFORE_UNSET_GODMODE, oformat => 'db_time');

    my $multistates_filter_protected =
      [qw(internal_mobile_app_rtb mobile_app_rtb video_an_site_fullscreen video_an_site_inpage video_an_site_instream)];

    my $data = $self->partner_db->query->select(
        fields => [qw(id create_date is_custom_bk_data bk_data multistate), $opts{page_id_field_name}],
        filter => [AND => [['create_date' => '<' => \$date], ['is_custom_bk_data' => '=' => \1],],],
        table => $opts{block_table},
      )->join(
        fields => [$opts{campaign_id_field_name}],
        filter => [
            multistate => 'NOT IN' => \$opts{campaign_model}->get_multistates_by_filter(
                in_array($opts{accessor}, $multistates_filter_protected) ? 'protected' : 'read_only or protected'
            )
        ],
        join_on => [$opts{campaign_id_field_name} => '=' => {$opts{page_id_field_name} => $opts{block_table}}],
        table   => $opts{campaign_table},
      )->left_join(
        alias   => 'unedited_blocks',
        fields  => ['action'],
        filter  => [id => '=' => \undef],
        join_on => [
            AND => [
                [elem_id                       => '=' => {id                            => $opts{block_table}}],
                [$opts{log_elem_page_id_field} => '=' => {$opts{campaign_id_field_name} => $opts{campaign_table}}],
            ]
        ],
        table => $self->partner_db->query->select(
            fields => ['id', 'action', 'dt', 'elem_id', $opts{log_elem_page_id_field}],
            table  => $opts{block_action_log},
            filter => [AND => [[action => '=' => \'edit'], [dt => '>' => \$date],],],
          )->limit(1),
      )->limit($opts{limit})->get_all();

    return [
        map {$_->{public_id} = sprintf("%s%s-%s", $opts{prefix}, $_->{$opts{campaign_id_field_name}}, $_->{id}); $_}
        grep {$_} @$data
    ];
}

1;
